// KreateCD - CD recording software for the K desktop environment
//
// 1999-2000 by Alexander Feigl <Alexander.Feigl@gmx.de>
//
// This file is subject to the terms and conditions of the GNU General      
// Public License.  See the file COPYING in the main directory of the       
// KreateCD distribution for more details.                                     

#include "IsoOptions.h"
#include "IsoImage.h"
#include "IsoOptions.moc"

#include <kapp.h>
#include <klocale.h>
#include <kconfig.h>

#include <qlayout.h>
#include <qpushbutton.h>
#include <qcheckbox.h>
#include <qradiobutton.h>
#include <qlabel.h>
#include <qbuttongroup.h>
#include <qlineedit.h>

#include <strings.h>
#include <stdio.h>

ISOOptions::ISOOptions( QWidget *parent, const char *name,
			ISOImage *track):QWidget(parent,name) {
  KConfig *config;
  QVBoxLayout *mainlayout;
  QButtonGroup *isoopt,*presetopt,*stringopt;
  QGridLayout *isogrid,*presetgrid,*stringgrid;

  if (track!=0) {
    iso=track;
    configMode=false;
  } else {
    iso=new ISOImage();
    configMode=true;
  }

  this->setCaption(i18n("KreateCD - ISO image options"));

  config=kapp->config();

  mainlayout=new QVBoxLayout(this);

  mainlayout->addStretch(1);

  // build string lines

  stringopt=new QButtonGroup(i18n("ISO strings"),this);
  stringopt->setAlignment(AlignLeft);
  stringopt->setMinimumSize(300,60);
  mainlayout->addWidget(stringopt);
  mainlayout->addStretch(1);

  stringgrid=new QGridLayout(stringopt,10,2,20,5);



  appIDLabel=new QLabel(i18n("Application ID"),stringopt);
  appIDLabel->setFixedSize(appIDLabel->sizeHint());
  appIDEdit=new QLineEdit(stringopt,0);
  appIDEdit->setMaxLength(128);
  appIDEdit->setMinimumSize(appIDEdit->sizeHint());
  stringgrid->addWidget(appIDLabel,1,0,AlignLeft);
  stringgrid->addWidget(appIDEdit,1,1);
  connect(appIDEdit,SIGNAL(textChanged(const QString &)),
          iso,SLOT(setApplicationID(const QString &)));

  sysIDLabel=new QLabel(i18n("System ID"),stringopt);
  sysIDLabel->setFixedSize(sysIDLabel->sizeHint());
  sysIDEdit=new QLineEdit(stringopt,0);
  sysIDEdit->setMaxLength(32);
  sysIDEdit->setMinimumSize(sysIDEdit->sizeHint());
  stringgrid->addWidget(sysIDLabel,3,0,AlignLeft);
  stringgrid->addWidget(sysIDEdit,3,1);
  connect(sysIDEdit,SIGNAL(textChanged(const QString &)),
          iso,SLOT(setSystemID(const QString &)));

  volIDLabel=new QLabel(i18n("Volume ID"),stringopt);
  volIDLabel->setFixedSize(volIDLabel->sizeHint());
  volIDEdit=new QLineEdit(stringopt,0);
  volIDEdit->setMaxLength(32);
  volIDEdit->setMinimumSize(volIDEdit->sizeHint());
  stringgrid->addWidget(volIDLabel,5,0,AlignLeft);
  stringgrid->addWidget(volIDEdit,5,1);
  connect(volIDEdit,SIGNAL(textChanged(const QString &)),
          iso,SLOT(setVolumeID(const QString &)));

  preIDLabel=new QLabel(i18n("Preparer"),stringopt);
  preIDLabel->setFixedSize(preIDLabel->sizeHint());
  preIDEdit=new QLineEdit(stringopt,0);
  preIDEdit->setMaxLength(128);
  preIDEdit->setMinimumSize(preIDEdit->sizeHint());
  stringgrid->addWidget(preIDLabel,7,0,AlignLeft);
  stringgrid->addWidget(preIDEdit,7,1);
  connect(preIDEdit,SIGNAL(textChanged(const QString &)),
          iso,SLOT(setPreparerID(const QString &)));

  pubIDLabel=new QLabel(i18n("Publisher"),stringopt);
  pubIDLabel->setFixedSize(pubIDLabel->sizeHint());
  pubIDEdit=new QLineEdit(stringopt,0);
  pubIDEdit->setMaxLength(128);
  pubIDEdit->setMinimumSize(pubIDEdit->sizeHint());
  stringgrid->addWidget(pubIDLabel,9,0,AlignLeft);
  stringgrid->addWidget(pubIDEdit,9,1);
  connect(pubIDEdit,SIGNAL(textChanged(const QString &)),
          iso,SLOT(setPublisherID(const QString &)));


  // build ISO options presets

  presetopt=new QButtonGroup(i18n("ISO option presets"),this);
  presetopt->setAlignment(AlignLeft);
  presetopt->setMinimumSize(300,60);
  mainlayout->addWidget(presetopt);
  mainlayout->addStretch(1);

  presetgrid=new QGridLayout(presetopt,5,5,20,0);

  unixButton=new QRadioButton(i18n("Unix RockRidge extensions"),presetopt);
  unixButton->setFixedSize(unixButton->sizeHint());
  presetgrid->addWidget(unixButton,1,1, AlignLeft);
  connect(unixButton,SIGNAL(clicked()),this,SLOT(doUnix()));

  unixWinButton=new QRadioButton(i18n("Unix RockRidge + Windows joliet"),
				 presetopt);
  unixWinButton->setFixedSize(unixWinButton->sizeHint());
  presetgrid->addWidget(unixWinButton,3,1, AlignLeft);
  connect(unixWinButton,SIGNAL(clicked()),this,SLOT(doUnixWin()));

  winButton=new QRadioButton(i18n("Windows joliet extensions"),presetopt);
  winButton->setFixedSize(winButton->sizeHint());
  presetgrid->addWidget(winButton,1,3, AlignLeft);
  connect(winButton,SIGNAL(clicked()),this,SLOT(doWin()));

  msDOSButton=new QRadioButton(i18n("MS-DOS standard"),presetopt);
  msDOSButton->setFixedSize(msDOSButton->sizeHint());
  presetgrid->addWidget(msDOSButton,3,3, AlignLeft);
  connect(msDOSButton,SIGNAL(clicked()),this,SLOT(doMSDos()));

  {
    int i;
    for (i=0;i<5;i+=2) {
      presetgrid->setColStretch(i,(i!=0)?4:1);
    }
    for (i=0;i<5;i+=2) {
      presetgrid->setRowStretch(i,1);
    }
  }

  // build ISO options checkboxes

  isoopt=new QButtonGroup(i18n("ISO options"),this);
  isoopt->setAlignment(AlignLeft);
  isoopt->setMinimumSize(300,60);
  mainlayout->addWidget(isoopt);


  isogrid=new QGridLayout(isoopt,11,5,20,0);

  tdotButton=new QCheckBox(i18n("Omit trailing dots"),isoopt);
  tdotButton->setFixedSize(tdotButton->sizeHint());
  isogrid->addWidget(tdotButton,1,1, AlignLeft);
  connect(tdotButton,SIGNAL(toggled(bool)),iso,SLOT(setOmitTrailingDot(bool)));

  noDeepButton=new QCheckBox(i18n("No deep directory relocation"),isoopt);
  noDeepButton->setFixedSize(noDeepButton->sizeHint());
  isogrid->addWidget(noDeepButton,3,1, AlignLeft);
  connect(noDeepButton,SIGNAL(toggled(bool)),iso,SLOT(setNoDeepRelocation(bool)));

  longNameButton=new QCheckBox(i18n("Long ISO-9660 names"),isoopt);
  longNameButton->setFixedSize(longNameButton->sizeHint());
  isogrid->addWidget(longNameButton,1,3, AlignLeft);
  connect(longNameButton,SIGNAL(toggled(bool)),iso,SLOT(setLongISONames(bool)));

  leadDotButton=new QCheckBox(i18n("Allow leading dots"),isoopt);
  leadDotButton->setFixedSize(leadDotButton->sizeHint());
  isogrid->addWidget(leadDotButton,3,3, AlignLeft);
  connect(leadDotButton,SIGNAL(toggled(bool)),iso,SLOT(setAllowLeadingDot(bool)));

  omitVerButton=new QCheckBox(i18n("Omit ISO-9660 versions"),isoopt);
  omitVerButton->setFixedSize(omitVerButton->sizeHint());
  isogrid->addWidget(omitVerButton,5,1, AlignLeft);
  connect(omitVerButton,SIGNAL(toggled(bool)),iso,SLOT(setOmitVersions(bool)));

  transTabButton=new QCheckBox(i18n("Create TRANS.TBL"),isoopt);
  transTabButton->setFixedSize(transTabButton->sizeHint());
  isogrid->addWidget(transTabButton,5,3, AlignLeft);
  connect(transTabButton,SIGNAL(toggled(bool)),iso,SLOT(setMakeTransTab(bool)));

  rockButton=new QCheckBox(i18n("Full RockRidge extensions"),isoopt);
  rockButton->setFixedSize(rockButton->sizeHint());
  isogrid->addWidget(rockButton,7,1, AlignLeft);
  connect(rockButton,SIGNAL(toggled(bool)),iso,SLOT(setWithRockRidge(bool)));

  anoRockButton=new QCheckBox(i18n("Anonymous RockRidge extensions"),isoopt);
  anoRockButton->setFixedSize(anoRockButton->sizeHint());
  isogrid->addWidget(anoRockButton,7,3, AlignLeft);
  connect(anoRockButton,SIGNAL(toggled(bool)),iso,SLOT(setWithAnonymousRockRidge(bool)));

  jolietButton=new QCheckBox(i18n("Joliet extensions"),isoopt);
  jolietButton->setFixedSize(jolietButton->sizeHint());
  isogrid->addWidget(jolietButton,9,1, AlignLeft);
  connect(jolietButton,SIGNAL(toggled(bool)),iso,SLOT(setWithJoliet(bool)));

  allButton=new QCheckBox(i18n("Include ALL files"),isoopt);
  allButton->setFixedSize(allButton->sizeHint());
  isogrid->addWidget(allButton,9,3, AlignLeft);
  connect(allButton,SIGNAL(toggled(bool)),iso,SLOT(setAllFiles(bool)));

  {
    int i;
    for (i=0;i<5;i+=2) {
      isogrid->setColStretch(i,(i!=0)?4:1);
    }
    for (i=0;i<11;i+=2) {
      isogrid->setRowStretch(i,1);
    }
  }


  // make the rest
 
  mainlayout->addSpacing(5);
  mainlayout->addStretch(1);

  isogrid->activate();
  presetgrid->activate();
  mainlayout->activate();

  updateWindow();
  connect(iso,SIGNAL(imageChanged()),this,SLOT(updateWindow()));
}


ISOOptions::~ISOOptions(void) {
  if (configMode==true) {
    delete iso;
  } 
}

void ISOOptions::updateWindow(void) {
  appIDEdit->setText(iso->getApplicationID());
  preIDEdit->setText(iso->getPreparerID());
  pubIDEdit->setText(iso->getPublisherID());
  volIDEdit->setText(iso->getVolumeID());
  sysIDEdit->setText(iso->getSystemID());

  switch (iso->getOptionsDefault()) {
  case ISOImage::Unix:
    unixButton->setChecked(true);
    break;
  case ISOImage::UnixWin:
    unixWinButton->setChecked(true);
    break;
  case ISOImage::Win:
    winButton->setChecked(true);
    break;
  case ISOImage::MSDos:
    msDOSButton->setChecked(true);
    break;
  default:
    unixButton->setChecked(false);
    unixWinButton->setChecked(false);
    winButton->setChecked(false);
    msDOSButton->setChecked(false);
    break;
  }

  tdotButton->setChecked(iso->getOmitTrailingDot());
  noDeepButton->setChecked(iso->getNoDeepRelocation());
  longNameButton->setChecked(iso->getLongISONames());
  leadDotButton->setChecked(iso->getAllowLeadingDot());
  omitVerButton->setChecked(iso->getOmitVersions());
  rockButton->setChecked(iso->getWithRockRidge());
  anoRockButton->setChecked(iso->getWithAnonymousRockRidge());
  transTabButton->setChecked(iso->getMakeTransTab());
  jolietButton->setChecked(iso->getWithJoliet());
  allButton->setChecked(iso->getAllFiles());
}

void ISOOptions::doUnix(void) {
  iso->setOptionsDefault(ISOImage::Unix);
}

void ISOOptions::doUnixWin(void) {
  iso->setOptionsDefault(ISOImage::UnixWin);
}

void ISOOptions::doWin(void) {
  iso->setOptionsDefault(ISOImage::Win);
}

void ISOOptions::doMSDos(void) {
  iso->setOptionsDefault(ISOImage::MSDos);
}

void ISOOptions::applySettings(void) {
  KConfig *config;
  config=kapp->config();
  config->setGroup("ISOOptions");
  iso->saveConfig(config);
}






