// KreateCD - CD recording software for the K desktop environment
//
// 1999-2000 by Alexander Feigl <Alexander.Feigl@gmx.de>
//
// This file is subject to the terms and conditions of the GNU General
// Public License.  See the file COPYING in the main directory of the
// KreateCD distribution for more details.

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "ProcessInterface.h"
#include "Fork.h"

#include <kprocess.h>

#include <qtimer.h>

#include <stdio.h>
#include <string.h>

#include "ProcessInterface.moc"

ProcessInterface::~ProcessInterface(void) {
  isKilled=1;
  emit processDeletion(this);
  if (process!=0) closeProcess();
}

ProcessInterface::ProcessInterface(void) {
  forkProcess=false;
  forkAudio=false;
  blockTimer=false;
  process=0;
  outPending=false;
  errPending=false;
  outTimed=0;
  errTimed=0;
  lineTimeout=8;
  outputTimer=0;
  isKilled=0;
}

ProcessInterface::ProcessInterface(int (*fx)(int argc,char **argv)) {
  forkProcess=true;
  forkAudio=false;
  blockTimer=false;
  process=0;
  forkFunc=fx;
  outPending=false;
  errPending=false;
  outTimed=0;
  errTimed=0;
  lineTimeout=5;
  outputTimer=0;
  isKilled=0;
}

ProcessInterface::ProcessInterface(int (*fx)(ProcessInterface *afi,
					     const char *fn),
				   const char *fnx) {
  forkProcess=true;
  forkAudio=true;
  blockTimer=false;
  process=0;
  forkFuncAudio=fx;
  audioForkArg=fnx;
  outPending=false;
  errPending=false;
  outTimed=0;
  errTimed=0;
  lineTimeout=5;
  outputTimer=0;
  isKilled=0;
}

void ProcessInterface::setForkArgument(const char *arg) {
  audioForkArg=arg;
}


void ProcessInterface::prepareProcess() {
  if (!forkProcess) {
    process=new KProcess();
  } else {
    if (!forkAudio) {
      process=new Fork(forkFunc);
    } else {
      process=new Fork(forkFuncAudio,this,audioForkArg);
    }
  }
  stdOutBuffer[0]='\0';
  stdErrBuffer[0]='\0';
  isKilled=0;

  connect(process,SIGNAL(receivedStderr(KProcess *,char *,int)),
          this,SLOT(processStderr(KProcess *,char *,int)));
  connect(process,SIGNAL(receivedStdout(KProcess *,char *,int)),
          this,SLOT(processStdout(KProcess *,char *,int)));
  connect(process,SIGNAL(processExited(KProcess *)),
          this,SLOT(processLeave(KProcess *)));

  outputTimer=new QTimer();
  outputTimer->start(100,false);
  connect(outputTimer,SIGNAL(timeout()),this,SLOT(triggerTimer()));

}

int ProcessInterface::startProcess(void) {
  bool couldStart;

  couldStart=process->start(KProcess::NotifyOnExit,KProcess::All);

  if (!couldStart) return(-1);
  return(0);
}

void ProcessInterface::closeProcess(void) {
  if (process!=0) {
    if (!forkProcess) {
      delete (process);
    } else {
      delete (((Fork *)process));
    }
  }
  if (outputTimer!=0) delete(outputTimer);
  process=0;
  outputTimer=0;
}


void ProcessInterface::cancelPressed(void) {
  processCancel();
}

void ProcessInterface::processLeave(KProcess *) {
  int rv2,rv;
  rv2=processExited();
  rv=processCleanup(isKilled?-1:rv2);
  emit processTerminated(this,rv);
}

void ProcessInterface::processStdout(KProcess *,char *buffer,int bufflen) {
  int i;
  char *buf,*buf2;

  i=bufflen;
  buf=buffer;
  buf2=stdOutBuffer;
  while (*buf2!=0) ++buf2;
  while (i>0) {
    if ( (*buf=='\n') || (*buf=='\r') ) {
      *buf2=0;
      buf2=stdOutBuffer;
      blockTimer=true;
      if (!processStdoutLine(stdOutBuffer)) {
        process->kill(SIGINT);
        return;
      }
      stdOutBuffer[0]=0;
      outPending=false;
      outTimed=0;
      blockTimer=false;
    } else {
      *(buf2++)=*buf;
      *buf2=0;
      outPending=true;
    }
    ++buf;
    --i;
  }
}

void ProcessInterface::processStderr(KProcess *,char *buffer,int bufflen) {
  int i;
  char *buf,*buf2;

  i=bufflen;
  buf=buffer;
  buf2=stdErrBuffer;
  while (*buf2!=0) ++buf2;
  while (i>0) {
    if ( (*buf=='\n') || (*buf=='\r') ) {
      *buf2=0;
      buf2=stdErrBuffer;
      blockTimer=true;
      if (!processStderrLine(stdErrBuffer)) {
        process->kill(SIGINT);
        return;
      }
      stdErrBuffer[0]=0;
      errPending=false;
      errTimed=0;
      blockTimer=false;
    } else {
      *(buf2++)=*buf;
      *buf2=0;
      errPending=true;
    }
    ++buf;
    --i;
  }
}

bool ProcessInterface::processStdoutLine(char *) {
  return(true);
}

bool ProcessInterface::processStderrLine(char *) {
  return(true);
}

int  ProcessInterface::processExited(void) {
  return(1);
}

ProcessInterface &ProcessInterface::operator<<(const char *arg) {
  *process<<arg;
  return(*this);
}

void ProcessInterface::writeStdin(char *buffer, int len) {
  process->writeStdin(buffer,len);
}


void ProcessInterface::sendSignal(int signal) {
  process->kill(signal);
}

void ProcessInterface::triggerTimer(void) {
  if ( (!blockTimer) && (outPending==true) ) {
    outTimed++; 
    if (outTimed>=lineTimeout) {
      blockTimer=true;
      if (!processStdoutLine(stdOutBuffer)) {
	process->kill(SIGINT);
	return;
      }
      stdOutBuffer[0]=0;
      outPending=false;
      outTimed=0;
      blockTimer=false;
    }
  }

  if ( (!blockTimer) && (errPending==true) ) {
    errTimed++;
    if (errTimed>=lineTimeout) {
      blockTimer=true;
      if (!processStderrLine(stdErrBuffer)) {
	process->kill(SIGINT);
	return;
      }
      stdErrBuffer[0]=0;
      errPending=false;
      errTimed=0;
      blockTimer=false;
    }
  }
}

void ProcessInterface::setLineTimeout(int timeout) {
  lineTimeout=timeout;
}

void ProcessInterface::reportAction(QString action) {
  emit processAction(action);
}

void ProcessInterface::reportProgress(long int val,long int max) {
  emit processProgress(val,max);
}

void ProcessInterface::reportSecondProgress(long int total,long int offset) {
  emit processSecondProgress(total,offset);
}

void ProcessInterface::printStatusLine(QString line) {
  emit processStatus(line);
}

void ProcessInterface::processCancel(void) {
  if (process!=0) {
    isKilled=1;
    process->kill(SIGKILL);
  }
}

void ProcessInterface::waitExit(void) {
  int status;
  if (process!=0) waitpid(process->pid(),&status,0);
}

int ProcessInterface::processCleanup(int rv) {
  closeProcess();
  return(rv);
}

void ProcessInterface::reportBuffer(int percent) {
  emit processBuffer(percent);
}
