# Copyright (C) 2009-2010 LottaNZB Development Team
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 3.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.

import gtk

from threading import Lock
from kiwi.ui.delegates import GladeDelegate

from lottanzb.plugins import PluginBase, PluginConfig, PluginPrefsTabBase
from lottanzb.util import gproperty, _

try:
    import appindicator
except ImportError:
    appindicator = None

class Config(PluginConfig):
    """The plug-in is enabled by default."""
    
    enabled = gproperty(type=bool, default=True)


class Plugin(PluginBase):
    title = _("Panel menu")
    description = _("Control LottaNZB using a menu in the panel.")
    author = _("LottaNZB Development Team")
    
    # Contains a `MenuContainer` object that represents the actual icon in
    # the panel. On Ubuntu 10.04 or later, this comes in the form of an
    # application indicator. On other systems, a simple `gtk.StatusIcon` is
    # used instead.
    menu_container = None
    
    # The `GladeDelegate` object holding the `gtk.Menu` as a property `menu`.
    menu_delegate = None
    
    # The `gui.main.Window` object.
    main_window = None
    
    # Simple property that indicates whether the main window is minimized
    # to the task list.
    # It's the cached value from the previous `window-state-event` emitted by
    # the main window as it cannot be accessed directly.
    main_window_minimized = False
    
    # Ensure that the `toggle_main_window` is not called recursively,
    # potentially leading to an infinite loop.
    _main_window_toggle_lock = Lock()
    
    def refresh(self):
        """
        Make sure that the application is not entirely shut down if the main
        window is closed while the plug-in is enabled.
        """
        
        if self.main_window is not None:
            self.main_window.quit_on_close = not self.enabled
    
    def load(self):
        """
        Create the `menu_container` if necessary and make sure that it's
        visible.
        """
        
        if self.menu_container is None:
            if self.main_window is not None:
                # Choose the right `MenuContainer`.
                menu_container_classes = (IndicatorMenuContainer,
                    StatusIconMenuContainer)
                
                for menu_container_class in menu_container_classes:
                    try:
                        self.menu_container = menu_container_class("lottanzb",
                            self.menu_delegate.menu, self.main_window)
                    except:
                        pass
                    else:
                        break
                
                # Left-clicking on the `gtk.StatusIcon` should cause the main
                # window to be toggled. For the new-fasioned application
                # indicators, it will just open the associated menu.
                # TODO: This unelegant code breaks the otherwise okay-ish
                # abstraction of `MenuContainer`, but it works.
                if isinstance(self.menu_container, StatusIconMenuContainer):
                    self.menu_container._icon.connect("activate",
                        self.toggle_main_window)
        else:
            self.menu_container.set_visible(True)
        
        PluginBase.load(self)
    
    def unload(self):
        """Hide the `menu_container`."""
        
        if self.menu_container:
            self.menu_container.set_visible(False)
        
        PluginBase.unload(self)
    
    def toggle_main_window(self, *args):
        """
        Unminimize the window if previously minimized, hide the window if
        previously shown and vice versa.
        
        Also update the `Show main menu` checkbox in the panel menu, if
        necessary.
        """
        
        if self._main_window_toggle_lock.acquire(False):
            try:
                if self.main_window_minimized:
                    self.main_window.toplevel.deiconify()
                elif self.main_window_visible:
                    self.main_window.hide()
                else:
                    self.main_window.show()
                
                self.update_open_checkbox()
            finally:
                self._main_window_toggle_lock.release()
                
        return True
    
    def on_main_window_ready(self, main_window):
        """Register the 'Close' file menu entry as well as the panel menu."""
        
        self.main_window = main_window
        
        close_menu_item = gtk.ImageMenuItem(gtk.STOCK_CLOSE)
        close_menu_item.connect("activate", self.toggle_main_window)
        
        self.insert_menu_item(main_window.file_menu, close_menu_item, -2)
        
        # When clicking the close button of the window, only hide the window
        # instead of destroying it.
        self.connect_when_enabled(
            self.main_window.toplevel,
            "delete-event",
            self.toggle_main_window
        )
        
        # When minimizing the main window, deactivate the corresponding checkbox
        # in the panel menu.
        self.connect_when_enabled(
            self.main_window.toplevel,
            "window-state-event",
            self.on_main_window_state_event
        )
        
        self.menu_delegate = MenuDelegate()
        
        # `update_open_checkbox` needs to be called before the signal handler
        # is installed. Otherwise, the main window will be hidden automatically
        # on startup.
        self.update_open_checkbox()
        self.update_pause_checkbox()
        
        self.menu_delegate.open.connect("toggled", self.toggle_main_window)
        self.menu_delegate.add.connect("activate", self.on_add_activated)
        self.menu_delegate.pause.connect("toggled", self.on_pause_toggled)
        self.menu_delegate.quit.connect("activate", self.on_quit_activated)
        self.menu_delegate.show_preferences.connect("activate",
            self.on_show_preferences_activated)
        
        self.main_window.pause.connect("activate", self.update_pause_checkbox)
        
        if self.enabled:
            self.load()
    
    def on_main_window_state_event(self, window, event):
        """
        If the main window has been minimized make sure that the
        "Show main window" checkbox in the panel is deactivated.
        
        Otherwise, it needs to be activated.
        """
        
        state = gtk.gdk.WINDOW_STATE_ICONIFIED
        
        if event.changed_mask & state:
            self.main_window_minimized = bool(event.new_window_state & state)
            
            # Make sure that `toggle_main_window` isn't executed because of the
            # call to `update_open_checkbox`.
            if self._main_window_toggle_lock.acquire(False):
                try:
                    self.update_open_checkbox()
                finally:
                    self._main_window_toggle_lock.release()
        
        return True
    
    def on_add_activated(self, *args):
        """
        Show the 'Add file' dialog.
        
        Wrapper method that ensures that no unwanted arguments are passed to the
        corresponding `gui.main.Window` method.
        """
        
        self.main_window.show_add_dialog()
    
    def on_pause_toggled(self, widget):
        """Pause or resume downloads."""
        
        self.main_window.pause.set_active(widget.get_active())
    
    def on_quit_activated(self, *args):
        """
        Quit the application.
        
        Wrapper method that ensures that no unwanted arguments are passed to the
        corresponding `core.App` method.
        """
        
        self.app.quit()
    
    def on_show_preferences_activated(self, *args):
        """
        Show the preferences window, but also jump to the "Plug-ins" tab and
        select the "Panel menu" item.
        
        This behaviour is meant to make it easy for a user to hide the panel
        item again and is recommended by
        https://wiki.ubuntu.com/CustomStatusMenuDesignGuidelines
        """
        
        if self.main_window is not None:
            self.main_window.show_preferences_window()
            
            prefs_window = self.main_window.prefs_window
            prefs_window.set_current_tab(prefs_window.plugins_tab)
            prefs_window.plugins_tab.plugin_list.select(self)
    
    def update_open_checkbox(self, *args):
        """
        If the main window is visibility of the main window is changed by the
        user by closing or minimizing the window, this method changes the
        'Show main window' checkbox in the panel menu accordingly.
        """
        
        enabled = self.main_window_visible and not self.main_window_minimized
        
        self.menu_delegate.open.set_active(enabled)
    
    def update_pause_checkbox(self, *args):
        """
        If the downloads are paused from the main window, also update the
        checkbox in the panel menu.
        """
        
        paused = self.main_window.pause.get_active()
        
        self.menu_delegate.pause.set_active(paused)
    
    @property
    def main_window_visible(self):
        """
        Simple shortcut property that returns the visibility of the main
        window.
        """
        
        return self.main_window.toplevel.get_property("visible")


class MenuDelegate(GladeDelegate):
    """
    This UI file contains a nearly empty gtk.Window, which isn't actually
    used, but necessary because Kiwi doesn't allow delegates with a gtk.Menu
    root object.
    """
    
    gladefile = "plugin_panel_menu"


class MenuContainer(object):
    def __init__(self, application, menu, main_window):
        self.application = application
        self.menu = menu
        self.main_window = main_window
    
    def set_visible(self, visible):
        raise NotImplementedError


class IndicatorMenuContainer(MenuContainer):
    def __init__(self, application, menu, main_window):
        MenuContainer.__init__(self, application, menu, main_window)
        
        self._indicator = appindicator.Indicator(self.application,
            self.application, appindicator.CATEGORY_APPLICATION_STATUS)
        
        self._indicator.set_status(appindicator.STATUS_ACTIVE)
        self._indicator.set_menu(self.menu)
    
    def set_visible(self, visible):
        if visible:
            status = appindicator.STATUS_ACTIVE
        else:
            status = appindicator.STATUS_PASSIVE
        
        self._indicator.set_status(status)


class StatusIconMenuContainer(MenuContainer):
    def __init__(self, application, menu, main_window):
        MenuContainer.__init__(self, application, menu, main_window)
        
        self._icon = None
        self._do_with_rgb_colormap(self._create_icon)
        
    def set_visible(self, visible):
        def show_icon():
            self._icon.set_visible(visible)
        
        self._do_with_rgb_colormap(show_icon)
    
    def _do_with_rgb_colormap(self, function):
        """
        Workaround for bug http://bugzilla.gnome.org/show_bug.cgi?id=501842
        
        This method ensures that the default RGB color map is active while the
        passed function is being executed and returns to the previous color
        map afterwards.
        """
        
        # We cannot do it without the gtk.Screen provided by the main window.
        screen = self.main_window.toplevel.get_screen()
        current_colormap = screen.get_default_colormap()
        
        try:
            assert current_colormap == screen.get_rgba_colormap()
        except (AttributeError, AssertionError):
            rgba_support = False
        else:
            screen.set_default_colormap(screen.get_rgb_colormap())
            rgba_support = True
        
        function()
        
        # Restore the RGBA color map if necessary.
        if rgba_support:
            screen.set_default_colormap(current_colormap)
    
    def _show_menu(self, icon, button, activate_time):
        """Display the status icon menu."""
        
        position = gtk.status_icon_position_menu
        
        self.menu.popup(None, None, position, button, \
            activate_time, icon)
    
    def _create_icon(self):
        """
        Helper method that creates the `gtk.StatusIcon` and
        registers all event handlers.
        """
        
        self._icon = gtk.StatusIcon()
        self._icon.set_from_icon_name(self.application)
        self._icon.connect("popup-menu", self._show_menu)
