""" Manages all dynamic importing. """


# Standard library imports.
import sys

# Enthought library imports.
from enthought.pyface.message_dialog import error
from enthought.traits.api import Event, HasTraits


class ImportManager(HasTraits):
    """ Manages all dynamic importing.

    This class is intended to make debugging easier than it would be if
    imports happened from arbitrary places.
    """
    # It's just a guess, but I think using an import manager to do all imports
    # will make debugging easier (as opposed to just letting imports happen from
    # all over the place).


    #### 'ImportManager' interface ############################################

    # Fired when a symbol is imported.
    symbol_imported = Event

    ###########################################################################
    # 'ImportManager' interface.
    ###########################################################################

    def import_symbol(self, symbol_path):
        """ Imports the symbol defined by *symbol_path*.

        Parameters
        ----------
        symbol_path : a string in the form 'foo.bar.baz'
            The module path to a symbol to import

        The *symbol_path* value is turned into an import statement
        ``'from foo.bar import baz'``
        (i.e., the last component of the name is the symbol name, the rest
        is the module path to load it from).
        """

        components = symbol_path.split('.')

        module_name = '.'.join(components[:-1])
        symbol_name = components[-1]

        module = __import__(module_name, globals(), locals(), [symbol_name])
    
        symbol = getattr(module, symbol_name)

        # Event notification.
        self.symbol_imported = symbol

        return symbol

    def get_class(self, class_path):
        """ Returns the class defined by *class_path*.

        Returns **None** if the class has not yet been loaded.

        """

        # Only check if the class name has at least a partial hierarchy.
        #
        # fixme: Comment should say why!
        if '.' in class_path:
            components = class_path.split('.')

            module_name = '.'.join(components[:-1])
            class_name  = components[-1]

            # The class is loaded if its module has been imported and the class
            # is defined in the module dictionary.
            module = sys.modules.get(module_name, None)
            if module is not None and hasattr(module, class_name):
                klass = getattr(module, class_name)

            else:
                klass = None

        else:
            klass = None

        return klass

#### EOF ######################################################################
