""" Preference helper classes. """


# Major package imports.
import wx


class PreferenceHelper:
    """ Abstract base class for all preference helpers. """

    ###########################################################################
    # 'PreferenceHelper' class interface.
    ###########################################################################

    def handles(klass, value):
        """ Returns True if the value is handled by this helper. """

        raise NotImplementedError

    handles = classmethod(handles)
    
    def get_preference_value(klass, value):
        """ Returns the value to store in the preferences. """

        raise NotImplementedError

    get_preference_value = classmethod(get_preference_value)

    ###########################################################################
    # 'PreferenceHelper' interface.
    ###########################################################################

    def get_editor_value(self, value):
        """ Returns the value used by a preference editor. """

        raise NotImplementedError
    
    
class FontPreferenceHelper(PreferenceHelper):
    """ A helper for wx font preferences.

    This helper is required because wx fonts cannot be unpickled before the
    wx event loop has been started!

    """

    # The class that this helper 
    def __init__(self, font):
        """ Creates a new helper for a wx font. """

        self.point_size = font.GetPointSize()
        self.family     = font.GetFamily()
        self.style      = font.GetStyle() 
        self.weight     = font.GetWeight()
        self.underline  = font.GetUnderlined()
        self.face_name  = font.GetFaceName()

        return

    ###########################################################################
    # 'PreferenceHelper' class interface.
    ###########################################################################

    def handles(klass, value):
        """ Returns True if the value is handled by this helper. """

        return isinstance(value, wx.Font)

    handles = classmethod(handles)
    
    def get_preference_value(klass, value):
        """ Returns the value to store in the preferences. """

        return klass(value)

    get_preference_value = classmethod(get_preference_value)
    
    ###########################################################################
    # 'PreferenceHelper' interface.
    ###########################################################################

    def get_editor_value(self, value):
        """ Returns the value used by a preference editor. """

        font = wx.Font(
            self.point_size,
            self.family,
            self.style,
            self.weight,
            self.underline,
            self.face_name
        )

        return font

#### EOF ######################################################################
