/*
 * <kupdate.c>
 *
 * Handle interaction with kupdate(d) kernel daemon.
 *
 * Copyright (C) 2000 Daniel Kobras
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * $Id: kupdate.c,v 1.11 2002/01/22 19:34:11 nold Exp $
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "kupdate.h"
#include "bug.h"
#include "util.h"
#include "noflushd.h"

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <dirent.h>
#include <string.h>
#include <ctype.h>
#include <signal.h>
#include <errno.h>
#include <sys/param.h>		/* for HZ. (Yeah, I know...) */
#include <sys/types.h>

/* libc6 exports a bdflush interface, for prior versions we have to
 * wrap the syscall on our own.
 */

#ifdef HAVE_SYS_KDAEMON_H
#include <sys/kdaemon.h>	/* for bdflush() */
#else
#include <syscall.h>
#ifndef SYS_bdflush
#error No bdflush syscall present. You lose.
#endif
static _syscall2(int, bdflush, int, __func, long int, __data)
#endif	


/*
 * Here's a simple yet effective fallback in case everything goes haywire
 * and kflushd is left messed up. In fact, this is bdflush's/update's
 * fallback behaviour as well.
 */

static void last_resort(long interval)
{
	while (1==1) {
		sync();
		sleep(interval);
	}
}

	
/*
 * Ugly fumbling through /proc to determine the PID of the kupdate kernel
 * thread. My, do I hate writing parsers! I wonder how you do it without
 * /proc or /proc mounted on a different node. Humm... [peeking through
 * pidof source] looks like you're simply screwed. Weird. But then thank
 * god the days of messing with /dev/kmem have gone.
 */

static void get_pid(kupdate_t k)
{
	DIR	*proc, *pid_dir;
	FILE	*f=NULL;
	char	*line, *s_pid, *name;
	char	namebuf[267] = "/proc/";	/* d_name[] length of 256 bytes
						   is hardcoded in <direntry.h>.
						   We need to store at max 
						   /proc/<d_name[]>/stat. */
	int	pid, kpid=-1;
	struct dirent	*procent;

	proc = opendir("/proc");
	if (!proc) {
		perror("Unable to open /proc");
		exit(1);
	}

	/* More space efficient than using scandir()... */ 
	while ((procent = readdir(proc))) {
		if (!isdigit((int)(procent->d_name[0])))
			continue;
		strcpy(namebuf+6, procent->d_name);
		if (!(pid_dir = opendir(namebuf)))
			continue;
		closedir(pid_dir);
		strcat(namebuf, "/stat");
		if (f)
			fclose(f);
		f = fopen(namebuf, "r");
		line=get_line(f);
		if (!line)
			continue;
		name=_get_entry(line, 1);
		if (!name) {
			release_line(line);
			continue;
		}
		/* An ordinary user might have started a process named kupdate
		 * but it will never get a lower pid than the kupdate kernel
		 * thread. Uh, versions later than 2.4.2 seem to call it
		 * kupdated. Alan's valentine? Lovely...
		 */
		if (!strcmp(name, "(kupdate)") || !strcmp(name, "(kupdated)")) {
			s_pid=_get_entry(line, 0);
			if (!s_pid)
				BUG("No pid info on kupdate");
			pid=atoi(s_pid);
			if((pid < kpid) || (kpid == -1)) {
				kpid=pid;
				if (k->stat)
					fclose(k->stat);
				k->stat=f;
				f=NULL;
			}
			DEBUG("Detected process %s at pid %d", name, pid);
		}
		release_line(line);
		
	}
	closedir(proc);
	if (kpid == -1) {
		ERR("Could not determine PID of kupdate");
		exit(1);
	}
	k->pid=(pid_t) kpid;
}

/* Check if kupdate is really stopped. */
static int check_stop(kupdate_t k)
{
	char *line;
	char *state;
	int ret;
	
	fflush(k->stat);
	rewind(k->stat);
	line=get_line(k->stat);
	if (!line)
		BUG("Could not read kupdate stat");
	
	state=_get_entry(line, 2);
	if (!state)
		BUG("Could not read kupdate state");
	
	ret=(state[0]=='T');
	
	release_line(line);
	
	return ret;
}

/*
 * get/set_interval are simple helper functions to wrap [gs]etting the
 * basic flush interval of kupdate.
 */

static long get_interval(void)
{
	long interval;
	int ret;

	ret = bdflush(2 + (1 << 3), (long) (&interval));
	if (ret || interval <= 0) {
		INFO("Bogus kupdate interval %ld, setting to %d",
		       interval, 5*HZ);
		interval = 5*HZ;
	}
	return interval;
}

static int set_interval(long interval)
{
	int ret;

	ret = bdflush(2 + (1 << 3) + 1, interval);
	if (ret)
		ERR("Could not set bdflush parameters");
	return ret;
}

/* 
 * Set up kupdate struct. Determines pid of kupdate kernel daemon.
 */

kupdate_t kupdate_init(void)
{
	kupdate_t k;

	k=malloc(sizeof(struct kupdate_s));
	if (!k)
		return NULL;

	k->stat=NULL;
	get_pid(k);
	k->interval=get_interval();
	k->stopped=k->interval ? 0 : 1;

	return k;
}

long kupdate_get_interval(kupdate_t k)
{
	return (k->interval+HZ-1)/HZ;
}

/*
 * Handle background flushing of dirty buffers to disk via starting or 
 * stopping the kupdate kernel thread. kupdate is either (re)started or
 * stopped, depending on current setting in struct *k.
 * There's one ugly thing: kupdate expects its params scaled by HZ. Which
 * is quite funny as HZ is a kernel thingie and nowhere exported to userspace
 * (and rightfully so). 
 */

void __kupdate_stop(kupdate_t k)
{
	long left;

	if (!k)
		return;

	k->interval = get_interval();
	set_interval(0);
	kill(k->pid, SIGSTOP);	/* The signal itself is ignored, but it will
				 * cause kupdate to wakeup now and detect 
				 * interval=0. */

	/* Make sure kupdate is really asleep. We poll 5 times per second and
	 * wait at most 10 kupdate wakeup cycles. (During heavy system load,
	 * kupdate wakeup might be delayed.) */
	left = (10*k->interval*5+HZ-1)/HZ;
	while (!check_stop(k) && left > 0) {
		usleep(200000);
		left--;
	}

	if (!left) {
		ERR("Could not stop kupdate. Expect lousy spindown times.");
		return;
	}
		
	k->stopped=1;
	DEBUG("kupdate stopped");

}
	
void __kupdate_start(kupdate_t k)
{
	if (!k)
		return;

	set_interval(k->interval);
	if(-1 == kill(k->pid, SIGCONT)) {
		ERR("Could not restart kupdate");
		ERR("Falling back to sync()");
		last_resort(k->interval);
	}

	k->stopped=0;
	
	DEBUG("kupdate restarted with interval %ld.", k->interval);
}
