/*
 * Copyright (C) 1999-2001  Richard Hult 
 * 
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the 
 * License, or (at your option) any later version. 
 * 
 * This program is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <gnome.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include "main.h"
#include "schematic.h"
#include "sim-settings.h"
#include "sheet-private.h"
#include "node-store.h"
#include "node.h"
#include "wire.h"
#include "part-private.h"
#include "part-property.h"
#include "netlist.h"

typedef struct {
	gint node_nr;
	GHashTable *pins;
	GHashTable *models;
	GSList     *gnd_list;
	GSList     *mark_list;
	
	GList      *node_and_number_list;

	NodeStore  *store;
} NetlistData;

typedef struct {
	gint node_nr;
	gchar *name;
} Marker;

typedef struct {
	gint       node_nr;
	Node      *node;
} NodeAndNumber;

static void node_traverse (Node *node, NetlistData *data);
static void wire_traverse (Wire *wire, NetlistData *data);

static void
node_foreach_reset (gpointer key, gpointer value, gpointer user_data)
{
	Node *node = value;

	node_set_visited (node, FALSE);
}

static void
wire_traverse (Wire *wire, NetlistData *data)
{
	GSList *nodes;

	g_return_if_fail (wire != NULL);
	g_return_if_fail (IS_WIRE (wire));

	if (wire_is_visited (wire))
		return;

	wire_set_visited (wire, TRUE);

	for (nodes = wire_get_nodes (wire); nodes; nodes = nodes->next) {
		Node *node = nodes->data;

		node_traverse (node, data);
	}
}

static void
node_traverse (Node *node, NetlistData *data)
{
	GSList *wires, *pins;
	gchar *prop;
	NodeAndNumber *nan;

	g_return_if_fail (node != NULL);
	g_return_if_fail (IS_NODE (node));

	if (node_is_visited (node))
		return;

	node_set_visited (node, TRUE);

	/*
	 * Keep track of netlist nr <---> Node.
	 */
	nan = g_new0 (NodeAndNumber, 1);
	nan->node_nr = data->node_nr;
	nan->node = node;
	data->node_and_number_list = g_list_prepend (data->node_and_number_list, nan);

	/*
	 * Traverse the pins at this node.
	 */ 
	for (pins = node->pins; pins; pins = pins->next) {
		Pin *pin = pins->data;

		/*
		 * First see if the pin belongs to an "internal", special part.
		 */
		prop = part_get_property (pin->part, "internal");
		if (prop) {
			if (!g_strcasecmp (prop, "marker")) {
				Marker *marker;
				gchar *name, *value;

				name = part_get_property (pin->part, "name");

				if (!name) {
					g_free (prop);
					continue;
				}
				
				value = part_property_expand_macros (pin->part, name);
				g_free (name);

				if (!value)
					continue;
				
				marker = g_new0 (Marker, 1);
				marker->node_nr = data->node_nr;
				marker->name = value;
				data->mark_list = g_slist_prepend (data->mark_list, marker);
			} else if (!g_strcasecmp (prop, "ground")) {
				data->gnd_list = g_slist_prepend (data->gnd_list, GINT_TO_POINTER (data->node_nr));
			} else if (!g_strncasecmp (prop, "jumper", 5)) { /* Either jumper2 or jumper4. */
				Node *opposite_node;
				Pin opposite_pin;
				gint pin_nr, opposite_pin_nr;
				SheetPos pos;
				Pin *jumper_pins;
				gint num_pins;

				opposite_pin_nr = -1;
				num_pins = part_get_num_pins (pin->part);
				jumper_pins = part_get_pins (pin->part);
				for (pin_nr = 0; pin_nr < num_pins; pin_nr++) {
					if (&jumper_pins[pin_nr] == pin) {
						opposite_pin_nr = pin_nr;
						break;
					}
				}

				switch (opposite_pin_nr) {
				case 0:
					opposite_pin_nr = 1;
					break;
				case 1:
					opposite_pin_nr = 0;
					break;
				case 2:
					opposite_pin_nr = 3;
					break;
				case 3:
					opposite_pin_nr = 2;
					break;
				default: 
					g_assert (TRUE);
					break;
				}

				opposite_pin = jumper_pins[opposite_pin_nr];
				
				item_data_get_pos (ITEM_DATA (pin->part), &pos);
				pos.x += opposite_pin.offset.x;
				pos.y += opposite_pin.offset.y;
				
				opposite_node = node_store_get_node (data->store, pos);

#if 0
				if (node_is_visited (opposite_node)) {
					GList *list;

					/* Set the node name on the current node to the same as the already  visited node. */
					for (list = data->node_and_number_list; list; list = list->next) {
						NodeAndNumber *opposite_nan = list->data;

						if (opposite_nan->node == opposite_node)
							nan->node_nr = opposite_nan->node_nr;
					}
				}
#endif				
				node_traverse (opposite_node, data);
			}
			
			g_free (prop);
			continue;
		}

		/*
		 * Keep track of models to include. Needs to be freed when the
		 * hash table is no longer needed.
		 */
		prop = part_get_property (pin->part, "model");
		if (prop) {
			if (!g_hash_table_lookup (data->models, prop))
				g_hash_table_insert (data->models, prop, NULL);
		}
		
		g_hash_table_insert (data->pins, pin, GINT_TO_POINTER (data->node_nr));
	}
	
	/*
	 * Traverse the wires at this node.
	 */
	for (wires = node->wires; wires; wires = wires->next) {
		Wire *wire = wires->data;
		wire_traverse (wire, data);
	}
}

static void
node_foreach_traverse (gpointer key, gpointer value, NetlistData *data)
{
	Node *node = value;

	/*
	 * Only visit nodes that are not already visited.
	 */
	if (!node_is_visited (node))
		node_traverse (node, data);
	else
		return;

	data->node_nr++;
}

static gint
compare_marker (gconstpointer a, gconstpointer b)
{
	const Marker *ma;
	gint node_nr;

	ma = a;
	node_nr = GPOINTER_TO_INT (b);

	if (ma->node_nr == node_nr)
		return 0;
	else
		return 1;
}

static void
foreach_model_write (char *model, gpointer user_data, FILE *f)
{
	char *str, *tmp1, *tmp2;

	tmp1 = g_concat_dir_and_file (OREGANO_MODELDIR, model);
	tmp2 = g_strconcat (tmp1, ".model", NULL);

	str = g_strdup_printf (".include %s\n", tmp2);
	fputs (str, f);

	g_free (tmp1);
	g_free (tmp2);
	g_free (str);
}

static void
foreach_model_free (gpointer key, char *model, gpointer user_data)
{
	g_free (key);
}

static char *
linebreak (char *str)
{
	char **split, *tmp;
	GString *out;
	int i;

	split = g_strsplit (str, "\\", 0);	

	out = g_string_new ("");

	i = 0;
	while (split[i] != NULL) {
		if (split[i][0] == 'n') {
			if (strlen (split[i]) > 1) {
				out = g_string_append_c (out, '\n');
				out = g_string_append (out, split[i] + 1);
			}
		} else {
			out = g_string_append (out, split[i]);
		}
		
		i++;
	}

	g_strfreev (split);
	tmp = out->str;
	g_string_free (out, FALSE); /* Don't free the string data. */
	return tmp;
}

gchar *
nl_generate (Schematic *sm, gchar *name)
{
	NetlistData *data;
	GList *parts, *wires, *list;
	Part *part;
	gint pin_nr, num_pins, num_nodes, num_gnd_nodes, i, j;
	Pin *pins;
	gchar *template, **template_split;
	NodeStore *store;
	gchar **node2real;
	FILE *f;
	char *tmp_filename = NULL;
	gchar *filename;
	gint fd;
	SimSettings *sim_settings;

	schematic_log_clear (sm); 

	if (!name) {
		tmp_filename = g_concat_dir_and_file (g_get_tmp_dir (), "oregXXXXXX");
		fd = mkstemp (tmp_filename);
		f = fdopen (fd, "w");
		filename = tmp_filename;
	} else {
		filename = g_strdup (name);
		f = fopen (name, "w");
	}

	if (f == NULL) {
		g_free (filename);
		return NULL;
	}

	{
		char *title;
		GList *list;
		SimSettings *s;

		title = schematic_get_filename (sm);
		fputs ("* ",f);
		fputs (title ? title : "Title: <unset>", f);
		fputs ("\n"
		       "*----------------------------------------------"
		       "\n\n", f);
		s =  schematic_get_sim_settings (sm);
		list = sim_settings_get_options (s);
		if ( list ) {
		   fputs (".OPTIONS\n",f);
		   while (list) {
		      SimOption *so = list->data;
		      fprintf (f,"+     %s=%s\n",so->name,so->value);
		      list=list->next;
		   }
		   fputs ("\n"
			 "*----------------------------------------------"
			 "\n\n"
		      ,f);
		}
	}

	store = schematic_get_store (sm);

	node_store_node_foreach (store, (GHFunc *) node_foreach_reset, NULL);
	for (wires = store->wires; wires; wires = wires->next) {
		Wire *wire = wires->data;
		wire_set_visited (wire, FALSE);
	}

	data = g_new0 (NetlistData, 1);
	data->pins = g_hash_table_new (g_direct_hash, g_direct_equal);
	data->models = g_hash_table_new (g_str_hash, g_str_equal);
	data->node_nr = 1;
	data->gnd_list = NULL;
	data->mark_list = NULL;
	data->node_and_number_list = NULL;
	data->store = store;

	node_store_node_foreach (store, (GHFunc *) node_foreach_traverse, data);

	num_gnd_nodes = g_slist_length (data->gnd_list);

	if (num_gnd_nodes == 0) {
		schematic_log_append (sm, _("No ground node. Aborting.\n"));
		schematic_log_show (sm); 

		fclose (f);
		unlink (filename);
		g_free (filename);
		filename = NULL;
		goto bail_out;
	}		

	num_nodes = data->node_nr - 1;

	/*
	 * Build an array for going from node nr to "real node nr",
	 * where gnd nodes are nr 0 and the rest of the nodes are
	 * 1, 2, 3, ...
	 */
	node2real = g_new0 (gchar*, num_nodes + 1);

	for (i = 1, j = 1; i <= num_nodes; i++) {
		GSList *mlist;

		if (g_slist_find (data->gnd_list, GINT_TO_POINTER (i))) {
			node2real[i] = g_strdup ("0");
		} else if ((mlist = g_slist_find_custom (data->mark_list, GINT_TO_POINTER (i), compare_marker))) {
			Marker *marker = mlist->data;
			node2real[i] = g_strdup (marker->name);
		} else {
			node2real[i] = g_strdup_printf ("%d", j++);
		}
	}
	
	/*
	 * Fill in the netlist node names for all the used nodes.
	 */ 
	for (list = data->node_and_number_list; list; list = list->next) {
		NodeAndNumber *nan = list->data;

		g_free (nan->node->netlist_node_name);
		nan->node->netlist_node_name = g_strdup (node2real[nan->node_nr]);
	}

	for (parts = store->parts; parts; parts = parts->next) {
		gchar *tmp, *internal;
		GString *str;

		part = parts->data;
		internal = part_get_property (part, "internal");
		if (internal) {
			g_free (internal);
			continue;
		}

		tmp = part_get_property (part, "template");
		if (!tmp) {
			continue;
		}

		template = part_property_expand_macros (part, tmp);
		/*g_print ("Template: '%s'\n"
			 "macro   : '%s'\n",
			 tmp, template);*/
		g_free (tmp);

		tmp = linebreak (template);
		g_free (template);
		template = tmp;
		
		num_pins = part_get_num_pins (part);
		pins = part_get_pins (part);
		
		template_split = g_strsplit (template, " ", 0);
		g_free (template);
		template = NULL;

		str = g_string_new ("");

		i = 0;
		while (template_split[i] != NULL && template_split[i][0] != '%') {
			g_string_append (str, template_split[i++]);
			g_string_append_c (str , ' ');
			/*g_print ("str: %s\n", str->str);*/
		}

		/*g_print ("Reading %d pins.\n)", num_pins);*/
		
		for (pin_nr = 0; pin_nr < num_pins; pin_nr++) {
			gint node_nr;

			node_nr = GPOINTER_TO_INT (g_hash_table_lookup (data->pins, &pins[pin_nr]));
			if (!node_nr) {
				g_warning ("Couln't find part, pin_nr %d.", pin_nr);
			} else {
				gchar *tmp;
				tmp = node2real[node_nr];
				g_string_append (str, tmp);
				g_string_append_c (str, ' ');
				/*g_print ("str: %s\n", str->str);*/
				i++;
			}
			
			while (template_split[i] != NULL) {
				if (template_split[i][0] == '%')
					break;
				
				g_string_append (str, template_split[i]);
				g_string_append_c (str, ' ');
				/*g_print ("str: %s\n", str->str);*/
				i++;
			}
		}

		/*g_print ("Done with pins, i = %d\n", i);*/
		
		while (template_split[i] != NULL) {
			g_string_append (str, template_split[i]);
			g_string_append_c (str, ' ');
			/*g_print ("str: %s\n", str->str);*/
			i++;
		}
		
		g_strfreev (template_split);
		fputs (str->str, f);
		fputs ("\n", f);
		g_string_free (str, TRUE);
	}
	
	for (i = 0; i < num_nodes + 1; i++) {
		g_free (node2real[i]);
	}
	
	g_free (node2real);
	
	fputs ("\n"
	       "*----------------------------------------------"
	       "\n\n",f);
	
	g_hash_table_foreach (data->models, (GHFunc) foreach_model_write, f);
	
	/*
	 * Make sure we get ascii output.
	 * FIXME: Use binary to speed things up and simplify parsing?
	 */
	fputs ("\n"
	      "*----------------------------------------------"
	      "\n\n",f);
	fputs (".control\n", f);
	fputs ("set filetype=ascii\n", f);
	fputs (".endc\n", f);

	/* FIXME: move this to simulate.c */
	sim_settings = schematic_get_sim_settings (sm);
	if (sim_settings_get_trans (sim_settings)) {
		gchar *tmp;
		gdouble start, stop, step;
		gboolean step_enable;

	 	start = sim_settings_get_trans_start (sim_settings);
 		stop = sim_settings_get_trans_stop (sim_settings);
 		step = sim_settings_get_trans_step (sim_settings);
		step_enable = sim_settings_get_trans_step_enable (sim_settings);

		/* STEP STOP START MAX. */
		if (step_enable) // FIXME: is this right? check spice, first arg was 1e-3 before.
			tmp = g_strdup_printf (".tran %g %g %g %g\n", step, stop, start, step);
		else
			tmp = g_strdup_printf (".tran %g %g %g\n", step, stop, start);

		fputs (tmp, f);
		g_free (tmp);

	}
	if ( sim_settings_get_dc (sim_settings) ) {
		gchar *tmp=0,*vin1,*vin2;
		vin1 = sim_settings_get_dc_vsrc (sim_settings,0);
		vin2 = sim_settings_get_dc_vsrc (sim_settings,1);
		if ( vin1 ) {
			if ( vin2 ) 
				tmp = g_strdup_printf (
					".dc %s %g %g %g %s %g %g %g\n",
					vin1,
					sim_settings_get_dc_start (sim_settings,0),
					sim_settings_get_dc_stop (sim_settings,0),
					sim_settings_get_dc_step (sim_settings,0),
					vin2,
					sim_settings_get_dc_start (sim_settings,1),
					sim_settings_get_dc_stop (sim_settings,1),
					sim_settings_get_dc_step (sim_settings,1));
			else 
				tmp = g_strdup_printf (
					".dc %s %g %g %g\n",
					vin1,
					sim_settings_get_dc_start (sim_settings,0),
					sim_settings_get_dc_stop (sim_settings,0),
					sim_settings_get_dc_step (sim_settings,0));
			fputs (tmp,f);
			g_free (tmp);
		}
	}
	if ( sim_settings_get_ac (sim_settings) ) {
		gchar *tmp;
		tmp = 
			g_strdup_printf (
				".ac %s %d %g %g\n",
				sim_settings_get_ac_type (sim_settings),
				sim_settings_get_ac_npoints (sim_settings),
				sim_settings_get_ac_start (sim_settings),
				sim_settings_get_ac_stop (sim_settings)
				);
		fputs (tmp,f);
		g_free (tmp);
	}
	


	/* Debug op analysis. */
	fputs (".op\n", f);

 	fputs ("\n.END\n", f);

 	fclose (f);

 bail_out:
	g_hash_table_foreach (data->models, (GHFunc)foreach_model_free, NULL);
	g_hash_table_destroy (data->models);

	g_hash_table_destroy (data->pins);

	for (list = data->node_and_number_list; list; list = list->next) {
		NodeAndNumber *nan = list->data;
		g_free (nan);
	}
	g_list_free (data->node_and_number_list);

	g_free (data);

	return filename;
}

