/*
 * part-property.c
 *
 * 
 * Author: 
 *  Richard Hult <rhult@hem.passagen.se>
 * 
 *  http://www.dtek.chalmers.se/~d4hult/oregano/ 
 * 
 * Copyright (C) 1999,2000  Richard Hult 
 * 
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the 
 * License, or (at your option) any later version. 
 * 
 * This program is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#include <glib.h>
#include "part.h"
#include "part-property.h"

/*------------------------------------------------------------------------
 * Gets the name of a macro variable.
 * It also returns
 *     cls1 - first conditional clause
 *     cls2 - second clause
 *     sz   - returns number of characters parsed
 *------------------------------------------------------------------------*/
static char *get_macro_name (const char *str, char **cls1, char **cls2, int *sz)
{
	char tmp[512], separators[] = { ",.;/|()" };
	char *id;
	const char *q, *qend;
	char *qn, *csep = NULL;
	size_t sln;
	int rc = 0;

	sln = strlen (str) + 1;
	*sz = 0;
	*cls1 = *cls2 = 0;
	qend = str + sln;
	id = g_malloc0 (sln);

	/* Get the name */
	for (q = str, qn = id; *q && *q != ' ' && !(csep = strchr (separators, *q)); q++) {
		if ( q > qend ) {
			g_warning ("Expand macro error.");     
			rc = 1;
			break;
		}
		*qn++ = *q;
	}

	/* if error found, return here */   
	if (rc)
		goto error;
	
	/* Look for conditional clauses */
	*qn = 0;
	if (csep) {
		/* get the first one */
	        q++; /* skip the separator and store the clause in tmp */
		for (qn = tmp; *q && *q != *csep; q++)
			*qn++ = *q;

		if (!*q)
			goto error;

		*qn = 0;
		*cls1 = g_strdup(tmp);
		q++; /* skip the end-of-clause separator */

		/* Check for the second one */
		if ( *q && (csep = strchr (separators, *q))) {
		        q++; /* skip the separator and store in tmp*/
			for (qn = tmp; *q && *q != *csep; q++)
				*qn++ = *q;
			
			if (!*q) {
				g_free (*cls1);
				*cls1 = 0;
				goto error;
			}

			*qn = 0;
			*cls2 = g_strdup (tmp);  
			q++; /* skip the end-of-clause separator */
		}
	}

	*sz = (int) (q - str);
	return id;

 error:
	g_free (id);
	return 0;
}

char *
part_property_expand_macros (Part *part, char *string)
{
	static char mcode[] = {"@?~#&"};
	char *value;
	char *tmp0, *temp, *pbuf, *qn, *q0, *qend, *t0;
	char *cls1, *cls2;
	char buffer[512];
	size_t sln;
	
	g_return_val_if_fail (part != NULL, NULL);
	g_return_val_if_fail (IS_PART (part), NULL);
	g_return_val_if_fail (string != NULL, NULL);
	
	cls1 = cls2 = q0 = 0;
	/* Rules:
	   @<id>             value of <id>. If no value, error
	   &<id>             value of <id> if <id> is defined
	   ?<id>s...s        text between s...s separators if <id> defined
	   ?<id>s...ss...s   text between 1st s...s separators if <id> defined
	   else 2nd s...s clause
	   ~<id>s...s        text between s...s separators if <id> undefined
	   ~<id>s...ss...s   text between 1st s...s separators if <id> undefined
	   else 2nd s...s clause
	   #<id>s...s        text between s...s separators if <id> defined, but
	   delete rest of tempalte if <id> undefined
	   
	   Separatos can be any of (, . ; / |) For an opening-closing pair of
	   separators the same character ahs to be used.
	   
	   Examples: R^@refdes %1 %2 @value
	   V^@refdes %+ %- SIN(@offset @ampl @freq 0 0)
	   ?DC|DC @DC|
	*/
	pbuf = buffer;
	tmp0 = temp = g_strdup (string);
	qend = temp + strlen (temp);
	
	for (pbuf = buffer, temp = string; *temp;) {
		/*
		 * Look for any of the macro char codes.
		 */
		if (strchr (mcode, *temp)) {
			qn = get_macro_name (temp + 1, &cls1, &cls2, &sln);
			if (qn == 0)
				return 0;
			value = part_get_property (part, qn);
			if ((*temp == '@' || *temp == '&') && value) {
				pbuf += sprintf (pbuf, "%s", value);
			} else if (*temp =='&' && !value) {
				g_warning ("expand macro error: macro %s undefined", qn);
				g_free (qn);
				return 0;
			} else if (*temp == '?' || *temp == '~') {
				if (cls1 == 0) {
					g_warning ("error in template: %s", temp);
					g_free (qn);
					return 0;
				}
				q0 = (value
				      ? (*temp == '?' ? cls1 : cls2)
				      : (*temp == '?' ? cls2 : cls1)
					);
				if (q0) {
					t0 = part_property_expand_macros (part, q0);
					if (!t0) {
						g_warning ("error in template: %s", temp);
						g_free (qn);
					}
					pbuf += sprintf (pbuf, "%s", t0);
					g_free (t0);      
				}
			} else if (*temp=='#') {
				if (value) {
					t0 = part_property_expand_macros (part, value);
					if (!t0) {
						g_warning ("error in template: %s", temp);
						g_free (qn);
					}
					pbuf += sprintf (pbuf, "%s", t0);
					g_free (t0);             
				} else 
					*(temp + sln) = 0;
			}
			temp += 1 + sln;
			g_free (qn);
			g_free (cls1);
			g_free (cls2);
		} else {
			if ( *temp== '\\' ) {
				temp++;
				switch (*temp) {
				case 'n': *pbuf++ = '\n'; break;
				case 't': *pbuf++ = '\t'; break;
				case 'r': *pbuf++ = '\r'; break;
				case 'f': *pbuf++ = '\f'; break;
				}
				temp++;
			} else
				*pbuf++ = *temp++;
		}
	}
	
	g_free (tmp0);
	*pbuf = 0;
	
	return g_strdup (buffer);
}


#if 0
//---------------------
char *
part_property_expand_macros (Part *part, char *string)
{
	char *name;
	char *value;
	char *temp;
	char buffer[512];
	gint in_index = 0, out_index = 0;

	g_return_val_if_fail (part != NULL, NULL);
	g_return_val_if_fail (IS_PART (part), NULL);
	g_return_val_if_fail (string != NULL, NULL);

	/* Examples: R^@refdes %1 %2 @value
	 *           V^@refdes %+ %- SIN(@offset @ampl @freq 0 0)
	 */

	temp = string;
	in_index = 0;
	out_index = 0;

	while (*temp != 0) {
		/* Is it a macro? */
		if (temp[0] == '@'){
			int i = 0;
			/* Find the end of the macro. */
			while (1){
				if (temp[i] == ' ' || temp[i] == '(' || temp[i] == ')' || temp[i] == 0)
					break;
				else {
					i++;
					if (i > strlen (string)){
						g_warning ("expand macro error.");
						break;
					}
				}
			}

			/* Perform a lookup on the macro. */
			name = g_strndup (temp + 1, i - 1);
			value = part_get_property (part, name);
			g_free (name);

			if (value) {
				snprintf (buffer + out_index, 16, "%s ", value);
				out_index += strlen (value);
				in_index += i + 1;
			}
			temp += i;
		} else{
			buffer[out_index] = *temp;
			out_index++;
			in_index++;
			temp++;
		}			
	}

	buffer[out_index] = '\0';
	return g_strdup (buffer);
}
#endif
