/*
 * sim-engine.c
 *
 * Simulate engine class, interface to SPICE.
 * 
 * Author: 
 *  Richard Hult <rhult@hem.passagen.se>
 * 
 *  http://www.dtek.chalmers.se/~d4hult/oregano/ 
 * 
 * Copyright (C) 1999,2000  Richard Hult 
 * 
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the 
 * License, or (at your option) any later version. 
 * 
 * This program is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <gnome.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/wait.h>
#include <sys/stat.h>

#include "main.h" /* get rid of this... */
#include "sim-settings.h"
#include "sim-engine.h"
#include "schematic.h"

#define READ 0
#define WRITE 1

static void sim_engine_class_init (SimEngineClass *klass);
static void sim_engine_init (SimEngine *engine);

static void data_input_cb (SimEngine *engine, gint source, GdkInputCondition condition);
static void error_input_cb (SimEngine *engine, gint source, GdkInputCondition condition);

enum {
	DONE,
	ABORTED,
	CANCELLED,
	LAST_SIGNAL
};
typedef enum {
	STATE_IDLE,
	IN_VARIABLES,
	IN_VALUES,
	STATE_ABORT
} ParseDataState;

static gchar *analysis_names[] = {
	N_("Operating Point")            ,
	N_("Transient Analysis")         ,
	N_("DC transfer characteristic") ,
	N_("AC Analysis")                ,
	N_("Transfer Function")          ,
	N_("Distortion Analysis")        ,
	N_("Noise Analysis")             ,
	N_("Pole-Zero Analysis")         ,
	N_("Sensitivity Analysis")       ,
	N_("Unknown Analysis")           ,
	NULL
};

#define SP_TITLE      "Title:"
#define SP_DATE       "Date:"
#define SP_PLOT_NAME  "Plotname:"
#define SP_FLAGS      "Flags:"
#define SP_N_VAR      "No. Variables:"
#define SP_N_POINTS   "No. Points:"
#define SP_COMMAND    "Command:"
#define SP_VARIABLES  "Variables:"
#define SP_BINARY     "Binary:"
#define SP_VALUES     "Values:" 
#define IS_THIS_ITEM(str,item)  (!strncmp(str,item,strlen(item)))  


static guint sim_engine_signals [LAST_SIGNAL] = { 0 };
static GtkObject *parent_class = NULL;

guint
sim_engine_get_type (void)
{
	static guint sim_engine_type = 0;

	if (!sim_engine_type) {
		static const GtkTypeInfo sim_engine_info = {
			"SimEngine",
			sizeof (SimEngine),
			sizeof (SimEngineClass),
			(GtkClassInitFunc) sim_engine_class_init,
			(GtkObjectInitFunc) sim_engine_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};
		
		sim_engine_type = gtk_type_unique (GTK_TYPE_OBJECT, &sim_engine_info);
	}

	return sim_engine_type;
}

static void
sim_engine_destroy (GtkObject *object)
{
	SimEngine *engine;
	SimulationData *sdat;
	GList     *analysis;
	int i, num;

	engine = SIM_ENGINE (object);

	analysis = engine->analysis;
	for ( ; analysis; analysis=analysis->next ) {
		sdat = SIM_DATA(analysis->data);
		num = sdat->n_variables;

		g_free (sdat->var_names);
		g_free (sdat->var_units);

	for (i = 0; i < num; i++) {
			if (sdat->data[i])
				g_array_free (sdat->data[i], FALSE);
	}	
		g_free (sdat->data);

		g_free (sdat->min_data);
		g_free (sdat->max_data);
		g_free(sdat);
	}		
	g_list_free(engine->analysis);
	/*
	 * Chain up. 
	 */
	GTK_OBJECT_CLASS (parent_class)->destroy (object);
}

static void
sim_engine_class_init (SimEngineClass *klass)
{
	GtkObjectClass *object_class;

	object_class = (GtkObjectClass *)klass;
	parent_class = gtk_type_class (GTK_TYPE_OBJECT);

	object_class->destroy = sim_engine_destroy;

	sim_engine_signals [DONE] = 
		gtk_signal_new ("done",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SimEngineClass, done),
				gtk_marshal_NONE__NONE,
				GTK_TYPE_NONE, 0);

	sim_engine_signals [ABORTED] = 
		gtk_signal_new ("aborted",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SimEngineClass, aborted),
				gtk_marshal_NONE__NONE,
				GTK_TYPE_NONE, 0);

	sim_engine_signals [CANCELLED] = 
		gtk_signal_new ("cancelled",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SimEngineClass, cancelled),
				gtk_marshal_NONE__NONE,
				GTK_TYPE_NONE, 0);

	gtk_object_class_add_signals (object_class, sim_engine_signals, LAST_SIGNAL);
}

static void
sim_engine_init (SimEngine *engine)
{
	engine->running = FALSE;
	engine->progress = 0.0;

	engine->child_pid = 0;

	engine->data_input_id = 0;
	engine->error_input_id = 0;

	engine->inputfp = NULL;
	engine->outputfp = NULL;
	engine->errorfp = NULL;
	
	engine->has_warnings = FALSE;
	engine->has_errors = FALSE;
	engine->aborted = FALSE;

	engine->num_analysis=0;
	engine->analysis=0;
}

SimEngine *
sim_engine_new (Schematic *sm)
{
	SimEngine *engine;

	engine = gtk_type_new (sim_engine_get_type ());

	engine->sm = sm;
	engine->sim_settings = schematic_get_sim_settings (sm);

	return engine;
}

void
sim_engine_start_with_file (SimEngine *engine, const gchar *netlist)
{
	int netlist_fd;

	if (pipe (engine->to_child) < 0)
		g_error ("Error creating stdin pipe");
	if (pipe (engine->to_parent) < 0)
		g_error ("Error creating stdout pipe");
	if (pipe (engine->to_parent_error) < 0)
		g_error ("Error creating stderr pipe");
	
	engine->child_pid = fork ();
	if (engine->child_pid == 0) {
		gchar *spice = oregano.simexec;
		gchar *args[4] = { oregano.simexec, "-s", "-n", NULL };

		/* The child executes here. */
		netlist_fd = open (netlist, O_RDONLY);
		if (netlist_fd == -1)
			g_error ("Error opening netlist.");
		
		close (engine->to_child[WRITE]);
		close (engine->to_parent[READ]);
		close (engine->to_parent_error[READ]);

		dup2 (netlist_fd, STDIN_FILENO);
		close (netlist_fd);
		close (engine->to_child[READ]);

		/* Map stderr and stdout to their pipes. */
		dup2 (engine->to_parent[WRITE], STDOUT_FILENO);
		dup2 (engine->to_parent_error[WRITE], STDERR_FILENO);
		close (engine->to_parent[WRITE]);
		close (engine->to_parent_error[WRITE]);

		setlocale (LC_NUMERIC, "C");

		/* Execute spice. */
		execvp (spice, args);

		/* We should never get here. */
		g_warning ("Error executing spice.");
		_exit (1);
	}
	
	if (engine->child_pid == -1) {
		g_warning ("Could not fork child process.");
		return;
	}

	/* The parent executes here. */

	schematic_log_clear (engine->sm);

	/* Set up communication. */
	close (engine->to_child[READ]);
	close (engine->to_parent[WRITE]);
	close (engine->to_parent_error[WRITE]);
	
	engine->errorfp = fdopen (engine->to_parent_error[READ], "r");
	engine->inputfp = fdopen (engine->to_parent[READ], "r");
	engine->outputfp = fdopen (engine->to_child[WRITE], "w");
	if (engine->inputfp == NULL || engine->outputfp == NULL) {
		g_warning ("Error executing child process");
		return;
	}

	engine->running = TRUE;
	engine->data_input_id = gdk_input_add (
		engine->to_parent[READ],
		GDK_INPUT_READ,
		(GdkInputFunction) data_input_cb,
		(gpointer) engine);

	engine->error_input_id = gdk_input_add (
		engine->to_parent_error[READ],
		GDK_INPUT_READ,
		(GdkInputFunction) error_input_cb,
		(gpointer) engine);	
}

void
sim_engine_stop (SimEngine *engine)
{
	int status;

	g_return_if_fail (engine != NULL);
	g_return_if_fail (IS_SIM_ENGINE (engine));

	if (!engine->running)
		return;

	gdk_input_remove (engine->data_input_id);
	gdk_input_remove (engine->error_input_id);
	engine->data_input_id = 0;
	engine->error_input_id = 0;
	
	kill (engine->child_pid, SIGKILL);
	waitpid (engine->child_pid, &status, WUNTRACED);
	engine->child_pid = 0;
	
	fclose (engine->errorfp);
	fclose (engine->inputfp);
	fclose (engine->outputfp);
	
	engine->running = FALSE;
}

static void
data_input_cb (SimEngine *engine, gint source, GdkInputCondition condition) 
{
	static gchar buf[256];
	static SimulationData *sdata = NULL;
	gint status, iter;
	gdouble val, np1, np2;
	gchar **tmp = NULL;	
	gchar *send;

	if (!(condition & GDK_INPUT_READ))
		return;

	if (fgets (buf, 255, engine->inputfp) == NULL){
		engine->running = FALSE;

		gdk_input_remove (engine->data_input_id);
		gdk_input_remove (engine->error_input_id);

		kill (engine->child_pid, SIGKILL);
		waitpid (engine->child_pid, &status, WUNTRACED);
		
		fclose (engine->errorfp);
		fclose (engine->inputfp);
		fclose (engine->outputfp);

		if (!sdata || engine->aborted || engine->num_analysis < 2) {
			if (engine->num_analysis < 2) {
				schematic_log_append (engine->sm, 
						      "\n\n### Too few or none analysis found ###\n\n");
	}
			engine->has_errors = TRUE;
			gtk_signal_emit_by_name (GTK_OBJECT (engine), "aborted");
		}
		else {
			gtk_signal_emit_by_name (GTK_OBJECT (engine), "done");
		}
		
		return;
	}

	/* We are getting the simulation title */
	if (IS_THIS_ITEM (buf, SP_TITLE)) {
		sdata = SIM_DATA (g_new0 (Analysis, 1));		
		engine->analysis = g_list_append (engine->analysis, sdata);
		engine->num_analysis++;
	} 
	/* The date of the analysis */
	else if (IS_THIS_ITEM (buf, SP_DATE)) {
		sdata->state = STATE_IDLE;
	} 
	/*
	   The name of the plot.    
	   We use that to detrminethe type of analysis
	 */
	else if ( IS_THIS_ITEM (buf,SP_PLOT_NAME) ) {
		gint i;
		gchar    *analysis = buf+strlen(SP_PLOT_NAME)+1;

		sdata->state = STATE_IDLE;
		sdata->type  = ANALYSIS_UNKNOWN;
		for (i = 0; analysis_names[i]; i++) 
			if (IS_THIS_ITEM (analysis, analysis_names[i])) {
				sdata->type = i;
				break;
		}

		switch ( sdata->type ) {
		case TRANSIENT:
			ANALYSIS(sdata)->transient.sim_length =
				sim_settings_get_trans_stop  (engine->sim_settings) -
				sim_settings_get_trans_start (engine->sim_settings);
			ANALYSIS(sdata)->transient.step_size = 
				sim_settings_get_trans_step (engine->sim_settings);
			break;
		case AC:
			ANALYSIS(sdata)->ac.start = 
				sim_settings_get_ac_start (engine->sim_settings);
			ANALYSIS(sdata)->ac.stop  = 
				sim_settings_get_ac_stop  (engine->sim_settings);
			ANALYSIS(sdata)->ac.sim_length = 
				sim_settings_get_ac_npoints (engine->sim_settings);
			break;
		case OP_POINT:
		case DC_TRANSFER:
			np1 = np2 = 1.;
			ANALYSIS(sdata)->dc.start1 = 
				sim_settings_get_dc_start (engine->sim_settings,0);
			ANALYSIS(sdata)->dc.stop1  = 
				sim_settings_get_dc_stop  (engine->sim_settings,0);
			ANALYSIS(sdata)->dc.step1  = 
				sim_settings_get_dc_step  (engine->sim_settings,0);
			ANALYSIS(sdata)->dc.start2 = 
				sim_settings_get_dc_start (engine->sim_settings,1);
			ANALYSIS(sdata)->dc.stop2  = 
				sim_settings_get_dc_stop  (engine->sim_settings,1);
			ANALYSIS(sdata)->dc.step2  = 
				sim_settings_get_dc_step  (engine->sim_settings,1);

			np1 = (ANALYSIS(sdata)->dc.stop1-ANALYSIS(sdata)->dc.start1) /
				ANALYSIS(sdata)->dc.step1;
			if ( ANALYSIS(sdata)->dc.step2 != 0. ) {
				np2 = (ANALYSIS(sdata)->dc.stop2-ANALYSIS(sdata)->dc.start2) /
					ANALYSIS(sdata)->dc.step2;
	}	
			ANALYSIS(sdata)->dc.sim_length = np1 * np2; 
			break;
		case TRANSFER:
		case DISTORTION:
		case NOISE:
		case POLE_ZERO:
		case SENSITIVITY:
			break;
		case ANALYSIS_UNKNOWN:
			g_error("Unknown analysis: %s",analysis);
			break;
	}
	}
	/* Flags */
	else if (IS_THIS_ITEM(buf, SP_FLAGS) ) {
		
	}
	/* Command */
	else if (IS_THIS_ITEM (buf, SP_COMMAND)) {

	}
	/* Number of variables  */
	else if (IS_THIS_ITEM (buf, SP_N_VAR)) { 
		gint i, n = atoi (buf + strlen (SP_N_VAR));

		sdata->state = STATE_IDLE;
		sdata->n_variables = n;
		sdata->got_points  = 0;
		sdata->got_var     = 0;
		sdata->var_names   = (char**) g_new0 (gpointer, n);
		sdata->var_units   = (char**) g_new0 (gpointer, n);
		sdata->data        = (GArray**) g_new0 (gpointer, n);
		for (i = 0; i < n; i++) 
			sdata->data[i] = 
				g_array_new (TRUE, TRUE, sizeof (double));
		sdata->min_data = g_new (double, n);
		sdata->max_data = g_new (double, n);
		for (i = 0; i < n; i++) {
			sdata->min_data[i] =  G_MAXDOUBLE;
			sdata->max_data[i] = -G_MAXDOUBLE;
		}
	}

	/*  
	    Number of points
	    For some reason is 0 when spice runs in server mode...
	*/
	else if (IS_THIS_ITEM (buf, SP_N_POINTS)) { 
		sdata->state = STATE_IDLE;
		sdata->n_points = atoi (buf + strlen (SP_N_POINTS));
	}
	/*  List with the name of the variables and their units */
	else if (IS_THIS_ITEM (buf, SP_VARIABLES)) { 
		sdata->state = IN_VARIABLES;		

	} 
	/* Data in binary format. */
	else if (IS_THIS_ITEM (buf, SP_BINARY)) { 
		sdata->state = IN_VALUES;
		sdata->binary = TRUE;
		sdata->got_var = 0;
		sdata->got_points = 0;
	} 
	/*  Data in ascii */
	else if (IS_THIS_ITEM (buf, SP_VALUES)) { 
		sdata->state = IN_VALUES;
		sdata->binary = FALSE;
		sdata->got_var = 0;
		sdata->got_points = 0;
	} 
	/*  We should be either getting the varible names or the data */
	else {
		if (engine->analysis == 0) {
			if (strlen (buf) > 1)
				schematic_log_append (engine->sm, buf); 			
			return;
		}

		switch (sdata->state) {
		case IN_VARIABLES:
		tmp = g_strsplit (buf, "\t", 0);
			sdata->var_names[sdata->got_var] = g_strdup (tmp[2]);
			sdata->var_units[sdata->got_var] = g_strdup (tmp[3]);
			send = strchr(sdata->var_units[sdata->got_var], '\n');
			if (send)
				*send = 0;
			g_strfreev (tmp);
			
			if ((sdata->got_var + 1) < sdata->n_variables)
				sdata->got_var++;
			break;

		case IN_VALUES:
			if (sdata->got_var) 
				sscanf(buf, "\t%lf", &val);
			else
				sscanf(buf, "%d\t\t%lf", &iter, &val);

			if ( sdata->got_var == 0) {
				switch (sdata->type) {
				case TRANSIENT:
					engine->progress = val /
						ANALYSIS(sdata)->transient.sim_length;
					
					break;
				case AC:
					engine->progress = (val - ANALYSIS(sdata)->ac.start) /
						ANALYSIS(sdata)->ac.sim_length;
					break;
				case DC_TRANSFER:
					engine->progress = ((gdouble) iter) /
						ANALYSIS(sdata)->ac.sim_length;
				default:
					break;
				}
				if (engine->progress > 1.0)
					engine->progress = 1.0;
			}

			sdata->data[sdata->got_var] =
				g_array_append_val (
					sdata->data[sdata->got_var],
					val);
			
			/* Update the minimal and maximal values so far. */
			if (val < sdata->min_data[sdata->got_var])
				sdata->min_data[sdata->got_var] = val;
			if (val > sdata->max_data[sdata->got_var])
				sdata->max_data[sdata->got_var] = val;
		
			/* Check for the end of the point. */
			if (sdata->got_var + 1 == sdata->n_variables) {
				sdata->got_var = 0;
				sdata->got_points++;
			}
			else
				sdata->got_var++;

			break;
		default:
			if (strlen (buf) > 1) {
				if (strstr (buf,"abort"))
					sdata->state = STATE_ABORT;
				schematic_log_append (engine->sm, buf); 
		}
			break;
			
		}
	}
}

static void
error_input_cb (SimEngine *engine, gint source, GdkInputCondition condition)
{
	static gchar buf[256]; 

	if (!(condition & GDK_INPUT_READ))
		return;

	/*
	 * Store the error messages.
	 */
	if (fgets (buf, 255, engine->errorfp) != NULL) {
		if (strncmp (buf, "@@@", 3) != 0) {
			if (strstr (buf, "abort")) {
				engine->aborted = TRUE;
			}
			
			schematic_log_append (engine->sm, buf); 
			engine->has_warnings = TRUE;
		}
	}
}

gboolean
sim_engine_get_op_value (SimEngine *engine, char *node_name, double *value)
{
	int i;
	GList *analysis;
	SimulationData *sdat = NULL;
	/*
	 * FIXME: build a hash table with the values.
	 */
	
	g_return_val_if_fail (engine != NULL, FALSE);
	g_return_val_if_fail (IS_SIM_ENGINE (engine), FALSE);
	g_return_val_if_fail (node_name != NULL, FALSE);
	
	/* Look for the OP analysis. */
	for (analysis = engine->analysis; analysis; analysis = analysis->next ) {
		if (SIM_DATA (analysis->data)->type == OP_POINT) {
			sdat = SIM_DATA (analysis->data);
			break;
		}
	}

	for (i = 0; i < sdat->n_variables; i++) {
		if (g_strcasecmp (sdat->var_names[i], node_name) == 0) {
			*value = g_array_index (sdat->data[i], double, 0);

			return TRUE;
		}	
	}

	return FALSE;
}

gchar *
sim_engine_analysis_name (SimulationData *sdat) 
{
	if (sdat == NULL)
		return g_strdup (_(analysis_names[ANALYSIS_UNKNOWN]));

	return g_strdup (_(analysis_names[sdat->type]));
} 
