/*
 * sim-settings.c
 *
 * 
 * Author: 
 *  Richard Hult <rhult@hem.passagen.se>
 * 
 *  http://www.dtek.chalmers.se/~d4hult/oregano/ 
 * 
 * Copyright (C) 1999,2000  Richard Hult 
 * 
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the 
 * License, or (at your option) any later version. 
 * 
 * This program is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <unistd.h>
#include <ctype.h>
#include <gnome.h>
#include <glade/glade.h>
#include <gtk/gtkclist.h>
#include <gtk/gtkentry.h>
#include "main.h"
#include "sim-settings.h"
#include "schematic.h"
#include "schematic-view.h"
#include "dialogs.h"
#include "oregano-utils.h"

typedef enum {
        TRANSIENT_PAGE,
        FOURIER_PAGE,
	DC_SWEEP_PAGE,
	AC_PAGE,
//        TEMPERATURE_PAGE,
        OPTIONS_PAGE,
} SettingPages;

struct _SimSettingsPriv {
	/* Transient analysis. */
        GtkWidget *w_trans_enable, *w_trans_start, *w_trans_stop;
	GtkWidget *w_trans_step, *w_trans_step_enable;
	gboolean trans_enable;
	gchar *trans_start;
	gchar *trans_stop;
	gchar *trans_step;
	gboolean trans_step_enable;

	/*  AC   */
	GtkWidget *w_ac_enable,*w_ac_type,*w_ac_npoints,*w_ac_start,*w_ac_stop; 
	gboolean ac_enable;
	gchar   *ac_type;
	gchar   *ac_npoints;
	gchar   *ac_start;
	gchar   *ac_stop;

	/*  DC  */
	GtkWidget *w_dc_enable,*w_dc_vin1,*w_dc_start1,*w_dc_stop1,*w_dc_step1,
		*w_dc_vin2,*w_dc_start2,*w_dc_stop2,*w_dc_step2;
	gboolean dc_enable;
	gchar   *dc_vin1;
	gchar   *dc_start1,*dc_stop1,*dc_step1;
	gchar   *dc_vin2;
	gchar   *dc_start2,*dc_stop2,*dc_step2;

	/* Fourier analysis. Replace with something sane later. */
	GtkWidget *w_four_enable,*w_four_freq,*w_four_vout,*w_four_combo,*w_four_add;
	gboolean four_enable;
        gchar *four_freq;
        gchar *four_vout;

	/*  Options   */
        GtkEntry *w_opt_value;
        GtkCList  *w_opt_list;
        GList *options;
};


static SimOption default_options[] = {
        {"TEMP" ,NULL},

        {"GMIN" ,NULL},
        {"ABSTOL" ,NULL},
        {"CHGTOL" ,NULL},
        {"RELTOL" ,NULL},
        {"VNTOL" ,NULL},

        {"ITL1" ,NULL},
        {"ITL2" ,NULL},
        {"ITL4", NULL},

        {"PIVREL", NULL},
        {"PIVTOL", NULL},

        {"TNOM", NULL},
        {"TRTOL", NULL},

        {"DEFAD", NULL},
        {"DEFAS", NULL},
        {"DEFL", NULL},
        {"DEFW", NULL},
        {NULL, NULL}
};


static void
set_options_in_list (gchar *key, gchar *val, GtkCList *cl) {
        int i;
        char *opt;

        for (i = 0; i< cl->rows; i++) {
                gtk_clist_get_text (cl, i, 0, &opt);
                if (!strcmp (opt, key) ) {
                        gtk_clist_set_text (cl, i, 1, val);
                }
        }
}

static void
get_options_from_list (SimSettings *s) {
        int i;
        gchar *opt,*val;
        SimOption *so;
        GtkCList *cl = s->priv->w_opt_list;

        /*  Empty the list   */
        while (s->priv->options) {
                so = s->priv->options->data;
                if (so) {
                        g_free (so->name);
                        g_free (so->value);
                        g_list_remove (s->priv->options,so);
                        g_free (so);
                }
                s->priv->options = s->priv->options->next;
        }

        for (i = 0; i < cl->rows; i++) {
                val = 0;
                gtk_clist_get_text (cl, i, 0, &opt);
                gtk_clist_get_text (cl, i, 1, &val);
                if (val) {
                        SimOption *so = g_new0 (SimOption,1);
                        so->name = g_strdup (opt);
                        so->value = g_strdup (val);
                        s->priv->options = g_list_append (s->priv->options, so);
                }
        }
}

static void
select_opt_callback (GtkCList        *clist,
                     gint             row,
                     gint             column,
                     GdkEvent        *event,
                     SimSettings     *s)
{
        gchar *value = "";
        gtk_clist_get_text (s->priv->w_opt_list, row, 1, &value);
        gtk_entry_set_text (s->priv->w_opt_value, value);

}

static void 
option_setvalue (GtkWidget *w, SimSettings *s)
{
        gchar *value = gtk_entry_get_text (s->priv->w_opt_value);

        if (value) {
                if (s->priv->w_opt_list->selection) {
                        int index = (int) s->priv->w_opt_list->selection->data;
                        gtk_clist_set_text (s->priv->w_opt_list, index, 1, value);
                        gnome_property_box_set_modified (GNOME_PROPERTY_BOX (s->pbox), TRUE);
                }
        }
}

static void 
option_remove (GtkWidget *w, SimSettings *s)
{
        if ( s->priv->w_opt_list->selection ) {
                int index = (int)s->priv->w_opt_list->selection->data;
                gtk_clist_set_text (s->priv->w_opt_list,index,1,NULL);
        }
}

static void 
trans_enable_cb (GtkWidget *widget, SimSettings *s)
{
	gboolean enable, step_enable;

	enable = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget));
	step_enable = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (s->priv->w_trans_step_enable));

	gtk_widget_set_sensitive (s->priv->w_trans_start, enable);
	gtk_widget_set_sensitive (s->priv->w_trans_stop, enable);
	gtk_widget_set_sensitive (s->priv->w_trans_step, enable & step_enable);
	gtk_widget_set_sensitive (s->priv->w_trans_step_enable, enable);
	
	gnome_property_box_changed (GNOME_PROPERTY_BOX (s->pbox));
}

static void 
trans_step_enable_cb (GtkWidget *widget, SimSettings *s)
{
	gboolean enable, step_enable;

	step_enable = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget));
	enable = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (s->priv->w_trans_enable));

	gtk_widget_set_sensitive (s->priv->w_trans_step, step_enable & enable);
	
	gnome_property_box_changed (GNOME_PROPERTY_BOX (s->pbox));
}

static void 
entry_changed_cb (GtkWidget *widget, SimSettings *s)
{
	gnome_property_box_changed (GNOME_PROPERTY_BOX (s->pbox));
}

static void
ac_enable_cb (GtkWidget *widget, SimSettings *s)
{
	gboolean enable;
	enable = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget));
	gtk_widget_set_sensitive (s->priv->w_ac_type,enable);
	gtk_widget_set_sensitive (s->priv->w_ac_npoints,enable);
	gtk_widget_set_sensitive (s->priv->w_ac_start,enable);
	gtk_widget_set_sensitive (s->priv->w_ac_stop,enable);

	gnome_property_box_changed (GNOME_PROPERTY_BOX (s->pbox));
}

static void
dc_enable_cb (GtkWidget *widget, SimSettings *s)
{
	gboolean enable;
	enable = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget));
	gtk_widget_set_sensitive (s->priv->w_dc_vin1,enable);
	gtk_widget_set_sensitive (s->priv->w_dc_start1,enable);
	gtk_widget_set_sensitive (s->priv->w_dc_stop1,enable);
	gtk_widget_set_sensitive (s->priv->w_dc_step1,enable);
	gtk_widget_set_sensitive (s->priv->w_dc_vin2,enable);
	gtk_widget_set_sensitive (s->priv->w_dc_start2,enable);
	gtk_widget_set_sensitive (s->priv->w_dc_stop2,enable);
	gtk_widget_set_sensitive (s->priv->w_dc_step2,enable);
	gnome_property_box_changed (GNOME_PROPERTY_BOX (s->pbox));
}

static void
four_enable_cb (GtkWidget *widget, SimSettings *s)
{
	gboolean enable;
	enable = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget));
	gtk_widget_set_sensitive (s->priv->w_four_freq,enable);
	gtk_widget_set_sensitive (s->priv->w_four_vout,enable);
	gtk_widget_set_sensitive (s->priv->w_four_combo,enable);
	gtk_widget_set_sensitive (s->priv->w_four_add,enable);

	gnome_property_box_changed (GNOME_PROPERTY_BOX (s->pbox));
}

static void 
four_add_vout_cb (GtkWidget *widget, SimSettings *s)
{

}

static void
apply_callback (GtkWidget *w, int page_nr, SimSettings *s)
{
        switch (page_nr) {
        case TRANSIENT_PAGE:
		s->priv->trans_enable = 
			gtk_toggle_button_get_active (
				GTK_TOGGLE_BUTTON (s->priv->w_trans_enable));

		if (s->priv->trans_start) g_free (s->priv->trans_start);
		s->priv->trans_start = 
			gtk_editable_get_chars (
				GTK_EDITABLE (s->priv->w_trans_start), 0, -1);

		if (s->priv->trans_stop) g_free (s->priv->trans_stop);
                s->priv->trans_stop = 
			gtk_editable_get_chars (
				GTK_EDITABLE (s->priv->w_trans_stop), 0, -1);

                if (s->priv->trans_step) g_free (s->priv->trans_step);
                s->priv->trans_step = 
			gtk_editable_get_chars (
				GTK_EDITABLE (s->priv->w_trans_step), 0, -1);

		s->priv->trans_step_enable = 
			gtk_toggle_button_get_active (
				GTK_TOGGLE_BUTTON (s->priv->w_trans_step_enable));
                
                break;
        case FOURIER_PAGE:
                break;
	case DC_SWEEP_PAGE:
		s->priv->dc_enable = 
			gtk_toggle_button_get_active (				
				GTK_TOGGLE_BUTTON (s->priv->w_dc_enable));

		if ( s->priv->dc_vin1 ) g_free (s->priv->dc_vin1);
		s->priv->dc_vin1 =
			g_strdup (
				gtk_entry_get_text (
					GTK_ENTRY (
						GTK_COMBO (s->priv->w_dc_vin1)->entry)));
		
		if ( s->priv->dc_start1 ) g_free (s->priv->dc_start1);
		s->priv->dc_start1 = 
			g_strdup (
				gtk_entry_get_text (
					GTK_ENTRY (s->priv->w_dc_start1)));

		if ( s->priv->dc_stop1 ) g_free (s->priv->dc_stop1);
		s->priv->dc_stop1 = 
			g_strdup (
				gtk_entry_get_text (
					GTK_ENTRY (s->priv->w_dc_stop1)));

		if ( s->priv->dc_step1 ) g_free (s->priv->dc_step1);
		s->priv->dc_step1 = 
			g_strdup (
				gtk_entry_get_text (
					GTK_ENTRY (s->priv->w_dc_step1)));

		s->priv->dc_vin2 =
			g_strdup (
				gtk_entry_get_text (
					GTK_ENTRY (
						GTK_COMBO (s->priv->w_dc_vin2)->entry)));
		
		if ( s->priv->dc_start2 ) g_free (s->priv->dc_start2);
		s->priv->dc_start2 = 
			g_strdup (
				gtk_entry_get_text (
					GTK_ENTRY (s->priv->w_dc_start2)));

		if ( s->priv->dc_stop2 ) g_free (s->priv->dc_stop2);
		s->priv->dc_stop2 = 
			g_strdup (
				gtk_entry_get_text (
					GTK_ENTRY (s->priv->w_dc_stop2)));

		if ( s->priv->dc_step2 ) g_free (s->priv->dc_step2);
		s->priv->dc_step2 = 
			g_strdup (
				gtk_entry_get_text (
					GTK_ENTRY (s->priv->w_dc_step2)));


		break;
	case AC_PAGE:
		s->priv->ac_enable = 
			gtk_toggle_button_get_active (
				GTK_TOGGLE_BUTTON (s->priv->w_ac_enable));

		if (s->priv->ac_type ) g_free (s->priv->ac_type);
		s->priv->ac_type = 
			g_strdup (
				gtk_entry_get_text (
					GTK_ENTRY( s->priv->w_ac_type)));

		if ( s->priv->ac_npoints ) g_free (s->priv->ac_npoints);
		s->priv->ac_npoints = g_strdup (
			gtk_entry_get_text (
				GTK_ENTRY( s->priv->w_ac_npoints)));

		if ( s->priv->ac_start ) g_free (s->priv->ac_start);
		s->priv->ac_start = g_strdup (
			gtk_entry_get_text (
				GTK_ENTRY( s->priv->w_ac_start)));

		if ( s->priv->ac_stop ) g_free (s->priv->ac_stop);
		s->priv->ac_stop = g_strdup (
			gtk_entry_get_text (
				GTK_ENTRY( s->priv->w_ac_stop)));

		break;
/*        case TEMPERATURE_PAGE:
                break;
*/
        case OPTIONS_PAGE:              

                /* get the options */
                get_options_from_list (s);
                break;
        default:
                break;
	}
}

static int 
close_callback (GtkWidget *widget, SimSettings *s)
{
	if (s->priv->trans_enable) {
		if (sim_settings_get_trans_start (s) >= sim_settings_get_trans_stop (s)) {
			oregano_error (_("Transient analysis:\n\n"
					 "Start time must be less than stop time."));
			return TRUE;
		}
	}

	s->pbox = NULL;
	return FALSE;
}

static int 
delete_event_cb (GtkWidget *widget, GdkEvent *event, gpointer data)
{
	return FALSE;
}

SimSettings *
sim_settings_new (Schematic *sm)
{
	SimSettings *s;

	s = g_new0 (SimSettings, 1);
	s->sm = sm;

	s->priv = g_new0 (SimSettingsPriv, 1);

	/*
	 * Set some default settings.
	 */
	/*  transient   */
	s->priv->trans_enable = TRUE;
	s->priv->trans_start = g_strdup ("0 s");
	s->priv->trans_stop = g_strdup ("5 ms");
	s->priv->trans_step = g_strdup ("0.1 ms");
	s->priv->trans_step_enable = FALSE;

	/*  ac   */
	s->priv->ac_enable = FALSE;
	s->priv->ac_type   = g_strdup ("DEC");
	s->priv->ac_npoints= g_strdup ("50");
	s->priv->ac_start  = g_strdup ("1");
	s->priv->ac_stop   = g_strdup ("1 Meg");

	/*  dc   */
	s->priv->dc_enable = FALSE;
	s->priv->dc_vin1   = g_strdup ("");
	s->priv->dc_start1 = g_strdup ("");
	s->priv->dc_stop1  = g_strdup ("");
	s->priv->dc_step1  = g_strdup ("");
	s->priv->dc_vin2   = g_strdup ("");
	s->priv->dc_start2 = g_strdup ("");
	s->priv->dc_stop2  = g_strdup ("");
	s->priv->dc_step2  = g_strdup ("");
	
	/*   fourier   */
	s->priv->four_enable = FALSE;

        
        s->priv->options=0;

	return s;
}

gboolean
sim_settings_get_trans (SimSettings *sim_settings)
{
	return sim_settings->priv->trans_enable;
}

gdouble
sim_settings_get_trans_start (SimSettings *sim_settings)
{
	gchar *text = sim_settings->priv->trans_start;
	return oregano_strtod (text, 's');
}

gdouble
sim_settings_get_trans_stop (SimSettings *sim_settings)
{
	gchar *text = sim_settings->priv->trans_stop;
	return oregano_strtod (text, 's');
}

gdouble
sim_settings_get_trans_step (SimSettings *sim_settings)
{
	gchar *text = sim_settings->priv->trans_step;
	return oregano_strtod (text, 's');
}

gdouble
sim_settings_get_trans_step_enable (SimSettings *sim_settings)
{
	return sim_settings->priv->trans_step_enable;
}

void
sim_settings_set_trans (SimSettings *sim_settings, gboolean enable)
{
	sim_settings->priv->trans_enable = enable;
}

void
sim_settings_set_trans_start (SimSettings *sim_settings, gchar *str)
{
	if (sim_settings->priv->trans_start)
		g_strdup (sim_settings->priv->trans_start);
	sim_settings->priv->trans_start = g_strdup (str);
}

void
sim_settings_set_trans_stop (SimSettings *sim_settings, gchar *str)
{
	if (sim_settings->priv->trans_stop)
		g_strdup (sim_settings->priv->trans_stop);
	sim_settings->priv->trans_stop = g_strdup (str);
}

void
sim_settings_set_trans_step (SimSettings *sim_settings, gchar *str)
{
	if (sim_settings->priv->trans_step)
		g_strdup (sim_settings->priv->trans_step);
	sim_settings->priv->trans_step = g_strdup (str);
}

void
sim_settings_set_trans_step_enable (SimSettings *sim_settings, gboolean enable)
{
	sim_settings->priv->trans_step_enable = enable;
}

gboolean   sim_settings_get_ac (SimSettings *sim_settings) {
	return sim_settings->priv->ac_enable;
}
gchar     *sim_settings_get_ac_type (SimSettings *sim_settings) {
	return sim_settings->priv->ac_type;
}
gint       sim_settings_get_ac_npoints (SimSettings *sim_settings) {
	return atoi (sim_settings->priv->ac_npoints);
}
gdouble    sim_settings_get_ac_start (SimSettings *sim_settings) {
	return oregano_strtod (sim_settings->priv->ac_start, 's');
}
gdouble    sim_settings_get_ac_stop (SimSettings *sim_settings) {
	return oregano_strtod (sim_settings->priv->ac_stop,'s');
}   

void  sim_settings_set_ac (SimSettings *sim_settings,gboolean enable) {
	sim_settings->priv->ac_enable = enable;
}       
void  sim_settings_set_ac_type(SimSettings *sim_settings,gchar *str) {
	if ( sim_settings->priv->ac_type ) 
		g_free ( sim_settings->priv->ac_type );	
	sim_settings->priv->ac_type = g_strdup (str);		
}   
void  sim_settings_set_ac_npoints(SimSettings *sim_settings,gchar *str) {
	if ( sim_settings->priv->ac_npoints ) 
		g_free ( sim_settings->priv->ac_npoints );	
	sim_settings->priv->ac_npoints = g_strdup (str);		

}
void sim_settings_set_ac_start (SimSettings *sim_settings,gchar *str) 
{
	if (sim_settings->priv->ac_start) 
		g_free (sim_settings->priv->ac_start);	
	sim_settings->priv->ac_start = g_strdup (str);		
}  

void sim_settings_set_ac_stop(SimSettings *sim_settings,gchar *str) 
{
	if (sim_settings->priv->ac_stop) 
		g_free (sim_settings->priv->ac_stop);	
	sim_settings->priv->ac_stop = g_strdup (str);		
}   
 
gboolean sim_settings_get_dc (SimSettings *sim_settings) 
{
	return sim_settings->priv->dc_enable;
}

gchar     *sim_settings_get_dc_vsrc (SimSettings *sim_settings,gint i) 
{
	gchar *tmp = (i==0 ? sim_settings->priv->dc_vin1 : sim_settings->priv->dc_vin2);
	return (tmp && *tmp ? tmp : NULL);	
}

gdouble sim_settings_get_dc_start (SimSettings *sim_settings, gint i) 
{
	return oregano_strtod ( i==0
				? sim_settings->priv->dc_start1
				: sim_settings->priv->dc_start2, 's');
}

gdouble sim_settings_get_dc_stop (SimSettings *sim_settings,gint i) 
{
	return oregano_strtod ( i==0
				? sim_settings->priv->dc_stop1
				: sim_settings->priv->dc_stop2, 's');
}

gdouble    sim_settings_get_dc_step  (SimSettings *sim_settings,gint i) 
{
	return oregano_strtod ( i==0
				? sim_settings->priv->dc_step1
				: sim_settings->priv->dc_step2, 's');
}

void  sim_settings_set_dc (SimSettings *sim_settings, gboolean enable) 
{
	sim_settings->priv->dc_enable = enable;
}

void  sim_settings_set_dc_vsrc (SimSettings *sim_settings, gint i, gchar *str) 
{
	gchar **val = (i==0 
		       ? &sim_settings->priv->dc_vin1
		       : &sim_settings->priv->dc_vin2);
	if ( *val ) g_free (*val);
	*val = g_strdup (str);
}

void  sim_settings_set_dc_start (SimSettings *sim_settings, gint i, gchar *str) {
	gchar **val = (i==0 
		       ? &sim_settings->priv->dc_start1
		       : &sim_settings->priv->dc_start2);
	if ( *val ) g_free (*val);
	*val = g_strdup (str);
}

void  sim_settings_set_dc_stop  (SimSettings *sim_settings, gint i, gchar *str) 
{
	gchar **val = (i==0 
		       ? &sim_settings->priv->dc_stop1
		       : &sim_settings->priv->dc_stop2);
	if (*val) g_free (*val);
	*val = g_strdup (str);
}

void  sim_settings_set_dc_step  (SimSettings *sim_settings, gint i, gchar *str) 
{
	gchar **val = (i==0 
		       ? &sim_settings->priv->dc_step1
		       : &sim_settings->priv->dc_step2);
	if (*val) g_free (*val);
	*val = g_strdup (str);
}

void
sim_settings_show (GtkWidget *widget, SchematicView *sv)
{
        int i;
	GtkWidget *w, *pbox;
        GtkCList *opt_list;
	GladeXML *gui;
	SimSettings *s;
	Schematic *sm;

        GList *list;
	GList *items,*sources=NULL,*ltmp;

	g_return_if_fail (sv != NULL);

	sm = schematic_view_get_schematic (sv);
	s = schematic_get_sim_settings (sm);

	/* Only allow one instance of the property box per schematic. */
	if (s->pbox){
		gdk_window_raise (GTK_WIDGET (s->pbox)->window);
 		return;
	}		

	if (!g_file_exists (OREGANO_GLADEDIR "/sim-settings.glade")) {
		oregano_error (_("Could not create simulation settings dialog."));
		return;
	}

	gui = glade_xml_new (OREGANO_GLADEDIR "/sim-settings.glade", "toplevel");
	if (!gui) {
		oregano_error (_("Could not create simulation settings dialog."));
		return;
	}

	w = glade_xml_get_widget (gui, "toplevel");
	if (!w) {
		oregano_error (_("Could not create simulation settings dialog."));
		return;
	}

	pbox = w;
	s->pbox = GNOME_PROPERTY_BOX (pbox);

	gtk_signal_connect (GTK_OBJECT (pbox), "delete_event",
			    GTK_SIGNAL_FUNC (delete_event_cb), s);

        /*  Prepare options list   */
        s->priv->w_opt_value = GTK_ENTRY (glade_xml_get_widget (gui, "opt_value"));
        opt_list = s->priv->w_opt_list  = GTK_CLIST (glade_xml_get_widget (gui, "option_list"));
        gtk_clist_column_titles_passive (opt_list);
        gtk_clist_set_column_justification (opt_list, 1, GTK_JUSTIFY_RIGHT);
        for (i = 0; default_options[i].name; i++)
                gtk_clist_append (opt_list, (gchar**)&default_options[i]);
        
        /* Set the optinos already stored */
        list = s->priv->options;
        while (list) {
                SimOption *so = list->data;
                if (so)
                        set_options_in_list (so->name,
                                             so->value,
                                             opt_list);
                list = list->next;
        }
        
        gtk_signal_connect (GTK_OBJECT (opt_list), "select_row",
                            GTK_SIGNAL_FUNC (select_opt_callback), s);

        w = glade_xml_get_widget (gui, "opt_accept");
        gtk_signal_connect (GTK_OBJECT(w),"clicked",
                            GTK_SIGNAL_FUNC (option_setvalue), s);
        w = glade_xml_get_widget (gui, "opt_remove");
        gtk_signal_connect (GTK_OBJECT(w),"clicked",
                            GTK_SIGNAL_FUNC (option_remove), s);

	/* Setup callbacks. */
	gtk_signal_connect (GTK_OBJECT (pbox), "apply",
			    GTK_SIGNAL_FUNC (apply_callback), s);
	gtk_signal_connect (GTK_OBJECT (pbox), "close",
			    GTK_SIGNAL_FUNC (close_callback), s);

	/*  Transient   */
	w = glade_xml_get_widget (gui, "trans_start");
	if (s->priv->trans_start)
		gtk_entry_set_text (GTK_ENTRY (w), s->priv->trans_start);
	s->priv->w_trans_start = w;
	gtk_signal_connect (GTK_OBJECT (w), "changed",
			    GTK_SIGNAL_FUNC (entry_changed_cb), s);

	w = glade_xml_get_widget (gui, "trans_stop");
	if (s->priv->trans_stop)
		gtk_entry_set_text (GTK_ENTRY (w), s->priv->trans_stop);
	s->priv->w_trans_stop = w;
	gtk_signal_connect (GTK_OBJECT (w), "changed",
			    GTK_SIGNAL_FUNC (entry_changed_cb), s);

	w = glade_xml_get_widget (gui, "trans_step");
	if (s->priv->trans_step)
		gtk_entry_set_text (GTK_ENTRY (w), s->priv->trans_step);
	s->priv->w_trans_step = w;
	gtk_signal_connect (GTK_OBJECT (w), "changed",
			    GTK_SIGNAL_FUNC (entry_changed_cb), s);

	w = glade_xml_get_widget (gui, "trans_enable");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w), s->priv->trans_enable);
	s->priv->w_trans_enable = w;

	w = glade_xml_get_widget (gui, "trans_step_enable");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w), s->priv->trans_step_enable);
	s->priv->w_trans_step_enable = w;

	gtk_signal_connect (GTK_OBJECT (s->priv->w_trans_enable), "clicked",
			    GTK_SIGNAL_FUNC (trans_enable_cb), s);

	gtk_signal_connect (GTK_OBJECT (s->priv->w_trans_step_enable), "clicked",
			    GTK_SIGNAL_FUNC (trans_step_enable_cb), s);

	trans_enable_cb (s->priv->w_trans_enable, s);
	trans_step_enable_cb (s->priv->w_trans_step_enable, s);
	
	/* AC  */
        w = glade_xml_get_widget (gui, "ac_enable");
        gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w), 
				      s->priv->ac_enable);
	s->priv->w_ac_enable=w;
        gtk_signal_connect (GTK_OBJECT (w), "clicked",
                            GTK_SIGNAL_FUNC (ac_enable_cb), s);

	w =  glade_xml_get_widget (gui, "ac_type");
	if ( s->priv->ac_type ) 
		gtk_entry_set_text ( GTK_ENTRY (GTK_COMBO (w)->entry),
				     s->priv->ac_type
			);
	s->priv->w_ac_type = GTK_COMBO (w)->entry;
        gtk_signal_connect (GTK_OBJECT (GTK_COMBO (w)->entry), "changed",
                            GTK_SIGNAL_FUNC (entry_changed_cb), s);
	
	w = glade_xml_get_widget (gui, "ac_npoints");
	gtk_entry_set_text ( GTK_ENTRY (w), s->priv->ac_npoints);
	s->priv->w_ac_npoints = w;
        gtk_signal_connect (GTK_OBJECT (w), "changed",
                            GTK_SIGNAL_FUNC (entry_changed_cb), s);

	w = glade_xml_get_widget (gui, "ac_start");
	gtk_entry_set_text ( GTK_ENTRY (w), s->priv->ac_start);
	s->priv->w_ac_start = w;
        gtk_signal_connect (GTK_OBJECT (w), "changed",
                            GTK_SIGNAL_FUNC (entry_changed_cb), s);

	w = glade_xml_get_widget (gui, "ac_stop");
	gtk_entry_set_text ( GTK_ENTRY (w), s->priv->ac_stop);
	s->priv->w_ac_stop = w;
        gtk_signal_connect (GTK_OBJECT (w), "changed",
                            GTK_SIGNAL_FUNC (entry_changed_cb), s);

	ac_enable_cb (s->priv->w_ac_enable,s);

	/*  DC   */
	/*  Get list of sources */
	items = node_store_get_parts (schematic_get_store (sm));
	for ( ; items; items = items->next ) {
	   gchar *temp = part_get_property (items->data,"template");
	   gchar *name = part_get_property (items->data,"refdes");
	   if (temp) {
	      gchar c = tolower(*temp);
	      if (c=='v' || c=='i' || 
		  c=='e' || c=='f' || c=='g' || c=='h' || c=='b' ) {
		      gchar *vsrc = g_strdup_printf ("%c_%s",*temp,name);
		      sources = g_list_prepend (sources,vsrc);
	      }
	   }
	}

        w = glade_xml_get_widget (gui, "dc_enable");
	s->priv->w_dc_enable = w;
        gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w), 
				      s->priv->dc_enable);
        gtk_signal_connect (GTK_OBJECT (w), "clicked",
                            GTK_SIGNAL_FUNC (dc_enable_cb), s);

	w = glade_xml_get_widget (gui, "dc_vin1");
	s->priv->w_dc_vin1 = w;
	if (sources) 
		gtk_combo_set_popdown_strings (GTK_COMBO (w), sources);
	gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (w)->entry),s->priv->dc_vin1);
	gtk_signal_connect (GTK_OBJECT (GTK_COMBO (w)->entry), "changed",
                            GTK_SIGNAL_FUNC (entry_changed_cb), s);

	w = glade_xml_get_widget (gui, "dc_vin2");
	s->priv->w_dc_vin2 = w;
	if (sources)
		gtk_combo_set_popdown_strings (GTK_COMBO (w),sources);
	gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (w)->entry),s->priv->dc_vin2);
	gtk_signal_connect (GTK_OBJECT (GTK_COMBO (w)->entry), "changed",
                            GTK_SIGNAL_FUNC (entry_changed_cb), s); 

	w = glade_xml_get_widget (gui, "dc_start1");
	s->priv->w_dc_start1 = w;
	gtk_entry_set_text ( GTK_ENTRY (w), s->priv->dc_start1);
        gtk_signal_connect (GTK_OBJECT (w), "changed",
                            GTK_SIGNAL_FUNC (entry_changed_cb), s);

	w = glade_xml_get_widget (gui, "dc_stop1");
	s->priv->w_dc_stop1 = w;
	gtk_entry_set_text ( GTK_ENTRY (w), s->priv->dc_stop1);
        gtk_signal_connect (GTK_OBJECT (w), "changed",
                            GTK_SIGNAL_FUNC (entry_changed_cb), s);

	w = glade_xml_get_widget (gui, "dc_step1");
	s->priv->w_dc_step1 = w;
	gtk_entry_set_text ( GTK_ENTRY (w), s->priv->dc_step1);
        gtk_signal_connect (GTK_OBJECT (w), "changed",
                            GTK_SIGNAL_FUNC (entry_changed_cb), s);

	w = glade_xml_get_widget (gui, "dc_start2");
	s->priv->w_dc_start2 = w;
	gtk_entry_set_text ( GTK_ENTRY (w), s->priv->dc_start2);
        gtk_signal_connect (GTK_OBJECT (w), "changed",
                            GTK_SIGNAL_FUNC (entry_changed_cb), s);

	w = glade_xml_get_widget (gui, "dc_stop2");
	s->priv->w_dc_stop2 = w;
	gtk_entry_set_text ( GTK_ENTRY (w), s->priv->dc_stop2);
        gtk_signal_connect (GTK_OBJECT (w), "changed",
                            GTK_SIGNAL_FUNC (entry_changed_cb), s);
	w = glade_xml_get_widget (gui, "dc_step2");
	s->priv->w_dc_step2 = w;
	gtk_entry_set_text ( GTK_ENTRY (w), s->priv->dc_step2);
        gtk_signal_connect (GTK_OBJECT (w), "changed",
                            GTK_SIGNAL_FUNC (entry_changed_cb), s);

	dc_enable_cb( s->priv->w_dc_enable,s);

	for (ltmp = sources; ltmp; ltmp = ltmp->next) 
		g_free (ltmp->data);	
	g_list_free (sources);

	/* Fourier    */
	w = glade_xml_get_widget (gui, "fourier_enable");
        gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w), 
				      s->priv->four_enable);
	s->priv->w_four_enable=w;
        gtk_signal_connect (GTK_OBJECT (w), "clicked",
                            GTK_SIGNAL_FUNC (four_enable_cb), s);

	w = glade_xml_get_widget (gui, "fourier_freq");
	s->priv->w_four_freq = w;
	gtk_signal_connect (GTK_OBJECT (w), "changed",
                            GTK_SIGNAL_FUNC (entry_changed_cb), s);

	w = glade_xml_get_widget (gui, "fourier_vout");
	s->priv->w_four_vout = w;
	gtk_signal_connect (GTK_OBJECT (w), "changed",
                            GTK_SIGNAL_FUNC (entry_changed_cb), s);

	w = glade_xml_get_widget (gui, "fourier_select_out");
	s->priv->w_four_combo = w;
	gtk_signal_connect (GTK_OBJECT (GTK_COMBO (w)->entry), "changed",
                            GTK_SIGNAL_FUNC (entry_changed_cb), s);

	w = glade_xml_get_widget (gui, "fourier_add");
	s->priv->w_four_add = w;
	gtk_signal_connect (GTK_OBJECT (w), "clicked",
                            GTK_SIGNAL_FUNC (four_add_vout_cb), s);

	four_enable_cb (s->priv->w_four_enable,s);
}

GList *
sim_settings_get_options (SimSettings *s) 
{
        return s->priv->options;
}

void sim_settings_add_option (SimSettings *s, SimOption *opt) 
{
        GList *list=s->priv->options;

        /* Remove the option if already in the list. */
        while (list) {
                SimOption *so=list->data;
                if (so && !strcmp (opt->name,so->name)) {
                        g_free (so->name);
                        g_free (so->value);     
                        g_list_remove (s->priv->options, so);
                        g_free (so);
                }
                list = list->next;
        }

        s->priv->options = g_list_append (s->priv->options, opt);
}

