/**
  @file btsdp-dbus.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2006 Nokia Corporation. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License, version 2, as
  published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

*/
#include <stdio.h>
#include <stdlib.h>
#include <glib.h>

#include <dbus/dbus.h>

#include "log.h"
#include "btsdp-bt.h"
#include "bt-dbus.h"
#include "dbus-helper.h"
#include "btsdp-dbus.h"

extern GMainLoop *event_loop;

static char *sdp_bda = NULL;

#define EXIT_TIMEOUT 5000 /* milliseconds */

static GSource *exit_timer = NULL;

static gboolean exit_cb(gpointer user_data) {
    error("btsdp running but no requests received.");
    g_main_loop_quit(event_loop);
    return FALSE;
}

static void set_exit_timer(void)
{
    if (exit_timer)
        g_source_destroy(exit_timer);
    exit_timer = g_timeout_source_new(EXIT_TIMEOUT);
    g_source_set_callback(exit_timer, exit_cb, NULL, NULL);
    (void) g_source_attach(exit_timer, NULL);
}

static void remove_exit_timer(void)
{
    if (exit_timer) {
        g_source_destroy(exit_timer);
        exit_timer = NULL;
    }
}

static void add_sdp_arg(sdp_info_t *info, DBusMessageIter *iter)
{
    DBusMessageIter sub;
    const char *name = info->name ? info->name : "";

    debug("%s, channel %u, ttl %d, \"%s\"",
            info->service, info->channel, info->ttl, info->name);

    if (!dbus_message_iter_open_container(iter, DBUS_TYPE_STRUCT, NULL, &sub))
        die("Out of memory during dbus_message_iter_open_container");

    dbus_message_iter_append_basic(&sub, DBUS_TYPE_BYTE, &info->type);
    dbus_message_iter_append_basic(&sub, DBUS_TYPE_STRING, &info->service);
    dbus_message_iter_append_basic(&sub, DBUS_TYPE_BYTE, &info->channel);
    dbus_message_iter_append_basic(&sub, DBUS_TYPE_UINT32, &info->ttl);
    dbus_message_iter_append_basic(&sub, DBUS_TYPE_STRING, &name);

    if (!dbus_message_iter_close_container(iter, &sub))
        die("Out of memory during dbus_message_iter_close_container");
}

static void send_services_reply(DBusMessage    *message,
                                DBusConnection *connection,
                                GError         *err, 
                                GSList         *services)
{
    DBusMessageIter iter, sub;
    DBusMessage *reply;

    if (err) {
        reply = new_dbus_error_gerr(message, err);
        if(!send_and_unref(connection, reply))
            error("Unable to send connect_failed error reply");
        goto out;
    }

    debug("%d supported services", g_slist_length(services));

    reply = new_dbus_method_return(message);

    dbus_message_iter_init_append(reply, &iter);
    dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &sdp_bda);

    if (!dbus_message_iter_open_container(&iter, DBUS_TYPE_ARRAY, "(ysyus)", &sub))
        die("Out of memory during dbus_message_iter_open_container");

    g_slist_foreach(services, (GFunc)add_sdp_arg, &sub);

    if (!dbus_message_iter_close_container(&iter, &sub))
        die("Out of memory during dbus_message_iter_close_container");

    if (!send_and_unref(connection, reply))
        error("Unable to send rfcomm services reply");

    g_slist_foreach(services, (GFunc)free_sdp_info, NULL);
    g_slist_free(services);

out:
    dbus_message_unref(message);
    g_free(sdp_bda);
    sdp_bda = NULL;
    g_main_loop_quit(event_loop);
}

static DBusHandlerResult get_services_request(DBusMessage    *message,
                                              DBusConnection *connection)
{
    char *bda;
    GError *err = NULL;
    DBusMessage *reply;
    DBusMessageIter iter;
    GSList *l = NULL;
    sdp_cb_data_t *data;

    remove_exit_timer();

    debug("get_services_request()");

    if (sdp_bda) {
        reply = new_dbus_error(message, BTSDP_ERROR_QUERY_IN_PROGRESS, NULL);
        if(!send_and_unref(connection, reply))
            error("Unable to send query_in_progress error reply");
        return DBUS_HANDLER_RESULT_HANDLED;
    }

    dbus_message_iter_init(message, &iter);
    /* We can't use get_dbus_iter_args because after the call the iter points on the
     * second argument if the message had two arguments, but on the third argument if 
     * the message had more than two arguments */
    if (dbus_message_iter_get_arg_type(&iter) != DBUS_TYPE_STRING) {
        error("Invalid parameters for get_services");
        if (!send_invalid_args(connection, message))
            error("Unable to send invalid_args error");
        g_main_loop_quit(event_loop);
        return DBUS_HANDLER_RESULT_HANDLED;
    }

    dbus_message_iter_get_basic(&iter, &bda);

    while (dbus_message_iter_next(&iter)) {
        char *svc;

        if (dbus_message_iter_get_arg_type(&iter) != DBUS_TYPE_STRING)
            break;

        dbus_message_iter_get_basic(&iter, &svc);
        debug("Adding string to search list: %s", svc);
        l = g_slist_append(l, svc);
    }

    data = g_new0(sdp_cb_data_t, 1);
    data->func       = send_services_reply;
    data->message    = message;
    data->connection = connection;

    dbus_message_ref(message);

    if (!get_services(bda, l, data, &err)) {
        g_slist_free(l);
        dbus_message_unref(message);
        g_free(data);
        reply = new_dbus_error_gerr(message, err);
        if(!send_and_unref(connection, reply))
            error("Unable to send connect_failed error reply");
        g_clear_error(&err);
        g_main_loop_quit(event_loop);
        return DBUS_HANDLER_RESULT_HANDLED;
    }

    g_slist_free(l);

    sdp_bda = g_strdup(bda);

    return DBUS_HANDLER_RESULT_HANDLED;
}


/* Handler for D-Bus requests */
static DBusHandlerResult btsdp_req_handler(DBusConnection     *connection,
                                           DBusMessage        *message,
                                           void               *user_data)
{
    const char *dest;

    if (dbus_message_get_type(message) != DBUS_MESSAGE_TYPE_METHOD_CALL)
        return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;

    dest = dbus_message_get_destination(message);
    if (!g_str_equal(dest, BTSDP_SERVICE)) {
        debug("Received D-Bus message not addressed to me.");
        return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    }

    if (dbus_message_is_method_call(message,
                                    BTSDP_REQ_INTERFACE,
                                    BTSDP_GET_SERVICES_REQ))
        return get_services_request(message, connection);

    return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

static DBusObjectPathVTable btsdp_req_vtable = {
    .message_function    = btsdp_req_handler,
    .unregister_function = NULL
};

/* Create bindings for D-Bus handlers */
void init_dbus_handlers(DBusConnection *connection)
{
    dbus_bool_t ret;
    ret = dbus_connection_register_object_path(connection,
                                               BTSDP_REQ_PATH,
                                               &btsdp_req_vtable,
                                               NULL);
    if (ret == FALSE)
        error("register_object_path(req) failed");

    set_exit_timer();
}

void destroy_dbus_handlers(DBusConnection *connection)
{
    dbus_connection_unregister_object_path(connection, BTSDP_REQ_PATH);
}
