/**
  @file sdp-query.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2006 Nokia Corporation. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License, version 2, as
  published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

*/
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <errno.h>
#include <getopt.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <glib.h>

#include <bluetooth/bluetooth.h>
#include <bluetooth/hci.h>
#include <bluetooth/hci_lib.h>
#include <bluetooth/sdp.h>
#include <bluetooth/sdp_lib.h>

#include "log.h"

typedef struct {
    int ch;
} SdpServiceInfo;

struct uuid16_map_t {
    const char *svc;
    uint16_t uuid;
} uuid16_map[] = {
    { "SPP", SERIAL_PORT_SVCLASS_ID    },
    { "DUN", DIALUP_NET_SVCLASS_ID     },
    { "FTP", OBEX_FILETRANS_SVCLASS_ID },
    { "OPP", OBEX_OBJPUSH_SVCLASS_ID   },
    { "SAP", SAP_SVCLASS_ID            },
    { "HID", HID_SVCLASS_ID            },
    { "HSP", HEADSET_SVCLASS_ID        },
    { NULL,  0 }
};

static const unsigned char
nokia_ftp_uuid[16] = { 0x00, 0x00, 0x50, 0x05, 0x00, 0x00, 0x10, 0x00,
                       0x80, 0x00, 0x00, 0x02, 0xee, 0x00, 0x00, 0x01 };

static const char *svc_from_uuid(uuid_t *uuid)
{
    struct uuid16_map_t *u16e;

    switch (uuid->type) {
        case SDP_UUID16:
            for (u16e = uuid16_map; u16e->svc != NULL; u16e++) {
                if (u16e->uuid == uuid->value.uuid16)
                    return u16e->svc;
            }
            return NULL;
        case SDP_UUID32:
            for (u16e = uuid16_map; u16e->svc != NULL; u16e++) {
                if (u16e->uuid == uuid->value.uuid32)
                    return u16e->svc;
            }
            return NULL;
        case SDP_UUID128:
            if (memcmp(&uuid->value.uuid128, nokia_ftp_uuid, 16) == 0)
                return "NFTP";
        default:
           return NULL; 
    }
}

static uuid_t *uuid_from_svc(uuid_t *uuid, const char *svc)
{
    int len = strlen(svc);
    struct uuid16_map_t *u16e;

    if (len < 2)
        return NULL;

    if (strncmp(svc, "0x", 2) == 0) {
        if (len == 6)
            return sdp_uuid16_create(uuid, (uint16_t)strtol(svc, NULL, 0));
        else if (len == 10)
            return sdp_uuid32_create(uuid, strtol(svc, NULL, 0));
        else
            return NULL;
    }

    /* SDP_UUID128 */
    if (len == 36 && svc[8] == '-' && svc[13] == '-' && svc[18] == '-' && svc[23] == '-') {
        uint8_t val[16];
        uint32_t data0, data4;
        uint16_t data1, data2, data3, data5;

        sscanf(svc, "%08x-%04hx-%04hx-%04hx-%08x%04hx",
                &data0, &data1, &data2, &data3, &data4, &data5);

        data0 = htonl(data0);
        data1 = htons(data1);
        data2 = htons(data2);
        data3 = htons(data3);
        data4 = htonl(data4);
        data5 = htons(data5);

        memcpy(&val[0], &data0, 4);
        memcpy(&val[4], &data1, 2);
        memcpy(&val[6], &data2, 2);
        memcpy(&val[8], &data3, 2);
        memcpy(&val[10], &data4, 4);
        memcpy(&val[14], &data5, 2);

        return sdp_uuid128_create(uuid, val);
    }

    for (u16e = uuid16_map; u16e->svc != NULL; u16e++) {
        if (g_str_equal(u16e->svc, svc))
            return sdp_uuid16_create(uuid, u16e->uuid);
    }

    if (g_str_equal(svc, "NFTP"))
        return sdp_uuid128_create(uuid, nokia_ftp_uuid);

    return NULL;
}

static gboolean uuid_to_service_name(uuid_t *uuid, char *name, int len)
{
    const char *svc;

    svc = svc_from_uuid(uuid);

    if (svc) {
        strncpy(name, svc, len);
        return TRUE;
    }

    switch (uuid->type) {
        case SDP_UUID16:
            snprintf(name, len, "0x%04X", uuid->value.uuid16);
            break;
        case SDP_UUID32:
            snprintf(name, len, "0x%08X", uuid->value.uuid32);
            break;
        default:
            if (sdp_uuid2strn(uuid, name, len) < 0)
                return FALSE;
    }

    return TRUE;
}

static void add_new_service(sdp_record_t *rec)
{
    sdp_list_t *classes, *protos = NULL;
    uuid_t uuid;
    int port = -1;
    uint32_t ttl = 0;
    char svc[37], name[64] = "", type = 0;

    if (sdp_get_service_classes(rec, &classes) < 0)
        goto out;

    memcpy(&uuid, classes->data, sizeof(uuid));

    if (!uuid_to_service_name(&uuid, svc, sizeof(svc)))
        goto out;

    if (sdp_get_access_protos(rec, &protos) < 0)
        goto out;

    port = sdp_get_proto_port(protos, RFCOMM_UUID);
    if (port > 0) {
        type = 'r';
        goto found;
    }

    port = sdp_get_proto_port(protos, L2CAP_UUID);
    if (port > 0)
        type = 'l';
    else
        goto out;

found:
    sdp_get_service_ttl(rec, &ttl);
    sdp_get_service_name(rec, name, sizeof(name));

    printf("%c|%s|%d|%d|%s\n", type, svc, port, ttl, name);
    fflush(stdout);

out:
    if (classes)
        sdp_list_free(classes, free);
    if (protos) {
        sdp_list_foreach(protos, (sdp_list_func_t)sdp_list_free, NULL);
        sdp_list_free(protos, NULL);
    }
}

static void get_sdp_info(uuid_t *uuid, sdp_session_t *sess)
{
    uint32_t range = 0x0000ffff;
    sdp_list_t *attrid, *search, *rsp, *next;

    search = sdp_list_append(0, uuid);

    if (sdp_service_search_req(sess, search, 40, &rsp)) {
        error("Service Discovery failed: %s\n", g_strerror(errno));
        sdp_list_free(search, 0);
        return;
    }

    attrid = sdp_list_append(0, &range);

    for (; rsp; rsp = next) {
        uint32_t *handle;
        sdp_record_t *rec;

        handle = rsp->data;

        rec = sdp_service_attr_req(sess, *handle, SDP_ATTR_REQ_RANGE, attrid);
        if (rec) {
            add_new_service(rec);
            sdp_record_free(rec);
	}

        next = rsp->next;
        free(handle);
        free(rsp);
    }

    sdp_list_free(attrid, 0);
    sdp_list_free(search, 0);
}

static void query_error(void)
{
    printf("Error\n");
    fflush(stdout);
}

int main (int argc, char *argv[])
{
    int i;
    uuid_t *uuid;
    bdaddr_t ba;
    sdp_session_t *sess;
    GPtrArray *arr;
   
    open_log("sdp-query", TRUE);

    arr = g_ptr_array_new();

    if (argc < 2) {
        query_error();
        die("Not enough arguments");
    }

    if (str2ba(argv[1], &ba) < 0) {
        query_error();
        die("Invalid BDA");
    }

    for (i = 2; i < argc; i++) {
        uuid = g_new(uuid_t, 1);

        if (!uuid_from_svc(uuid, argv[i])) {
            error("Invalid service: %s", argv[i]);
            g_free(uuid);
            break;
        }

        g_ptr_array_add(arr, uuid);
    }

    sess = sdp_connect(BDADDR_ANY, &ba, SDP_RETRY_IF_BUSY);
    if (sess == NULL) {
        query_error();
        die("SDP connect failed");
    }

    if (arr->len == 0) {
        uuid = g_new(uuid_t, 1);
        sdp_uuid16_create(uuid, PUBLIC_BROWSE_GROUP);
        g_ptr_array_add(arr, uuid);
    }

    g_ptr_array_foreach(arr, (GFunc)get_sdp_info, sess);

    sdp_close(sess); 

    exit(EXIT_SUCCESS);
}

