/* ----------------------------------------------------------------------------
 * pbb_lib.c
 * main funtions of pbb library
 *
 * Copyright 2002 Matthias Grimm
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 * ----------------------------------------------------------------------------*/

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <syslog.h>

#include "pbb.h"

struct libbase libdata;     /* library basic data */

/*---------------------------------------------------------------------
 * void init_libpbbuttons ()
 *
 * This function initialises the basic library structure and
 * have to be called before any library funcion is used.
 *---------------------------------------------------------------------*/

void
init_libpbb ()
{
	struct libbase *base = &libdata;
	int n;

	base->serverport = 0;
	base->msgport = 0;
	base->ptagcount = 0;
	base->count = 0;
	base->daemon = PBBDS_NONE; /* don't detach by default */
	base->filtermode = 0;  /* no filtering by default */
	base->uid = geteuid();   /* uid from current process */
	base->pidfile[0] = '\0';
	base->rc = 0;
	base->timeforcmd = 10;   /* 10 seconds by default */
	taglist_init (base->ptags);

	for (n=0; n < MAXCLIENTS; n++)
		base->client[n] = 0;
}

void
exit_libpbb ()
{
	struct libbase *base = &libdata;

	if (base->daemon == PBBDS_CHILD)
		unlink (base->pidfile);
}

/* This function is for compatibility reasons. It should
 * be called as early as possible to show the library that
 * this process becomes a daemon so that error messages
 * could be redirected into a logfile instead of printing
 * them to console. It also saves the pidfile name for
 * later use.
 */
int
prepare_daemon (char *name, char *pidfile, int flags)
{
	struct libbase *base = &libdata;
	int err;

	err = check_devorfile(pidfile, TYPE_FILE);
	/* If the caller set PBBDF_FORCE, it wouldn't matter
	   if the pidfile already exists. The caller takes
	   care about it.
	 */
	if (err == 0 && flags == PBBDF_FORCE)
		err = E_NOEXIST;

	if (err == E_NOEXIST) {
		strncpy (base->pidfile, pidfile, sizeof(base->pidfile));
		base->daemon = PBBDS_PREPARE;
		openlog (name, 0, LOG_DAEMON);
	}
	return err;
}

int
create_daemon ()
{
	struct libbase *base = &libdata;
	int pid = 0;
	FILE *fh;

	if (base->daemon == PBBDS_PREPARE) {
		if ((pid = fork()) == 0) {
			base->daemon = PBBDS_CHILD; /* this is the child now */
			fclose(stdin);
			fclose(stdout);
			fclose(stderr);
			chdir("/");
			setsid();
		} else {
			base->daemon = PBBDS_PARENT; /* this is the parent now */
			/* the parent process have to save the pid */
			if ((fh = fopen (base->pidfile, "w")) != 0) {
				fprintf (fh, "%i\n", pid);
				fclose (fh);
			}
		}
	}
	return pid;
}

