/*	String_Utilities

PIRL CVS ID: String_Utilities.java,v 1.7 2012/04/16 06:15:35 castalia Exp

Copyright (C) 2004-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package PIRL.Strings;


/**	The <I>String_Utilities</I> are a collection of static functions
   that operate on String objects to provide various commonly used
   operations.
<p>
	@author		Bradford Castalia, UA/PIRL
	@version	1.7
*/
public class String_Utilities
{
/**	Class identification with revision number and date.
*/
public static final String
	ID = "PIRL.Strings.String_Utilities (1.7 2012/04/16 06:15:35)";
	

//  DEBUG control.
private static final int
	DEBUG_OFF			= 0,
	DEBUG_UTILITIES		= 1 << 1,
	DEBUG_ALL			= -1,

	DEBUG       		= DEBUG_OFF;


private String_Utilities ()
{}

/**	Trims all delimiter characters from both ends of a string.
<p>
	@param	string		The String to trim.
	@param	delimiter	The character to be trimmed off the ends.
	@return	The string, sans leading and trailing characers.
*/
public static String Trim
	(
	String	string,
	char	delimiter
	)
{
if (string == null)
	return string;
int
	end;
for (end = string.length () - 1;
	 end >= 0;
	 end--)
	if (string.charAt (end) != delimiter)
		break;
end++;
int
	start;
for (start = 0;
	 start < end;
	 start++)
	if (string.charAt (start) != delimiter)
		break;
return string.substring (start, end);
}

/**	Gets that portion of a String preceeding a delimiter.
<p>
	@param	string	The String from which to get the leading portion.
	@param	delimiter	The delimiter character.
	@return	The portion of the String before the delimiter, or null
		if the delimiter is not present.
*/
public static String Preceeding
	(
	String	string,
	char	delimiter
	)
{
if ((DEBUG & DEBUG_UTILITIES) != 0)
	System.out.println
		(">>> Preceeding: '" + delimiter + "' in " + string);
if (string != null)
	{
	int
		index;
	if ((index = string.indexOf (delimiter)) >= 0)
		{
		string = string.substring (0, index);
		if (string.length () == 0)
			string = null;
		}
	else
		string = null;
	}
if ((DEBUG & DEBUG_UTILITIES) != 0)
	System.out.println ("    preceeding: " + string);
return string;
}

/**	The default number of decimal digits used by the {@link
	#Amount_Magnitude(long, int)} function.
*/
public static final int
	DEFAULT_DECIMAL_DIGITS	= 2,
	MAX_DECIMAL_DIGITS		= 12;

private static final char
	MAGNITUDE[] = {' ', 'K', 'M', 'G', 'T', 'P'};

/**	Gets that portion of a String following a delimiter.
<p>
	@param	string	The String from which to get the trailing portion.
	@param	delimiter	The delimiter character.
	@return	The portion of the String following the delimiter, or null
		if the delimiter is not present.
*/
public static String Following
	(
	String	string,
	char	delimiter
	)
{
if ((DEBUG & DEBUG_UTILITIES) != 0)
	System.out.println (">>> Following: '" + delimiter + "' in " + string);
if (string != null)
	{
	int
		index;
	if ((index = string.indexOf (delimiter)) >= 0)
		{
		string = string.substring (++index);
		if (string.length () == 0)
			string = null;
		}
	else
		string = null;
	}
if ((DEBUG & DEBUG_UTILITIES) != 0)
	System.out.println ("    following: " + string);
return string;
}

/**	Provide a binary magnitude String representation of an amount.
<p>
	An amount that is less than 1024 is returned as its String
	representation. Otherwise it is divided by 1024 until its value is
	less than 1024, up to a maximum of five times. The number of
	divisions determines the binary magnitude of the amount, the first
	character of which is then appended to the String representation of
	the reduced value:
<p><ol>
<li>Kilo
<li>Mega
<li>Giga
<li>Tera
<li>Peta
</ol><p>
	Negative values are treated as positive values with the negative
	sign applied to the final result.
<p>
	@param	amount	The amount to be represented.
	@param	decimal_digits	The number of decimal digits in the reduced
		amount representation. If this is negative the {@link
		#DEFAULT_DECIMAL_DIGITS} will be used. The number of digits will
		be limited to a maximum of {@link #MAX_DECIMAL_DIGITS}.
	@return	A String representation of the amount with a binary magnitude
		suffix character if the amount was not less than 1024.
*/
public static String Amount_Magnitude
	(
	long	amount,
	int		decimal_digits
	)
{
if (amount <  1024 &&
	amount > -1024)
	return String.valueOf (amount);

if (decimal_digits < 0)
	decimal_digits = DEFAULT_DECIMAL_DIGITS;
else
if (decimal_digits > MAX_DECIMAL_DIGITS)
	decimal_digits = MAX_DECIMAL_DIGITS;
String
	format = "%." + decimal_digits + 'f';

int
	magnitude;
long
	number = amount;
if (number < 0)
	number = -number;

for (magnitude = 0,
	 decimal_digits = 1;
	 magnitude < MAGNITUDE.length &&
	 number >= 1024;
	 magnitude++,
	 number >>>= 10,
	 decimal_digits *= 1024);

return
	String.format (format, (double)amount / decimal_digits)
		+ MAGNITUDE[magnitude];
}

/**	Provide a binary magnitude String representation of an amount.
<p>
	The {@link #DEFAULT_DECIMAL_DIGITS} number of decimal digits in the
	reduced amount representation will be used.
<p>
	@param	amount	The amount to be represented.
	@return	A String representation of the amount with a binary magnitude
		suffix character if the amount was not less than 1024.
	@see	#Amount_Magnitude(long, int)
*/
public static String Amount_Magnitude
	(
	long	amount
	)
{return Amount_Magnitude (amount, -1);}


}
