/****h* Docuwala/ROBODoc
 * FUNCTION
 *   ROBODoc is intended to be a replacement for the original AutoDocs
 *   program.  ROBODoc will extract the procedure comment headers
 *   from a source file, and put them into a separate documentation file.
 *   For further information read the documentation in the archive.
 * COPYRIGHT
 *   Copyright (C) 1994-2004
 *   Frans Slothouber, Petteri Kettunen, and Jacco van Weert.
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA  02111-1307  USA
 * AUTHOR
 *   See AUTHORS in the archive.
 * CREATION DATE
 *   20-Dec-94  Jacco van Weert.
 * MODIFICATION HISTORY
 *   See ChangeLog in the archive
 *   Latest version can be found on
 *     http://robodoc.sourceforge.net
 *     http://www.xs4all.nl/~rfsber/Robo/
 *     http://freshmeat.net/
 * BUGS
 *   Other bugs?
 *     Catch them in a jar and send them to rfsber -(at)- xs4all.nl
 *
 ****
 * $Id: robodoc.c,v 1.57 2003/12/30 17:39:36 gumpu Exp $
 */

#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include "robodoc.h"
#include "globals.h"
#include "headers.h"
#include "util.h"
#include "links.h"
#include "part.h"
#include "analyser.h"
#include "generator.h"
#include "document.h"
#include "directory.h"
#include "roboconfig.h"

#ifdef DMALLOC
#include <dmalloc.h>
#endif

/* Functions local to this file. */

T_RB_DocType        RB_Find_DocType( void );
long                RB_Find_Actions( struct RB_Document *document );
int                 RB_Find_Option( char *option );
static int          RB_PathBegin_Check( char *path );
static void         RB_Path_Check( char *sourcepath, char *docpath );
static int          RB_Stat_Path( char required, char *path );
static int          RB_Check_Options( void );
char               *RB_Find_Parameterized_Option( char *actionname );
char               *RB_Find_In_Argv_Parameterized_Option( int argc, char** argv, char* optionname );
static char*        RB_General_Find_Parameterized_Option( int n, char** options, char* optionname );
static char*        RB_Fix_Path( char* path );


char                RB_VER[] =
    "$VER: robodoc " VERSION " (" __DATE__
    ") (c) by Maverick Software Development 1994-2004";


/****h* ROBODoc/UserInterface
 * FUNCTION
 *   This module contains functions to parse the
 *   command line and inform the user about any errors.
 *****
 */


/****v* UserInterface/use [3.0h]
 * NAME
 *   use -- usage string
 * FUNCTION
 *   Inform the user how to use ROBODoc.
 * AUTHOR
 *   Koessi
 * SOURCE
 */

char                use[] =
    "ROBODoc Version " VERSION "    autodocs formatter ($Revision: 1.57 $)\n"
    "(c) 1994-2004 Frans Slothouber, Petteri Kettunen, and Jacco van Weert\n"
    "ROBODoc comes with ABSOLUTELY NO WARRANTY.\n"
    "This is free software, and you are welcome to redistribute it\n"
    "under certain conditions; type `robodoc -c' for details.\n" "\n";
char                use_usage[] =
    "Usage:\n"
    "   robodoc --src <directory> --doc <directory> --multidoc [type] [options]\n"
    "   robodoc --src <directory> --doc <file>      --singledoc [type] [options]\n"
    "   robodoc --src <file>      --doc <file>      --singlefile [type] [options]\n"
    "Type:\n" "   --html, --rtf, --latex, --ascii, --dbxml\n"; 
char                use_options1[] =
    "Options:\n"
    "   --sections       Add sections and subsections.\n"
    "   --internal       Also include internal headers.\n"
    "   --internalonly   Only include internal headers.\n"
    "   --nosource       Do not include SOURCE items.\n"
    "   --index          Add an index.\n"
    "   --toc            Add a table of contents.\n"
    "   --folds          Use fold marks.\n";
char                use_options2[] =
    "   --tell           ROBODoc will tell you what it is doing.\n"
    "   --nodesc         Do not descent into subdirectories.\n"
    "   --ext EXTENSION  Set extension for generated files.\n"
    "   --cmode          Use ANSI C grammar in source items (html only).\n"
    "   --charset NAME   Add character encoding information (html only).\n"
    "   --tab NUMBER     Set the tab size.\n"
    "   --version        Print version info and exit.\n";
char                use_options3[] =
    "   --css            Specify the stylesheet to use.\n"
    "   --rc             Specify an alternate configuration file.\n"
    "   --lock           Recognize only one header marker per file.\n";
char                use_authors[] =
    "Authors/Contributors: Frans Slothouber <rfsber@xs4all.nl>, Jacco van Weert,\n"
    "  Petteri Kettunen, Bernd Koesling, Thomas Aglassinger, Anthon Pang,\n"
    " Stefan Kost, David Druffner, Sasha Vasko, Kai Hofmann.\n";

/********/

char                short_use[] =
    "Usage:\n"
    "   robodoc --src <directory> --doc <directory> --multidoc   [type] [options]\n"
    "   robodoc --src <directory> --doc <file>      --singledoc  [type] [options]\n"
    "   robodoc --src <file>      --doc <file>      --singlefile [type] [options]\n"
    "Use robodoc --help for more information\n";


/****v* UserInterface/ok_options
 * NAME
 *   ok_options -- allowed command-line options
 * SYNOPSIS
 *   static char *ok_options[]
 * FUNCTION
 *   An array of all allowed command-line options.  If you add any
 *   options add its name here too.
 * SOURCE
 */

static char        *ok_options[] = {
    "--rc",
    "--src",
    "--doc",
    "--html",
    "--latex",
    "--ascii",
    "--rtf",
/*    "--troff",  TODO */
    "--dbxml",
/*    "--dbsgml",   TODO */
    "--singledoc",
    "--singlefile",
    "--multidoc",
    "--sections",
    "--internal",
    "--internalonly",
    "--toc",
    "--index",
    "--nosource",
    "--tabsize",
    "--tell",
    "--folds",
    "--nodesc",
    "--cmode",
    "--charset",
    "--ext",
    "--help",
    "--css",
    "--version",
    "-c",
    "--lock",
    ( char * ) NULL
};

/*****/



/****v* UserInterface/copying [3.1]
 * FUNCTION
 *   Information about the ROBODoc licence.
 * AUTHOR
 *   Frans
 * HISTORY
 *   2003-02-25/petterik: corrected link to GNU copyleft.
 *******
 */

char                copying[] =
    "\n"
    " Distributed under the GNU GENERAL PUBLIC LICENSE\n"
    "   TERMS AND CONDITIONS FOR COPYING, DISTRIBUTION AND MODIFICATION\n"
    " See the source archive for a copy of the complete licence\n"
    " If you do not have it you can get it from URL\n"
    " http://www.gnu.org/copyleft/gpl.html\n";

/****f* UserInterface/RB_Stat_Path
 * FUNCTION
 *   Check the given path against required type.
 * RETURN VALUE
 *   0 if all ok, otherwise 1.
 * BUGS
 *   Should check if symbolic link points to a directory or to a file.
 * SOURCE
 */

static int
RB_Stat_Path( char required, char *path )
{
    struct stat         st;

    if ( stat( path, &st ) < 0 )
    {
        RB_Panic( "RB_Stat_Path: can not stat '%s'\n", path );
    }

    switch ( ( ( st.st_mode ) & S_IFMT ) )
    {
        case S_IFDIR:
            if ( required == 'd' )
            {
                return 0;
            }
            break;
        case S_IFREG:
            if ( required == 'f' )
            {
                return 0;
            }
            break;
            /* TODO case S_IFLNK: chdir() */
        default:
            break;
    }                           /* end switch */

    return 1;
}

/*******/


/****f* UserInterface/main [2.0d]
 * FUNCTION
 *   Get and parse the arguments.  Analyse document and generate the
 *   documentation. Everything starts from here.
 * SYNOPSIS
 *   main (int argc, char **argv)
 * SOURCE
 */

int
main( int argc, char **argv )
{
    int                 result = 0;
    struct RB_Document *document = NULL;
    struct RB_Directory *srctree = NULL;
    char               *tabsizestr = NULL;

    whoami = argv[0];           /* global me,myself&i */

    /* Read the configuration file. This might contain
       addition options. */
    RB_SetCurrentFile( NULL );
    RB_ReadConfiguration( argc, argv, 
            RB_Find_In_Argv_Parameterized_Option( argc, argv, "--rc" ) );

    if ( RB_Check_Options(  ) == EXIT_FAILURE )
    {
        return EXIT_FAILURE;
    }

    if ( RB_Find_Option( "-c" ) )
    {
        printf( "%s", copying );
        return EXIT_SUCCESS;
    }

    if ( RB_Find_Option( "--version" ) )
    {
        printf( "%s\n", VERSION );
        return EXIT_SUCCESS;
    }

    if ( RB_Find_Option( "--help" ) )
    {
        printf( "%s%s%s%s%s%s", use, use_usage, use_options1, use_options2, use_options3,
                use_authors );
        return EXIT_SUCCESS;
    }

    output_mode = RB_Find_DocType();  /* one of the globals that are still left */
    if ( output_mode == UNKNOWN )
    {
        printf( "%s", short_use );
        return EXIT_SUCCESS;
    }

    /* First the basics. */
    document = RB_Get_RB_Document(  );

    document->doctype = output_mode;
    document->actions = RB_Find_Actions( document );
    document->charset = RB_Find_Parameterized_Option( "--charset" );
    document->extension = RB_Find_Parameterized_Option( "--ext" );
    document->css     = RB_Find_Parameterized_Option( "--css" );

    tabsizestr = RB_Find_Parameterized_Option( "--tabsize" );
    if ( tabsizestr )
    {
        tab_size = atoi( tabsizestr );
    }

    if ( !document->extension )
    {
        document->extension = RB_Get_Default_Extension( document->doctype );
    }
    course_of_action = document->actions;

    /* Test if there is a --src and --doc */
    if ( !RB_Find_Parameterized_Option( "--src" ) ) 
    {
        printf( "Error: you need to specify a source"
                " file or directory with --src.\n" );
        printf( "%s", short_use );
        return EXIT_FAILURE;
    }
    if ( !RB_Find_Parameterized_Option( "--doc" ) ) 
    {
        printf( 
          "Error: you need to specify a documentation file"
          " or directory with --doc.\n" );
        printf( "%s", short_use );
        return EXIT_FAILURE;
    }

    /* What mode are we using? */
    if ( RB_Find_Option( "--multidoc" ) )
    {
        char               *srcrootname;        /* directory */
        char               *docrootname;

        srcrootname = RB_Fix_Path( RB_Find_Parameterized_Option( "--src" ) );
        if ( RB_Stat_Path( 'd', srcrootname ) )
        {
            printf( "Error: %s is not a directory\n", srcrootname );
            printf( "%s", short_use );
            return EXIT_FAILURE;
        }
        document->srcroot = RB_Get_RB_Path( srcrootname );

        docrootname = RB_Fix_Path( RB_Find_Parameterized_Option( "--doc" ) );
        RB_Path_Check( srcrootname, docrootname );

        document->docroot = RB_Get_RB_Path( docrootname );

        srctree = RB_Get_RB_Directory( srcrootname, docrootname );
        document->srctree = srctree;

        RB_Document_Create_Parts( document );
        RB_Analyse_Document( document );
        RB_Generate_Documentation( document );

        RB_Free_RB_Path( document->srcroot );
        document->srcroot = 0;
        RB_Free_RB_Path( document->docroot );
        document->docroot = 0;
        RB_Free_RB_Directory( srctree );
        document->srctree = 0;
    }
    else if ( RB_Find_Option( "--singledoc" ) )
    {
        char               *srcrootname;        /* directory */

        srcrootname = RB_Fix_Path( RB_Find_Parameterized_Option( "--src" ) );
        if ( RB_Stat_Path( 'd', srcrootname ) )
        {
            printf( "Error: %s is not a file\n", srcrootname );
            printf( "%s", short_use );
            return EXIT_FAILURE;
        };
        document->srcroot = RB_Get_RB_Path( srcrootname );

        document->docroot = 0;
        document->singledoc_name = RB_Fix_Path( RB_Find_Parameterized_Option( "--doc" ) );

        srctree = RB_Get_RB_Directory( srcrootname, NULL );
        document->srctree = srctree;

        RB_Document_Create_Parts( document );
        RB_Analyse_Document( document );
        RB_Generate_Documentation( document );

        RB_Free_RB_Directory( srctree );
    }
    else if ( RB_Find_Option( "--singlefile" ) )
    {
        char*               srcfile;    /* file */
        char*               docfile;    /* file */

        document->docroot = 0;
        docfile = RB_Find_Parameterized_Option( "--doc" );
        assert( docfile );
        document->singledoc_name = RB_Fix_Path( docfile );
        srcfile = RB_Fix_Path( RB_Find_Parameterized_Option( "--src" ) );
        if ( RB_Stat_Path( 'f', srcfile ) )
        {
            printf( "Error: %s is not a file\n", srcfile );
            printf( "%s", short_use );
            return EXIT_FAILURE;
        };

        document->srctree = RB_Get_RB_SingleFileDirectory( srcfile );
        document->srcroot =
            RB_Get_RB_Path( document->srctree->first_path->name );

        RB_Document_Create_Parts( document );
        RB_Analyse_Document( document );
        RB_Generate_Documentation( document );

        RB_Free_RB_Directory( document->srctree );
    }
    else
    {
        printf( "%s", short_use );
        printf
            ( "\n\nError: Use either --multidoc, --singledoc, or --singlefile\n" );
    }

    RB_Free_RB_Document( document );
    RB_Free_Configuration(  );
    return result;
}

/*******/


/****f* UserInterface/RB_Path_Check
 * FUNCTION
 *   Test the validity of the doc and source path. The doc path should
 *   not be a sub directory of the source path otherwise the generated
 *   documentation will be part of the generated documentation if
 *   robodoc is run more than once.
 * SYNOPSIS
 *   void RB_Path_Check( char *sourcepath, char *docpath )
 * INPUTS
 *   o sourcepath -- the path to the source files.
 *   o docpath    -- the path to the documentation files.
 * OUTPUT
 *   o error messages
 *****
 */ 

static void
RB_Path_Check( char *sourcepath, char *docpath )
{
    if ( docpath )
    {
        int                 dl;
        int                 sl;

        dl = strlen( docpath );
        sl = strlen( sourcepath );
        if ( dl >= sl )
        {
            int                 i;
            int                 equal = TRUE;

            for ( i = 0; i < sl; ++i )
            {
                if ( docpath[i] != sourcepath[i] )
                {
                    equal = FALSE;
                    break;
                }
            }
            if ( equal && ( dl == sl ) )
            {
                RB_Panic
                    ( "The source path and document path can not be equal\n" );
            }
            else
            {
                /* OK  */
            }
        }
    }
}

/****f* UserInterface/RB_PathBegin_Check
 * FUNCTION
 *   Checks the validity of a path.
 *   A path should start with
 *     ./
 *   or
 *     /
 *   or
 *     have a ':' some where
 * SYNOPSIS
 *   int RB_PathBegin_Check( char* path )
 * INPUTS
 *   o path -- the path to be cheked.
 * RESULT
 *   FALSE -- path is not OK.
 *   TRUE  -- path is OK.
 ******
 */

static int
RB_PathBegin_Check( char *path )
{
    int                 result = FALSE;
    int                 l = strlen( path );

    if ( l >= 2 )
    {
        result = ( ( path[0] == '.' ) && ( path[1] == '/' ) ) ||
            ( path[0] == '/' ) || ( strchr( path, ':' ) != NULL );
    }
    else
    {
        /* Empty */
    }
    return result;
}


/****f* UserInterface/RB_Find_Option
 * FUNCTION
 *   Search configuration.options for a specific option.
 * SYNOPSIS
 *   int RB_Find_Option( char *option )
 * INPUTS
 *   o option -- the option to be found.
 * RESULT
 *   TRUE  -- option does exist
 *   FALSE -- option does not exist
 ******
 */

int
RB_Find_Option( char *option )
{
    unsigned int        parameter_nr;
    int                 found = FALSE;

    for ( parameter_nr = 0;
          parameter_nr < configuration.options.number; parameter_nr++ )
    {
        if ( !RB_Str_Case_Cmp( configuration.options.names[parameter_nr], option ) )
        {
            found = TRUE;
            break;
        }
    }
    return found;
}

/****f* UserInterface/RB_Fix_Path
 * FUNCTION
 *   Add a "./" to a path if it does not start with a "./" or does not
 *   contain a ":".  Adding a "./" simplifies the creating of relative
 *   links during the generation process.
 * SYNOPSIS
 *   char* RB_Fix_Path( char* path )
 * INPUTS
 *   o path -- the path to be fixed.
 * RESULT
 *   A pointer to a newly allocated string containing the path.
 * SOURCE
 */

static char* RB_Fix_Path( char* path )
{
    char* result = 0;
    if ( ! RB_PathBegin_Check( path ) )
    {
        int l = strlen( path );
        char* prefix = "./";
        l += strlen( prefix ) + 1;
        result = malloc( l );
        assert( result );
        result[ 0 ] = '\0';
        strcat( result, prefix );
        strcat( result, path );
    }
    else
    {
        result = strdup( path );
    }
    return result;
}

/*****/

/****f* UserInterface/RB_Check_Options
 * FUNCTION
 *   Check the spelling of all the options in configuration.options[].
 * SYNOPSIS
 *  int RB_Check_Options( void )
 * RESULT
 *   EXIT_SUCCESS -- all options are correctly spelled.
 *   EXIT_FAILURE -- one of more options are misspelled.
 * SOURCE
 */

static int
RB_Check_Options( void )
{
    char                ok, *arg, **opts;
    unsigned int        parameter_nr;

    for ( parameter_nr = 0;
          parameter_nr < configuration.options.number; parameter_nr++ )
    {
        arg = configuration.options.names[parameter_nr];
        if ( arg[0] == '-' )
        {
            /* this arg is an option */
            ok = 0;
            opts = ok_options;
            while ( *opts )
            {
                if ( strcmp( arg, *opts ) == 0 )
                {
                    ok = 1;
                    break;
                }
                opts++;
            }
            if ( !ok )
            {
                printf( "Invalid argument: %s\n", arg );
                printf( "This might also be in your robodoc.rc file\n" );
                printf( "%s\n", short_use );
                return EXIT_FAILURE;
            }
        }
    }
    return EXIT_SUCCESS;
}

/******/



T_RB_DocType
RB_Find_DocType( void )
{
    T_RB_DocType        doctype = UNKNOWN;
    unsigned int        parameter_nr;

    for ( parameter_nr = 0;
          parameter_nr < configuration.options.number; parameter_nr++ )
    {

        if ( !RB_Str_Case_Cmp( configuration.options.names[parameter_nr],
                               "--html" ) )
        {
            doctype = HTML;
            break;
        }
        else if ( !RB_Str_Case_Cmp( configuration.options.names[parameter_nr],
                                    "--latex" ) )
        {
            doctype = LATEX;
            break;
        }
        else if ( !RB_Str_Case_Cmp( configuration.options.names[parameter_nr],
                                    "--ascii" ) )
        {
            doctype = ASCII;
            break;
        }
        else if ( !RB_Str_Case_Cmp( configuration.options.names[parameter_nr],
                                    "--rtf" ) )
        {
            doctype = RTF;
            break;
        }
        else if ( !RB_Str_Case_Cmp( configuration.options.names[parameter_nr],
                                    "--troff" ) )
        {
            doctype = TROFF;
            break;
        }
        else if ( !RB_Str_Case_Cmp( configuration.options.names[parameter_nr],
                                    "--dbxml" ) )
        {
            doctype = XMLDOCBOOK;
            break;
        }
        else if ( !RB_Str_Case_Cmp( configuration.options.names[parameter_nr],
                                    "--dbsgml" ) )
        {
            doctype = DBSGML;
            break;
        }
    }
    return doctype;
}



long
RB_Find_Actions( struct RB_Document *document )
{
    long                actions = 0;
    unsigned int        parameter_nr;

    for ( parameter_nr = 0;
          parameter_nr < configuration.options.number; parameter_nr++ )
    {
        if ( !RB_Str_Case_Cmp( configuration.options.names[parameter_nr],
                               "--singledoc" ) )
        {
            actions |= DO_SINGLEDOC;
        }
        if ( !RB_Str_Case_Cmp( configuration.options.names[parameter_nr],
                               "--singlefile" ) )
        {
            actions |= DO_SINGLEFILE;
        }
        else if ( !RB_Str_Case_Cmp( configuration.options.names[parameter_nr],
                                    "--multidoc" ) )
        {
            actions |= DO_MULTIDOC;
        }
        else if ( !RB_Str_Case_Cmp( configuration.options.names[parameter_nr],
                                    "--sections" ) )
        {
            actions |= DO_SECTIONS;
        }
        else if ( !RB_Str_Case_Cmp( configuration.options.names[parameter_nr],
                                    "--internal" ) )
        {
            actions |= DO_INCLUDE_INTERNAL;
        }
        else if ( !RB_Str_Case_Cmp( configuration.options.names[parameter_nr],
                                    "--internalonly" ) )
        {
            actions |= DO_INTERNAL_ONLY;
        }
        else if ( !RB_Str_Case_Cmp( configuration.options.names[parameter_nr],
                                    "--toc" ) )
        {
            actions |= DO_TOC;
        }
        else if ( !RB_Str_Case_Cmp( configuration.options.names[parameter_nr],
                                    "--index" ) )
        {
            actions |= DO_INDEX;
        }
        else if ( !RB_Str_Case_Cmp( configuration.options.names[parameter_nr],
                                    "--nosource" ) )
        {
            actions |= DO_NOSOURCE;
        }
        else if ( !RB_Str_Case_Cmp( configuration.options.names[parameter_nr],
                                    "--tell" ) )
        {
            actions |= DO_TELL;
        }
        else if ( !RB_Str_Case_Cmp( configuration.options.names[parameter_nr],
                                    "--folds" ) )
        {
            actions |= DO_FOLDS;
        }
        else if ( !RB_Str_Case_Cmp( configuration.options.names[parameter_nr],
                                    "--nodesc" ) )
        {
            actions |= DO_NODESC;
        }
        else if ( !RB_Str_Case_Cmp( configuration.options.names[parameter_nr],
                                    "--cmode" ) )
        {
            actions |= DO_CMODE;
        }
        else if ( !RB_Str_Case_Cmp( configuration.options.names[parameter_nr],
                                    "--lock" ) )
        {
            actions |= DO_LOCKHEADER;
        }
        else
        {
            /* Not an action */
        }
    }
    return actions;
}



/****f* UserInterface/RB_Find_Parameterized_Option
 * FUNCTION
 *   Search for an option of the form
 *     --a_option_name a_value
 *   in configuration.options.
 * SYNOPSIS
 *   value = RB_Find_Parameterized_Option( char *optionname )
 * INPUTS
 *   optionname -- the name of the option to search for.
 * RESULT
 *   NULL if the option is not found, a pointer to the value
 *   otherwise.
 * NOTES
 *   Results in a Panic if the option is found but
 *   no value is specified.
 * SOURCE
 */

char*
RB_Find_Parameterized_Option( char *optionname )
{
    return RB_General_Find_Parameterized_Option( configuration.options.number,
            &( configuration.options.names[0] ), optionname );
}

/******/


/****f* UserInterface/RB_Find_In_Argv_Parameterized_Option
 * FUNCTION
 *   Search for an option of the form
 *     --a_option_name a_value
 *   in argv.   The function is used to look for the
 *   --rc  option that can be used to specify an 
 *   alternate robodoc configuration file.
 * SYNOPSIS
 *   value = RB_Find_Parameterized_Option( 
 *      int argv, char** argv, *   char* optionname )
 * INPUTS
 *   argc -- the argument count as received by main().
 *   argv -- the array of argument values as received by main()
 *   optionname -- the name of the option to search for.
 * RESULT
 *   NULL if the option is not found, a pointer to the value
 *   otherwise.
 * NOTES
 *   Results in a Panic if the option is found but
 *   no value is specified.
 * SOURCE
 */

char*
RB_Find_In_Argv_Parameterized_Option( int argc, char** argv, char* optionname )
{
    return  RB_General_Find_Parameterized_Option( argc, argv, optionname );
}

/*****/


/****f* UserInterface/RB_General_Find_Parameterized_Option
 * FUNCTION
 *   Search for an option of the form
 *     --a_option_name a_value
 * SYNOPSIS
 *   value = RB_General_Find_Parameterized_Option( 
 *      int n, char** options, char* optionname )
 * INPUTS
 *   o n -- the number of options in the options array.
 *   o options -- the options array
 *   o optionname -- the name of the option to search for.
 * RESULT
 *   NULL if the option is not found, a pointer to the value
 *   otherwise.
 * NOTES
 *   Results in a Panic if the option is found but
 *   no value is specified.
 * SOURCE
 */

static char* RB_General_Find_Parameterized_Option( int n, char** options, char* optionname )
{
    int                 parameter_nr;
    char               *value = NULL;

    for ( parameter_nr = 0; parameter_nr < n; parameter_nr++ )
    {
        if ( !RB_Str_Case_Cmp( options[parameter_nr],
                               optionname ) )
        {
            if ( parameter_nr < n - 1 )
            {
                value = options[parameter_nr + 1];
                if ( value[0] == '-' )
                {
                    value = NULL;
                }
            }
            else
            {
                /* to few parameters. */
            }
            if ( !value )
            {
                RB_Panic( "you must be more specific"
                          " with the %s option\n", optionname );
            }
        }
    }
    return value;
}

/******/

