// Copyright (c) The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.

// Authors: Malolan Chetlur             mal@ececs.uc.edu
//          Jorgen Dahl                 dahlj@ececs.uc.edu
//          Dale E. Martin              dmartin@cliftonlabs.com
//          Radharamanan Radhakrishnan  ramanan@ececs.uc.edu
//          Dhananjai Madhava Rao       dmadhava@ececs.uc.edu
//          Philip A. Wilsey            phil.wilsey@uc.edu

//---------------------------------------------------------------------------
// 
// $Id: SimulationManagerImplementationBase.cpp
// 
//---------------------------------------------------------------------------


#include "SimulationManagerImplementationBase.h"
#include "Application.h"
#include <clutils/StringHashMap.h>
#include <functional>
#include <algorithm>

using std::unary_function;
using std::for_each;
using std::cerr;
using std::endl;

ostream *wout;
ostream *werr;

SimulationManagerImplementationBase::SimulationManagerImplementationBase(unsigned int numProcessors)
  : numberOfSimulationManagers(numProcessors),
    localArrayOfSimObjPtrs(0),
    localArrayOfSimObjIDs(0){}

SimulationManagerImplementationBase::~SimulationManagerImplementationBase(){
}

class InitObject : public unary_function<SimulationObject *,
		   void> {
public:
  void operator()( SimulationObject *toInit ){
    toInit->initialize();
  }
};

void
SimulationManagerImplementationBase::initializeObjects(){
  // now traverse the map of objects and call initialize

  vector<SimulationObject *> *objects = localArrayOfSimObjPtrs->getElementVector();
  InitObject initObject;
  for_each< vector<SimulationObject *>::iterator, InitObject &>( objects->begin(), 
								 objects->end(),
								 initObject );
  delete objects;
}

void
SimulationManagerImplementationBase::setNumberOfObjects(unsigned int numObjects) {
  numberOfObjects = numObjects;
}

void
SimulationManagerImplementationBase::finalizeObjects(){
  vector<SimulationObject *> *objects = localArrayOfSimObjPtrs->getElementVector();
  for( unsigned int i = 0; i < objects->size(); i++ ){
    (*objects)[i]->finalize();
  }
  delete objects;
}

void
SimulationManagerImplementationBase::receiveEvent( Event *,
                                                   SimulationObject *,
                                                   SimulationObject * ){
  shutdown("ERROR: receiveEvent() called in  SimulationManagerImplementationBase" );
}

void
SimulationManagerImplementationBase::simulate( const VTime & ){
  shutdown( "ERROR: simulate() called in SimulationManagerImplementationBase" );
}


void
SimulationManagerImplementationBase::registerSimulationObjects(){
  shutdown( "ERROR: registerSimulationObjects() called in SimulationManagerImplementationBase" );
}

void 
SimulationManagerImplementationBase::unregisterSimulationObjects (vector <SimulationObject*>* list){
  // some tasks this function is responsible for
};

// print out the name to simulation object ptr map
void
SimulationManagerImplementationBase::displayObjectMap(ostream& out){

  if(!localArrayOfSimObjPtrs->empty()){
    vector<string> *keys = localArrayOfSimObjPtrs->getKeyVector();
    vector<SimulationObject *> *objects = localArrayOfSimObjPtrs->getElementVector();
    for( unsigned int i = 0; i < objects->size(); i++ ){
      out << (*keys)[i] << ": " << (*objects)[i]->getObjectID();
    }
    delete objects;
    delete keys;
  }
  else {
    out << "Object Names to Object Pointers Map is empty" << endl;
  }
}

StringHashMap<SimulationObject *, true> *
SimulationManagerImplementationBase::partitionVectorToHashMap(vector<SimulationObject *> *vector){
  StringHashMap<SimulationObject *, true> *retval = new StringHashMap<SimulationObject *, true>;

  for( unsigned int i = 0; i < vector->size(); i++ ){
    retval->insert( (*vector)[i]->getName(), (*vector)[i] );
  }

  return retval;
}
