/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <math.h>

#include <qcheckbox.h>
#include <qpainter.h>
#include <qpixmap.h>
#include <qpushbutton.h>
#include <qslider.h>

#include "FilterDialog.h"
#include "GlobalIcon.h"
#include "images/filticon.xpm"

FilterDialog::FilterDialog(QWidget* parent, const char* name,
                           bool modal, WFlags fl)
: FilterDialogData(parent,name,modal,fl)
{
    setFixedSize(width(),height());
    setIcon(*mainIcon);

    enableChangedSignal(true);
    setButtons();
    paintFilterFunc();
}

void FilterDialog::enableChangedSignal(bool val)
{
    changedSignalEnabled = val;
}

void FilterDialog::emitChangedIfEnabled()
{
    if (changedSignalEnabled)
    {
        emit changed(config);
    }
}

void FilterDialog::setConfig(const emuConfig& inConfig)
{
    enableChangedSignal(false);
    config = inConfig;
    setButtons();
    enableChangedSignal(true);
}

void FilterDialog::setConfigDefault(const emuConfig& inConfig)
{
    configDefault = inConfig;
}

void FilterDialog::receiveConfig(const emuConfig& inConfig)
{
    // Check flags that can be set from other dialogs.
    if (config.emulateFilter != inConfig.emulateFilter)
    {
        config = inConfig;
        // Make sure we do not send values because we are the receiver.
        enableChangedSignal(false);
        isEnabled->setChecked(inConfig.emulateFilter);
        isEnabled->repaint(false);
        enableChangedSignal(true);
    }
}

void FilterDialog::setButtons()
{
    isEnabled->setChecked(config.emulateFilter);
    setSliderValues();
}

void FilterDialog::setSliderValues()
{
    para1Sld->setValue((int)((config.filterFs/6.0-1.0)));
    para2Sld->setValue((int)(config.filterFm-1.0));
    para3Sld->setValue((int)(config.filterFt*100.0));
}

void FilterDialog::updatePara1(int index)
{
    config.filterFs = 1.0+index*6.0;
    emitChangedIfEnabled();
}

void FilterDialog::updatePara2(int index)
{
    config.filterFm = 1.0+index;
    emitChangedIfEnabled();
}

void FilterDialog::updatePara3( int index )
{
    config.filterFt = index/100.0;
    emitChangedIfEnabled();
}

void FilterDialog::setDefaults()
{
    // Don't copy the full emuConfig because the rest of the default
    // doesn't get updated.
    config.emulateFilter = configDefault.emulateFilter;
    config.filterFs = configDefault.filterFs;
    config.filterFm = configDefault.filterFm;
    config.filterFt = configDefault.filterFt;

    setButtons();
    emitChangedIfEnabled();
}

void FilterDialog::setFilterFlag(bool val)
{
    if (val != config.emulateFilter)
    {
        config.emulateFilter = val;
        emitChangedIfEnabled();
    }
}

void FilterDialog::paintFilterFunc()
{
    QPixmap myFiltPic(filticon);
    QPainter paintBuf;
    paintBuf.begin(&myFiltPic);
    QColor myPenColor(0,0,255);  // rgb
    paintBuf.setPen(myPenColor);
    extern float filterTable[0x800];  // the only way
    for (int i=0; i<0x800; i++)
    {
        paintBuf.drawPoint((int)(i/15.75),(int)(filterTable[i]*100.0));
    }
    paintBuf.end();

    QPainter paint(this);
    paint.drawPixmap(220,0,myFiltPic);
}

void FilterDialog::paintEvent(QPaintEvent *)
{
    paintFilterFunc();
}
