/*
 * NASPRO - The NASPRO Architecture for Sound PROcessing
 * Portable runtime library
 *
 * Copyright (C) 2007-2014 Stefano D'Angelo
 *
 * See the COPYING file for license conditions.
 */

#include "internal.h"

#define MAX(a, b)	(((a) > (b)) ? (a) : (b))
#define MIN(a, b)	(((a) < (b)) ? (a) : (b))

_NACORE_DEF size_t
nacore_char_utf8_encode(char *utf8c, uint32_t cp)
{
	uint8_t *c;

	if ((cp > 0x10ffff) || ((cp >= 0xd800) && (cp <= 0xdfff))
	    || (cp == 0xfeff) || (cp == 0xfffe))
		return 0;

	c = (uint8_t *)utf8c;

	if (cp < 0x80)
	  {
		if (c != NULL)
			c[0] = cp & 0xff;
		return 1;
	  }
	if (cp < 0x800)
	  {
		if (c != NULL)
		  {
			c[0] = (cp >> 6) | 0xc0;
			c[1] = (cp & 0x3f) | 0x80;
		  }
		return 2;
	  }
	if (cp < 0x10000)
	  {
		if (c != NULL)
		  {
			c[0] = (cp >> 12) | 0xe0;
			c[1] = ((cp >> 6) & 0x3f) | 0x80;
			c[2] = (cp & 0x3f) | 0x80;
		  }
		return 3;
	  }

	if (c != NULL)
	  {
		c[0] = (cp >> 18) | 0xf0;
		c[1] = ((cp >> 12) & 0x3f) | 0x80;
		c[2] = ((cp >> 6) & 0x3f) | 0x80;
		c[3] = (cp & 0x3f) | 0x80;
	  }
	return 4;
}

_NACORE_DEF size_t
nacore_char_utf8_decode(const char *utf8c, uint32_t *cp)
{
	const uint8_t *c;

	c = (const uint8_t *)utf8c;

	if (c[0] < 0x80)			/* one byte */
	  {
		if (cp != NULL)
			*cp = c[0];
		return 1;
	  }

	if ((c[0] & 0xe0) == 0xc0)		/* two bytes */
	  {
		if ((c[1] & 0xc0) != 0x80)
			return 0;
		if (cp != NULL)
			*cp = ((c[0] & 0x1f) << 6) | (c[1] & 0x3f);
		return 2;
	  }
	if ((c[0] & 0xf0) == 0xe0)		/* three bytes */
	  {
		/* this does also reject surrogates (U+D800 to U+DFFF), BOM
		 * (U+FFEF) and inverse BOM (U+FFFE) */
		if ((((c[1] & 0xc0) != 0x80) || ((c[2] & 0xc0) != 0x80))
		    || ((c[0] == 0xed) && (c[1] >= 0xa0))
		    || ((c[0] == 0xef) && (c[1] == 0xbb) && (c[2] == 0xbf))
		    || ((c[0] == 0xef) && (c[1] == 0xbf) && (c[2] == 0xbe)))
			return 0;
		if (cp != NULL)
			*cp = ((c[0] & 0x0f) << 12) | ((c[1] & 0x3f) << 6)
			      | (c[2] & 0x3f);
		return 3;
	  }
	if ((c[0] & 0xf8) == 0xf0)		/* four bytes */
	  {
		if (((c[1] & 0xc0) != 0x80) || ((c[2] & 0xc0) != 0x80)
		    || ((c[3] & 0xc0) != 0x80))
			return 0;
		if (cp != NULL)
			*cp = ((c[0] & 0x07) << 18) | ((c[1] & 0x3f) << 12)
			      | ((c[2] & 0x3f) << 6) | (c[3] & 0x3f);
	  }

	return 0;
}

_NACORE_DEF size_t
nacore_char_utf16_encode(uint16_t *utf16c, uint32_t cp)
{
	if ((cp > 0x10ffff) || ((cp >= 0xd800) && (cp <= 0xdfff))
	    || (cp == 0xfeff) || (cp == 0xfffe))
		return 0;

	if (cp < 0x10000)
	  {
		if (utf16c != NULL)
			*utf16c = cp;
		return 2;
	  }

	if (utf16c != NULL)
	  {
		cp -= 0x10000;
		utf16c[0] = 0xd8 | ((cp >> 10) & 0x3ff);
		utf16c[1] = 0xdc | (cp & 0x3ff);
	  }
	return 4;
}

_NACORE_DEF size_t
nacore_char_utf16_decode(const uint16_t *utf16c, uint32_t *cp)
{
	if ((*utf16c == 0xfeff) || (*utf16c == 0xfffe))
		return 0;
	if ((*utf16c >= 0xd800) && (*utf16c <= 0xdbff))
	  {
		if (cp != NULL)
			*cp = 0x10000 + (((utf16c[0] & 0x3ff) << 10)
					 | (utf16c[1] & 0x3ff));
		return 4;
	  }

	if (cp != NULL)
		*cp = *utf16c;
	return 2;
}

_NACORE_DEF size_t
nacore_string_utf8_to_utf16_len(const char *str_utf8)
{
	const char *c8;
	uint32_t cp;
	size_t s, ret;

	for (c8 = str_utf8, ret = 0; *c8 != '\0'; c8 += s)
	  {
		s = nacore_char_utf8_decode(c8, &cp);
		if (s == 0)
		  {
			s = 1;
			continue;
		  }

		ret += nacore_char_utf16_encode(NULL, cp);
	  }

	return ret;
}

_NACORE_DEF void
nacore_string_utf8_to_utf16(uint16_t *buf, const char *str_utf8)
{
	const char *c8;
	uint16_t *c16;
	uint32_t cp;
	size_t s8, s16;

	for (c8 = str_utf8, c16 = buf; *c8 != '\0'; c8 += s8, c16 += s16)
	  {
		s8 = nacore_char_utf8_decode(c8, &cp);
		if (s8 == 0)
		  {
			s8 = 1;
			s16 = 0;
			continue;
		  }

		s16 = nacore_char_utf16_encode(c16, cp) >> 1;
	  }

	*c16 = 0;
}

_NACORE_DEF uint16_t *
nacore_string_utf8_to_utf16_a(const char *str_utf8)
{
	uint16_t *ret;
	size_t len;

	len = nacore_string_utf8_to_utf16_len(str_utf8);

	ret = malloc(len + 2);
	if (ret == NULL)
		return NULL;

	nacore_string_utf8_to_utf16(ret, str_utf8);

	return ret;
}

_NACORE_DEF size_t
nacore_string_utf16_to_utf8_len(const uint16_t *str_utf16)
{
	const uint16_t *c16;
	uint32_t cp;
	size_t s, ret;

	for (c16 = str_utf16, ret = 0; *c16 != 0; c16 += s)
	  {
		s = nacore_char_utf16_decode(c16, &cp) >> 1;
		if (s == 0)
		  {
			s = 2;
			continue;
		  }

		ret += nacore_char_utf8_encode(NULL, cp);
	  }

	return ret;
}

_NACORE_DEF void
nacore_string_utf16_to_utf8(char *buf, const uint16_t *str_utf16)
{
	const uint16_t *c16;
	char *c8;
	uint32_t cp;
	size_t s8, s16;

	for (c8 = buf, c16 = str_utf16; *c16 != 0; c8 += s8, c16 += s16)
	  {
		s16 = nacore_char_utf16_decode(c16, &cp) >> 1;
		if (s16 == 0)
		  {
			s16 = 2;
			s8 = 0;
			continue;
		  }

		s8 = nacore_char_utf8_encode(c8, cp);
	  }

	*c8 = '\0';
}

_NACORE_DEF char *
nacore_string_utf16_to_utf8_a(const uint16_t *str_utf16)
{
	char *ret;
	size_t len;

	len = nacore_string_utf16_to_utf8_len(str_utf16);

	ret = malloc(len + 1);
	if (ret == NULL)
		return NULL;

	nacore_string_utf16_to_utf8(ret, str_utf16);

	return ret;
}

_NACORE_DEF size_t
nacore_string_get_size(const char *s, void *unused)
{
	return strlen(s) + 1;
}

_NACORE_DEF size_t
nacore_strnlen(const char *s, size_t maxlen)
{
	size_t ret;

	for (ret = 0; s[ret] != '\0'; ret++)
		if (ret == maxlen)
			break;

	return ret;
}

_NACORE_DEF char *
nacore_strdup(const char *s, void *unused)
{
	char *ret;
	size_t len;

	len = strlen(s) + 1;
	ret = malloc(len);
	if (ret == NULL)
		return NULL;
	memcpy(ret, s, len);

	return ret;
}

/* Length modifier enumeration. */
typedef enum
  {
	lm_none,
	lm_hh,
	lm_h,
	lm_l,
	lm_ll,
	lm_L,
	lm_j,
	lm_z,
	lm_t
  } length_mod;

/* Conversion specifier enumeration. */
typedef enum
  {
	cs_d = 0,
	cs_o,
	cs_u,
	cs_x,
	cs_X,
	cs_e,
	cs_E,
	cs_f,
	cs_F,
	cs_g,
	cs_G,
	cs_a,
	cs_A,
	cs_c,
	cs_s,
	cs_p,
	cs_n,
	cs_pc
  } conv_specifier;

/* Conversion flags. */
#define FLAG_DASH	1
#define FLAG_ZERO	(1 << 1)
#define FLAG_MINUS	(1 << 2)
#define FLAG_SPACE	(1 << 3)
#define FLAG_PLUS	(1 << 4)

/* Conversion specification type. */
typedef struct
  {
	char			flags;
	int			min_width;
	int			precision;	/* -1 == no precision given */
	length_mod		lm;
	conv_specifier		cs;
  } conv_spec;

/*
 * Parses minimum field width or precision values (also handles the "*" and
 * "*m$" cases).
 *
 * *c should point to the beginning of the string to parse and will be set to
 * point to the character right after the value specification.
 *
 * ap should point to the current argument which is used in the "*" and "*m$"
 * cases and will advance one position only in the "*" case.
 */
static int
get_width_prec(const char **c, va_list *ap)
{
	int ret;
	int m;
	int i;
	va_list m_ap;

	ret = 0;
	switch (**c)
	  {
		case '-':
			do
				(*c)++;
			while (isdigit(**c));
			break;
		case '*':
			(*c)++;
			if (isdigit(**c))
			  {
				m = 0;
				while (**c != '$')
				  {
					  m = 10 * m + **c - '0';
					  (*c)++;
				  }
				(*c)++;

				va_copy(m_ap, *ap);
				for (i = 0; i < m; i++)
					ret = va_arg(m_ap, int);
				va_end(m_ap);
			  }
			else
				ret = va_arg(*ap, int);
			break;
		default:
			while (isdigit(**c))
			  {
				ret = 10 * ret + **c - '0';
				(*c)++;
			  }
			break;
	  }

	return ret;
}

/*
 * Parses conversion specifications (not including the first '%' character).
 *
 * *c should point to the beginning of the string to parse and will be set to
 * point to the character right after the conversion specification.
 *
 * ap should point to the current argument which is used in the "*" and "*m$"
 * cases of precision and/or minimum width specifications and will advance only
 * in "*" cases.
 */
static conv_spec
get_conv_spec(const char **c, va_list *ap)
{
	conv_spec ret =
	  {
		/* .flags	= */ 0,
		/* .min_width	= */ 0,
		/* .precision	= */ -1,
		/* .lm		= */ lm_none,
		/* .cs		= */ cs_pc
	  };

	/* %% */
	if (**c == '%')
	  {
		(*c)++;
		return ret;
	  }

	/* flags */
	while (1)
	  {
		switch (**c)
		  {
			  case '#':
				ret.flags |= FLAG_DASH;
				break;
			  case '0':
				ret.flags |= FLAG_ZERO;
				break;
			  case '-':
				ret.flags |= FLAG_MINUS;
				break;
			  case ' ':
				ret.flags |= FLAG_SPACE;
				break;
			  case '+':
				ret.flags |= FLAG_PLUS;
				break;
			  default:
				goto flags_loop_end;
				break;
		  }

		(*c)++;
	  }
flags_loop_end:

	/* minimum field width */
	if (isdigit(**c) || (**c == '*'))
		ret.min_width = get_width_prec(c, ap);

	/* precision */
	if (**c == '.')
	  {
		(*c)++;
		ret.precision = get_width_prec(c, ap);
	  }

	/* length modifier */
	switch (**c)
	  {
		case 'h':
			(*c)++;
			if (**c == 'h')
			  {
				(*c)++;
				ret.lm = lm_hh;
			  }
			else
				ret.lm = lm_h;
			break;
		case 'l':
			(*c)++;
			if (**c == 'l')
			  {
				(*c)++;
				ret.lm = lm_ll;
			  }
			else
				ret.lm = lm_l;
			break;
		case 'L':
			(*c)++;
			ret.lm = lm_L;
			break;
		case 'j':
			(*c)++;
			ret.lm = lm_j;
			break;
		case 't':
			(*c)++;
			ret.lm = lm_t;
			break;
		case 'z':
			(*c)++;
			ret.lm = lm_z;
			break;
		default:
			ret.lm = lm_none;
			break;
	  }

	/* conversion specifier */
	switch (**c)
	  {
		case 'd':
		case 'i':
			ret.cs = cs_d;
			break;
		case 'u':
			ret.cs = cs_u;
			break;
		case 'o':
			ret.cs = cs_o;
			break;
		case 'x':
			ret.cs = cs_x;
			break;
		case 'X':
			ret.cs = cs_X;
			break;
		case 'e':
			ret.cs = cs_e;
			break;
		case 'E':
			ret.cs = cs_E;
			break;
		case 'f':
			ret.cs = cs_f;
			break;
		case 'F':
			ret.cs = cs_F;
			break;
		case 'g':
			ret.cs = cs_g;
			break;
		case 'G':
			ret.cs = cs_G;
			break;
		case 'a':
			ret.cs = cs_a;
			break;
		case 'A':
			ret.cs = cs_A;
			break;
		case 'c':
			ret.cs = cs_c;
			break;
		case 's':
			ret.cs = cs_s;
			break;
		case 'p':
			ret.cs = cs_p;
			break;
		default: /* 'n' */
			ret.cs = cs_n;
			break;
	  }

	(*c)++;

	return ret;
}

/* Gets a signed integer value from ap according to the given length
 * modifier. */
static intmax_t
conv_val_lli(length_mod lm, va_list *ap)
{
	intmax_t v;

	switch (lm)
	  {
		case lm_hh:
			v = (char)va_arg(*ap, int);
			break;
		case lm_h:
			v = (short int)va_arg(*ap, int);
			break;
		case lm_l:
			v = va_arg(*ap, long int);
			break;
		case lm_ll:
			v = va_arg(*ap, long long int);
			break;
		case lm_j:
			v = va_arg(*ap, intmax_t);
			break;
		case lm_z:
			v = va_arg(*ap, size_t);
			break;
		case lm_t:
			v = va_arg(*ap, ptrdiff_t);
			break;
		default: /* lm_none */
			v = va_arg(*ap, int);
			break;
	  }

	return v;
}

/* Gets an unsigned integer value from ap according to the given length
 * modifier. */
static uintmax_t
conv_val_ulli(length_mod lm, va_list *ap)
{
	uintmax_t v;

	switch (lm)
	  {
		case lm_hh:
			v = (unsigned char)va_arg(*ap, unsigned int);
			break;
		case lm_h:
			v = (unsigned short int)va_arg(*ap, unsigned int);
			break;
		case lm_l:
			v = va_arg(*ap, unsigned long int);
			break;
		case lm_ll:
			v = va_arg(*ap, unsigned long long int);
			break;
		case lm_j:
			v = va_arg(*ap, uintmax_t);
			break;
		case lm_z:
			v = va_arg(*ap, size_t);
			break;
		case lm_t:
			v = va_arg(*ap, ptrdiff_t);
			break;
		default: /* lm_none */
			v = va_arg(*ap, unsigned int);
			break;
	  }

	return v;
}

/* Gets a floating point value from ap according to the given length
 * modifier. */
static long double
conv_val_ld(length_mod lm, va_list *ap)
{
	return (lm == lm_L) ? va_arg(*ap, long double) : va_arg(*ap, double);
}

/* Adds left padding to *str according to the given conversion specification cs
 * and the length of the unpadded string indicated by len. *str will point to
 * the character after the padding spaces. */
static void
pad_left(char **str, conv_spec *cs, int len)
{
	int i;

	if (!(cs->flags & (FLAG_ZERO | FLAG_MINUS)))
	  {
		for (i = len; i < cs->min_width; i++)
		  {
			**str = ' ';
			(*str)++;
		  }
	  }
}

/* Adds zero padding to *str according to the given conversion specification cs
 * and the length of the unpadded string indicated by len. *str will point to
 * the character after the padding zeros. */
static void
pad_zero(char **str, conv_spec *cs, int len)
{
	int i;

	if (cs->flags & FLAG_ZERO)
	  {
		for (i = len; i < cs->min_width; i++)
		  {
			**str = '0';
			(*str)++;
		  }
	  }
}

/* Adds right padding to *str according to the given conversion specification cs
 * and the length of the unpadded string indicated by len. *str will point to
 * the character after the padding spaces. */
static void
pad_right(char **str, conv_spec *cs, int len)
{
	int i;

	if (cs->flags & FLAG_MINUS)
	  {
		for (i = len; i < cs->min_width; i++)
		  {
			**str = ' ';
			(*str)++;
		  }
	  }
}

/* Conversion functions. */

static int
conv_d(char *str, conv_spec *cs, va_list *ap, const char *dec_sep,
       size_t dec_sep_len)
{
	intmax_t v, c;
	int ret, i;

	v = conv_val_lli(cs->lm, ap);

	ret = 0;
	c = v;
	do
	  {
		c /= 10;
		ret++;
	  }
	while (c != 0);

	if ((cs->precision == 0) && (v == 0))
		ret = 0;
	else
		ret = MAX(ret, cs->precision);

	if ((cs->flags & (FLAG_SPACE | FLAG_PLUS)) || (v < 0))
		ret++;

	if (str == NULL)
		return MAX(ret, cs->min_width);

	pad_left(&str, cs, ret);

	if (v < 0)
	  {
		*str = '-';
		str++;
	  }
	else if (cs->flags & FLAG_PLUS)
	  {
		*str = '+';
		str++;
	  }
	else if (cs->flags & FLAG_SPACE)
	  {
		*str = ' ';
		str++;
	  }

	pad_zero(&str, cs, ((cs->flags & (FLAG_PLUS | FLAG_SPACE)) || (v < 0))
			   ? ret + 1 : ret);

	if (!((cs->precision == 0) && (v == 0)))
	  {
		c = (v < 0) ? -1 : 1;
		for (i = 1; v / c >= 10; i++, c *= 10) ;

		if (i < cs->precision)
			for (; i < cs->precision; cs->precision--)
			  {
				*str = '0';
				str++;
			  }

		if (v == 0)
		  {
			*str = '0';
			str++;
		  }
		else
		  {
			do
			  {
				*str = v / c + '0';
				str++;
				v %= c;
				c /= 10;
			  }
			while (c != 0);
		  }
	  }

	pad_right(&str, cs, ret);

	return MAX(ret, cs->min_width);
}

static int
conv_o(char *str, conv_spec *cs, va_list *ap, const char *dec_sep,
       size_t dec_sep_len)
{
	uintmax_t v, c;
	int ret, i;

	v = conv_val_ulli(cs->lm, ap);

	ret = 0;
	c = v;
	do
	  {
		c >>= 3;
		ret++;
	  }
	while (c != 0);

	if ((cs->precision == 0) && (v == 0))
		ret = 0;
	else if (cs->precision > ret)
		ret = cs->precision;
	else if (cs->flags & FLAG_DASH)
		ret++;

	if (str == NULL)
		return MAX(ret, cs->min_width);

	pad_left(&str, cs, ret);
	pad_zero(&str, cs, ret);

	if (!((cs->precision == 0) && (v == 0)))
	  {
		c = (~(c & 0)) << 3;
		for (i = 1; v & c; i++, c <<= 3) ;
		c = 1;
		c <<= 3 * (i - 1);

		if (i < cs->precision)
			for (; i < cs->precision; cs->precision--)
			  {
				*str = '0';
				str++;
			  }
		else if (cs->flags & FLAG_DASH)
		  {
			*str = '0';
			str++;
		  }

		if (v == 0)
		  {
			*str = '0';
			str++;
		  }
		else
		  {
			do
			  {
				*str = v / c + '0';
				str++;
				v %= c;
				c >>= 3;
			  }
			while (c != 0);
		  }
	  }

	pad_right(&str, cs, ret);

	return MAX(ret, cs->min_width);
}

static int
conv_u(char *str, conv_spec *cs, va_list *ap, const char *dec_sep,
       size_t dec_sep_len)
{
	uintmax_t v, c;
	int ret, i;

	v = conv_val_ulli(cs->lm, ap);

	ret = 0;
	c = v;
	do
	  {
		c /= 10;
		ret++;
	  }
	while (c != 0);

	if ((cs->precision == 0) && (v == 0))
		ret = 0;
	else
		ret = MAX(ret, cs->precision);

	if (str == NULL)
		return MAX(ret, cs->min_width);

	pad_left(&str, cs, ret);
	pad_zero(&str, cs, ret);

	if (!((cs->precision == 0) && (v == 0)))
	  {
		for (i = 1, c = 1; v / c >= 10; i++, c *= 10) ;

		if (i < cs->precision)
			for (; i < cs->precision; cs->precision--)
			  {
				*str = '0';
				str++;
			  }

		if (v == 0)
		  {
			*str = '0';
			str++;
		  }
		else
		  {
			do
			  {
				*str = v / c + '0';
				str++;
				v %= c;
				c /= 10;
			  }
			while (c != 0);
		  }
	  }

	pad_right(&str, cs, ret);

	return MAX(ret, cs->min_width);
}

static int
conv_xX(char *str, conv_spec *cs, va_list *ap, char X)
{
	uintmax_t v, c;
	int ret, i;
	char x;

	v = conv_val_ulli(cs->lm, ap);

	ret = 0;
	c = v;
	do
	  {
		c >>= 4;
		ret++;
	  }
	while (c != 0);

	if ((cs->precision == 0) && (v == 0))
		ret = 0;
	else
		ret = MAX(ret, cs->precision)
		      + ((cs->flags & FLAG_DASH) ? 2 : 0);;

	if (str == NULL)
		return MAX(ret, cs->min_width);

	pad_left(&str, cs, ret);
	pad_zero(&str, cs, ret);

	if (!((cs->precision == 0) && (v == 0)))
	  {
		c = (~(c & 0)) << 4;
		for (i = 1; v & c; i++, c <<= 4) ;
		c = 1;
		c <<= 4 * (i - 1);

		if (cs->flags & FLAG_DASH)
		  {
			*str = '0';
			str++;
			*str = X ? 'X' : 'x';
			str++;
		  }

		if (i < cs->precision)
			for (; i < cs->precision; cs->precision--)
			  {
				*str = '0';
				str++;
			  }

		if (v == 0)
		  {
			*str = '0';
			str++;
		  }
		else
		  {
			do
			  {
				x = v / c;
				*str = x + ((x < 10)
					    ? '0' : ((X ? 'A' : 'a') - 10));
				str++;
				v %= c;
				c >>= 4;
			  }
			while (c != 0);
		  }
	  }

	pad_right(&str, cs, ret);

	return MAX(ret, cs->min_width);
}

static int
conv_x(char *str, conv_spec *cs, va_list *ap, const char *dec_sep,
       size_t dec_sep_len)
{
	return conv_xX(str, cs, ap, 0);
}

static int
conv_X(char *str, conv_spec *cs, va_list *ap, const char *dec_sep,
       size_t dec_sep_len)
{
	return conv_xX(str, cs, ap, 1);
}

static int
conv_eE(char *str, conv_spec *cs, va_list *ap, const char *dec_sep,
	size_t dec_sep_len, char E)
{
	long double v, p, d;
	short int exp, dexp, pe, de;
	int ret;

	v = conv_val_ld(cs->lm, ap);

	if (cs->precision == -1)
		cs->precision = 6;

	ret = 3;
	exp = 0;
	dexp = 2;

	if (!isinf(v) && !isnan(v))
	  {
		if (v < -0.0)
		  {
			exp = floor(log10l(-v));
			p = powl(10.0, exp - cs->precision);
			v = roundl(v / p) * p;
			exp = floor(log10l(-v));
		  }
		else if (v > 0.0)
		  {
			exp = floor(log10l(v));
			p = powl(10.0, exp - cs->precision);
			v = roundl(v / p) * p;
			exp = floor(log10l(v));
		  }

		if ((cs->precision != 0) || (cs->flags & FLAG_DASH))
			ret += dec_sep_len + cs->precision;

		if (exp < 0)
			dexp = floor(log10(-exp) + 1.0);
		else if (exp > 0)
			dexp = floor(log10(exp) + 1.0);

		if (dexp < 2)
			dexp = 2;

		ret += dexp;
	  }

	if (signbit(v) || (cs->flags & FLAG_PLUS) || (cs->flags & FLAG_SPACE))
		ret++;

	if (str == NULL)
		return MAX(ret, cs->min_width);

	pad_left(&str, cs, ret);

	if (signbit(v))
	  {
		*str = '-';
		str++;
	  }
	else if (cs->flags & FLAG_PLUS)
	  {
		*str = '+';
		str++;
	  }
	else if (cs->flags & FLAG_SPACE)
	  {
		*str = ' ';
		str++;
	  }

	if (!isnan(v) && !isinf(v))
		pad_zero(&str, cs,
			 ((cs->flags & (FLAG_PLUS | FLAG_SPACE)) || signbit(v))
			 ? ret + 1 : ret);

	if (isnan(v))
	  {
		strncpy(str, E ? "NAN" : "nan", 3);
		str += 3;
	  }
	else if (isinf(v))
	  {
		strncpy(str, E ? "INF" : "inf", 3);
		str += 3;
	  }
	else if ((v == 0.0) || (v == -0.0))
	  {
		*str = '0';
		str++;

		if ((cs->precision != 0) || (cs->flags & FLAG_DASH))
		  {
			strncpy(str, dec_sep, dec_sep_len);
			str += dec_sep_len;

			for (; cs->precision != 0; cs->precision--)
			  {
				*str = '0';
				str++;
			  }
		  }

		strncpy(str, E ? "E+00" : "e+00", 4);
		str += 4;
	  }
	else
	  {
		if (signbit(v))
			v = -v;

		p = powl(10.0, exp);

		d = v / p;

		*str = (char)d + '0';
		str++;

		if ((cs->precision != 0) || (cs->flags & FLAG_DASH))
		  {
			strncpy(str, dec_sep, dec_sep_len);
			str += dec_sep_len;

			for (; cs->precision != 0; cs->precision--)
			  {
				v -= (long double)((char)d) * p;
				p /= 10.0L;
				d = v / p;
				*str = (char)d + '0';
				str++;
			  }
		  }

		*str = E ? 'E' : 'e';
		str++;

		if (exp >= 0)
			*str = '+';
		else
			*str = '-';
		str++;

		if (exp < 0)
			exp = -exp;

		for (pe = 1, dexp--; dexp != 0; pe *= 10, dexp--) ;

		for (; pe != 0; pe /= 10)
		  {
			de = exp / pe;
			*str = (char)de + '0';
			str++;
			exp -= de * pe;
		  }
	  }

	pad_right(&str, cs, ret);

	return MAX(ret, cs->min_width);
}

static int
conv_e(char *str, conv_spec *cs, va_list *ap, const char *dec_sep,
       size_t dec_sep_len)
{
	return conv_eE(str, cs, ap, dec_sep, dec_sep_len, 0);
}

static int
conv_E(char *str, conv_spec *cs, va_list *ap, const char *dec_sep,
       size_t dec_sep_len)
{
	return conv_eE(str, cs, ap, dec_sep, dec_sep_len, 1);
}

static int
conv_fF(char *str, conv_spec *cs, va_list *ap, const char *dec_sep,
	size_t dec_sep_len, char F)
{
	long double v, p;
	short int d;
	char c;
	int ret;

	v = conv_val_ld(cs->lm, ap);

	if (cs->precision == -1)
		cs->precision = 6;

	ret = 0;
	d = 1;

	if (isinf(v) || isnan(v))
		ret = 3;
	else
	  {
		if (v > 0.0)
		  {
			p = powl(10.0, -cs->precision);
			v = roundl(v / p) * p;
		  }
		else if (v < -0.0)
		  {
			p = powl(10.0, -cs->precision);
			v = roundl(v / p) * p;
		  }

		if (v <= -10.0)
			d = floor(log10l(-v) + 1.0);
		else if (v >= 10.0)
			d = floor(log10l(v) + 1.0);

		ret += d;

		if ((cs->precision != 0) || (cs->flags & FLAG_DASH))
			ret += dec_sep_len + cs->precision;
	  }

	if (signbit(v) || (cs->flags & FLAG_PLUS) || (cs->flags & FLAG_SPACE))
		ret++;

	if (str == NULL)
		return MAX(ret, cs->min_width);

	pad_left(&str, cs, ret);

	if (signbit(v))
	  {
		*str = '-';
		str++;
	  }
	else if (cs->flags & FLAG_PLUS)
	  {
		*str = '+';
		str++;
	  }
	else if (cs->flags & FLAG_SPACE)
	  {
		*str = ' ';
		str++;
	  }

	if (!isnan(v) && !isinf(v))
		pad_zero(&str, cs,
			 ((cs->flags & (FLAG_PLUS | FLAG_SPACE)) || signbit(v))
			 ? ret + 1 : ret);

	if (isnan(v))
	  {
		strncpy(str, F ? "NAN" : "nan", 3);
		str += 3;
	  }
	else if (isinf(v))
	  {
		strncpy(str, F ? "INF" : "inf", 3);
		str += 3;
	  }
	else
	  {
		if (signbit(v))
			v = -v;

		p = powl(10.0, d - 1);

		if (p < 1.0)
		  {
			*str = '0';
			str++;
		  }

		for (; p >= 1.0; p /= 10.0L)
		  {
			c = v / p;
			*str = c + '0';
			str++;
			v -= (long double)c * p;
		  }

		if ((cs->precision != 0) || (cs->flags & FLAG_DASH))
		  {
			strncpy(str, dec_sep, dec_sep_len);
			str += dec_sep_len;

			for (; cs->precision != 0; cs->precision--)
			  {
				c = v / p;
				*str = c + '0';
				str++;
				v -= (long double)c * p;
				p /= 10.0L;
			  }
		  }
	  }

	pad_right(&str, cs, ret);

	return MAX(ret, cs->min_width);
}

static int
conv_f(char *str, conv_spec *cs, va_list *ap, const char *dec_sep,
       size_t dec_sep_len)
{
	return conv_fF(str, cs, ap, dec_sep, dec_sep_len, 0);
}

static int
conv_F(char *str, conv_spec *cs, va_list *ap, const char *dec_sep,
       size_t dec_sep_len)
{
	return conv_fF(str, cs, ap, dec_sep, dec_sep_len, 1);
}

static int
conv_gG(char *str, conv_spec *cs, va_list *ap, const char *dec_sep,
	size_t dec_sep_len, char G)
{
	long double v, vc, p, d, min;
	short int exp, dexp, pe, de;
	int pc, ret;

	v = conv_val_ld(cs->lm, ap);

	if (cs->precision == -1)
		cs->precision = 6;
	else if (cs->precision == 0)
		cs->precision = 1;

	ret = 3;
	exp = 0;
	dexp = 2;

	if (!isinf(v) && !isnan(v))
	  {
		vc = v; /* make gcc happy */

		if (v < -0.0)
		  {
			exp = floor(log10l(-v));
			p = powl(10.0, exp - cs->precision);
			vc = roundl(v / p) * p;
			exp = floor(log10l(-vc));
		  }
		else if (v > 0.0)
		  {
			exp = floor(log10l(v));
			p = powl(10.0, exp - cs->precision);
			vc = roundl(v / p) * p;
			exp = floor(log10l(vc));
		  }

		if ((exp < -4) || (exp >= cs->precision))
		  {
			v = vc;

			if (cs->flags & FLAG_DASH)
				ret += dec_sep_len + cs->precision;
			else
			  {
				p = powl(10.0, exp);
				vc = v;
				d = vc / p;
				vc -= (long double)((char)d) * p;
				p /= 10.0L;

				min = powl(10.0, exp - cs->precision + 1);

				if ((vc >= min) || (vc <= -min))
				  {
					ret++;

					for (pc = cs->precision - 1;
					     ((vc >= min) || (vc <= -min))
					     && (pc != 0); pc--, ret++)
					  {
						d = vc / p;
						vc -= (long double)((char)d)
						      * p;
						p /= 10.0L;
					  }
				  }
			  }

			if (exp < 0)
				dexp = floor(log10(-exp) + 1.0);
			else if (exp > 0)
				dexp = floor(log10(exp) + 1.0);

			if (dexp < 2)
				dexp = 2;

			ret += dexp;
		  }
		else
		  {
			p = powl(10.0, -cs->precision);
			v = roundl(v / p) * p;

			if (v <= -10.0)
				ret = floor(log10l(-v) + 1.0);
			else if (v >= 10.0)
				ret = floor(log10l(v) + 1.0);
			else
				ret = 1;

			if (cs->flags & FLAG_DASH)
				ret += dec_sep_len + cs->precision;
			else
			  {
				vc = fabsl(vc);
				vc -= floorl(vc);

				min = powl(10.0, ret - cs->precision);
				if ((v < 1.0) && (v > -1.0))
					min /= 10.0L;

				if (vc >= min)
				  {
					ret++;

					p = 0.1L;
					for (pc = cs->precision;
					     (vc >= min) && (pc != 0); pc--,
					     ret++)
					  {
						d = vc / p;
						vc -= (long double)((char)d)
						      * p;
						p /= 10.0L;
					  }
				  }
			  }
		  }
	  }

	if (signbit(v) || (cs->flags & FLAG_PLUS) || (cs->flags & FLAG_SPACE))
		ret++;

	if (str == NULL)
		return MAX(ret, cs->min_width);

	pad_left(&str, cs, ret);

	if (signbit(v))
	  {
		*str = '-';
		str++;
	  }
	else if (cs->flags & FLAG_PLUS)
	  {
		*str = '+';
		str++;
	  }
	else if (cs->flags & FLAG_SPACE)
	  {
		*str = ' ';
		str++;
	  }

	if (!isnan(v) && !isinf(v))
		pad_zero(&str, cs,
			 ((cs->flags & (FLAG_PLUS | FLAG_SPACE)) || signbit(v))
			 ? ret + 1 : ret);

	if (isnan(v))
	  {
		strncpy(str, G ? "NAN" : "nan", 3);
		str += 3;
	  }
	else if (isinf(v))
	  {
		strncpy(str, G ? "INF" : "inf", 3);
		str += 3;
	  }
	else
	  {
		if (signbit(v))
			v = -v;

		if ((exp < -4) || (exp >= cs->precision))
		  {
			p = powl(10.0, exp);

			d = v / p;
			v -= (long double)((char)d) * p;
			p /= 10.0L;

			*str = (char)d + '0';
			str++;

			min = powl(10.0, exp - cs->precision + 1);

			if ((cs->flags & FLAG_DASH)
			    || ((v >= min) || (v <= -min)))
			  {
				strncpy(str, dec_sep, dec_sep_len);
				str += dec_sep_len;

				for (; (cs->precision != 0)
				       && ((cs->flags & FLAG_DASH)
					   || (!(cs->flags & FLAG_DASH)
					       && ((v >= min)
					           || (v <= -min))));
				     cs->precision--)
				  {
					d = v / p;
					*str = (char)d + '0';
					str++;
					v -= (long double)((char)d) * p;
					p /= 10.0L;
				  }
			  }

			*str = G ? 'E' : 'e';
			str++;

			if (exp >= 0)
				*str = '+';
			else
				*str = '-';
			str++;

			if (exp < 0)
				exp = -exp;

			for (pe = 1, dexp--; dexp != 0; pe *= 10, dexp--) ;

			for (; pe != 0; pe /= 10)
			  {
				de = exp / pe;
				*str = (char)de + '0';
				str++;
				exp -= de * pe;
			  }
		  }
		else
		  {
			if (v != 0.0)
				p = powl(10.0, floor(log10l(v) + 1.0) - 1.0);
			else
				p = 1.0;

			min = p / powl(10.0, cs->precision);
			if ((v < 1.0) && (v > -1.0))
				min /= 10.0L;

			if (p < 1.0)
			  {
				*str = '0';
				str++;
			  }

			for (; p >= 1.0; p /= 10.0L)
			  {
				de = v / p;
				*str = (char)de + '0';
				str++;
				v -= (long double)de * p;
			  }

			if ((cs->flags & FLAG_DASH)
			    || ((v <= -min) || (v >= min)))
			  {
				strncpy(str, dec_sep, dec_sep_len);
				str += dec_sep_len;

				p = 0.1L;
				for (; (cs->precision != 0)
				       && ((cs->flags & FLAG_DASH)
					   || (!(cs->flags & FLAG_DASH)
					       && ((v >= min)
					           || (v <= -min))));
				     cs->precision--)
				  {
					de = v / p;
					*str = (char)de + '0';
					str++;
					v -= (long double)de * p;
					p /= 10.0L;
				  }
			  }
		  }
	  }

	pad_right(&str, cs, ret);

	return MAX(ret, cs->min_width);
}

static int
conv_g(char *str, conv_spec *cs, va_list *ap, const char *dec_sep,
       size_t dec_sep_len)
{
	return conv_gG(str, cs, ap, dec_sep, dec_sep_len, 0);
}

static int
conv_G(char *str, conv_spec *cs, va_list *ap, const char *dec_sep,
       size_t dec_sep_len)
{
	return conv_gG(str, cs, ap, dec_sep, dec_sep_len, 1);
}

static int
conv_aA(char *str, conv_spec *cs, va_list *ap, const char *dec_sep,
	size_t dec_sep_len, char A)
{
	long double v, p, d, vc;
	short int exp, dexp, pe, de;
	int ret;

	v = conv_val_ld(cs->lm, ap);

	exp = 0;
	dexp = 1;

	if (isinf(v) || isnan(v))
		ret = 3;
	else
	  {
		if (v < -0.0)
			exp = floor(log2l(-v));
		else if (v > 0.0)
			exp = floor(log2l(v));

		if (cs->precision != -1)
		  {
			if (v < -0.0)
			  {
				p = powl(16.0, (double)exp / 4.0
					       - (double)cs->precision);
				v = roundl(v / p) * p;
				exp = floor(log2l(-v));
			  }
			else if (v > 0.0)
			  {
				p = powl(16.0, (double)exp / 4.0
					       - (double)cs->precision);
				v = roundl(v / p) * p;
				exp = floor(log2l(v));
			  }

			ret = 5;
			if ((cs->precision != 0) || (cs->flags & FLAG_DASH))
				ret += dec_sep_len + cs->precision;
		  }
		else
		  {
			ret = 5;

			p = powl(16.0, (double)exp / 4.0);

			vc = v;
			d = vc / p;
			vc -= (long double)((char)d) * p;

			if (((vc != 0.0) && (vc != -0.0))
			    || (cs->flags & FLAG_DASH))
				ret += dec_sep_len;

			while ((vc != 0.0) && (vc != -0.0))
			  {
				p /= 16.0L;
				d = vc / p;
				vc -= (long double)((char)d) * p;
				ret++;
			  }
		  }

		if (exp < 0)
			dexp = floor(log10(-exp) + 1.0);
		else if (exp > 0)
			dexp = floor(log10(exp) + 1.0);

		if (dexp < 1)
			dexp = 1;

		ret += dexp;
	  }

	if (signbit(v) || (cs->flags & FLAG_PLUS) || (cs->flags & FLAG_SPACE))
		ret++;

	if (str == NULL)
		return MAX(ret, cs->min_width);

	pad_left(&str, cs, ret);

	if (signbit(v))
	  {
		*str = '-';
		str++;
	  }
	else if (cs->flags & FLAG_PLUS)
	  {
		*str = '+';
		str++;
	  }
	else if (cs->flags & FLAG_SPACE)
	  {
		*str = ' ';
		str++;
	  }

	if (!isnan(v) && !isinf(v))
		pad_zero(&str, cs,
			 ((cs->flags & (FLAG_PLUS | FLAG_SPACE)) || signbit(v))
			 ? ret + 1 : ret);

	if (isnan(v))
	  {
		strncpy(str, A ? "NAN" : "nan", 3);
		str += 3;
	  }
	else if (isinf(v))
	  {
		strncpy(str, A ? "INF" : "inf", 3);
		str += 3;
	  }
	else
	  {
		if (signbit(v))
			v = -v;

		*str = '0';
		str++;
		*str = A ? 'X' : 'x';
		str++;

		p = powl(16.0, (double)exp / 4.0);

		d = v / p;
		*str = (char)d + ((d < 10.0) ? '0' : (((A ? 'A' : 'a') - 10)));
		str++;
		v -= (long double)((char)d) * p;

		if ((cs->precision != -1)
		    && ((cs->precision != 0) || (cs->flags & FLAG_DASH)))
		  {
			strncpy(str, dec_sep, dec_sep_len);
			str += dec_sep_len;

			for (; cs->precision != 0; cs->precision--)
			  {
				p /= 16.0L;
				d = v / p;
				*str = (char)d + ((d < 10.0)
						  ? '0' : (((A ? 'A' : 'a')
							    - 10)));
				str++;
				v -= (long double)((char)d) * p;
			  }
		  }
		else if ((v != 0.0) && (v != -0.0) && (cs->precision == -1))
		  {
			strncpy(str, dec_sep, dec_sep_len);
			str += dec_sep_len;

			while ((v != 0.0) && (v != -0.0))
			  {
				p /= 16.0L;
				d = v / p;
				*str = (char)d + ((d < 10.0)
						  ? '0' : (((A ? 'A' : 'a')
							    - 10)));
				str++;
				v -= (long double)((char)d) * p;
			  }
		  }

		*str = A ? 'P' : 'p';
		str++;

		if (exp >= 0)
			*str = '+';
		else
			*str = '-';
		str++;

		if (exp < 0)
			exp = -exp;

		for (pe = 1, dexp--; dexp != 0; pe *= 10, dexp--) ;

		for (; pe != 0; pe /= 10)
		  {
			de = exp / pe;
			*str = (char)de + '0';
			str++;
			exp -= de * pe;
		  }
	  }

	pad_right(&str, cs, ret);

	return MAX(ret, cs->min_width);
}

static int
conv_a(char *str, conv_spec *cs, va_list *ap, const char *dec_sep,
       size_t dec_sep_len)
{
	return conv_aA(str, cs, ap, dec_sep, dec_sep_len, 0);
}

static int
conv_A(char *str, conv_spec *cs, va_list *ap, const char *dec_sep,
       size_t dec_sep_len)
{
	return conv_aA(str, cs, ap, dec_sep, dec_sep_len, 1);
}

/* FIXME: wint_t ("%lc") not supported */
static int
conv_c(char *str, conv_spec *cs, va_list *ap, const char *dec_sep,
       size_t dec_sep_len)
{
	unsigned char v;

	v = va_arg(*ap, int);

	if (str == NULL)
		return MAX(1, cs->min_width);

	pad_left(&str, cs, 1);

	*str = v;
	str++;

	pad_right(&str, cs, 1);

	return MAX(1, cs->min_width);
}

/* FIXME: const wchar_t * ("%ls") not supported */
static int
conv_s(char *str, conv_spec *cs, va_list *ap, const char *dec_sep,
       size_t dec_sep_len)
{
	const char *v;
	int ret;

	v = va_arg(*ap, const char *);

	ret = (cs->precision >= 0) ? nacore_strnlen(v, cs->precision)
				   : strlen(v);

	if (str == NULL)
		return MAX(ret, cs->min_width);

	pad_left(&str, cs, ret);

	memcpy(str, v, ret);
	str += ret;

	pad_right(&str, cs, ret);

	return MAX(ret, cs->min_width);
}

static int
conv_p(char *str, conv_spec *cs, va_list *ap, const char *dec_sep,
       size_t dec_sep_len)
{
	cs->flags |= FLAG_DASH;
	cs->lm = lm_l;

	return conv_x(str, cs, ap, dec_sep, dec_sep_len);
}

/* Conversion functions' table (index of function according to
 * enum conv_specifier. */
static int (*conv_func[])(char *str, conv_spec *cs, va_list *ap,
			  const char *dec_sep, size_t dec_sep_len) =
  {
	conv_d, conv_o, conv_u, conv_x, conv_X, conv_e, conv_E, conv_f, conv_F,
	conv_g, conv_G, conv_a, conv_A, conv_c, conv_s, conv_p
  };

static int
_vasprintf(char **strp, const char *fmt, va_list ap, char *dec_sep,
	   size_t dec_sep_len)
{
	int ret;
	const char *c;
	char *s, *rets;
	va_list ap2;
	conv_spec cs;
	int *n;
	int i;

	/* calculate needed memory space */
	va_copy(ap2, ap);

	c = fmt;
	ret = 0;
	while (*c != '\0')
	  {
		if (*c != '%')
		  {
			/* not a conversion specification */
			ret++, c++;
			continue;
		  }

		/* conversion specification, get needed space */
		c++;
		cs = get_conv_spec(&c, &ap2);
		if (cs.cs != cs_n)
			ret += conv_func[cs.cs](NULL, &cs, &ap2, dec_sep,
						dec_sep_len);
	  }

	va_end(ap2);

	if (strp == NULL)
		return ret;

	/* allocate memory and perform conversions */
	rets = malloc(ret + 1);
	if (rets == NULL)
	  {
		*strp = NULL;
		return ret;
	  }

	va_copy(ap2, ap);

	s = rets;
	c = fmt;
	ret = 0;
	while (*c != '\0')
	  {
		if (*c != '%')
		  {
			/* not a conversion specification */
			*s = *c;
			ret++, c++, s++;
			continue;
		  }

		/* conversion specification, perform conversion */
		c++;
		cs = get_conv_spec(&c, &ap2);
		if (cs.cs == cs_n)
		  {
			/* %n */
			n = va_arg(ap, int *);
			*n = ret;
		  }
		else
		  {
			i = conv_func[cs.cs](s, &cs, &ap2, dec_sep,
					     dec_sep_len);
			ret += i;
			s += i;
		  }
	  }

	*s = '\0';
	*strp = rets;

	va_end(ap2);

	return ret;
}

_NACORE_DEF NACORE_FORMAT_VPRINTF(2) int
nacore_vasprintf(char **strp, const char *fmt, va_list ap)
{
	size_t dec_sep_len = _nacore_locale_dec_sep_len();
	char dec_sep[dec_sep_len + 1];

	_nacore_locale_dec_sep_get(dec_sep);

	return _vasprintf(strp, fmt, ap, dec_sep, dec_sep_len);
}

_NACORE_DEF NACORE_FORMAT_VPRINTF(2) int
nacore_vasprintf_nl(char **strp, const char *fmt, va_list ap)
{
	return _vasprintf(strp, fmt, ap, ".", 1);
}

_NACORE_DEF NACORE_FORMAT_PRINTF(2, 3) int
nacore_asprintf(char **strp, const char *fmt, ...)
{
	va_list ap;
	int ret;
	size_t dec_sep_len = _nacore_locale_dec_sep_len();
	char dec_sep[dec_sep_len + 1];

	_nacore_locale_dec_sep_get(dec_sep);

	va_start(ap, fmt);
	ret = _vasprintf(strp, fmt, ap, dec_sep, dec_sep_len);
	va_end(ap);

	return ret;
}

_NACORE_DEF NACORE_FORMAT_PRINTF(2, 3) int
nacore_asprintf_nl(char **strp, const char *fmt, ...)
{
	va_list ap;
	int ret;

	va_start(ap, fmt);
	ret = _vasprintf(strp, fmt, ap, ".", 1);
	va_end(ap);

	return ret;
}

_NACORE_DEF nacore_list
nacore_string_split(const char *s, const char *sep, nacore_filter_cb filter_cb,
		    void *filter_opaque)
{
	nacore_list ret;
	const char *p;
	char *buf, *tmp;
	size_t buf_size, s_len, sep_len;

	ret = nacore_list_new((nacore_get_size_cb)nacore_string_get_size);
	if (ret == NULL)
		return NULL;

	for (buf = NULL, buf_size = 0, sep_len = strlen(sep),
	     p = strstr(s, sep); p != NULL; s = p + sep_len, p = strstr(s, sep))
	  {
		s_len = (size_t)(p - s);

		if (buf_size <= s_len)
		  {
			tmp = realloc(buf, s_len + 1);
			if (tmp == NULL)
				goto err;

			buf = tmp;
			buf_size = s_len + 1;
		  }

		memcpy(buf, s, s_len);
		buf[s_len] = '\0';

		if (filter_cb != NULL)
			if (filter_cb(buf, filter_opaque) == 0)
				continue;

		if (nacore_list_append(ret, NULL, buf) == NULL)
			goto err;
	  }

	if (buf != NULL)
	  {
		free(buf);
		buf = NULL;
	  }

	if (filter_cb != NULL)
		if (filter_cb(s, filter_opaque) == 0)
			return ret;

	if (nacore_list_append(ret, NULL, (void *)s) == NULL)
		goto err;

	return ret;

err:
	if (buf != NULL)
		free(buf);
	nacore_list_free(ret, NULL, NULL);
	return NULL;
}
