/***********************************************************************
* Name:
*        dapiproc.c
*
* Function:
*        Functions invoked via menu selections
*
* Environment:
*        This is a PLATFORM-INDEPENDENT source file. As such it may
*        contain no dependencies on any specific operating system
*        environment or hardware platform.
*
* Description:
*        The file contains the functions that are invoked from the
*        menu processing code and tables.
***********************************************************************/
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "dsmapitd.h"      /* Tivoli Storage Manager API type definitions        */
#include "dsmapifp.h"      /* Tivoli Storage Manager API function prototypes.    */
#include "dsmrc.h"         /* Tivoli Storage Manager API rc definitions.         */
#include "dapitype.h"      /* Tivoli Storage Manager Sample API type definitions.*/
#include "dapiutil.h"
#include "dapiproc.h"

/*----------------------------------------------------------------------+
| Private routines
.----------------------------------------------------------------------*/
static dsInt16_t get_input(dialog *line);
static dsInt16_t process_dialog(func_proto func, dialog *sel_dialog,
                                dsUint16_t parm1);
static void output_optioninfo();
static int  free_dialog_buffs(dialog *sel_dialog);
static void output_sessinfo(void);

/*----------------------------------------------------------------------+
| Public routines
.----------------------------------------------------------------------*/
dsInt8_t no_restore = 0, no_retrieve = 0;

/*----------------------------------------------------------------------+
| Name:    smpapi_exit()
|
| Action:  Performs any cleanup necessary before exiting the sample api
|          main menu processing.
|
| Input:   sel_dialog  - Pointer to dialog table if any.
|
| Returns: RC_OK            - Successful
+----------------------------------------------------------------------*/
dsInt16_t smpapi_exit(dialog *sel_dialog)

{
   dsInt16_t rc;
   rc = dsmTerminate(dsmHandle);
   rc = dsmCleanUp(DSM_SINGLETHREAD);
   dsmHandle = 0;
   memset(&dsmSessInfo,0x00,sizeof(ApiSessInfo));  /* Zero out block.     */
   return 0;
}

/*----------------------------------------------------------------------+
| Name:    smpapi_backup()
|
| Action:  Prompts user for backup objects to create and then sends those
|          objects to the DSM server.
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK            - Successful
|          others           - Any other valid Tivoli Storage Manager api rc.
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t smpapi_backup (dialog *sel_dialog)
{
   dsUint16_t rc = 0;

   if (!sel_dialog)
   {
      printf("*** Error in coding the backup dialog! ***\n");
   }
   else 
      rc = process_dialog((func_proto)perform_send,sel_dialog,Backup_Send);

   return rc;
}

/*----------------------------------------------------------------------+
| Name:    smpapi_archive()
|
| Action:  Prompts user for archive objects to create and then sends those
|          objects to the DSM server.
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK            - Successful
|          others           - Any other valid Tivoli Storage Manager api rc.
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t smpapi_archive (dialog *sel_dialog)
{
   dsUint16_t rc = 0;

   if (!sel_dialog)
   {
      printf("*** Error in coding the archive dialog! ***\n");
   }
   else 
      rc = process_dialog((func_proto)perform_send,sel_dialog,Archive_Send);

   return rc;
}

/*----------------------------------------------------------------------+
| Name:    smpapi_restore(dialog *sel_dialog)
|
| Action:  Prompts user for items to restore and then does so.
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK            - Successful
|          others           - Any other valid Tivoli Storage Manager api rc.
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t smpapi_restore(dialog *sel_dialog)
{
   dsUint16_t rc = 0;

   if (!sel_dialog)
   {
      printf("*** Error in coding the restore dialog! ***\n");
   }
   else 
      rc = process_dialog((func_proto)perform_query_get,sel_dialog,Restore_Get);

   return rc;
}

/*----------------------------------------------------------------------+
| Name:    smpapi_restore2(dialog *sel_dialog)
|
| Action:  Automatically restores all items without prompt.
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK            - Successful
|          others           - Any other valid Tivoli Storage Manager api rc.
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t smpapi_restore2(dialog *sel_dialog)
{
   dsUint16_t rc = 0;

   no_restore = 1;

   if (!sel_dialog) 
   {
      printf("*** Error in coding the restore dialog! ***\n");
   }
   else 
      rc = process_dialog((func_proto)perform_query_get,sel_dialog,Restore_Get);

   no_restore = 0;

   return rc;
}

/*----------------------------------------------------------------------+
| Name:    smpapi_retrieve(dialog *sel_dialog)
|
| Action:  Prompts user for items to retrieve and then does so.
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK            - Successful
|          others           - Any other valid Tivoli Storage Manager api rc.
|          RC_NO_MEMORY     - Memory allocation call failed.
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t smpapi_retrieve(dialog *sel_dialog)
{
   dsUint16_t rc = 0;

   if (!sel_dialog)
   {
      printf("*** Error in coding the restore dialog! ***\n");
   }
   else 
      rc = process_dialog((func_proto)perform_query_get, sel_dialog, Retrieve_Get);

   return rc;
}

/*----------------------------------------------------------------------+
| Name:    smpapi_retrieve2(dialog *sel_dialog)
|
| Action:  Prompts user for items to retrieve and then does so without
|          further action.
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK            - Successful
|          others           - Any other valid Tivoli Storage Manager api rc.
|          RC_NO_MEMORY     - Memory allocation call failed.
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t smpapi_retrieve2(dialog *sel_dialog)
{
   dsUint16_t rc = 0;

   no_retrieve = 1;

   if (!sel_dialog)
   {
      printf("*** Error in coding the restore dialog! ***\n");
   }
   else 
      rc = process_dialog((func_proto)perform_query_get,sel_dialog,Retrieve_Get);

   no_retrieve = 0;

   return rc;
}

/*----------------------------------------------------------------------+
| Name:    smpapi_signon()
|
| Action:  Prompts user for backup objects to create and then sends those
|          objects to the DSM server.
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK            - Successful
|          others           - Any other valid Tivoli Storage Manager api rc.
|          RC_NO_MEMORY     - Memory allocation call failed.
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t smpapi_signon (dialog *sel_dialog)
{
   dsUint16_t rc = 0;

   if (!sel_dialog)
   {
      printf("*** Error in coding the signon dialog! ***\n");
   }
   else 
      rc = process_dialog(perform_signon,sel_dialog,0);

   return rc;
}

/*----------------------------------------------------------------------+
| Name:    smpapi_ex_signon()
|
| Action:  Prompts user for backup objects to create and then sends those
|          objects to the DSM server.
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK            - Successful
|          others           - Any other valid Tivoli Storage Manager api rc.
|          RC_NO_MEMORY     - Memory allocation call failed.
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t smpapi_ex_signon (dialog *sel_dialog)
{
   dsUint16_t rc = 0;

   if (!sel_dialog)
   {
      printf("*** Error in coding the extended signon dialog! ***\n");
   }
   else 
      rc = process_dialog(perform_signon,sel_dialog,ex_signon);

   return rc;
}

/*----------------------------------------------------------------------+
| Name:    smpapi_ex_signon_app()
|
| Action:  Prompts user for SignOn information plus application version
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK            - Successful
|          others           - Any other valid Tivoli Storage Manager api rc.
|          RC_NO_MEMORY     - Memory allocation call failed.
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t smpapi_ex_signon_app(dialog *sel_dialog)
{
   dsUint16_t rc = 0;

   if (!sel_dialog)
   {
      printf("*** Error in coding the extended signon dialog! ***\n");
   }
   else 
      rc = process_dialog(perform_signon, sel_dialog, ex_app_signon);

   return rc;
}


/*----------------------------------------------------------------------+
| Name:    smpapi_qry_archive(dialog *sel_dialog)
|
| Action:  Prompts user for filespace(s) to query and then calls
|          perform_query_fs to do the actual query.
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK            - Successful
|          others           - Any other valid Tivoli Storage Manager api rc.
|          RC_NO_MEMORY     - Memory allocation call failed.
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t smpapi_qry_archive (dialog *sel_dialog)
{
   dsUint16_t rc = 0;

   if (!sel_dialog)
   {
      printf("*** Error in coding the query archive dialog! ***\n");
   }
   else 
      rc = process_dialog(perform_query_get,sel_dialog,Archive_Get);

   return rc;
}

/*----------------------------------------------------------------------+
| Name:    smpapi_qry_backup(dialog *sel_dialog)
|
| Action:  Prompts user for filespace(s) to query and then calls
|          perform_query_fs to do the actual query.
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK            - Successful
|          others           - Any other valid Tivoli Storage Manager api rc.
|          RC_NO_MEMORY     - Memory allocation call failed.
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t smpapi_qry_backup (dialog *sel_dialog)
{
   dsUint16_t rc = 0;

   if (!sel_dialog)
   {
      printf("*** Error in coding the query backup dialog! ***\n");
   }
   else 
      rc = process_dialog(perform_query_get,sel_dialog,Backup_Get);

   return rc;
}

/*----------------------------------------------------------------------+
| Name:    smpapi_del_backup(dialog *sel_dialog)
|
| Action:  Prompts user for filespace(s) to query and then calls
|          perform_query_fs to do the actual query.
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK            - Successful
|          others           - Any other valid Tivoli Storage Manager api rc.
|          RC_NO_MEMORY     - Memory allocation call failed.
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t smpapi_del_backup (dialog *sel_dialog)
{
   dsUint16_t rc = 0;

   if (!sel_dialog)
   {
      printf("*** Error in coding the query backup dialog! ***\n");
   }
   else 
      rc = process_dialog(perform_query_get,sel_dialog,Backup_Delete);

   return rc;
}

/*----------------------------------------------------------------------+
| Name:    smpapi_del_archive(dialog *sel_dialog)
|
| Action:  Prompts user for filespace(s) to query and then calls
|          perform_query_fs to do the actual query.
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK            - Successful
|          others           - Any other valid Tivoli Storage Manager api rc.
|          RC_NO_MEMORY     - Memory allocation call failed.
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t smpapi_del_archive (dialog *sel_dialog)
{
   dsUint16_t rc = 0;

   if (!sel_dialog)
   {
      printf("*** Error in coding the query backup dialog! ***\n");
   }
   else 
      rc = process_dialog(perform_query_get,sel_dialog,Archive_Delete);

   return rc;
}
/*----------------------------------------------------------------------+
| Name:    smpapi_del_obj(dialog *sel_dialog)
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK            - Successful
|          others           - Any other valid Tivoli Storage Manager api rc.
|          RC_NO_MEMORY     - Memory allocation call failed.
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t smpapi_del_obj (dialog *sel_dialog)
{
   dsUint16_t rc = 0;

   if (!sel_dialog)
   {
      printf("*** Error in coding the query backup dialog! ***\n");
   }
   else 
      rc = process_dialog(perform_del_obj,sel_dialog,del_obj);

return rc;
}

/*----------------------------------------------------------------------+
| Name:    smpapi_ret_event(dialog *sel_dialog)
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK            - Successful
|          others           - Any other valid Tivoli Storage Manager api rc.
|          RC_NO_MEMORY     - Memory allocation call failed.
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t smpapi_ret_event (dialog *sel_dialog)
{
   dsUint16_t rc = 0;

   if (!sel_dialog)
   {
      printf("*** Error in coding the query backup dialog! ***\n");
   }
   else 
      rc = process_dialog(perform_ret_event,sel_dialog,ret_event);

   return rc;
}

/*----------------------------------------------------------------------+
| Name:    smpapi_sleep(dialog *sel_dialog)
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK            - Successful
|          others           - Any other valid Tivoli Storage Manager api rc.
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t smpapi_sleep (dialog *sel_dialog)
{
   dsUint16_t rc = 0;

   if (!sel_dialog)
   {
      printf("*** Error in coding the query backup dialog! ***\n");
   }
   else 
      rc = process_dialog(perform_sleep,sel_dialog,0);

   return rc;
}

/*----------------------------------------------------------------------+
| Name:    smpapi_rename_obj(dialog *sel_dialog)
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK            - Successful
|          others           - Any other valid Tivoli Storage Manager api rc.
|          RC_NO_MEMORY     - Memory allocation call failed.
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t smpapi_rename_obj (dialog *sel_dialog)
{
   dsUint16_t rc = 0;

   if (!sel_dialog)
   {
      printf("*** Error in coding the rename obj dialog! ***\n");
   }
   else 
      rc = process_dialog(perform_rename_obj,sel_dialog,rename_obj);

   return rc;
}

/*----------------------------------------------------------------------+
| Name:    smpapi_qry_fs(dialog *sel_dialog)
|
| Action:  Prompts user for filespace(s) to query and then calls
|          perform_query_fs to do the actual query.
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK            - Successful
|          others           - Any other valid Tivoli Storage Manager api rc.
|          RC_NO_MEMORY     - Memory allocation call failed.
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t smpapi_qry_fs (dialog *sel_dialog)
{
   dsUint16_t rc = 0;

   if (!sel_dialog)
   {
      printf("*** Error in coding the query filespace dialog! ***\n");
   }
   else 
      rc = process_dialog(perform_query_fs,sel_dialog,fs_query);

   return rc;
}

/*----------------------------------------------------------------------+
| Name:    smpapi_del_fs(dialog *sel_dialog)
|
| Action:  Prompts user for filespace(s) to delete and then calls
|          perform_query_fs to do the actual query/delete.
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK            - Successful
|          others           - Any other valid Tivoli Storage Manager api rc.
|          RC_NO_MEMORY     - Memory allocation call failed.
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t smpapi_del_fs (dialog *sel_dialog)
{
   dsUint16_t rc = 0;

   if (!sel_dialog)
   {
      printf("*** Error in coding the query filespace dialog! ***\n");
   }
   else 
      rc = process_dialog(perform_query_fs,sel_dialog,fs_delete);

   return rc;
}
/*----------------------------------------------------------------------+
| Name:    smpapi_del_access(dialog *sel_dialog)
|
| Action:  Prompts user for filespace(s) to delete and then calls
|          perform_query_fs to do the actual query/delete.
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK            - Successful
|          others           - Any other valid Tivoli Storage Manager api rc.
|          RC_NO_MEMORY     - Memory allocation call failed.
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t smpapi_del_access (dialog *sel_dialog)
{
   dsUint16_t rc = 0;

   if (!sel_dialog)
   {
      printf("*** Error in coding the delete access dialog! ***\n");
   }
   else 
      rc = process_dialog(perform_del_access,sel_dialog,fs_delete);

   return rc;
}

/*----------------------------------------------------------------------+
| Name:    smpapi_update_obj(dialog *sel_dialog)
|
| Action:  Prompts user for a backedup object to update then calls
|          perform_update_obj to do the actual update.
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK            - Successful
|          others           - Any other valid Tivoli Storage Manager api rc.
|          RC_NO_MEMORY     - Memory allocation call failed.
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t smpapi_update_obj (dialog *sel_dialog)
{
   dsUint16_t rc = 0;

   if (!sel_dialog)
   {
      printf("*** Error in coding the update object dialog! ***\n");
   }
   else 
      rc = process_dialog(perform_update_obj,sel_dialog,0);

   return rc;
}

/*----------------------------------------------------------------------+
| Name:    smpapi_update_objEx(dialog *sel_dialog)
|
| Action:  Prompts user for a backedup object to update then calls
|          perform_update_obj to do the actual update.
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK            - Successful
|          others           - Any other valid Tivoli Storage Manager api rc.
|          RC_NO_MEMORY     - Memory allocation call failed.
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t smpapi_update_objEx (dialog *sel_dialog)
{
   dsUint16_t rc = 0;

   if (!sel_dialog)
   {
      printf("*** Error in coding the update objectEx dialog! ***\n");
   }
   else 
      rc = process_dialog(perform_update_objEx,sel_dialog,0);

   return rc;
}
/*----------------------------------------------------------------------+
| Name:    smpapi_update_arc(dialog *sel_dialog)
|
| Action:  Prompts user for a archive objects to update then calls
|          perform_update_arc to do the actual update.
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK            - Successful
|          others           - Any other valid Tivoli Storage Manager api rc.
|          RC_NO_MEMORY     - Memory allocation call failed.
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t smpapi_update_arc (dialog *sel_dialog)
{
   dsUint16_t rc = 0;

   if (!sel_dialog)
   {
      printf("*** Error in coding the update object dialog! ***\n");
   }
   else 
      rc = process_dialog(perform_update_arc,sel_dialog,0);

   return rc;
}

/*----------------------------------------------------------------------+
| Name:    smpapi_update_fs(dialog *sel_dialog)
|
| Action:  Prompts user for a backedup object to update then calls
|          perform_update_fsj to do the actual update.
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK            - Successful
|          others           - Any other valid Tivoli Storage Manager api rc.
|          RC_NO_MEMORY     - Memory allocation call failed.
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t smpapi_update_fs (dialog *sel_dialog)
{
   dsUint16_t rc = 0;

   if (!sel_dialog)
   {
      printf("*** Error in coding the update file space dialog! ***\n");
   }
   else 
      rc = process_dialog(perform_update_fs,sel_dialog,0);


   return rc;
}

/*----------------------------------------------------------------------+
| Name:    smpapi_log_event(dialog *sel_dialog)
|
| Action:  Prompts user for a message and a file to log event to
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK            - Successful
|          others           - Any other valid Tivoli Storage Manager api rc.
|          RC_NO_MEMORY     - Memory allocation call failed.
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t smpapi_log_event (dialog *sel_dialog)
{
   dsUint16_t rc = 0;

   if (!sel_dialog)
   {
      printf("*** Error in coding the log Event dialog! ***\n");
   }
   else 
      rc = process_dialog(perform_log_event, sel_dialog, log_event);

   return rc;
}
/*----------------------------------------------------------------------+
| Name:    smpapi_setup(dialog *sel_dialog)
|
| Action:  Prompts user for a message and a file to log event to
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK            - Successful
|          others           - Any other valid Tivoli Storage Manager api rc.
|          RC_NO_MEMORY     - Memory allocation call failed.
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t smpapi_setup (dialog *sel_dialog)
{
   dsUint16_t rc = 0;

   if (!sel_dialog)
   {
      printf("*** Error in coding the setup dialog! ***\n");
   }
   else 
      rc = process_dialog(perform_setup, sel_dialog, setup);

   return rc;
}


/*----------------------------------------------------------------------+
| Name:    smpapi_set_access(dialog *sel_dialog)
|
| Action:  Prompts user for a message and a file to log event to
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK            - Successful
|          others           - Any other valid Tivoli Storage Manager api rc.
|          RC_NO_MEMORY     - Memory allocation call failed.
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t smpapi_set_access (dialog *sel_dialog)
{
   dsUint16_t rc = 0;

   if (!sel_dialog)
   {
      printf("*** Error in coding the set_access dialog! ***\n");
   }
   else 
      rc = process_dialog(perform_set_access, sel_dialog, set_access);

   return rc;
}

/*----------------------------------------------------------------------+
| Name:    smpapi_qry_mc(dialog *sel_dialog)
|
| Action:  Prompts user for mgmt classes to query and then calls
|          perform_query_mc to do the actual query.
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK            - Successful
|          others           - Any other valid Tivoli Storage Manager api rc.
|          RC_NO_MEMORY     - Memory allocation call failed.
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t  smpapi_qry_mc (dialog *sel_dialog)
{
   dsUint16_t rc = 0;

   if (!sel_dialog)
   {
      printf("*** Error in coding the query filespace dialog! ***\n");
   }
   else 
      rc = process_dialog(perform_query_mc,sel_dialog,0);
   
   return rc;
}

/*----------------------------------------------------------------------+
| Name:    smpapi_qry_sess(dialog *sel_dialog)
|
| Action:  Displays session values returned from a query session.
|
| Input:   sel_dialog  - Null pointer for this call
|
| Returns: RC_OK            - Successful
|          others           - Any other valid Tivoli Storage Manager api rc.
|          RC_NO_MEMORY     - Memory allocation call failed.
|
| Side
| Effects: Resets the global variable dsmSessInfo that is set upon
|          session signon.  The query is re-issued here for
|          completeness of the sample and testing.
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t smpapi_qry_sess(dialog *sel_dialog)
{
   dsInt16_t rc;

   if (!dsmHandle)
   {
      printf("You must signon before performing this action.\n");
      return(0);
   }

   memset(&dsmSessInfo,0x00,sizeof(ApiSessInfo));  /* Zero out block.     */
   dsmSessInfo.stVersion = ApiSessInfoVersion;     /* Init struct version */
   rc = dsmQuerySessInfo(dsmHandle,                /* Our session handle  */
                         &dsmSessInfo);            /* Output structure.   */
   if (rc) 
   {
      printf("*** Error on issueing Query Session \n");
      rcApiOut(dsmHandle, rc);
   }
   else 
      output_sessinfo();

   return 0;
}

/*----------------------------------------------------------------------+
| Name:    smpapi_qry_version(dialog *sel_dialog)
|
| Action:  Displays version the API being called is at.
|
| Input:   sel_dialog  - Null pointer for this call
|
| Returns: RC_OK            - Successful
|
| Side
| Effects: Resets the global variable apiLibVer that is set with the API
|          version upon entry to the sample application.  It is done
|          here to actually re-issue the query.
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t smpapi_qry_version(dialog* sel_dialog)
{

   memset(&apiLibVer,0x00,sizeof(dsmApiVersionEx));
   dsmQueryApiVersionEx(&apiLibVer);
   printf("\nAPI Version = %d.%d.%d.%d\n\n",apiLibVer.version,
                                         apiLibVer.release,
                                         apiLibVer.level,
                                         apiLibVer.subLevel);

   return 0;
}
/*----------------------------------------------------------------------+
| Name:    smpapi_qry_access()
|
| Action:  Displays session values returned from a query session.
|
| Input:   sel_dialog  - Null pointer for this call
|
| Returns: RC_OK            - Successful
|          others           - Any other valid Tivoli Storage Manager api rc.
|          RC_NO_MEMORY     - Memory allocation call failed.
|
| Side
| Effects: Resets the global variable dsmSessInfo that is set upon
|          session signon.  The query is re-issued here for
|          completeness of the sample and testing.
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t smpapi_qry_access(dialog* sel_dialog)
{
   dsInt16_t rc;
   dsUint16_t numberOfRules;
   qryRespAccessData *accessListP;
   dsInt16_t index =0;
   char  path[DSM_MAX_FSNAME_LENGTH + DSM_MAX_HL_LENGTH + DSM_MAX_LL_LENGTH+1];

   if (!dsmHandle)
   {
      printf("You must signon before performing this action.\n");
      return(0);
   }

   rc = dsmQueryAccess(dsmHandle, &accessListP, &numberOfRules); 
   if ( rc |= DSM_RC_OK)
   {
      printf("\n*** Error on issueing Query Access : \n");
      rcApiOut(dsmHandle, rc);
   }
   else
   {

      printf(" id    Type      Node        User        Path\n");
      printf("-----  ------    ------------------------------\n");
      while(numberOfRules)
      {
         strcpy(path, accessListP->objName.fs);
         strcat(path, accessListP->objName.hl);
         strcat(path, accessListP->objName.ll);
         printf("%3d    %-9s %-10s  %-10s  %s \n",accessListP->ruleNumber, 
               accessListP->accessType == atBackup ? "BACKUP" : "ARCHIVE",
               accessListP->node, accessListP->owner, path  );
         accessListP++;
         numberOfRules--;
      }
   }
   return rc;
}


/*----------------------------------------------------------------------+
| Name:    smpapi_qry_cliOptions(dialog *sel_dialog)
|
| Action:  Displays session values returned from a query session.
|
| Input:   sel_dialog  - Null pointer for this call
|
| Returns: RC_OK            - Successful
|          others           - Any other valid Tivoli Storage Manager api rc.
|          RC_NO_MEMORY     - Memory allocation call failed.
|
| Side
| Effects: Resets the global variable dsmSessInfo that is set upon
|          session signon.  The query is re-issued here for
|          completeness of the sample and testing.
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t smpapi_qry_cliOptions(dialog *sel_dialog)
{
   dsInt16_t rc;
   optStruct dsmOpt;

   rc = dsmQueryCliOptions( &dsmOpt);            /* Output structure.   */

   if (rc) 
   {
      printf("*** Error on issueing Query Option files.\n");
      rcApiOut(dsmHandle, rc);
   }
   else
   {
      printf("\n");
      printf("DSMI_DIR               >%s<\n", dsmOpt.dsmiDir);
      printf("DSMI_CONFIG            >%s<\n", dsmOpt.dsmiConfig);
      printf("serverName             >%s<\n", dsmOpt.serverName);
      printf("commMethod              %d\n",  dsmOpt.commMethod);
      printf("serverAddress          >%s<\n", dsmOpt.serverAddress);
      printf("nodeName               >%s<\n", dsmOpt.nodeName);
      printf("compress                %d\n",   dsmOpt.compression);
      printf("compressalways          %d\n",   dsmOpt.compressalways);
      printf("passwordAccess          %d\n",   dsmOpt.passwordAccess);
      return 0;
   }
   return rc;
}

/*----------------------------------------------------------------------+
| Name:    smpapi_qry_sess(dialog *sel_dialog)
|
| Action:  Displays session values returned from a query session.
|
| Input:   sel_dialog  - Null pointer for this call
|
| Returns: RC_OK            - Successful
|          others           - Any other valid Tivoli Storage Manager api rc.
|          RC_NO_MEMORY     - Memory allocation call failed.
|
| Side
| Effects: Resets the global variable dsmSessInfo that is set upon
|          session signon.  The query is re-issued here for
|          completeness of the sample and testing.
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t smpapi_qry_sessOptions(dialog *sel_dialog)
{
   dsInt16_t rc = 0;
   optStruct dsmOpt;

   if (!dsmHandle)
   {
      printf("You must signon before performing this action.\n");
      return(0);
   }

   rc = dsmQuerySessOptions(dsmHandle,           /* Our session handle  */
                            &dsmOpt);            /* Output structure.   */
   if (rc) printf("*** Error on issueing Query Session Options .  Rc = %d\n",rc);
   else
   {
      printf("\n");
      printf("DSMI_DIR               >%s<\n", dsmOpt.dsmiDir);
      printf("DSMI_CONFIG            >%s<\n", dsmOpt.dsmiConfig);
      printf("serverName             >%s<\n", dsmOpt.serverName);
      printf("commMethod              %d\n",  dsmOpt.commMethod);
      printf("serverAddress          >%s<\n", dsmOpt.serverAddress);
      printf("nodeName               >%s<\n", dsmOpt.nodeName);
      printf("compress                %d\n",   dsmOpt.compression);
      printf("compressalways          %d\n",   dsmOpt.compressalways);
      printf("passwordAccess          %d\n",   dsmOpt.passwordAccess);
      return 0;
   }

   return rc;
}

/*----------------------------------------------------------------------+
| Name:    process_dialog()
|
| Action:  Displays the passed dialog, and after all entries entered
|          calls the passed function with the passed dialog.
|          On return from the called function the dialog memory will
|          be freed for another invocation.
|
| Input:   func()      - Pointer to the function to invoke
|          sel_dialog  - Pointer to dialog table to process.
|
| Returns: RC_OK            - Successful
|          RC_NO_MEMORY     - Memory allocation call failed.
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
static dsInt16_t process_dialog(func_proto func,
                     dialog *sel_dialog,
                     dsUint16_t parm1)
{
   dsUint16_t   i;
   dsUint8_t    done;
   dsUint8_t    done2;
   dsUint16_t   rc = 0;
   char     s[INPUT_BUFFER_SIZE];
   dialog   *dlg_item;

   i=0;
   done = FALSE;
   while (!done)
   {
      dlg_item = &sel_dialog[i];               /* Specific item to work on  */
      if (dlg_item->item_type != DSMAPI_END)   /* In not at end of dialog   */
      {
         /*=== extra () to suppress warning ===*/
         if ((rc = get_input(dlg_item)))          /* Put msg out, check for error*/
         {
            printf("** Error on data input **  rc = %d\n",rc);
            done = TRUE;
         }
         else 
            i++;                              /* Next item.                */
      }
      else                                     /* At end of dialog.         */
      {
         done2 = FALSE;
         while (!done2)
         {
            printf("Are the above responses correct (y/n/q)? \n");
            ourGetS(s);
            switch (*s)
            {
               case 'n' :
               case 'N' :
                  free_dialog_buffs(sel_dialog);
                  done2 = TRUE;             /* Done with inner loop */
                  break;
               case 'q' :
               case 'Q' :
                  done = TRUE;              /* Done with outer loop */
                  done2 = TRUE;             /* Done with inner loop */
                  break;
               case 'y' :
               case 'Y' :
               default  :
                  rc = func(sel_dialog,parm1);
                  done = TRUE;              /* Done with outer loop */
                  done2 = TRUE;             /* Done with inner loop */
               } /* switch */
         } /* while */
         i=0;
      } /* else */
   } /* while */
   free_dialog_buffs(sel_dialog);
   return rc;
}


/*----------------------------------------------------------------------+
| Name:    smpapi_qry_prx_auth()
|
| Action:  Displays session values returned from a proxy authority query.
|
| Input:   sel_dialog  - Null pointer for this call
|
| Returns: RC_OK            - Successful
|          others           - Any other valid Tivoli Storage Manager api rc.
|          RC_NO_MEMORY     - Memory allocation call failed.
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t smpapi_qry_prx_auth(dialog* sel_dialog)
{
   dsUint16_t rc = 0;

   return perform_query_prx(sel_dialog, qtProxyNodeAuth);
   
}

/*----------------------------------------------------------------------+
| Name:    smpapi_qry_prx_peers()
|
| Action:  Displays session values returned from a proxy authority query.
|
| Input:   sel_dialog  - Null pointer for this call
|
| Returns: RC_OK            - Successful
|          others           - Any other valid Tivoli Storage Manager api rc.
|          RC_NO_MEMORY     - Memory allocation call failed.
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
dsInt16_t smpapi_qry_prx_peers(dialog* sel_dialog)
{
   dsUint16_t rc = 0;

   if (!sel_dialog)
   {
      printf("*** Error in coding the query filespace dialog! ***\n");
   }
   else 
      rc = process_dialog(perform_query_prx,sel_dialog,qtProxyNodePeer);
   
   return rc;
}

/*----------------------------------------------------------------------+
| Name: smpapi_query_exit
.----------------------------------------------------------------------*/
dsInt16_t smpapi_query_exit(dialog *sel_dialog)
{
   return 0;
}

/*----------------------------------------------------------------------+
| Name: smpapi_utilities_exit
.----------------------------------------------------------------------*/
dsInt16_t smpapi_utilities_exit(dialog *sel_dialog)
{
   return 0;
}

/*----------------------------------------------------------------------+
| Private routines
.----------------------------------------------------------------------*/

/*----------------------------------------------------------------------+
| Name:    get_input
|
| Action:  Prompts user for input of passed dialog line.  Input is then
|          put into malloc'd storage in the passed buffer.
|
| Returns:
|     RC_OK            - Successful
+----------------------------------------------------------------------*/
static dsInt16_t get_input(dialog *line)
{
   char    *input_buff;
   dsUint32_t  len;
   dsInt16_t   rc = 0;

   input_buff = (char*)malloc(INPUT_BUFFER_SIZE);
   printf("%s",line->item_desc); fflush(stdout);
   ourGetS(input_buff);
   len = strlen(input_buff);

   if (!(line->item_buff = (char*)malloc(len + 1)))
      rc = RC_NO_MEMORY;
   else
      strcpy(line->item_buff,input_buff);
      
   free(input_buff);
   return rc;

}

/*----------------------------------------------------------------------+
| Name:    free_dialog_buffs
|
| Action:  Free any buffers for dialogs and reset to NULL.
|
| Returns:
|     RC_OK            - Successful
+----------------------------------------------------------------------*/
static int  free_dialog_buffs(dialog *sel_dialog)
{
   dialog *dlg;
   dsUint16_t i;
   dsUint8_t done;

   i=0;
   done = FALSE;
   while (!done)
   {
      dlg = &sel_dialog[i];
      if (dlg->item_type != DSMAPI_END)
      {
         if (dlg->item_buff)
         {
            free(dlg->item_buff);
            dlg->item_buff = NULL;
            i++;
         }
      }
      else 
         done = TRUE;
   }
   return 0;
}

/*----------------------------------------------------------------------+
| Name:    output_sessinfo()
|
| Action:  Printf the session info structure.  Refer to the dsmapi.h
|          file for format and possible values for the structure.
|
| Input:   None
|
| Returns: None
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
static void output_sessinfo(void)
{
   char t[50];

   printf("\ndsmQuerySessInfo Values:\n");
   printf("  Server Information:\n");
   printf("    Server name   : %s\n",dsmSessInfo.adsmServerName);
   printf("    Server Host   : %s\n",dsmSessInfo.serverHost);
   printf("    Server port   : %u\n",dsmSessInfo.serverPort);
   printf("    Server date   : %i/%i/%i  %i:%i:%i\n",
                         dsmSessInfo.serverDate.year,
                  (dsInt16_t)dsmSessInfo.serverDate.month,
                  (dsInt16_t)dsmSessInfo.serverDate.day,
                  (dsInt16_t)dsmSessInfo.serverDate.hour,
                  (dsInt16_t)dsmSessInfo.serverDate.minute,
                  (dsInt16_t)dsmSessInfo.serverDate.second);
   printf("    Server type   : %s\n",dsmSessInfo.serverType);
   printf("    Server version: %i.%i.%i.%i\n",dsmSessInfo.serverVer,
                                              dsmSessInfo.serverRel,
                                              dsmSessInfo.serverLev,
                                              dsmSessInfo.serverSubLev);
   printf("    Server Archive Retention Protection : %s\n",
          dsmSessInfo.archiveRetentionProtection ? "YES" : "NO");

   printf("  Replication Server Infomation\n"
          "    Home Server name        : %s\n"
          "    Replication Server name : %s\n"
          "       Host                 : %s\n"
          "       Port                 : %d\n"
          "    Fail over status        : %s\n",
          dsmSessInfo.homeServerName, 
          dsmSessInfo.replServerName, 
          dsmSessInfo.replServerHost,
          dsmSessInfo.replServerPort,
          dsmSessInfo.failOverCfgType == failOvrNotConfigured ? "Not Configured" : 
             dsmSessInfo.failOverCfgType == failOvrConfigured ?  "Configured" :
             dsmSessInfo.failOverCfgType == failOvrConnectedToReplServer ?  "Connected to replication server" :
             "unknown" );
   
   
   
   printf("  Client Information:\n");
   printf("    Client node type          : %s\n",dsmSessInfo.nodeType);
   printf("    Client filespace delimiter: %c\n",dsmSessInfo.fsdelim);
   printf("    Client hl & ll delimiter  : %c\n",dsmSessInfo.hldelim);

   switch (dsmSessInfo.compression)
   {
      case COMPRESS_YES :
         strcpy(t,"ON"); 
         break;
      case COMPRESS_NO :
         strcpy(t,"OFF"); 
         break;
      case COMPRESS_CD :
         strcpy(t,"Client determined"); 
         break;
      default :
         strcpy(t,"Unknown compress state.");
   }
   printf("    Client compression        : %s (%iu)\n",t,(dsUint16_t)dsmSessInfo.compression);

   switch (dsmSessInfo.archDel)
   {
      case ARCHDEL_YES :
         strcpy(t,"Client can delete archived objects"); 
         break;
      case ARCHDEL_NO :
         strcpy(t,"Client CANNOT delete archived objects"); 
         break;
      default :
         strcpy(t,"Unknown archive delete state");
   }
   printf("    Client archive delete     : %s\n",t);


   switch (dsmSessInfo.backDel)
   {
      case BACKDEL_YES :
         strcpy(t,"Client can delete backup objects"); 
         break;
      case ARCHDEL_NO :
         strcpy(t,"Client CANNOT delete backup objects"); 
         break;
      default :
         strcpy(t,"Unknown backup delete state");
   }
   printf("    Client backup delete      : %s\n",t);
   printf("    Maximum objects in multiple object transactions: %u\n",
                   dsmSessInfo.maxObjPerTxn);
   printf("    Lan free Enabled          : %s\n",dsmSessInfo.lanFreeEnabled ? "YES" : "NO");
   printf("    Deduplication             : %s\n",dsmSessInfo.dedupType ==  dedupClientOrServer ?
                             "Client Or Server" : "Server Only");

   printf("  General session info:\n");
   printf("    Node           : %s\n",dsmSessInfo.id);
   printf("    Access Node    : %s\n",dsmSessInfo.accessNode);
   printf("    Owner          : %s\n",dsmSessInfo.owner);
   printf("    API Config file: %s\n",dsmSessInfo.confFile);
   
   printf("  Policy Information:\n");
   printf("    Domain name                   : %s\n",dsmSessInfo.domainName);
   printf("    Policyset name                : %s\n",dsmSessInfo.policySetName);
   printf("    Policy activation date        : %i/%i/%i  %i:%i:%i\n",
                             dsmSessInfo.polActDate.year,
                  (dsInt16_t)dsmSessInfo.polActDate.month,
                  (dsInt16_t)dsmSessInfo.polActDate.day,
                  (dsInt16_t)dsmSessInfo.polActDate.hour,
                  (dsInt16_t)dsmSessInfo.polActDate.minute,
                  (dsInt16_t)dsmSessInfo.polActDate.second);
   printf("    Default management class      : %s\n",dsmSessInfo.dfltMCName);
   printf("    Backup retention grace period : %u days\n",
                    dsmSessInfo.gpBackRetn);
   printf("    Archive retention grace period: %u days\n",
                    dsmSessInfo.gpArchRetn);
}

/*----------------------------------------------------------------------+
| Name:    output_optioninfo()
|
| Action:  Printf the session info structure.  Refer to the dsmapi.h
|          file for format and possible values for the structure.
|
| Input:   None
|
| Returns: None
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
static void output_otioninfo()
{
   printf("so far so good \n");
}

