// Licensed to the Software Freedom Conservancy (SFC) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The SFC licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

package org.openqa.selenium.bidi;

import static java.util.logging.Level.INFO;
import static org.openqa.selenium.concurrent.Lazy.lazy;

import com.google.auto.service.AutoService;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Optional;
import java.util.function.Predicate;
import java.util.logging.Logger;
import org.openqa.selenium.Capabilities;
import org.openqa.selenium.concurrent.Lazy;
import org.openqa.selenium.remote.AugmenterProvider;
import org.openqa.selenium.remote.ExecuteMethod;
import org.openqa.selenium.remote.RemoteExecuteMethod;
import org.openqa.selenium.remote.http.ClientConfig;
import org.openqa.selenium.remote.http.HttpClient;

@SuppressWarnings({"rawtypes", "RedundantSuppression"})
@AutoService(AugmenterProvider.class)
public class BiDiProvider implements AugmenterProvider<HasBiDi> {
  private static final Logger LOG = Logger.getLogger(BiDiProvider.class.getName());

  @Override
  public Predicate<Capabilities> isApplicable() {
    return caps -> getBiDiUrl(caps).isPresent();
  }

  @Override
  public Class<HasBiDi> getDescribedInterface() {
    return HasBiDi.class;
  }

  @Override
  public HasBiDi getImplementation(Capabilities caps, ExecuteMethod executeMethod) {
    final Lazy<BiDi> biDi = lazy(() -> establishBiDiConnection(caps, executeMethod));

    LOG.log(
        INFO,
        "WebDriver augmented with BiDi interface; connection will not be verified until first"
            + " use.");

    return new HasBiDi() {
      @Override
      public Optional<BiDi> maybeGetBiDi() {
        return biDi.getIfInitialized();
      }

      @Override
      public BiDi getBiDi() {
        return biDi.get();
      }
    };
  }

  private BiDi establishBiDiConnection(Capabilities caps, ExecuteMethod executeMethod) {
    URI wsUri = getBiDiUrl(caps).orElseThrow(() -> new BiDiException("BiDi not supported"));
    ClientConfig clientConfig =
        ((RemoteExecuteMethod) executeMethod).getWrappedDriver().getClientConfig();

    HttpClient.Factory clientFactory = HttpClient.Factory.createDefault();
    ClientConfig wsConfig = clientConfig.baseUri(wsUri);
    HttpClient wsClient = clientFactory.createClient(wsConfig);
    Connection connection = new Connection(wsClient, wsUri.toString());

    return new BiDi(connection, wsConfig.wsTimeout());
  }

  private Optional<URI> getBiDiUrl(Capabilities caps) {
    Object biDiCapability = caps.getCapability("webSocketUrl");
    Optional<String> webSocketUrl = Optional.ofNullable((String) biDiCapability);

    return webSocketUrl.map(
        uri -> {
          try {
            return new URI(uri);
          } catch (URISyntaxException e) {
            return null;
          }
        });
  }
}
