/*
 * Copyright (c) 2015, 2025, Oracle and/or its affiliates.
 *
 * This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License, version 2.0, as published by
 * the Free Software Foundation.
 *
 * This program is designed to work with certain software that is licensed under separate terms, as designated in a particular file or component or in
 * included license documentation. The authors of MySQL hereby grant you an additional permission to link the program and your derivative works with the
 * separately licensed software that they have either included with the program or referenced in the documentation.
 *
 * Without limiting anything contained in the foregoing, this file, which is part of MySQL Connector/J, is also subject to the Universal FOSS Exception,
 * version 1.0, a copy of which can be found at http://oss.oracle.com/licenses/universal-foss-exception.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License, version 2.0, for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

package com.mysql.cj.result;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.nio.ByteBuffer;

import com.mysql.cj.Constants;
import com.mysql.cj.conf.PropertySet;

/**
 * A {@link ValueFactory} to create {@link java.math.BigDecimal} instances.
 */
public class BigDecimalValueFactory extends AbstractNumericValueFactory<BigDecimal> {

    int scale;
    boolean hasScale;

    public BigDecimalValueFactory(PropertySet pset) {
        super(pset);
    }

    public BigDecimalValueFactory(PropertySet pset, int scale) {
        super(pset);
        this.scale = scale;
        this.hasScale = true;
    }

    @Override
    public BigDecimal createFromLong(long l) {
        return adjustResult(BigDecimal.valueOf(l));
    }

    @Override
    public BigDecimal createFromBigInteger(BigInteger i) {
        return adjustResult(new BigDecimal(i));
    }

    @Override
    public BigDecimal createFromDouble(double d) {
        return adjustResult(BigDecimal.valueOf(d));
    }

    @Override
    public BigDecimal createFromBigDecimal(BigDecimal d) {
        return adjustResult(d);
    }

    @Override
    public BigDecimal createFromBit(byte[] bytes, int offset, int length) {
        return new BigDecimal(new BigInteger(ByteBuffer.allocate(length + 1).put((byte) 0).put(bytes, offset, length).array()));
    }

    @Override
    public String getTargetTypeName() {
        return BigDecimal.class.getName();
    }

    /**
     * Adjust the result value by replacing new "zero" instances by {@link BigDecimal} cached ones, and applying the scale, if appropriate.
     *
     * @param d
     *            value
     * @return result
     */
    private BigDecimal adjustResult(BigDecimal d) {
        if (d.signum() == 0) {
            d = Constants.BIG_DECIMAL_ZERO.setScale(d.scale());
        }
        if (this.hasScale) {
            try {
                return d.setScale(this.scale);
            } catch (ArithmeticException ex) {
                // try this if above fails
                return d.setScale(this.scale, BigDecimal.ROUND_HALF_UP);
            }
        }

        return d;
    }

}
