/*
 * Copyright (c) 2018, 2025, Oracle and/or its affiliates. All rights reserved.
 * 
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * The contents of this file are subject to the terms of either the Universal Permissive License
 * v 1.0 as shown at https://oss.oracle.com/licenses/upl
 *
 * or the following license:
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted
 * provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions
 * and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other materials provided with
 * the distribution.
 * 
 * 3. Neither the name of the copyright holder nor the names of its contributors may be used to
 * endorse or promote products derived from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY
 * WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.openjdk.jmc.ui.misc;

import java.util.Arrays;

import org.eclipse.swt.dnd.ByteArrayTransfer;
import org.eclipse.swt.dnd.Clipboard;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.dnd.TransferData;
import org.eclipse.swt.widgets.Display;

/**
 * Class for adding data to clipboard.
 */
public class ClipboardManager {

	private static class LocalObjectTransfer extends ByteArrayTransfer {

		private final static LocalObjectTransfer INSTANCE = new LocalObjectTransfer();
		private static final String TYPE_NAME = "local-object-transfer-format:" + System.currentTimeMillis() + ":" //$NON-NLS-1$ //$NON-NLS-2$
				+ INSTANCE.hashCode();
		private static final int TYPE_ID = registerType(TYPE_NAME);
		private static final byte[] TYPE_NAME_BYTES = TYPE_NAME.getBytes();

		private Object obj;

		@Override
		protected int[] getTypeIds() {
			return new int[] {TYPE_ID};
		}

		@Override
		protected String[] getTypeNames() {
			return new String[] {TYPE_NAME};
		}

		@Override
		protected void javaToNative(Object object, TransferData transferData) {
			super.javaToNative(TYPE_NAME_BYTES, transferData);
			obj = object;
		}

		@Override
		protected Object nativeToJava(TransferData transferData) {
			Object o = super.nativeToJava(transferData);
			if (o instanceof byte[] && Arrays.equals((byte[]) o, TYPE_NAME_BYTES)) {
				return obj;
			}
			return null;
		}

	}

	public static Transfer getClipboardLocalTransfer() {
		return LocalObjectTransfer.INSTANCE;
	}

	public static void setClipboardContents(Object[] data, Transfer[] dataTypes) {
		Clipboard clipboard = new Clipboard(Display.getCurrent());
		try {
			clipboard.setContents(data, dataTypes);
		} finally {
			clipboard.dispose();
		}
	}

	public static Object getClipboardContents(Transfer dataType) {
		Clipboard clipboard = new Clipboard(Display.getCurrent());
		try {
			return clipboard.getContents(dataType);
		} finally {
			clipboard.dispose();
		}
	}
}
