/*
 * Copyright (c) 2021, 2025, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

/*
 * @test
 * @summary Tests the FileServerHandler's mapping of request URI path to file
 *          system path
 * @library /test/lib
 * @build jdk.test.lib.Platform jdk.test.lib.net.URIBuilder
 * @run junit/othervm MapToPathTest
 */

import java.io.IOException;
import java.io.UncheckedIOException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse.BodyHandlers;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.logging.ConsoleHandler;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Stream;
import com.sun.net.httpserver.Filter;
import com.sun.net.httpserver.Headers;
import com.sun.net.httpserver.HttpHandlers;
import com.sun.net.httpserver.HttpServer;
import com.sun.net.httpserver.SimpleFileServer;
import com.sun.net.httpserver.SimpleFileServer.OutputLevel;
import jdk.test.lib.net.URIBuilder;
import jdk.test.lib.util.FileUtils;
import static java.lang.System.out;
import static java.net.http.HttpClient.Builder.NO_PROXY;
import static java.nio.file.StandardOpenOption.CREATE;

import org.junit.jupiter.api.AfterAll;
import static org.junit.jupiter.api.Assertions.assertEquals;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;

public class MapToPathTest {

    static final Path CWD = Path.of(".").toAbsolutePath();
    static final Path TEST_DIR = CWD.resolve("MapToPathTest").normalize();

    static final InetSocketAddress LOOPBACK_ADDR = new InetSocketAddress(InetAddress.getLoopbackAddress(), 0);
    static final Filter OUTPUT_FILTER = SimpleFileServer.createOutputFilter(out, OutputLevel.VERBOSE);

    static final boolean ENABLE_LOGGING = true;
    static final Logger LOGGER = Logger.getLogger("com.sun.net.httpserver");

    @BeforeAll
    public static void setup() throws IOException {
        if (ENABLE_LOGGING) {
            ConsoleHandler ch = new ConsoleHandler();
            LOGGER.setLevel(Level.ALL);
            ch.setLevel(Level.ALL);
            LOGGER.addHandler(ch);
        }
        if (Files.exists(TEST_DIR)) {
            FileUtils.deleteFileTreeWithRetry(TEST_DIR);
        }
        createDirectories(TEST_DIR);
    }

    private static void createDirectories(Path testDir) throws IOException {
        //      Create directory tree:
        //
        //      |-- TEST_DIR
        //          |-- foo
        //              |-- bar
        //                  |-- baz
        //                      |-- file.txt
        //              |-- file.txt
        //          |-- foobar
        //              |-- file.txt
        //          |-- file.txt

        Files.createDirectories(TEST_DIR);
        Stream.of("foo", "foobar", "foo/bar/baz").forEach(s -> {
            try {
                Path p = testDir.resolve(s);
                Files.createDirectories(p);
                Files.writeString(p.resolve("file.txt"), s, CREATE);
            } catch (IOException ioe) {
                throw new UncheckedIOException(ioe);
            }
        });
        Files.writeString(testDir.resolve("file.txt"), "testdir", CREATE);
    }

    @Test
    public void test() throws Exception {
        var client = HttpClient.newBuilder().proxy(NO_PROXY).build();
        {
            var handler = SimpleFileServer.createFileHandler(TEST_DIR);
            var server = HttpServer.create(LOOPBACK_ADDR, 10, "/", handler, OUTPUT_FILTER);
            server.start();
            try {
                var req1 = HttpRequest.newBuilder(uri(server, "/")).build();
                var res1 = client.send(req1, BodyHandlers.ofString());
                assertEquals(200, res1.statusCode());
                assertEquals("text/html; charset=UTF-8", res1.headers().firstValue("content-type").get());
                assertEquals(Long.toString(257L), res1.headers().firstValue("content-length").get());
                assertEquals(getLastModified(TEST_DIR), res1.headers().firstValue("last-modified").get());

                var req2 = HttpRequest.newBuilder(uri(server, "/../")).build();
                var res2 = client.send(req2, BodyHandlers.ofString());
                assertEquals(404, res2.statusCode());  // cannot escape root

                var req3 = HttpRequest.newBuilder(uri(server, "/foo/bar/baz/c://")).build();
                var res3 = client.send(req3, BodyHandlers.ofString());
                assertEquals(404, res3.statusCode());  // not found

                var req4 = HttpRequest.newBuilder(uri(server, "/foo/bar/baz/c:.//")).build();
                var res4 = client.send(req4, BodyHandlers.ofString());
                assertEquals(404, res4.statusCode());  // not found

                var req5 = HttpRequest.newBuilder(uri(server, "/foo/bar/baz/c:..//")).build();
                var res5 = client.send(req5, BodyHandlers.ofString());
                assertEquals(404, res5.statusCode());  // not found

                var req6 = HttpRequest.newBuilder(uri(server, "/foo/file:" + TEST_DIR.getParent())).build();
                var res6 = client.send(req6, BodyHandlers.ofString());
                assertEquals(404, res6.statusCode());  // not found

                var req7 = HttpRequest.newBuilder(uri(server, "/foo/bar/\\..\\../")).build();
                var res7 = client.send(req7, BodyHandlers.ofString());
                assertEquals(404, res7.statusCode());  // not found

                var req8 = HttpRequest.newBuilder(uri(server, "/foo")).build();
                var res8 = client.send(req8, BodyHandlers.ofString());
                assertEquals(301, res8.statusCode());  // redirect
                assertEquals("0", res8.headers().firstValue("content-length").get());
                assertEquals("/foo/", res8.headers().firstValue("location").get());
            } finally {
                server.stop(0);
            }
        }
        {
            var handler = SimpleFileServer.createFileHandler(TEST_DIR);
            var server = HttpServer.create(LOOPBACK_ADDR, 10, "/browse/", handler, OUTPUT_FILTER);
            server.start();
            try {
                var req1 = HttpRequest.newBuilder(uri(server, "/browse/file.txt")).build();
                var res1 = client.send(req1, BodyHandlers.ofString());
                assertEquals(200, res1.statusCode());
                assertEquals("testdir", res1.body());
                assertEquals("text/plain", res1.headers().firstValue("content-type").get());
                assertEquals(Long.toString(7L), res1.headers().firstValue("content-length").get());
                assertEquals(getLastModified(TEST_DIR.resolve("file.txt")), res1.headers().firstValue("last-modified").get());

                var req2 = HttpRequest.newBuilder(uri(server, "/store/file.txt")).build();
                var res2 = client.send(req2, BodyHandlers.ofString());
                assertEquals(404, res2.statusCode());  // no context found
            } finally {
                server.stop(0);
            }
        }
        {
            // Test "/foo/" context (with trailing slash)
            var handler = SimpleFileServer.createFileHandler(TEST_DIR.resolve("foo"));
            var server = HttpServer.create(LOOPBACK_ADDR, 10, "/foo/", handler, OUTPUT_FILTER);
            server.start();
            try {
                var req1 = HttpRequest.newBuilder(uri(server, "/foo/file.txt")).build();
                var res1 = client.send(req1, BodyHandlers.ofString());
                assertEquals(200, res1.statusCode());
                assertEquals("foo", res1.body());
                assertEquals("text/plain", res1.headers().firstValue("content-type").get());
                assertEquals(Long.toString(3L), res1.headers().firstValue("content-length").get());
                assertEquals(getLastModified(TEST_DIR.resolve("foo").resolve("file.txt")), res1.headers().firstValue("last-modified").get());

                var req2 = HttpRequest.newBuilder(uri(server, "/foobar/file.txt")).build();
                var res2 = client.send(req2, BodyHandlers.ofString());
                assertEquals(404, res2.statusCode());  // no context found

                var req3 = HttpRequest.newBuilder(uri(server, "/foo/../foobar/file.txt")).build();
                var res3 = client.send(req3, BodyHandlers.ofString());
                assertEquals(404, res3.statusCode());  // cannot escape context

                var req4 = HttpRequest.newBuilder(uri(server, "/foo/../..")).build();
                var res4 = client.send(req4, BodyHandlers.ofString());
                assertEquals(404, res4.statusCode());  // cannot escape root

                var req5 = HttpRequest.newBuilder(uri(server, "/foo/bar")).build();
                var res5 = client.send(req5, BodyHandlers.ofString());
                assertEquals(301, res5.statusCode());  // redirect
                assertEquals("0", res5.headers().firstValue("content-length").get());
                assertEquals("/foo/bar/", res5.headers().firstValue("location").get());
            } finally {
                server.stop(0);
            }
        }
        {
            // Test "/foo" context (without trailing slash)
            var handler = SimpleFileServer.createFileHandler(TEST_DIR.resolve("foo"));
            var server = HttpServer.create(LOOPBACK_ADDR, 10, "/foo", handler, OUTPUT_FILTER);
            server.start();
            try {
                var req1 = HttpRequest.newBuilder(uri(server, "/foo/file.txt")).build();
                var res1 = client.send(req1, BodyHandlers.ofString());
                assertEquals(200, res1.statusCode());
                assertEquals("foo", res1.body());
                assertEquals("text/plain", res1.headers().firstValue("content-type").get());
                assertEquals(Long.toString(3L), res1.headers().firstValue("content-length").get());
                assertEquals(getLastModified(TEST_DIR.resolve("foo").resolve("file.txt")), res1.headers().firstValue("last-modified").get());

                var req2 = HttpRequest.newBuilder(uri(server, "/foobar/")).build();
                var res2 = client.send(req2, BodyHandlers.ofString());
                assertEquals(404, res2.statusCode());  // handler prevents mapping to /foo/bar

                var req3 = HttpRequest.newBuilder(uri(server, "/foobar/file.txt")).build();
                var res3 = client.send(req3, BodyHandlers.ofString());
                assertEquals(404, res3.statusCode());  // handler prevents mapping to /foo/bar/file.txt

                var req4 = HttpRequest.newBuilder(uri(server, "/file.txt")).build();
                var res4 = client.send(req4, BodyHandlers.ofString());
                assertEquals(404, res4.statusCode());

                var req5 = HttpRequest.newBuilder(uri(server, "/foo/bar")).build();
                var res5 = client.send(req5, BodyHandlers.ofString());
                assertEquals(301, res5.statusCode());  // redirect
                assertEquals("0", res5.headers().firstValue("content-length").get());
                assertEquals("/foo/bar/", res5.headers().firstValue("location").get());

                var req6 = HttpRequest.newBuilder(uri(server, "/foo")).build();
                var res6 = client.send(req6, BodyHandlers.ofString());
                assertEquals(301, res6.statusCode());  // redirect
                assertEquals("0", res6.headers().firstValue("content-length").get());
                assertEquals("/foo/", res6.headers().firstValue("location").get());
            } finally {
                server.stop(0);
            }
        }
        {
            // Test that a request path segment that is a Windows root drive
            // does not circumvent access restrictions.
            //
            // For example, given the test directory tree:
            //
            //      |-- TEST_DIR
            //          |-- foo
            //              |-- bar  ----->>> if hidden, itself and any of its subdirectories are not accessible
            //                  |-- baz
            //                      |-- file.txt
            //      ...
            var handler = SimpleFileServer.createFileHandler(TEST_DIR);
            var server = HttpServer.create(LOOPBACK_ADDR, 10, "/", handler, OUTPUT_FILTER);
            server.start();
            try {
                var req1 = HttpRequest.newBuilder(uri(server, "/foo/bar/c:/baz/")).build();
                var res1 = client.send(req1, BodyHandlers.ofString());
                assertEquals(404, res1.statusCode());  // not found
            } finally {
                server.stop(0);
            }
        }
    }

    // Tests with a mixture of in-memory and file handlers.
    @Test
    public void multipleContexts() throws Exception {
        var rootHandler = HttpHandlers.of(200, Headers.of(), "root response body");
        var fooHandler = SimpleFileServer.createFileHandler(TEST_DIR.resolve("foo"));
        var foobarHandler = SimpleFileServer.createFileHandler(TEST_DIR.resolve("foobar"));
        var barHandler = HttpHandlers.of(200, Headers.of(), "bar response body");

        var server = HttpServer.create(LOOPBACK_ADDR, 0);
        server.createContext("/", rootHandler);
        server.createContext("/foo/", fooHandler);
        server.createContext("/bar/", barHandler);
        server.createContext("/foobar/", foobarHandler);
        server.start();
        var client = HttpClient.newBuilder().proxy(NO_PROXY).build();
        try {
            for (String uriPath : List.of("/", "/blah", "/xyz/t/z", "/txt") ) {
                out.println("uri.Path=" + uriPath);
                var req1 = HttpRequest.newBuilder(uri(server, uriPath)).build();
                var res1 = client.send(req1, BodyHandlers.ofString());
                assertEquals(200, res1.statusCode());
                assertEquals("root response body", res1.body());
            }
            {
                var req1 = HttpRequest.newBuilder(uri(server, "/foo/file.txt")).build();
                var res1 = client.send(req1, BodyHandlers.ofString());
                assertEquals(200, res1.statusCode());
                assertEquals("foo", res1.body());

                var req2 = HttpRequest.newBuilder(uri(server, "/foo/bar/baz/file.txt")).build();
                var res2 = client.send(req2, BodyHandlers.ofString());
                assertEquals(200, res2.statusCode());
                assertEquals("foo/bar/baz", res2.body());
            }
            {
                var req1 = HttpRequest.newBuilder(uri(server, "/foobar/file.txt")).build();
                var res1 = client.send(req1, BodyHandlers.ofString());
                assertEquals(200, res1.statusCode());
                assertEquals("foobar", res1.body());
            }
            for (String uriPath : List.of("/bar/", "/bar/t", "/bar/t/z", "/bar/index.html") ) {
                out.println("uri.Path=" + uriPath);
                var req1 = HttpRequest.newBuilder(uri(server, uriPath)).build();
                var res1 = client.send(req1, BodyHandlers.ofString());
                assertEquals(200, res1.statusCode());
                assertEquals("bar response body", res1.body());
            }
        } finally {
            server.stop(0);
        }
    }

    // Tests requests with queries, which are simply ignored by the handler
    @Test
    public void requestWithQuery() throws Exception {
        var client = HttpClient.newBuilder().proxy(NO_PROXY).build();
        var handler = SimpleFileServer.createFileHandler(TEST_DIR);
        var server = HttpServer.create(LOOPBACK_ADDR, 10, "/", handler, OUTPUT_FILTER);
        server.start();
        try {
            for (String query : List.of("x=y", "x=", "xxx", "#:?") ) {
                out.println("uri.Query=" + query);
                var req = HttpRequest.newBuilder(uri(server, "", query)).build();
                var res = client.send(req, BodyHandlers.ofString());
                assertEquals(200, res.statusCode());
                assertEquals("text/html; charset=UTF-8", res.headers().firstValue("content-type").get());
                assertEquals(Long.toString(257L), res.headers().firstValue("content-length").get());
                assertEquals(getLastModified(TEST_DIR), res.headers().firstValue("last-modified").get());
            }
        } finally {
            server.stop(0);
        }
    }

    @AfterAll
    public static void teardown() throws IOException {
        if (Files.exists(TEST_DIR)) {
            FileUtils.deleteFileTreeWithRetry(TEST_DIR);
        }
    }

    static URI uri(HttpServer server, String path) {
        return URIBuilder.newBuilder()
                .host("localhost")
                .port(server.getAddress().getPort())
                .scheme("http")
                .path(path)
                .buildUnchecked();
    }

    static URI uri(HttpServer server, String path, String query) {
        return URIBuilder.newBuilder()
                .host("localhost")
                .port(server.getAddress().getPort())
                .scheme("http")
                .path(path)
                .query(query)
                .buildUnchecked();
    }

    static String getLastModified(Path path) throws IOException {
        return Files.getLastModifiedTime(path).toInstant().atZone(ZoneId.of("GMT"))
                .format(DateTimeFormatter.RFC_1123_DATE_TIME);
    }
}
