# SPDX-License-Identifier: Apache-2.0
# Copyright 2019 The Meson development team

from __future__ import annotations

import os
import subprocess
from pathlib import Path
import typing as T

def ls_as_bytestream() -> bytes:
    if os.path.exists('.git'):
        return subprocess.run(['git', 'ls-tree', '-r', '--name-only', 'HEAD'],
                              stdout=subprocess.PIPE).stdout

    files = [str(p) for p in Path('.').glob('**/*')
             if not p.is_dir() and
             not next((x for x in p.parts if x.startswith('.')), None)]
    return '\n'.join(files).encode()


def cscope() -> int:
    ls = b'\n'.join([b'"%s"' % f for f in ls_as_bytestream().split()])
    return subprocess.run(['cscope', '-v', '-b', '-i-'], input=ls).returncode


def ctags() -> int:
    ls = ls_as_bytestream()
    return subprocess.run(['ctags', '-L-'], input=ls).returncode


def etags() -> int:
    ls = ls_as_bytestream()
    return subprocess.run(['etags', '-'], input=ls).returncode


def run(args: T.List[str]) -> int:
    tool_name = args[0]
    srcdir_name = args[1]
    os.chdir(srcdir_name)
    assert tool_name in {'cscope', 'ctags', 'etags'}
    res = globals()[tool_name]()
    assert isinstance(res, int)
    return res
