# (c) Copyright 2014-2015. CodeWeavers, Inc.

import os
import struct

import cxdiag
import cxobjc
import cxutils
import distversion


class SystemInfo(cxobjc.Proxy):
    pass


def get_bottle_information(bottle):
    toggles = [
        bottle.graphics_backend_state,
        bottle.vulkan_backend_state,
        bottle.is_dlss_enabled_state,
        bottle.is_high_resolution_enabled_state,
        bottle.is_msync_enabled_state,
        bottle.is_preview_enabled_state,
    ]

    info = {'Name': bottle.name,
            'Template': bottle.template,
            'Windows Version': bottle.windows_version,
            'Managed': str(bottle.is_managed),
            'Arch': bottle.arch,
            }
    info.update({k.strip(): v.strip() for k, v in (i.split(':') for i in toggles)})

    return info


def get_system_information(bottle=None, include_cxdiag=False):
    ''' Put the system information in a dictionary so we can store it in JSON format.'''
    info = {}

    if bottle:
        info['Bottle'] = get_bottle_information(bottle)

    info['Product'] = distversion.PRODUCT_NAME
    info['Version'] = distversion.PUBLIC_VERSION
    info['Install location'] = cxutils.CX_ROOT
    info['Bitness'] = '%s-bit' % (struct.calcsize('P') * 8)
    info['Desktop session'] = os.environ.get('DESKTOP_SESSION', '')

    try:
        from gi.repository import Gdk
        if hasattr(Gdk.Screen.get_default(), 'get_window_manager_name'):
            info['Window manager'] = Gdk.Screen.get_default().get_window_manager_name()
    except ImportError:
        pass

    info['Locale'] = {}
    lang = os.environ.get('LANG', '')
    info['Locale']['Locale'] = lang
    for var in ('LC_ADDRESS', 'LC_COLLATE', 'LC_CTYPE', 'LC_IDENTIFICATION',
                'LC_MONETARY', 'LC_MESSAGES', 'LC_MEASUREMENT', 'LC_NAME',
                'LC_NUMERIC', 'LC_PAPER', 'LC_TELEPHONE', 'LC_TIME', 'LC_ALL'):
        if var in os.environ and os.environ[var] != lang:
            info['Locale'][var] = os.environ[var]

    info['System Information'] = {}

    # System information on Linux
    for fname in ('/etc/os-release', '/etc/lsb-release'):
        if not os.path.exists(fname):
            continue

        with open(fname, 'r', encoding='utf8', errors='surrogateescape') as f:
            for line in f:
                key, _part, value = line.strip().partition('=')
                info['System Information'][key.strip('"')] = value.strip('"')
        break

    # System information on macOS
    cmd = ('system_profiler', '-detailLevel', 'mini', '-timeout', '1', 'SPDisplaysDataType', 'SPHardwareDataType', 'SPSoftwareDataType')
    retcode, out, _err = cxutils.run(cmd, stdout=cxutils.GRAB)
    if retcode == 0:
        for line in out.splitlines():
            key, _part, value = line.strip().partition(':')
            if value:
                info['System Information'][key.strip()] = value.strip()

    if include_cxdiag:
        cxdiag_info = cxdiag.get()
        info['Warnings'] = cxdiag_info.warnings
        info['Properties'] = cxdiag_info.properties
        info['Libraries'] = cxdiag_info.libs

    return info


@cxobjc.method(SystemInfo, 'systemInformationString')
def system_info_string(bottle=None):
    ret = ''
    for key, value in get_system_information(bottle=bottle, include_cxdiag=True).items():
        ret += '%s:\n' % key

        if isinstance(value, dict):
            for inner_key, inner_value in value.items():
                ret += '%s: %s\n' % (inner_key, inner_value)
        else:
            ret += '%s\n' % value

        ret += '\n'

    return ret


@cxobjc.method(SystemInfo, 'systemInformationStringForBottle_')
def _system_info_string_for_bottle(bottle):
    return system_info_string(bottle)
