/*
 * Copyright 2021-2025 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opentest4j.reporting.tooling.core.validator;

import org.intellij.lang.annotations.Language;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.io.TempDir;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.ValueSource;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.Path;

import static org.assertj.core.api.Assertions.assertThat;
import static org.opentest4j.reporting.tooling.core.validator.Severity.ERROR;
import static org.opentest4j.reporting.tooling.core.validator.Severity.WARNING;

class DefaultValidatorTests {

	@TempDir
	Path tempDir;

	@ParameterizedTest
	@ValueSource(strings = { "0.1.0", "0.2.0" })
	void validatesEventsXmlFile(String version) {
		var xmlFile = writeXml(tempDir.resolve("events.xml"),
			"<events xmlns='https://schemas.opentest4j.org/reporting/events/" + version + "'/>");

		var validationResult = new DefaultValidator().validate(xmlFile);

		assertThat(validationResult).isEmpty();
	}

	@ParameterizedTest
	@ValueSource(strings = { "0.1.0", "0.2.0" })
	void validatesHierarchicalXmlFile(String version) {
		var xmlFile = writeXml(tempDir.resolve("hierarchy.xml"),
			"<execution xmlns='https://schemas.opentest4j.org/reporting/hierarchy/" + version + "'/>\n");

		var validationResult = new DefaultValidator().validate(xmlFile);

		assertThat(validationResult).isEmpty();
	}

	@Test
	void resolvesThirdPartySchemasUsingCatalog() {
		writeXml(tempDir.resolve("schema.xsd"), """
				<xs:schema xmlns:xs="http://www.w3.org/2001/XMLSchema"
				      targetNamespace="urn:test:1"
				      elementFormDefault="qualified">
				    <xs:element name="test" type="xs:string"/>
				</xs:schema>
				""");
		var catalog = writeXml(tempDir.resolve("catalog.xml"), """
				<catalog xmlns="urn:oasis:names:tc:entity:xmlns:xml:catalog">
				    <system systemId="https://localhost/schema.xsd" uri="schema.xsd"/>
				</catalog>
				""");
		var xmlFile = writeXml(tempDir.resolve("test.xml"), """
				<test xmlns="urn:test:1"
				      xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
				      xsi:schemaLocation="urn:test:1 https://localhost/schema.xsd">
				    Hello World!
				</test>
				""");

		var validationResult = new DefaultValidator(catalog.toUri()).validate(xmlFile);

		assertThat(validationResult).isEmpty();
	}

	@Test
	void generatesWarningsAndErrors() {
		var xmlFile = writeXml(tempDir.resolve("test.xml"), """
				<test xmlns="urn:test:1"
				      xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
				      xsi:schemaLocation="urn:test:1 https://localhost/schema.xsd">
				    Hello World!
				</test>
				""");

		var validationResult = new DefaultValidator().validate(xmlFile);

		assertThat(validationResult.count(WARNING)).isEqualTo(1);
		assertThat(validationResult.count(ERROR)).isEqualTo(1);
	}

	private Path writeXml(Path file, @Language("xml") String content) {
		try {
			return Files.writeString(file, content);
		}
		catch (IOException e) {
			throw new UncheckedIOException(e);
		}
	}
}
