\name{communitydist}
\alias{communitydist}

%- Also NEED an `\alias' for EACH other topic documented here.
\title{Distances between communities}
\description{
  Constructs distances between communities: chord- (Cavalli-Sforza and
  Edwards, 1967), phiPT/phiST (Peakall and Smouse, 2012, Meirmans,
  2006),
  three versions of the shared allele distance between communities,
  and geographical distance between communities.
}

\usage{
communitydist(alleleobj,comvector="auto",distance="chord",
                          compute.geodist=TRUE,out.dist=FALSE,
                          grouping=NULL,geodist=NA,diploid=TRUE,
                          phiptna=NA,...)


}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{alleleobj}{if \code{diploid=TRUE}, an object of class
    \code{alleleobject} as produced by
    function  \code{\link[prabclus]{alleleinit}}. This has the required
    information on the individuals that are grouped into communities. In
    case \code{diploid=FALSE}, a list that needs to have components
    \code{n.variables} (number of loci), \code{alevels} (vector of
    allele names, see \code{\link{alleleinit}}) and
    \code{charmatrix} (matrix of characters with one row for every
    individual and one column for every locus giving the alleles; see
    examples below for how this can be constructed for a
    \code{prabobject} with presence-absence data).}
  \item{comvector}{either a vector of integers indicating to which
    community an individual belongs (these need to be numbered from 1 to
    a maximum number without interruption), or \code{"auto"}, which
    indicates that communities are automatically generated by the
    \code{\link{communities}}-function.}
  \item{distance}{one of \code{"chord"}, \code{"phipt"},
    \code{"shared.average"}, \code{"shared.chakraborty"},
    \code{"shared.problist"}. See Details.} 
  \item{compute.geodist}{logical, indicating whether geographical
    distances between communities should be generated.}
  \item{out.dist}{logical, indicating whether \code{dist}-objects are
    given out or rather distance matrices.}
  \item{grouping}{something that can be coerced into a factor, for
    passing on to \code{\link{communities}} in case that
    \code{comvector=="auto"}. This implies that individuals in different
    groups indicated by \code{grouping} cannot be together in the same
    community. Furthermore (also if \code{comvector} is something
    else), a vector of groups of communities will be computed, see
    output component \code{comgroup}. In any case individuals in
    different groups are not allowed to be in the same community.}
  \item{geodist}{matrix or \code{dist}-object providing geographical
    distances between individuals. Required if
    \code{compute.geodist==TRUE} or \code{comvector=="auto"}.}
  \item{diploid}{logical, indicating whether loci are diploid, see
    \code{alleleobj}.}
  \item{phiptna}{if \code{distance="phipt"}, value to be given out as
    phiPT-distance in case that
    the original definition amounts to 0/0 (particularly if communities
    have just one member).}
  \item{...}{optional arguments to be passed on to \code{\link{communities}}.}
}

\details{
  All genetic distances between communities are based on the information
  given in \code{alleleobj}; either on the alleles directly or on a genetic
  distance (\code{distmat}-component, see \code{\link{alleleinit}}). 
  The possible genetic distance measures between communities are as follows:
  \itemize{
    \item \code{"chord"}: chord-distance (Cavalli-Sforza and Edwards,
    1967)
    \item \code{"phipt"}: phiPT-distance implemented according to
    Peakall and Smouse, 2012. This also appears in the literature under
    the name phiST (Meirmans, 2006, although the definition there is
    incomplete and we are not sure whether this is identical).
    \item \code{"shared.average"}: average of between-community genetic
    distances.
    \item \code{"shared.chakraborty"}: between-community shared allele
    distance according to Chakraborty and Jin (1993).
    \item \code{"shared.problist"}: this implements the shared allele
    distance (Bowcock et al., 1994) for individuals directly for
    communities (one minus proportion of alleles shared by two
    communities averaged over loci).
  }
}
  

\value{
  list with components
  \item{comvector}{integer vector of length of the number of
    individuals, indicating their community membership.}
  \item{dist}{genetic distances between communities. Parameter \code{out.dist}
    determines whether this is a \code{dist}-object or a matrix.}
  \item{cgeodist}{if \code{compute.geodist}, geographical distance
  between communities defined as average distance of all pairs of
  individuals belonging to different ones of the two communities between
  which the distance is computed. Parameter \code{out.dist}
  determines whether this is a \code{dist}-object or a matrix.}
  \item{comgroup}{vector of length of the number of communities. If
  \code{grouping} was provided, this is a vector giving the group
  memberships of all communities, otherwise it is a vector of 1s.}
}

    
\references{
  Bowcock, A. M., Ruiz-Linares, A., Tomfohrde, J., Minch, E., Kidd, J. R.,
  Cavalli-Sforza, L. L. (1994) High resolution of human evolutionary
  trees with polymorphic microsatellites. \emph{Nature} 368, 455-457.
  
  Cavalli-Sforza, L. L. and Edwards, A. W. F. (1967) Phylogenetic
  Analysis - Models and Estimation Procedures. \emph{The American Journal of
  Human Genetics} 19, 233-257.

  Chakraborty, R. and Jin, L. (1993) Determination of relatedness between
  individuals using DNA fingerprinting. \emph{Human Biology} 65, 875-895.

  Meirmans, P. G. (2006) Using the AMOVA framework to estimate a
  standardized genetic differentiation measure. \emph{Evolution} 60, 2399-2402.
  
  Peakall, R. and Smouse P.E. (2012) GenAlEx Tutorial 2.
  \url{https://biology-assets.anu.edu.au/GenAlEx/Tutorials.html}
}

\author{Christian Hennig
  \email{christian.hennig@unibo.it}
  \url{https://www.unibo.it/sitoweb/christian.hennig/en}}

\seealso{
  \code{\link{communities}}; refer to \code{\link{phipt}} for
  computation of distances between specific pairs of communities.
  \code{\link{diploidcomlist}} produces relative frequencies for all
  alles of all loci in all communities (on which the chord- and the
  \code{"shared.problist"}-distances are based). 
}

\examples{
  options(digits=4)
  data(tetragonula)
  tnb <-
  coord2dist(coordmatrix=tetragonula.coord[83:120,],cut=50,
    file.format="decimal2",neighbors=TRUE)
  ta <- alleleconvert(strmatrix=tetragonula[83:120,])
  tai <- alleleinit(allelematrix=ta,neighborhood=tnb$nblist)
  tetraspec <- c(rep(1,11),rep(2,13),rep(3,14))
  tetracoms <-
  c(rep(1:3,each=3),4,5,rep(6:11,each=2),12,rep(13:19,each=2))
  c1 <- communitydist(tai,comvector=tetracoms,distance="chord",
    geodist=tnb$distmatrix,grouping=tetraspec)
  c2 <- communitydist(tai,comvector=tetracoms,distance="phipt",
    geodist=tnb$distmatrix,grouping=tetraspec,compute.geodist=FALSE)
  c3 <- communitydist(tai,comvector=tetracoms,distance="shared.average",
    geodist=tnb$distmatrix,grouping=tetraspec,compute.geodist=FALSE)
  c4 <- communitydist(tai,comvector=tetracoms,distance="shared.chakraborty",
    geodist=tnb$distmatrix,grouping=tetraspec,compute.geodist=FALSE)
  c5 <- communitydist(tai,comvector=tetracoms,distance="shared.problist",
    geodist=tnb$distmatrix,grouping=tetraspec,compute.geodist=FALSE)
  round(c1$cgeodist,digits=1)
  c1$comvector
  c2$comvector
  c3$comvector
  c4$comvector
  c5$comvector
  round(c1$dist,digits=2)
  round(c2$dist,digits=2)
  round(c3$dist,digits=2)
  round(c4$dist,digits=2)
  round(c5$dist,digits=2)  
}
\keyword{spatial}% __ONLY ONE__ keyword per line
\keyword{multivariate}% __ONLY ONE__ keyword per line
