/* $Id: LuaScript.cpp,v 1.7 2003/03/22 23:51:49 mrq Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2000 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <Modules/Lua/LuaScript.h>
#include <Modules/Lua/LuaWorld.h>
#include <Modules/Lua/LuaEntity.h>
#include <Ark/ArkSystem.h>

#include <sstream>

// memset/memcpy functions
#include <string.h>


extern "C"
{
#include <lua.h>
#include <lualib.h>
}

namespace Ark
{
   lua_State* g_LuaState;
   
   LuaScript::LuaScript ()
   {
      lua_State* L = lua_open(0);
      lua_baselibopen(L);
      lua_mathlibopen(L);
      
      g_LuaState = L;
      Luna<LuaEntity>::Register(L);
      Luna<LuaWorld>::Register(L);
      Luna<LuaEngine>::Register(L);


      ////////////
      // Set an entry in the registry.
      lua_getregistry(L);
      lua_pushstring (L, "LuaScript");
      lua_pushuserdata(L,this);
      lua_settable(L,-3);
      lua_pop (L, -1);
      
      lua_dostring(g_LuaState, "collectgarbage()");
   }
   
   LuaScript::~LuaScript ()
   {
     if (g_LuaState) 
	 lua_close(g_LuaState);
   }
   
   bool
   LuaScript::LoadLibrary (const String &file)
   {
      std::ostringstream stream;
      stream << "ark = Engine ();\n"
	     << "ark.world = World();\n"
	     << "ark.log = print;\n"
	     << "ark.true = 1;\n"
	     << "ark.false = 0;\n"
	     << "ark.play_once = " << ModelState::ONCE << ";\n"
	     << "ark.play_loop = " << ModelState::LOOP << ";\n"
	 ;
      
      lua_dostring (g_LuaState, stream.str().c_str());
      
      if (!Script::LoadLibrary (file))
	 return false;
      
      lua_dostring (g_LuaState, "ark.init();");
     
      return true;
   }
   
   bool
   LuaScript::LoadScript (const String& file)
   {
      AutoReadStream stream (file);
      std::ifstream& f = stream.Get();
      
      // use seekg/tellg to find size of buffer
      f.seekg(0, std::ios::end);
      const long size = f.tellg();
      f.seekg(0, std::ios::beg);
      
      if (size <= 0)
	 return false;
      
      char *buf = new char[size+1];
      assert (buf != NULL);
      
      memset(buf, 0, size+1);

      // Read the max number of chars possible
      f.read (buf, size);
      
      // Get the number of char read
      // On win32, there is a CRLF issue, so (readchars < size)
      const long readchars = f.gcount();
      
      // Check that we have read something
      if (readchars <= 0)
      {
	 delete[] buf;
	 return false;
      }
      
      lua_dobuffer(g_LuaState, buf, readchars, file.c_str());  
      return true;
   }
   
   bool
   LuaScript::NextFrame ()
   {
      lua_dostring (g_LuaState, "ark.nextframe();");
      return true;
   }

   bool
   LuaScript::Main ()
   {
      lua_dostring (g_LuaState, "ark.main();");
      return true;
   }
   
   EngineEntity *
   LuaScript::NewEntity ()
   {
      LuaEntity *ent;
      lua_getglobal (g_LuaState, "Entity");
      lua_call (g_LuaState, 0, 1);
      lua_getobject (g_LuaState, -1, &ent);
      lua_pop (g_LuaState, 1);
      
      return ent;
   }
   

   ////////////////////////////
   
   void lua_GetScript(lua_State *L, LuaScript **s)
   {
      lua_getregistry(L);
      lua_pushstring (L, "LuaScript");
      lua_gettable(L,-2);

      *s = (LuaScript*) lua_touserdata (L, -1);
      lua_pop (L, -2);
   }

   LuaEngine::LuaEngine(lua_State *L)
   {}

   LuaEngine::~LuaEngine()
   {
   }
   

   int
   LuaEngine::require (lua_State *L)
   {
      LuaScript *sc;
      String name = lua_tostring (L, -1);
      lua_pop (L,1);
     
      lua_GetScript(L,&sc);
      sc->LoadScript (name);
      return 0;
   }

   int
   LuaEngine::get_env (lua_State *L)
   {
      String name = lua_tostring (L, -1);
      lua_pop (L,1);
     
      lua_pushstring(L, Sys()->GetEnv(name).c_str());
      return 1;
   }


   int
   LuaEngine::version (lua_State *L)
   {
      return 0;
   }

   const char LuaEngine::className[] = "Engine";
   const Luna<LuaEngine>::RegType LuaEngine::Register[] = 
   {
      {"require",         &LuaEngine::require},
      {"version",         &LuaEngine::version},
      {"get_env",         &LuaEngine::get_env},
      {0}
   };


}

#define ARK_MODULE
#include <Ark/ArkFactoryReg.h>

class LuaScriptFactory : public Ark::ScriptFactory
{
   public:
      virtual ~LuaScriptFactory() {};
      virtual Ark::Script *NewScript() {return new Ark::LuaScript;}
};

ARK_REGISTER("ark::Script::Lua", LuaScriptFactory);
