/*
 * GPG Keys 
 * cmainwindow.cpp
 * (c) 2001 Peter Mathiasson <peter@mathiasson.nu>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2, as
 * published by the Free Software Foundation
 */

#include <qapp.h>
#include <qmainwindow.h>
#include <qlistview.h>
#include <qprocess.h>
#include <qmessagebox.h>
#include <qstatusbar.h>
#include <qmenubar.h>
#include <qpopupmenu.h>
#include <qvbox.h>
#include <qtabbar.h>
#include <qfont.h>
#include <qstringlist.h>
#include <qclipboard.h>
#include <qfiledialog.h>
#include "cmainwindow.h"
#include "cdlgproperties.h"
#include "cdlgabout.h"
#include "cdlgserversearch.h"
#include "cdlgkeyinfo.h"
#include "cdlgsign.h"
#include "clistviewitem.h"
#include "cgpgproc.h"
#include "ckeyserver.h"
#include "functions.h"
#include "config.h"

/* --------------------------------------------------------------------------------- */

enum {
    MNU_CONFIG,
    MNU_EXIT,

    MNU_KEYSERVER,
    MNU_IMPORT_FILE,
    MNU_IMPORT_CLIPBOARD,

    MNU_ABOUT,
    MNU_ABOUT_QT,

    MNU_PUBLIC_INFO,
    MNU_PUBLIC_SIGN,
    MNU_PUBLIC_DELETE,
    MNU_PUBLIC_UPLOAD,
    MNU_PUBLIC_TOCLIPBOARD,
    MNU_PUBLIC_TOFILE,

    MNU_POPUP_KEYSERVER
};

enum {
    EXPORT_SEND,
    EXPORT_FILE,
    EXPORT_CLIPBOARD
};

#define LVPOS_NAME      0
#define LVPOS_TRUST     1
#define LVPOS_DATE      2
#define LVPOS_TYPE      3
#define LVPOS_KEYID     4

extern CGPGProc *gnupgProc;

/* ---------------------------------------------------------------------------------
 * CMainWindow::CMainWindow
 *
 * This is the constructor for the main window. In here all class-global variables
 * and classes are created. It also creates all widgets used within the main window.
 * --------------------------------------------------------------------------------- */
CMainWindow::CMainWindow() : QMainWindow( 0, "CMainWindow" )
{
    /* Set Title */
    setCaption( "GPG Keys" );

    /* Create GPG Proc Object */
    gnupgProc = new CGPGProc( this );
    connect( gnupgProc, SIGNAL(message(QString,int)), this, SLOT(gnupgProcMessage(QString,int)) );

    /* Create Menubar/Statusbar */
    QMenuBar *menu = menuBar();
    statusBar();

    /* Create File Menu */
    menuFile = new QPopupMenu( this );
    addMenuItem( menuFile, tr( "&Options" ), MNU_CONFIG, CTRL+Key_O );
    menuFile->insertSeparator();
    addMenuItem( menuFile, tr( "E&xit" ), MNU_EXIT, CTRL+Key_Q );
    menu->insertItem( tr( "&File" ), menuFile );

    /* Create Commands Menu */
    menuCmd = new QPopupMenu( this );
    addMenuItem( menuCmd, tr( "Search &Keyserver" ), MNU_KEYSERVER, CTRL+Key_K );
    menuCmd->insertSeparator();
    addMenuItem( menuCmd, tr( "Import from &File" ), MNU_IMPORT_FILE, CTRL+Key_J );
#ifdef CLIPBOARD_SUPPORT
    addMenuItem( menuCmd, tr( "Import from &Clipboard" ), MNU_IMPORT_CLIPBOARD, CTRL+Key_V );
#endif
    menu->insertItem( tr( "&Command" ), menuCmd );

    /* Create Help Menu */
    menuHelp = new QPopupMenu( this );
    addMenuItem( menuHelp, tr( "&About" ), MNU_ABOUT, CTRL+Key_A );
    addMenuItem( menuHelp, tr( "About &Qt" ), MNU_ABOUT_QT );
    menu->insertItem( tr( "&Help" ), menuHelp );

    /* Create Public Key Popup Menu */
    menuPublic = new QPopupMenu( this );
    addMenuItem( menuPublic, tr( "&Information" ), MNU_PUBLIC_INFO, CTRL+Key_I );
    menuPublic->insertSeparator();
    addMenuItem( menuPublic, tr( "&Sign Key" ), MNU_PUBLIC_SIGN, CTRL+Key_S );
    addMenuItem( menuPublic, tr( "&Delete Key" ), MNU_PUBLIC_DELETE, CTRL+Key_D );
    menuPublic->insertSeparator();
#ifdef CLIPBOARD_SUPPORT
    addMenuItem( menuPublic, tr( "Export To &Clipboard" ), MNU_PUBLIC_TOCLIPBOARD, CTRL+Key_C );
#endif
    addMenuItem( menuPublic, tr( "Export To &File" ), MNU_PUBLIC_TOFILE, CTRL+Key_F );
    menuPublic->insertSeparator();
    addMenuItem( menuPublic, tr( "&Upload Key" ), MNU_PUBLIC_UPLOAD, CTRL+Key_U );
    addMenuItem( menuPublic, tr( "&Lookup Key" ), MNU_POPUP_KEYSERVER, CTRL+Key_L );

    /* Create Signature Popup Menu */
    menuSignature = new QPopupMenu( this );
    addMenuItem( menuSignature, tr( "&Lookup Key" ), MNU_POPUP_KEYSERVER, CTRL+Key_L );

    /* Create Vertical Box */
    QVBox *vbox = new QVBox( this );

    /* Create Listview */
    listMain = new QListView( vbox );    
    listMain->setRootIsDecorated( TRUE );
    listMain->addColumn( tr( "Name" ) );
    listMain->addColumn( tr( "Trust" ) );
    listMain->addColumn( tr( "Date" ) );
    listMain->addColumn( tr( "Type" ) );
    listMain->addColumn( tr( "Key ID" ) );
    connect( listMain, SIGNAL(contextMenuRequested(QListViewItem*,const QPoint&,int)),
             this, SLOT(contextMenuRequested(QListViewItem*,const QPoint&,int)) );

    /* Create Public/Secret TabBar */
    tabBar = new QTabBar( vbox );
    tabBar->setShape( QTabBar::TriangularBelow );
    tabBar->addTab( new QTab( tr( "&Public Keys" ) ) );
    tabBar->addTab( new QTab( tr( "P&rivate Keys" ) ) );
    connect( tabBar, SIGNAL(selected(int)), this, SLOT(tabSelected(int)) );

    /* Create Keyserver Object */
    keyServer = new CKeyServer( this );
    connect( keyServer, SIGNAL(statusChange(int)), this, SLOT(tcpStatusChange(int)) );
    connect( keyServer, SIGNAL(sendDone()), this, SLOT(tcpSendDone()) );

    /* Apply Configuration */
    listCommand = QString::null;
    applySettings();

    /* Set Central Widget */
    setCentralWidget( vbox );

    /* Resize and Move */
    resize( readConfigInt( "MainWindow/width" ), readConfigInt( "MainWindow/height" ) );
    move( readConfigInt( "MainWindow/x" ), readConfigInt( "MainWindow/y" ) );

}

/* ---------------------------------------------------------------------------------
 * CMainWindow::~CMainWindow
 *
 * This is the destructor for the main window. It's used to save window size and
 * position.
 * --------------------------------------------------------------------------------- */
CMainWindow::~CMainWindow()
{
    /* Save Window Size */
    QSize wndSize = size();
    writeConfigInt( "MainWindow/width", wndSize.width() );
    writeConfigInt( "MainWindow/height", wndSize.height() );
    
    /* Save Window Posistion */
    QPoint wndPos = pos();
    writeConfigInt( "MainWindow/x", wndPos.x() );
    writeConfigInt( "MainWindow/y", wndPos.y() );
}

/* ---------------------------------------------------------------------------------
 * CMainWindow::addMenuItem
 *
 * Helper function for creation of menu items
 * --------------------------------------------------------------------------------- */
void CMainWindow::addMenuItem( QPopupMenu *menu, const char *text, int menuItem, int accel )
{
        menu->insertItem( text, this, SLOT(menuEvent(int)), accel, menuItem );
}

/* ---------------------------------------------------------------------------------
 * CMainWindow::addMenuItem
 *
 * Helper function for creation of menu items
 * --------------------------------------------------------------------------------- */
void CMainWindow::addMenuItem( QPopupMenu *menu, QIconSet iconSet, const char *text, int menuItem,
                               int accel )
{
        menu->insertItem( iconSet, text, this, SLOT(menuEvent(int)), accel, menuItem );
}

/* ---------------------------------------------------------------------------------
 * CMainWindow::applySettings
 *
 * This function is called on startup and after a reconfiguration have been made to
 * ensure that the main window uses the latest settings
 * --------------------------------------------------------------------------------- */
void CMainWindow::applySettings()
{
    static int oldUtf8 = 2;

    /* Set Fonts */
    QFont stdFont = readConfigFont( "stdfont", STD_FONT );
    QFont fixedFont = readConfigFont( "fixedfont", FIX_FONT );
    setFont( stdFont );
    menuFile->setFont( stdFont );
    menuCmd->setFont( stdFont );
    menuHelp->setFont( stdFont );
    menuPublic->setFont( stdFont );
    menuSignature->setFont( stdFont );
    listMain->setFont( fixedFont );    
    
    /* Get List Command */
    QString oldListCommand = listCommand;
    if ( readConfigInt( "mainsigs", MAINSIGS ) ) {
        listCommand = "--list-sigs";
    } else {
        listCommand = "--list-keys";
    }

    /* Check for Utf8 Conversion */
    int utf8 = readConfigInt( "utf8userid", UTF8_USERID );
    if ( oldUtf8 == 2 ) {
        oldUtf8 = utf8;
    }

    /* Read Public Keys */
    if ( listCommand != oldListCommand || utf8 != oldUtf8 ) {
        readKeys( TRUE );
    }    

    /* Save Utf8 Setting */
    oldUtf8 = utf8;

}

/* ---------------------------------------------------------------------------------
 * CMainWindow::readKeys
 *
 * Executes GPG to read public or secret keys
 * --------------------------------------------------------------------------------- */
void CMainWindow::readKeys( bool publicKey )
{
    /* Disable TabBar */
    tabBar->setDisabled( TRUE );

    /* Clear ListView */
    listMain->clear();

    /* Create Process */
    gpgProc = new QProcess( this );
    gpgProc->addArgument( readConfigString( "gpgpath", GPGPATH ) );
    if ( publicKey ) {
        gpgProc->addArgument( listCommand );
    } else {
        gpgProc->addArgument( "--list-secret-keys" );
    }
    gpgProc->addArgument( "--with-colon" );
    connect( gpgProc, SIGNAL(processExited()), this, SLOT(processDone()) );
    connect( gpgProc, SIGNAL(readyReadStdout()), this, SLOT(readFromStdout()) );
    
    /* Run GPG */
    if ( !gpgProc->start() ) {
        runFailed( statusBar() );
        return;
    }

    /* Display Statusbar Message */
    if ( publicKey ) {
        statusBar()->message( tr( "Reading Public Keys..." ) );
    } else {
        statusBar()->message( tr( "Reading Secret Keys..." ) );
    }
}

/* ---------------------------------------------------------------------------------
 * CMainWindow::processDone
 *
 * This slot is trigged when the GPG List key process finishes
 * --------------------------------------------------------------------------------- */
void CMainWindow::processDone()
{
    /* Enable TabBar */
    tabBar->setDisabled( FALSE );

    /* Display Statusbar Message */
    statusBar()->message( tr( "Done!" ), 2000 );

}

/* ---------------------------------------------------------------------------------
 * CMainWindow::readFromStdout
 *
 * This slot reads and parses one or more line from the GPG keylisting process
 * --------------------------------------------------------------------------------- */
void CMainWindow::readFromStdout()
{
    QStringList entry;
    CListViewItem *itemTemp;
    static bool ignoreSig = FALSE;

    while ( gpgProc->canReadLineStdout() ) {
        
        /* Read/Split Line */
        entry = QStringList::split( ':', gpgProc->readLineStdout(), TRUE );

        /* Public Key */
        if ( entry[0] == "pub" ) {

            /* Create Public Key Entry */
            itemPub = new CListViewItem( listMain, CListViewItem::PUBLICKEY );
            itemPub->setText( LVPOS_NAME, fromUtf8( entry[9] ) );
            itemPub->setText( LVPOS_TRUST, getTrust( entry[1] ) );
            itemPub->setText( LVPOS_DATE, entry[5] );
            itemPub->setText( LVPOS_TYPE, entry[2] + "/" + getCipher( entry[3] ) );
            itemPub->setText( LVPOS_KEYID, entry[4] );

            /* Create UserID Entry */
            itemUid = new CListViewItem( itemPub, CListViewItem::USERID );
            itemUid->setText( LVPOS_NAME, fromUtf8( entry[9] ) );
            itemUid->setText( LVPOS_TRUST, getTrust( entry[1] ) );

            /* Include following signatures */
            ignoreSig = FALSE;

        }

        /* Subkey */
        else if ( entry[0] == "sub" ) {
            ignoreSig = TRUE;
        }


        /* User ID */
        else if ( entry[0] == "uid" ) {

            /* Create UserID Entry */
            itemUid = new CListViewItem( itemPub, CListViewItem::USERID );
            itemUid->setText( LVPOS_NAME, fromUtf8( entry[9] ) );
            itemUid->setText( LVPOS_TRUST, getTrust( entry[1] ) );

        }

        /* Signature */
        else if ( entry[0] == "sig" && !ignoreSig ) {

            /* Create Signature Entry */
            itemTemp = new CListViewItem( itemUid, CListViewItem::SIGNATURE );
            itemTemp->setText( LVPOS_NAME, fromUtf8( entry[9] ) );
            itemTemp->setText( LVPOS_TYPE, getCipher( entry[3] ) );
            itemTemp->setText( LVPOS_DATE, entry[5] );
            itemTemp->setText( LVPOS_KEYID, entry[4] );

        }
       
        /* Secret Key */
        if ( entry[0] == "sec" ) {

            /* Create Secret Key Entry */
            itemPub = new CListViewItem( listMain, CListViewItem::SECRETKEY );
            itemPub->setText( LVPOS_NAME, fromUtf8( entry[9] ) );
            itemPub->setText( LVPOS_TRUST, getTrust( entry[1] ) );
            itemPub->setText( LVPOS_DATE, entry[5] );
            itemPub->setText( LVPOS_TYPE, entry[2] + "/" + getCipher( entry[3] ) );
            itemPub->setText( LVPOS_KEYID, entry[4] );
            ignoreSig = FALSE;

        }

        /* Secret Sub Key */
        if ( entry[0] == "ssb" ) {
            ignoreSig = TRUE;
        }

    }
}

/* ---------------------------------------------------------------------------------
 * CMainWindow::menuEvent
 *
 * This slot is triggerd when someone selects a menu item
 * --------------------------------------------------------------------------------- */
void CMainWindow::menuEvent( int item )
{

    /* Get Current ListView Item */
    itemCurrent = (CListViewItem*)listMain->currentItem();


    switch( item ) {
        
    case MNU_CONFIG:            // File : Properties
        properties();
        break;

    case MNU_EXIT:              // File : Exit
        close();
        break;

    case MNU_KEYSERVER:         // Commands : Search Keyserver
        serverSearch( FALSE );
        break;

    case MNU_IMPORT_FILE:       // Commands : Import from File
        fromFile();
        break;

#ifdef CLIPBOARD_SUPPORT
    case MNU_IMPORT_CLIPBOARD:  // Commands : Import from Clipboard
        fromClipboard();
        break;
#endif

    case MNU_ABOUT:             // Help : About
        about();
        break;

    case MNU_ABOUT_QT:          // Help : About QT
        QMessageBox::aboutQt( this, tr( "About Qt" ) );
        break;

    case MNU_PUBLIC_INFO:       // Public Popup : Information
        keyInfo();
        break;

    case MNU_PUBLIC_SIGN:       // Public Popup : Sign
        signKey();
        break;

    case MNU_PUBLIC_DELETE:     // Public Popup : Delete
        doDelete();
        break;
    
    case MNU_PUBLIC_UPLOAD:     // Public Popup : Upload
        sendKey();
        break;

#ifdef CLIPBOARD_SUPPORT
    case MNU_PUBLIC_TOCLIPBOARD:// Public Popup : Export to Clipboard
        toClipboard();
        break;
#endif

    case MNU_PUBLIC_TOFILE:     // Public Popup : Export to File
        toFile();
        break;

    case MNU_POPUP_KEYSERVER:   // Multiple Popup : Keyserver
        serverSearch( TRUE );
        break;

    default:
        break;
    }
}

/* ---------------------------------------------------------------------------------
 * CMainWindow::tabSelected
 *
 * This slot executes when someone changes tab (public/secret key)
 * --------------------------------------------------------------------------------- */
void CMainWindow::tabSelected( int id )
{
    readKeys( id ? FALSE : TRUE );
}

/* ---------------------------------------------------------------------------------
 * CMainWindow::reReadPublic
 *
 * Re read public keys, this is executed after an import has occured
 * --------------------------------------------------------------------------------- */
void CMainWindow::reReadPublic()
{
    if ( tabBar->currentTab() == 0) {
        readKeys( TRUE );
    }    
}

/* ---------------------------------------------------------------------------------
 * CMainWindow::properties
 *
 * Displays properties dialog
 * --------------------------------------------------------------------------------- */
void CMainWindow::properties()
{
    CDlgProperties dlgProperties( this );
    if ( dlgProperties.exec() )
        applySettings();
}

/* ---------------------------------------------------------------------------------
 * CMainWindow::keyInfo
 *
 * Displays Key information dialog
 * --------------------------------------------------------------------------------- */
void CMainWindow::keyInfo()
{
    if ( checkCurrentItem() != CListViewItem::PUBLICKEY )
        return;            
    CDlgKeyInfo dlgKeyInfo( this, itemCurrent->text(LVPOS_KEYID), itemCurrent->text(LVPOS_NAME) );
    connect( &dlgKeyInfo, SIGNAL(newPublic()), this, SLOT(reReadPublic()) );
    dlgKeyInfo.exec();
}

/* ---------------------------------------------------------------------------------
 * CMainWindow::about
 *
 * Displays About dialog
 * --------------------------------------------------------------------------------- */
void CMainWindow::about()
{
    CDlgAbout dlgAbout( this );
    dlgAbout.exec();
}

/* ---------------------------------------------------------------------------------
 * CMainWindow::serverSearch
 *
 * Displays Keyserver Search dialog
 * --------------------------------------------------------------------------------- */
void CMainWindow::serverSearch( bool quickSearch )
{
    QString searchString = QString::null;
    int retval = checkCurrentItem();

    if ( quickSearch ) {
        if ( retval == CListViewItem::USERID || retval == CListViewItem::SECRETKEY )
            return;
        if ( retval != -1 )
            searchString = "0x" + itemCurrent->text(LVPOS_KEYID).right(8);
    }
    
    CDlgServerSearch dlgServerSearch( this, searchString );
    connect( &dlgServerSearch, SIGNAL(newPublic()), this, SLOT(reReadPublic()) );
    dlgServerSearch.exec();
}

/* ---------------------------------------------------------------------------------
 * CMainWindow::contextMenuRequested
 *
 * This slot is triggerd when you right click in the listview, it's used to display
 * a popup menu.
 * --------------------------------------------------------------------------------- */
void CMainWindow::contextMenuRequested( QListViewItem *tmpItem, const QPoint &pos, int )
{
    /* Abort if no item selected */
    if ( !tmpItem )
        return;
    
    /* Display Menu */
    itemCurrent = (CListViewItem*)tmpItem;
    switch ( itemCurrent->type ) {
    
    case CListViewItem::PUBLICKEY:
        menuPublic->exec( pos );
        break;

    case CListViewItem::SIGNATURE:
        menuSignature->exec( pos );
        break;

    default:
        break;
    
    }
}

/* ---------------------------------------------------------------------------------
 * CMainWindow::sendKey
 *
 * This functions is used to send a key to the keyserver
 * --------------------------------------------------------------------------------- */
void CMainWindow::sendKey()
{
    if ( checkCurrentItem() != CListViewItem::PUBLICKEY )
        return;

    /* Ask for confirmation */
    int retval = QMessageBox::warning( this, "GPG Keys", tr( "Really upload key?" ), QMessageBox::Yes,
                                       QMessageBox::No | QMessageBox::Escape | QMessageBox::Default );
    if ( retval == QMessageBox::Yes ) {
        exportOperation = EXPORT_SEND;
        exportKey( itemCurrent->text(LVPOS_KEYID) );
    }
}

/* ---------------------------------------------------------------------------------
 * CMainWindow::toClipboard
 *
 * This functions is used to export a key to the clipboard
 * --------------------------------------------------------------------------------- */
#ifdef CLIPBOARD_SUPPORT
void CMainWindow::toClipboard()
{
    if ( checkCurrentItem() != CListViewItem::PUBLICKEY )
        return;

    /* Ask for confirmation */
    int retval = QMessageBox::warning( this, "GPG Keys", tr( "Export to clipboard?" ), QMessageBox::Yes,
                                       QMessageBox::No | QMessageBox::Escape | QMessageBox::Default );
    if ( retval == QMessageBox::Yes ) {
        exportOperation = EXPORT_CLIPBOARD;
        exportKey( itemCurrent->text(LVPOS_KEYID) );
    }
}
#endif

/* ---------------------------------------------------------------------------------
 * CMainWindow::fromClipboard
 *
 * This functions is used to export a key to the clipboard
 * --------------------------------------------------------------------------------- */
#ifdef CLIPBOARD_SUPPORT
void CMainWindow::fromClipboard()
{
    /* Ask for confirmation */
    int retval = QMessageBox::warning( this, "GPG Keys", tr( "Import from clipboard?" ),
                                       QMessageBox::Yes, QMessageBox::No | QMessageBox::Escape |
                                       QMessageBox::Default );
    if ( retval == QMessageBox::Yes ) {
        gnupgProc->import( QApplication::clipboard()->text() );
    }
}
#endif

/* ---------------------------------------------------------------------------------
 * CMainWindow::toFile
 *
 * This functions is used to export a key to a file
 * --------------------------------------------------------------------------------- */
void CMainWindow::toFile()
{
    if ( checkCurrentItem() != CListViewItem::PUBLICKEY )
        return;

    //fileName = QFileDialog::getSaveFileName( QString::null, QString::null, this, 
    //                                         "exportdlg", tr( "Export key to..." ) );
    QFileDialog fileDlg( QString::null, QString::null, this );
    fileDlg.setMode( QFileDialog::AnyFile );
    fileDlg.setCaption( tr( "Export key to...") );
    fileDlg.setFont( readConfigFont( "stdfont", STD_FONT ) );
                                                     
    //if ( fileName != QString::null ) {
    if ( fileDlg.exec() == QDialog::Accepted ) {
        fileName = fileDlg.selectedFile();
        exportOperation = EXPORT_FILE;
        exportKey( itemCurrent->text(LVPOS_KEYID) );
    }
}

/* ---------------------------------------------------------------------------------
 * CMainWindow::fromFile
 *
 * This functions is used to import a key from a file
 * --------------------------------------------------------------------------------- */
void CMainWindow::fromFile()
{
    //fileName = QFileDialog::getOpenFileName( QString::null, QString::null, this, 
    //                                         "importdlg", tr( "Select key..." ) );
    QFileDialog fileDlg( QString::null, QString::null, this );
    fileDlg.setCaption( tr( "Select key...") );
    fileDlg.setFont( readConfigFont( "stdfont", STD_FONT ) );
                                                     
    //if ( fileName != QString::null ) {
    if ( fileDlg.exec() == QDialog::Accepted ) {
        fileName = fileDlg.selectedFile();

        QFile importFile;
        QString importData = "", line;

        importFile.setName( fileName );
        if ( !importFile.open( IO_ReadOnly ) ) {
            statusBar()->message( tr( "Fileopen failed!" ), 2000 );
        } else {
            QTextStream ts( &importFile );
            while( line = ts.readLine() ) {
                importData += line + "\n";
            }
            importFile.close();
            if ( gnupgProc->import( importData ) == CGPGProc::OK )
                reReadPublic();
        }

    }
}

/* ---------------------------------------------------------------------------------
 * CMainWindow::signKey
 *
 * Displays the sign key dialog
 * --------------------------------------------------------------------------------- */
void CMainWindow::signKey()
{
    if ( checkCurrentItem() != CListViewItem::PUBLICKEY )
        return;

    CDlgSign dlgSign( this, itemCurrent->text(LVPOS_KEYID) );
    connect( &dlgSign, SIGNAL(newPublic()), this, SLOT(reReadPublic()) );
    dlgSign.exec();
}

/* ---------------------------------------------------------------------------------
 * CMainWindow::doDelete
 *
 * This function is used to delete a key. First it ask the user for a confirmation
 * and then it runs gpg to actually remove the key from his keyring
 * --------------------------------------------------------------------------------- */
void CMainWindow::doDelete()
{
    if ( checkCurrentItem() != CListViewItem::PUBLICKEY )
        return;

    /* Confirm Messagebox */
    int retval = QMessageBox::warning( this, "GPG Keys", tr( "Proceed with delete?" ), QMessageBox::Yes,
                                       QMessageBox::No | QMessageBox::Escape | QMessageBox::Default );

    if ( retval == QMessageBox::Yes ) {
        
        /* Create Argument List */
        QStringList args;
        args << "--batch" << "--yes" << "--delete-key" << "0x" + itemCurrent->text(LVPOS_KEYID);
    
        /* Set Status Bar Message */
        statusBar()->message( tr( "Deleting Key..." ) );
    
        /* Run and Check Return Code */
        switch ( CGPGProc::run( this, args ) ) {
        
        case CGPGProc::RUNFAILED:
            runFailed( statusBar() );
            break;   
        case CGPGProc::OK:
            statusBar()->message( tr( "Key Deleted!" ), 3000 );
            listMain->takeItem( itemCurrent );
            break;    
        case CGPGProc::ERROR:
            statusBar()->message( tr( "Deletion Failed!" ), 3000 );
            break;
    
        }
    }    
}

/* ---------------------------------------------------------------------------------
 * CMainWindow::tcpStatusChange
 *
 * This slot is called when the tcp connection changes status
 * --------------------------------------------------------------------------------- */
void CMainWindow::tcpStatusChange( int newStatus )
{
    QString newText;
    switch( newStatus ) {    
    case CKeyServer::Idle:
        statusBar()->message( tr( "Idle" ), 2000 );
        return;
    case CKeyServer::Connecting:
        newText = tr( "Connecting..." );
        break;
    case CKeyServer::Sending:
        newText = tr( "Sending Request..." );
        break;
    case CKeyServer::Waiting:
        newText = tr( "Waiting for reply..." );
        break;
    case CKeyServer::Recieving:
        newText = tr( "Recieving data..." );
        break;
    default:
        break;
    
    }
    statusBar()->message( newText );
}

/* ---------------------------------------------------------------------------------
 * CMainWindow::tcpSendDone
 *
 * This is run after a key has been uploaded to the keyserver
 * --------------------------------------------------------------------------------- */
void CMainWindow::tcpSendDone()
{
    statusBar()->message( tr( "Key sucessfully sent" ), 2000 );
}

/* ---------------------------------------------------------------------------------
 * CMainWindow::exportKey
 *
 * This function starts GPG and dumps a key to memory
 * --------------------------------------------------------------------------------- */
void CMainWindow::exportKey( QString keyID )
{

    /* Disable TabBar */
    tabBar->setDisabled( TRUE );

    /* Create Process */
    gpgProc = new QProcess( this );
    gpgProc->addArgument( readConfigString( "gpgpath", GPGPATH ) );
    gpgProc->addArgument( "--armor" );
    gpgProc->addArgument( "--export" );
    gpgProc->addArgument( "0x" + keyID );
    connect( gpgProc, SIGNAL(processExited()), this, SLOT(exportProcDone()) );
    
    /* Run GPG */
    if ( !gpgProc->start() ) {
        runFailed( statusBar() );
        return;
    }

    /* Display Statusbar Message */
    statusBar()->message( tr( "Exporting key..." ) );

}

/* ---------------------------------------------------------------------------------
 * CMainWindow::exportProcDone
 *
 * This slot is triggerd when the exportKey procedure is finished,
 * and is used to read all data outputed to stardard out and carry out
 * the rest of the operation.
 * --------------------------------------------------------------------------------- */
void CMainWindow::exportProcDone()
{
    /* Get Data */
    QString exportData = QString::null;
    while ( gpgProc->canReadLineStdout() ) {
        exportData += gpgProc->readLineStdout() + '\n';
    }
    
    QFile exportFile;
    switch ( exportOperation ) {
    
    case EXPORT_SEND:       // Send to keyserver
        keyServer->sendKey( exportData );
        break;
#ifdef CLIPBOARD_SUPPORT
    case EXPORT_CLIPBOARD:  // Copy to clipboard
        QApplication::clipboard()->setText( exportData );
        statusBar()->message( tr( "Key exported to clipboard" ), 2000 );
        break;
#endif
    case EXPORT_FILE:       // Export to file
        exportFile.setName( fileName );
        if ( !exportFile.open( IO_WriteOnly ) ) {
            statusBar()->message( tr( "File creation failed!" ), 2000 );
        } else {
            exportFile.writeBlock( exportData, exportData.length() );
            statusBar()->message( tr( "Key exported!" ), 2000 );
            exportFile.close();
        }
        break;
    default:
        break;

    }

    /* Enable TabBar */
    tabBar->setDisabled( FALSE );

}

/* ---------------------------------------------------------------------------------
 * CMainWindow::checkCurrentItem
 *
 * This little function is used to determine the type of the current listitem
 * --------------------------------------------------------------------------------- */
int CMainWindow::checkCurrentItem()
{
    if ( !itemCurrent )
        return -1;
    return itemCurrent->type;
}

/* ---------------------------------------------------------------------------------
 * CMainWindow::gnupgProcMessage
 *
 * This function recieves messages from the gnupgProc class
 * --------------------------------------------------------------------------------- */
void CMainWindow::gnupgProcMessage( QString message, int timeout )
{
    if ( gnupgProc->owner == 0 ) {
        if ( timeout == 0) {
            statusBar()->message( message );
        } else {
            statusBar()->message( message, timeout );
        }
    }
}

