/*
  libu - a C++ widget library based on SDL (Simple Direct Layer)
  Copyright (C) 2002 Malcolm Walker
  Based on code copyright  (C) 1999  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
  
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/
#include "application.h"
#include "pushbutton.h"
#include "painter.h"
#include "rootwindow.h"
#include "cornercontainer.h"
#include "debug.h"

#include <cstdio>
#include <cstdlib>
#include <cmath>

using namespace wftk;

#define BPP 3
/* fineness 2..8 number of relevant bits in gamma value
 fineness = 2 uses a lookup table of 1024 which savely fits into the first level cache
 fineness = 8 uses a 64K lookup table which does perform worse on systems with small cache
              (e.g. a Celeron only has got 128K L2 cache and we already jam the cache data
               with all the pixels data ... )

 */
#define GAMMA_FINENESS 6
#define GAMMA_MASK 0xFC
 
/* fading blitter
   blit surface using given gamma value or even gamma map
   
*/
class FadingBlitter
{
 protected:
  static unsigned char *lut;
  static unsigned ref;
 public:
  FadingBlitter()
    {
      if(!ref)
	{
	  lut = new unsigned char [256*(1<<GAMMA_FINENESS)];
	  for(unsigned s=0; s< 256; s++)
	    for(unsigned g=0; g < 256; g++)
	      lut[s + g<<GAMMA_FINENESS] = (s + g<<(8-GAMMA_FINENESS) ) % 256;
	}
      ref++;
    }
  ~FadingBlitter()
    {
      ref--;
      if(!ref)
	delete [] lut;
    }
  void blit(Surface& dst, const Surface& src, unsigned char g)
    {
      assert(dst.pixelformat().asString() == src.pixelformat().asString());
      assert(dst.pixelformat().bpp() == BPP);
      assert(dst.width() == src.width());
      assert(dst.height() == src.height());
      dst.lock();
      src.lock();
      unsigned char* sPtr = (unsigned char*)src.pixels();
      unsigned char* dPtr = (unsigned char*)dst.pixels();
      unsigned offset = (g & GAMMA_MASK) << GAMMA_FINENESS;
      unsigned y = 0;
      unsigned x = 0;
      unsigned dstSkip = dst.pitch() - dst.width() * BPP;
      unsigned srcSkip = src.pitch() - src.width() * BPP;
      y = dst.height();
      while(y--)
	{
	  x = dst.width();
	  while(x--)
	    {
	      *dPtr++ = lut[*sPtr++ + offset];
	      *dPtr++ = lut[*sPtr++ + offset];
	      *dPtr++ = lut[*sPtr++ + offset];
	    }
	  dPtr += dstSkip;
	  sPtr += srcSkip;
	}
      dst.unlock();
      src.unlock();
    }
  void blit(Surface& dst, const Surface& src, const Surface& gamma_map)
    {
      assert(dst.pixelformat().asString() == src.pixelformat().asString());
      assert(dst.pixelformat().bpp() == BPP);
      assert(gamma_map.pixelformat().bpp() == 1);

      assert(dst.width() == gamma_map.width());
      assert(dst.width() == gamma_map.width());
      assert(dst.height() == src.height());
      assert(dst.height() == src.height());
      dst.lock();
      src.lock();
      gamma_map.lock();

      unsigned char* sPtr = (unsigned char*)src.pixels();
      unsigned char* dPtr = (unsigned char*)dst.pixels();
      unsigned char* gPtr = (unsigned char*)gamma_map.pixels();
     
      unsigned y = 0;
      unsigned x = 0;
      unsigned dstSkip = dst.pitch() - dst.width() * BPP;
      unsigned srcSkip = src.pitch() - src.width() * BPP;
      unsigned gammaSkip = gamma_map.pitch() - gamma_map.width();
      unsigned offset;
      
      y = dst.height();
      while(y--)
	{
	  x = dst.width();
	  while(x--)
	    {
	      offset = (*gPtr++ & GAMMA_MASK) << GAMMA_FINENESS;
	      *dPtr++ = lut[*sPtr++ + offset];
	      *dPtr++ = lut[*sPtr++ + offset];
	      *dPtr++ = lut[*sPtr++ + offset];
	    }
	  dPtr += dstSkip;
	  sPtr += srcSkip;
	  gPtr += gammaSkip;
	}
      dst.unlock();
      src.unlock();
      gamma_map.unlock();
    }
};

unsigned char* FadingBlitter::lut = NULL;
unsigned FadingBlitter::ref = 0;

class Terrain : public CornerContainer
{
 protected:
  ///
  virtual void draw(Surface&, const Point&, const Region&);
 private:
  ///
  Point w2p(float, float, float);
};

class TerrainApp : public Application
{
 public:
  TerrainApp(int argc, char** argv) : 
    Application(argc, argv)
    {
      new RootWindow(800,600);
  
      RootWindow::instance()->setTitle("LIBWFTK Texture Demo");

      //eg fontloading ....
      // These fonts are used by several gui elements,
      // so you should slways load text_font and button_font when
      // using widgets from libwftk.
      Font::registry.load("text_font","wf_opal.ttf, 18, 0xF0F0F0, 0x101010");
      Font::registry.load("button_font","wf_opal.ttf, 18, 0xF0F0F0, 0x101010");

      Surface::registry.load("button170","button.png");
      Surface::registry.load("button170_pressed","button_pressed.png");
      
      //      Surface::registry.load("mud","mud_dried.png");
      //       Surface::registry.load("mud25","mud_dried_shade25.png");
      //       Surface::registry.load("mud50","mud_dried_shade50.png");
      //       Surface::registry.load("mud75","mud_dried_shade75.png");

      Terrain* terrain = new Terrain();
      RootWindow::instance()->pack(terrain);

      //a pushbutton
      PushButton* quit_button = new PushButton("Quit");
      terrain->pack(quit_button);
    
      //connect this button's click event (SIGNAL) with the application's
      //quit action (SLOT)
      quit_button->clicked.connect(quitSlot());
    }
};

#define SIZE_X 15
#define SIZE_Y 15

Point 
Terrain::w2p(float x, float y, float z)
{ 
  int my = height();
  int mx = width() / 2;
  float scale_x = 40.0;
  float scale_y = 20.0; 
  float scale_z = 20.0;

  int _x = mx + (int)((x - y) * scale_x + 0.45);
  int _y = my - (int)((x + y) * scale_y + z * scale_z + 0.45);

  return Point(_x,_y);
}

void Terrain::draw(Surface& target, const Point& offset, const Region& r)
{
  Point p1,p2,p3,p4;
 
  Surface drawing(width(), height());
 
  float grid[SIZE_X][SIZE_Y];
  unsigned x,y;
  for(x = 0; x < SIZE_X; x++)
    {
      for(y = 0; y < SIZE_Y; y ++)
	{
	  grid [x][y] = sin( x );
	}
    }

  //clear surface
  drawing.clear();
  
  float z1,z2,z3,z4;
  
  Color colors[256];

  for(unsigned  n = 0; n < 256; n++)
    colors[n] = Color(n/4+32,n/4+32,n/4+32);

  Painter painter(&drawing);
  painter.setColor("blue");
  int c1,c2;
//  int c3,c4;
  
  float delta1, delta2;
  painter.setFilling(true);

  for(x = 0; x < SIZE_X-1; x++)
    for(y = 0; y < SIZE_Y-1; y ++)
      {
	z1 = grid[x][y+1];
	z2 = grid[x+1][y+1];
	z3 = grid[x+1][y];
	z4 = grid[x][y];

	delta1 = z1 - (z2+z4)/2.0;
	delta2 = (z2+z4)/2.0 - z3;
	
	c1 = 128 + (int) (delta1 * 256);
	c2 = 128 + (int) (delta2 * 256);
	
	//restrict values to range [0..255]
	if(c1 < 0) 
	  c1 = 0;
	if(c1 > 255)
	  c1 = 255;
	if(c2 < 0)
	  c2 = 0;
	if(c2 > 255)
	  c2 = 255;

// 	c1 = (unsigned char)(z1 * 256.0);
// 	c2 = (unsigned char)(z2 * 256.0);
// 	c3 = (unsigned char)(z3 * 256.0);
// 	c4 = (unsigned char)(z4 * 256.0);

	p1 = w2p(x,y+1,z1);
	p2 = w2p(x+1,y+1,z2);
	p3 = w2p(x+1,y,z3);
	p4 = w2p(x,y,z4);

	painter.trigon(p1,p2,p4,colors[c1]);
	painter.trigon(p2,p3,p4,colors[c2]);
// 	cerr << p1 << endl;
// 	cerr << p2 << endl;
// 	cerr << p3 << endl;
// 	cerr << p4 << endl;
// 	cerr << endl;
	// need to figure out colors for the edges
// 	painter.shadedTrigon(p1,p2,p3,colors[c1],colors[c2],colors[c3]);
// 	painter.shadedTrigon(p4,p3,p2,colors[c4],colors[c3],colors[c2]);

	
// 	if(z1 > z2)
// 	  if(z3 > z4)
// 	    Surface::registry.find("mud")->textureBlit(drawing,p1,p2,p3,p4);
// 	  else
// 	    Surface::registry.find("mud25")->textureBlit(drawing,p1,p2,p3,p4);
// 	else	  
// 	  if(z3 > z4)
// 	    Surface::registry.find("mud50")->textureBlit(drawing,p1,p2,p3,p4);
// 	  else
// 	    Surface::registry.find("mud75")->textureBlit(drawing,p1,p2,p3,p4);

	
      }

  drawing.blit(target, offset, r);
}

int main (int argc, char **argv)
{
  Debug::init(Debug::GENERIC);

  return TerrainApp(argc, argv).exec(); 
}

