/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2003 Ron Steinke <rsteinke@w-link.net>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "joystick.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "application.h"

#include <SDL/SDL.h>
#include <cassert>

bool wftk::Joystick::init_ = false;
std::vector<wftk::Joystick*> wftk::Joystick::joysticks_;

wftk::Joystick::Joystick(unsigned n) : num_(n)
{
  assert(number() > n);
  assert(!joysticks_[n]);
  joysticks_[n] = this;

  stick_ = SDL_JoystickOpen((int) n);
  assert(stick_);

  nAxes_ = (Uint8) SDL_JoystickNumAxes(stick_);
  nBalls_ = (Uint8) SDL_JoystickNumBalls(stick_);
  nHats_ = (Uint8) SDL_JoystickNumHats(stick_);
  nButtons_ = (Uint8) SDL_JoystickNumButtons(stick_);
}

wftk::Joystick::~Joystick()
{
  SDL_JoystickClose(stick_);

  assert(joysticks_[num_] == this);
  joysticks_[num_] = 0;
}

wftk::Joystick* wftk::Joystick::instance(unsigned n)
{
  if(n >= number()) // number() calls init() if necessary
    return 0;

  Joystick* j = joysticks_[n];
  return j ? j : new Joystick(n);
}

namespace { // empty namespace, don't export symbols
  // this structure takes advantage of the fact that
  // the first _three_ elements are the same for all
  // four SDL joystick event structures, not just the type
  struct JoyEvent {
    Uint8 type;
    Uint8 which;
    Uint8 elem;
  };
}

bool wftk::Joystick::handleEvent(const SDL_Event* event)
{
  // don't init Joystick instances here,
  // just drop the event if nobody's looking for it
  Uint8 which = ((const JoyEvent*) event)->which;
  // joysticks_.size() is zero if we haven't initialized yet
  Joystick* inst = which < joysticks_.size() ? joysticks_[which] : 0;
  // this checks for both which >= joysticks_.size() (usually joystick
  // subsystem not init) and for individual joysticks which haven't
  // been accessed (i.e., if only joystick zero is accessed, all other
  // elements in the array are null).
  if(!inst)
    return false;

  Uint8 elem = ((const JoyEvent*) event)->elem;

  switch(event->type) {
    case SDL_JOYAXISMOTION:
      return inst->axisEvent(elem, event->jaxis.value);
    case SDL_JOYBALLMOTION:
      return inst->ballEvent(elem, Point(event->jball.xrel, event->jball.yrel));
    case SDL_JOYHATMOTION:
      return inst->hatEvent(elem, event->jhat.value);
    case SDL_JOYBUTTONUP:
    case SDL_JOYBUTTONDOWN:
      return inst->buttonEvent(elem, event->jbutton.state == SDL_PRESSED);
    default:
      assert(false);
      return false;
  }
}

void wftk::Joystick::init()
{
  assert(!init_);
  assert(Application::instance()); // need SDL

  SDL_InitSubSystem(SDL_INIT_JOYSTICK);
  SDL_JoystickEventState(SDL_ENABLE);

  Application::instance()->destroyed.connect(SigC::slot(&shutdown));

  int joynum = SDL_NumJoysticks();
  assert(joynum >= 0);

  joysticks_.resize((unsigned) joynum);
  for(unsigned i = 0; i < joysticks_.size(); ++i)
    joysticks_[i] = 0;

  init_ = true;
}

void wftk::Joystick::shutdown()
{
  assert(init_);

  for(unsigned i = 0; i < joysticks_.size(); ++i)
    delete joysticks_[i];
  joysticks_.clear(); // in case only this subsystem is shut down

  SDL_JoystickEventState(SDL_IGNORE);
  SDL_QuitSubSystem(SDL_INIT_JOYSTICK);

  init_ = false;
}
