/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux 

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#ifndef _TIMER_H_
#define _TIMER_H_

#include <set>

#include <sigc++/object.h>
#if SIGC_MAJOR_VERSION == 1 && SIGC_MINOR_VERSION == 0
#include <sigc++/signal_system.h>
#else
#include <sigc++/signal.h>
#endif

#include <SDL/SDL_types.h>

#include "application.h"

namespace wftk {

/// A time class to escape SDL's 2^32 ms epoch limitations
class Time
{
 public:
  Time() : epoch_(0), ticks_(0) {}

  // default copy constructor, destructor, operator= are fine

  /// get the current time
  static Time now();

  /// get the SDL ticks (milliseconds) field of the time
  Uint32 ticks() const {return ticks_;}

  /// A structure used when parsing time into days, hours, ...
  struct Info
  {
    /// days field of time, can be negative for times before program start
    Sint32 days;
    /// hours field of time
    Uint8 hours;
    /// minutes field of time
    Uint8 minutes;
    /// seconds field of time
    Uint8 seconds;
    /// milliseconds field of time
    Uint16 msecs;
  };

  /// parse the time into days, hours, etc.
  Info info() const;

  /// check equality of two times
  bool operator==(const Time& t) const
	{return epoch_ == t.epoch_ && ticks_ == t.ticks_;}
  /// check inequality of two times
  bool operator!=(const Time& t) const {return !(*this == t);}
  /// check whether one time occured before another
  bool operator<(const Time& t) const
	{return (epoch_ == t.epoch_) ? (ticks_ < t.ticks_) : (epoch_ < t.epoch_);}
  /// check whether one time occured after another
  bool operator>(const Time& t) const {return t < *this;}
  /// check whether one time occured before another
  bool operator<=(const Time& t) const {return !(t < *this);}
  /// check whether one time occured after another
  bool operator>=(const Time& t) const {return !(*this < t);}

  /// find the difference between two times (only valid if the difference can fit in an int)
  int operator-(const Time& t) const;

  /// increment a timestamp
  Time& operator+=(Uint32);
  /// decrement a timestamp
  Time& operator-=(Uint32);
  /// increment a timestamp
  Time& operator+=(int val) {return (val >= 0)
	? operator+=((Uint32) val) : operator-=(negative_rectify(val));}
  /// decrement a timestamp
  Time& operator-=(int val) {return (val >= 0)
	? operator-=((Uint32) val) : operator+=(negative_rectify(val));}
  /// increment a timestamp
  Time operator+(Uint32 val) const {Time out(*this); out += val; return out;}
  /// decrement a timestamp
  Time operator-(Uint32 val) const {Time out(*this); out -= val; return out;}
  /// increment a timestamp
  Time operator+(int val) const {Time out(*this); out += val; return out;}
  /// decrement a timestamp
  Time operator-(int val) const {Time out(*this); out -= val; return out;}

 private:
  Time(Sint32 epoch, Uint32 ticks) : epoch_(epoch), ticks_(ticks) {}

  // returns -val, dealing with type limitations
  static unsigned negative_rectify(int val);

  // epoch is 2^32 milliseconds, equal to 49 days, 17 hours, 2 minutes,
  // 47 seconds, 296 milliseconds
  Sint32 epoch_; // epoch_ < 0 is time before program start
  Uint32 ticks_;
};

/// increment a timestamp
inline Time operator+(Uint32 val, const Time& t) {return t + val;}
/// increment a timestamp
inline Time operator+(int val, const Time& t) {return t + val;}

/// Timer handles known events that must happen (fairly) regularly
class Timer : virtual public SigC::Object
{
 public:
  /// Create a timer with a given timeout period (in milliseconds)
  Timer(Uint32 mseconds, bool runnning = true);
  ///
  ~Timer();

  class Alarm : public SigC::Signal1<void,unsigned int>
  {
   public:
    SigC::Connection connect(const SigC::Slot0<void>&);
    SigC::Connection connect(const SigC::Slot1<void,unsigned int>&);
  };

  /// Emitted when the timer is triggered, slots can optionally receive the ammout of time passed
  Alarm alarm;

  /// Set the timeout period
  void setInterval(unsigned int delta);
  /// Get the timeout period
  Uint32 interval() const {return deltaT_;}

  /// Run the timer
  void run();
  /// Halt the timer
  void halt();

  /// Trigger all timers for which sufficient time has passed
  static void processAllTimers();

  /**
   * returns the lesser of amount of time until the next timer and 'max'.
   * This handles the corner case where no timers are running more cleanly
   * than a simple 'get next time' function.
   **/
  static Uint32 limitWait(Uint32 max);
  
 private:
  // unimplemented
  Timer(const Timer&);
  Timer& operator=(const Timer&);

  ///
  void update(const Time&);
  ///
  bool running_;
  ///
  Uint32 deltaT_;
  ///
  Time nextEvent_;

  /// If there are any running timers, the next time one of the timers goes off
  static Time needUpdateBy_;
  /// Decrease needUpdateBy_ if necessary
  void setNeedUpdateBy(bool force = false);
  /// The set of currently running timers
  static std::set<Timer*> runningTimers_;

  typedef Application::FloatingEvent<Timer> BaseEvent;
  friend class BaseEvent;
  class Event : public BaseEvent
  {
   public:
    Event(Timer& timer, Uint32 time_passed) :
        BaseEvent(timer), timePassed_(time_passed) {}

    virtual void operator()();

   private:
    Uint32 timePassed_;
  };

  // so we can stop our alarms from being emitted if
  // we're destroyed
  BaseEvent* currentEvent_;
};

}

#endif
