/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "widget.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "debug.h"
#include "focus.h"
#include "surface.h"
#include "painter.h"

#include <assert.h>

namespace wftk {

Widget::Widget() :
  backgrnd_(0),
  color_("transparent"),
  colorDis_("darkgrey"),
  disabled_(false),
  tileBackground_(true),
  activateOnMouse_(false),
  activateOnClick_(false)
{
  setPackingInfo();
}

Widget::Widget(const Surface& backgrnd, bool tileIt) :
  backgrnd_(0),
  color_("transparent"),
  colorDis_("darkgrey"),
  disabled_(false),
  tileBackground_(true),
  activateOnMouse_(false),
  activateOnClick_(false)
{
  setBackground(backgrnd, tileIt);
  setPackingInfo();
}

Widget::Widget(Surface::Resource* backgrnd, bool tileIt) :
  backgrnd_(0),
  color_("transparent"),
  colorDis_("darkgrey"),
  disabled_(false),
  tileBackground_(true),
  activateOnMouse_(false),
  activateOnClick_(false)
{
  setBackground(backgrnd, tileIt);
  setPackingInfo();
}

Widget::Widget(const std::string& backgrnd, bool tileIt) :
  backgrnd_(0),
  color_("transparent"),
  colorDis_("darkgrey"),
  disabled_(false),
  tileBackground_(true),
  activateOnMouse_(false),
  activateOnClick_(false)
{
  setBackground(backgrnd, tileIt);
  setPackingInfo();
}

Widget::~Widget()
{
  Debug::channel(Debug::WIDGET_CREATE) << name() << " " << this << " has been hit for death." << Debug::endl;

  // if we still own focus, release it before dying !
  if(hasFocus())
    Focus::instance()->release(this);

  if(backgrnd_)
    backgrnd_->free();

  Debug::channel(Debug::WIDGET_CREATE) << name() << " vanishes." << Debug::endl;
}

void Widget::setBackground(const Surface& surf, bool tile)
{
  Surface* s = new Surface(surf);

  Surface::Resource* r = new Surface::Resource(s);
  setBackground(r, tile);
  r->free(); // since we now hold a reference through backgrnd_
}

void Widget::setBackground(Surface::Resource* backgrnd, bool tile)
{
  Debug::channel(Debug::DRAWING) << "Setting background for " << name() << Debug::endl;

  if(backgrnd == backgrnd_) { // bypass most of this function
    if(backgrnd_ && tile != tileBackground_) {
      tileBackground_ = tile;
      setScaled();
      invalidate();
    }
    return;
  }

  bool was_opaque = isOpaque();

  if(backgrnd_)
    backgrnd_->free();

  ///
  tileBackground_ = tile;
  // we don't copy the background....
  backgrnd_ = backgrnd;

  if(backgrnd_ && (backgrnd_->res()->width() == 0 || backgrnd_->res()->height() == 0))
    backgrnd_ = 0;

  if(backgrnd_)
    backgrnd_->bind();

  setScaled();

  if(isOpaque() != was_opaque) // set coverage
    handleResize(width(), height()); // chain to derived widgets' setCoverage() calls

  //we need update
  packingUpdate(); // the background image sets the widget's default size
  invalidate();
}

void Widget::setScaled()
{
  if(backgrnd_ && !backgrnd_->res()->empty() &&
    (backgrnd_->res()->width() != width() ||
    backgrnd_->res()->height() != height()) && !tileBackground_) {
    scaled_ = *backgrnd_->res(); // make a copy
    scaled_.scale(width(), height()); // and scale it
  }
  else if(!scaled_.empty())
    scaled_ = Surface(); // set it invalid, use backgrnd_ directly
}

void Widget::setBackground(const std::string& name, bool tile)
{
  Surface::Resource *res = Surface::registry.get(name);
  if(res)
    setBackground(res, tile);
}

void Widget::setColor(const Color &color)
{
  bool was_opaque = isOpaque();

  color_ = color;

  if(isOpaque() != was_opaque) // set coverage
    handleResize(width(), height()); // chain to derived widgets' setCoverage() calls

  //we need update
  invalidate();
}

void Widget::setColor(const std::string& name)
{
  // If the registry fails to find the color,
  // it returns transparent, which is the default anyway
  setColor(Color::registry.find(name));
}

void Widget::setDisabledColor(const Color& color)
{
    colorDis_ = color;
    invalidate();
}

void Widget::setDisabledColor(const std::string& name)
{
  Color::Resource *color = Color::registry.get(name);
  if(color)
    setDisabledColor(color->res());
}

bool Widget::isOpaque() const
{
  if(color_.a == Color::WFTK_OPAQUE)
    return true;

  if(!backgrnd_)
    return false;

  if(backgrnd_->res()->hasAlphaChannel())
    return false;

  if(backgrnd_->res()->usesColorKey())
    return false;

  return true;
}

void Widget::setPackingInfo()
{
  ScreenArea::setPackingInfo();

  if(backgrnd_) {
    Debug::channel(Debug::PACKING) <<
	"Setting " << name() << " prefered size from a background of size " <<
	backgrnd_->res()->rect() << Debug::endl;
    packing_info_.x.pref = backgrnd_->res()->width();
    packing_info_.y.pref = backgrnd_->res()->height();
  }
}

void Widget::enable()
{
  disabled_ = false;
  invalidate();
//  enabled.emit();
}

void Widget::disable()
{
  disabled_ = true;
  invalidate();
//  disabled.emit();
}

void Widget::drawAfter(Surface& target, const Point& offset, const Region& r)
{
  //when disabled greyout the widget
  if(disabled_)
    {
      Point p;
      Painter painter(&target);
      for(p.x=offset.x%2; (unsigned) p.x < width(); p.x+=2)
        for(p.y=offset.y%2; (unsigned) p.y < height(); p.y+=2)
          if(r.contains(p + offset))
            painter.setPixel(p + offset, colorDis_);
    }
}

void Widget::draw(Surface& surf, const Point& offset, const Region& r)
{
  assert(!isHidden());

  Debug out(Debug::DRAWING);
  out << "Inside draw function for " << name() << " " << this << Debug::endl;

  if(out)
    out << "Drawing " << (isOpaque() ? "opaque " : "transparent ")
	<< name() << " " << this << Debug::endl;

  bool have_background = backgrnd_ && !backgrnd_->res()->empty();
  bool need_bg_color = !have_background || backgrnd_->res()->hasAlphaChannel()
	|| backgrnd_->res()->usesColorKey();

  if(need_bg_color && color_.a != Color::WFTK_TRANSPARENT) {
    out << "Drawing background color " << color_ << " for widget " <<
      this << Debug::endl;
    surf.blend(r, color_);
  }

  if(have_background) {
    // have to call this here instead of in handleResize(), since derived
    // classes may override that and break stuff
    if(!tileBackground_ && (scaled_.width() != width() || scaled_.height() != height()))
      setScaled();

    const Surface* source = scaled_.empty() ? backgrnd_->res() : &scaled_;
    if(source->width() >= width() && source->height() >= height()) {
      out << "Doing a straight blit of the background surface" << Debug::endl;
      source->blit(surf, offset, r);
    }
    else {
      assert(tileBackground_);
      assert(source == backgrnd_->res());

      int w = backgrnd_->res()->width();
      int h = backgrnd_->res()->height();

      assert(w > 0 && h > 0);

      out << "Tiling a " << w << 'x' << h << " background onto a "
	<< surf.width() << 'x' << surf.height() << " surface" << Debug::endl;

      Point p;
      for(p.y = 0; (unsigned) p.y < height(); p.y += h) {
        for(p.x = 0; (unsigned) p.x < width(); p.x += w) {
          out << "Blitting tile at " << p << Debug::endl;
          //target, dest origin, src region
          backgrnd_->res()->blit(surf, p + offset, r);
        }
      }
    }
  }
}

void Widget::handleResize(Uint16 w, Uint16 h)
{
  Rect size(0, 0, w, h);
  setShape(size, isOpaque() ? size : Region());
}

void Widget::getResourceBackground(const std::string& name)
{
  setColor(name + "_color");
  setBackground(name + "_surface", false);
}

bool Widget::isEnabled() const
{
  for(const Widget* p = this; p != 0; p = dynamic_cast<const Widget*>(p->parent()))
    if (p->disabled_)
      return false;
  return true;
}

void Widget::checkGrabFocus()
{
  // Check if this widget or a child already has the focus
  for(ScreenArea* sc = Focus::instance()->currentFocus(); sc != 0; sc = sc->parent())
    if(sc == this)
      return;

  Focus::instance()->grab(this);
}

void Widget::setClickToFocus(bool val)
{
  bool couldFocus = acceptsFocus();

  activateOnClick_ = val;

  if(couldFocus != acceptsFocus()) {
    if(!couldFocus)
      Focus::instance()->addFocusable(this);
    else
      Focus::instance()->removeFocusable(this);
  }
}

void Widget::setPointToFocus(bool val)
{
  bool couldFocus = acceptsFocus();

  activateOnMouse_ = val;

  if(couldFocus != acceptsFocus()) {
    if(!couldFocus)
      Focus::instance()->addFocusable(this);
    else
      Focus::instance()->removeFocusable(this);
  }
}

} // namespace wftk
