#charset "us-ascii"

/* Copyright (c) 2000, 2002 Michael J. Roberts.  All Rights Reserved. */
/*
 *   TADS 3 Library - miscellaneous definitions
 *   
 *   This module contains miscellaneous definitions that don't have a
 *   natural grouping with any larger modules, and which aren't complex
 *   enough to justify modules of their own.  
 */

/* include the library header */
#include "adv3.h"


/* ------------------------------------------------------------------------ */
/*
 *   When a call is made to a property not defined or inherited by the
 *   target object, the system will automatically invoke this method.  The
 *   method will be invoked with a property pointer as its first argument,
 *   and the original arguments as the remaining arguments.  The first
 *   argument gives the property that was invoked and not defined by the
 *   object.  A typical definition in an object would look like this:
 *   
 *   propNotDefined(prop, [args]) { ... }
 *   
 *   If this method is not defined by the object, the system simply
 *   returns nil as the value of the undefined property evaluation or
 *   method invocation.
 */
property propNotDefined;
export propNotDefined;


/* ------------------------------------------------------------------------ */
/*
 *   Library Pre-Initializer.  This object performs the following
 *   initialization operations immediately after compilation is completed:
 *   
 *   - adds each defined Thing to its container's contents list
 *   
 *   - adds each defined Sense to the global sense list
 *   
 *   This object is named so that other libraries and/or user code can
 *   create initialization order dependencies upon it.  
 */
adv3LibPreinit: PreinitObject
    execute()
    {
        /* 
         *   visit every VocabObject, and run its vocabulary initializer
         *   (this routine will be defined in the language-specific part
         *   of the library to enter each object's vocabulary words into
         *   the dictionary) 
         */
        forEachInstance(VocabObject, { obj: obj.initializeVocab() });

        /* visit every Thing, and run its general initializer */
        forEachInstance(Thing, { obj: obj.initializeThing() });

        /* 
         *   visit every Sense, and add each one into the global sense
         *   list 
         */
        forEachInstance(Sense, { obj: libGlobal.allSenses += obj });

        /* initialize each Actor */
        forEachInstance(Actor, { obj: obj.initializeActor() });

        /* initialize the direction list */
        Direction.initializeDirectionClass();

        /* initialize the noise/odor notification daemon */
        new Daemon(SensoryEmanation, &noteSenseChanges, 1);

        /* set up a daemon for the current location */
        new Daemon(BasicLocation, &dispatchRoomDaemon, 1);

        /* 
         *   Initialize the status line daemon.  Set this daemon's event
         *   order to a high value so that it runs last, after all other
         *   daemons - we want this to be the last prompt daemon so that
         *   the status line is updated after any other daemons have done
         *   their jobs already, in case any of them move the player
         *   character to a new location or affect the score, or make any
         *   other changes that should be reflected on the status line.  
         */
        local sld = new PromptDaemon(statusLine, &showStatusLine);
        sld.eventOrder = 1000;

        /* 
         *   Attach the command sequencer output filter, the
         *   language-specific message parameter substitution filter, the
         *   style tag formatter filter, and the paragraph filter to the
         *   main output stream.  Stack them so that the paragraph manager
         *   is at the bottom, since the library tag filter can produce
         *   paragraph tags and thus needs to sit atop the paragraph
         *   filter.  Put the command sequencer above those, since it
         *   might need to write style tags.  Finally, put the sense
         *   context filter on top of those.  
         */
        mainOutputStream.addOutputFilter(paragraphManager);
        mainOutputStream.addOutputFilter(styleTagFilter);
        mainOutputStream.addOutputFilter(langMessageBuilder);
        mainOutputStream.addOutputFilter(commandSequencer);
        mainOutputStream.addOutputFilter(senseContext);

        /* 
         *   Attach our message parameter filter and style tag filter to
         *   the statusline stream.  We don't need a paragraph filter in
         *   the statusline.  
         */
        statuslineOutputStream.addOutputFilter(styleTagFilter);
        statuslineOutputStream.addOutputFilter(langMessageBuilder);

        /* likewise the text-mode statusline-right stream */
        statusRightOutputStream.addOutputFilter(styleTagFilter);
        statusRightOutputStream.addOutputFilter(langMessageBuilder);
    }

    /* 
     *   make sure the output streams we depend on are initialized before
     *   me (so that they set up properly internally) 
     */
    execBeforeMe = [mainOutputStream, statuslineOutputStream,
                    statusRightOutputStream]
;

/* ------------------------------------------------------------------------ */
/*
 *   Library Initializer.  This object performs the following
 *   initialization operations each time the game is started:
 *   
 *   - sets up the library's default output function 
 */
adv3LibInit: InitObject
    execute()
    {
        /* 
         *   Set up our default output function.  Note that we must do
         *   this during run-time initialization each time we start the
         *   game, rather than during pre-initialization, because the
         *   default output function state is not part of the load-image
         *   configuration. 
         */
        t3SetSay(say);
    }
;


/* ------------------------------------------------------------------------ */
/*
 *   Generic script object.  This class can be used to implement a simple
 *   state machine.  
 */
class Script: object
    /* 
     *   Get the current state.  This returns a value that gives the
     *   current state of the script, which is usually simply an integer.  
     */
    getState()
    {
        /* by default, return our state property */
        return curState;
    }

    /*
     *   Process the next step of the script.  This routine must be
     *   overridden to perform the action of the script.  This routine's
     *   action should call getState() to get our current state, and
     *   should update the internal state appropriately to take us to the
     *   next step after the current one.
     *   
     *   By default, we don't do anything at all.  
     */
    doScript()
    {
        /* override to carry out the script */
    }

    /* 
     *   Property giving our current state.  This should never be used
     *   directly; instead, getState() should always be used, since
     *   getState() can be overridden so that the state depends on
     *   something other than this internal state property. 
     */
    curState = 0

;

/* ------------------------------------------------------------------------ */
/*
 *   Text list.  This is a simple type of script that can be used to
 *   generate a sequence of different messages. 
 */
class TextList: Script
    /*
     *   Our message list.  This is simply a list of single-quoted
     *   strings.  Each time we're invoked, we'll choose one of these
     *   strings to display, based on our state.
     *   
     *   Elements of the list can be nil; a nil element indicates that
     *   nothing at all is to be displayed for that message position.
     *   This can be useful, for example, when messages are delivered with
     *   a daemon, but messages are not desired for every single turn.  
     */
    textStrings = []

    /* 
     *   Advance to the next state.  By default, we'll simply increment
     *   our internal state variable to the next index in the text string
     *   list; if we reach the end of the list, we'll loop back to the
     *   first message.  
     */
    advanceState()
    {
        /* increment our state index */
        ++curState;

        /* if that takes us past the last message, return to the first */
        if (curState > textStrings.length())
            curState = 1;
    }

    /* by default, start at the first string */
    curState = 1

    /* process the next step of the script */
    doScript()
    {
        local idx;
        
        /* 
         *   get our current state, treating it as an index into our text
         *   string list 
         */
        idx = getState();

        /* 
         *   If it's a valid index in our list, show the string.  If the
         *   entry is nil, it's a non-message step in the script, so show
         *   nothing in this case.
         */
        if (idx >= 1 && idx <= textStrings.length()
            && textStrings[idx] != nil)
            say(textStrings[idx]);

        /* advance to the next state */
        advanceState();
    }
;

/*
 *   Randomized text list.  This is similar to a regular text list, but
 *   chooses a message at random each time it's invoked.  
 */
class RandomTextList: TextList
    getState()
    {
        /* 
         *   Our current state is simply a random index into the text
         *   list.  We ignore the internal state property, since we choose
         *   a different random value every time we're called.
         *   
         *   Note that rand(n) returns a number from 0 to n-1 inclusive;
         *   since list indices run from 1 to list.length, add one to the
         *   result of rand(list.length) to get a value in the proper
         *   range for a list index.  
         */
        return rand(textStrings.length()) + 1;
    }
;

/*
 *   Slave text list.  This is a text list that takes its state from
 *   another text list object, so that this text list and the master text
 *   list automatically keep their states synchronized.  This can be
 *   useful when we have messages that reflect, for example, two different
 *   points of view on the same events: the messages for each point of
 *   view can be kept in a separate list, but the one list can be a slave
 *   of the other to ensure that the two lists are based on a common
 *   state. 
 */
class SlaveTextList: TextList
    /* my master text list object */
    masterObject = nil

    /* my state is simply the master list's state */
    getState() { return masterObject.getState(); }

    /* to advance my state, advance the master list's state */
    advanceState() { masterObject.advanceState(); }
;


/* ------------------------------------------------------------------------ */
/*
 *   Invoke the given function with the given values for the parser global
 *   variables gActor and gAction.  
 */
withParserGlobals(actor, action, func)
{
    local oldActor;
    local oldAction;
    
    /* remember the old action and actor, so we can restore them later */
    oldActor = gActor;
    oldAction = gAction;
    
    /* establish our actor and action as the global settings */
    gActor = actor;
    gAction = action;
    
    /* make sure we restore globals on the way out */
    try
    {
        /* invoke the callback and return the result */
        return (func)();
    }
    finally
    {
        /* restore the globals we changed */
        gActor = oldActor;
        gAction = oldAction;
    }
}

/* ------------------------------------------------------------------------ */
/*
 *   Library global variables 
 */
libGlobal: object
#ifdef SENSE_CACHE
    /* 
     *   Sense cache - we keep SenseInfo lists here, keyed by [pov,sense];
     *   we normally discard the cached information at the start of each
     *   turn, and disable caching entirely at the start of the "action"
     *   phase of each turn.  We leave caching disabled during each turn's
     *   action phase because this is the phase where simulation state
     *   changes are typically made, and hence it would be difficult to
     *   keep the cache coherent during this phase.
     *   
     *   When this is nil, it indicates that caching is disabled.  We only
     *   allow caching during certain phases of execution, when game state
     *   is not conventionally altered, so that we don't have to do a lot
     *   of work to keep the cache up to date.  
     */
    senseCache = nil

    /*
     *   Can-Touch cache - we keep CanTouchInfo entires here, keyed by
     *   [from,to].  This cache is the touch-path equivalent of the sense
     *   cache, and is enabled and disabled 
     */
    canTouchCache = nil

    /* 
     *   Connection list cache - this is a cache of all of the objects
     *   connected by containment to a given object. 
     */
    connectionCache = nil

    /* 
     *   Actor visual ambient cache - this keeps track of the ambient light
     *   level at the given actor. 
     */
    actorVisualAmbientCache = nil

    /* enable the cache, clearing any old cached information */
    enableSenseCache()
    {
        /* create a new, empty lookup table for the sense cache */
        senseCache = new LookupTable(32, 64);

        /* create the can-touch cache */
        canTouchCache = new LookupTable(32, 64);

        /* create the actor visual ambient cache */
        actorVisualAmbientCache = new LookupTable(32, 64);

        /* create a connection list cache */
        connectionCache = new LookupTable(32, 64);
    }

    /* disable the cache */
    disableSenseCache()
    {
        /* forget the cache tables */
        senseCache = nil;
        canTouchCache = nil;
        actorVisualAmbientCache = nil;
        connectionCache = nil;
    }
#endif

    /*
     *   List of all of the senses.  The library pre-initializer will load
     *   this list with a reference to each instance of class Sense. 
     */
    allSenses = []

    /*
     *   The current player character 
     */
    playerChar = nil

    /*
     *   Option flag: allow the player to use "you" and "me"
     *   interchangeably in referring to the player character.  We set this
     *   true by defalt, so that the player can refer to the player
     *   character in either the first or second person, as long as the
     *   player character normally uses either or these (in other words,
     *   this option is meaningless in a game when the narration refers to
     *   the player character in the third person).
     *   
     *   If desired, the game can set this flag to nil to force the player
     *   to use the correct pronoun to refer to the player character.
     *   
     *   We set the default to allow using "you" and "me" interchangeably
     *   because this will create no confusion in most games, and because
     *   most experienced IF players will be accustomed to using "me" to
     *   refer to the player character (because the majority of IF refers
     *   to the player character in the second person, and expects the
     *   player to conflate the player character with the player and hence
     *   to refer to the player character in the first person).  It is
     *   relatively unconventional for a game to refer to the player
     *   character in the first person in the narration, and thus to expect
     *   the player to use the second person to refer to the PC; as a
     *   result, experience players might tend to use the first person out
     *   of habit in such games, and might find it jarring to find the
     *   usage unacceptable.  Furthermore, in games that use a first-person
     *   narration, it seems unlikely that there will also be a
     *   second-person element to the narration; as long as both aren't
     *   present, it will cause no confusion for the game to accept either
     *   "you" or "me" as equivalent in commands.  However, the library
     *   provides this option in case such as situation does arise.  
     */
    allowYouMeMixing = true

    /*
     *   The current perspective object.  This is usually the actor
     *   performing the current command. 
     */
    pointOfView = nil

    /*
     *   The stack of point of view objects.  The last element of the
     *   vector is the most recent point of view after the current point
     *   of view.  
     */
    povStack = static new Vector(32)

    /* 
     *   The global score object.  We use a global for this, rather than
     *   referencing libScore directly, to allow the score module to be
     *   left out entirely if the game doesn't make use of scoring.  The
     *   score module should set this during pre-initialization.  
     */
    scoreObj = nil

    /* 
     *   The global Footnote class object.  We use a global for this,
     *   rather than referencing Footnote directly, to allow the footnote
     *   module to be left out entirely if the game doesn't make use of
     *   footnotes.  The footnote class should set this during
     *   pre-initialization.  
     */
    footnoteClass = nil

    /* the total number of turns so far */
    totalTurns = 0

    /* 
     *   verbose mode - if this is true, the full room description is
     *   displayed each time the player enters a room, regardless of
     *   whether or not the player has seen the room before; if this is
     *   nil, the full description is only displayed on the player's first
     *   entry to a room, and only the short description on re-entry 
     */
    verboseMode = true

    /* 
     *   flag: the parser is in 'debug' mode, in which it displays the
     *   parse tree for each command entered 
     */
    parserDebugMode = nil

    /*
     *   Most recent command, for 'undo' purposes.  This is the last
     *   command the player character performed, or the last initial
     *   command a player directed to an NPC.
     *   
     *   Note that if the player directed a series of commands to an NPC
     *   with a single command line, only the first command on such a
     *   command line is retained here, because it is only the first such
     *   command that counts as a player's turn in terms of the game
     *   clock.  Subsequent commands are executed by the NPC's on the
     *   NPC's own time, and do not count against the PC's game clock
     *   time.  The first command counts against the PC's clock because of
     *   the time it takes the PC to give the command to the NPC.  
     */
    lastCommandForUndo = ''

    /* 
     *   Most recent target actor phrase; this goes with
     *   lastCommandForUndo.  This is nil if the last command did not
     *   specify an actor (i.e., was implicitly for the player character),
     *   otherwise is the string the player typed specifying a target
     *   actor.  
     */
    lastActorForUndo = ''

    /*
     *   Current command information.  We keep track of the current
     *   command's actor and action here, as well as the verification
     *   result list and the command report list.  
     */
    curActor = nil
    curIssuingActor = nil
    curAction = nil
    curVerifyResults = nil

    /* the exitLister object, if included in the build */
    exitListerObj = nil
;

/* ------------------------------------------------------------------------ */
/*
 *   Set the common pronoun antecedents.
 *   
 *   The pronoun mechanism is extensible, so languages that use more
 *   (and/or less) than the common set of pronouns can add more functions
 *   like these as needed.  
 */

/* set the antecedent for the neuter singular pronoun ("it" in English) */
setIt(obj)
{
    gActor.setPronounAntecedent(PronounIt, obj);
}

/* set the antecedent for the masculine singular ("him") */
setHim(obj)
{
    gActor.setPronounAntecedent(PronounHim, obj);
}

/* set the antecedent for the feminine singular ("her") */
setHer(obj)
{
    gActor.setPronounAntecedent(PronounHer, obj);
}

/* set the antecedent list for the ungendered plural pronoun ("them") */
setThem(lst)
{
    gActor.setPronounAntecedent(PronounThem, lst);
}

/* ------------------------------------------------------------------------ */
/*
 *   Finish the game.  This can be called when an event occurs that ends
 *   the game, such as the player character's death, winning, or any other
 *   endpoint in the story.
 *   
 *   We'll prompt the user with options for how to proceed.  We'll always
 *   show the QUIT, RESTART, and RESTORE options; other options can be
 *   offered by listing one or more FinishOption objects in the 'extra'
 *   parameter, which is given as a list of FinishOption objects.
 *   
 *   We provide some standard additional finish options, such as
 *   FinishOptionUndo and FinishOptionCredits.  The game can subclass
 *   FinishOption to create its own additinal options as desired.  
 */
finishGame(extra)
{
    local lst;

    /* start with the standard options */
    lst = [finishOptionRestore, finishOptionRestart];

    /* add any additional options in the 'extra' parameter */
    if (extra != nil)
        lst += extra;

    /* always add 'quit' as the last option */
    lst += finishOptionQuit;

    /* keep going until we get a valid response */
promptLoop:
    for (;;)
    {
        local resp;
        
        /* show the options */
        finishOptionsLister.showListAll(lst, 0, 0);

        /* switch to no-command mode for reading the interactive input */
        "<.commandnone>";

        /* read a response */
        resp = inputManager.getInputLine(nil, nil);

        /* switch back to mid-command mode */
        "<.commandmid>";

        /* check for a match to each of the options in our list */
        foreach (local cur in lst)
        {
            /* if this one matches, process the option */
            if (cur.responseMatches(resp))
            {
                /* it matches - carry out the option */
                if (cur.doOption())
                {
                    /* 
                     *   they returned true - they want to continue asking
                     *   for more options
                     */
                    continue promptLoop;
                }
                else
                {
                    /* 
                     *   they returned nil - they want us to stop asking
                     *   for options and return to our caller 
                     */
                    return;
                }
            }
        }

        /*
         *   If we got this far, it means that we didn't get a valid
         *   option.  Display our "invalid option" message, and continue
         *   looping so that we show the prompt again and read a new
         *   option.  
         */
        libMessages.invalidFinishOption(resp);
    }
}

/*
 *   Finish Option class.  This is the base class for the abstract objects
 *   representing options offered by finishGame.  
 */
class FinishOption: object
    /* 
     *   The description, as displayed in the list of options.  For the
     *   default English messages, this is expected to be a verb phrase in
     *   infinitive form, and should show the keyword accepted as a
     *   response in all capitals: "RESTART", "see some AMUSING things to
     *   do", "show CREDITS". 
     */
    desc = ""

    /* our response keyword */
    responseKeyword = ''

    /* 
     *   a single character we accept as an alternative to our full
     *   response keyword, or nil if we don't accept a single-character
     *   response 
     */
    responseChar = nil
    
    /* 
     *   Match a response string to this option.  Returns true if the
     *   string matches our response, nil otherwise.  By default, we'll
     *   return true if the string exactly matches responseKeyword or
     *   exactly matches our responseChar (if that's non-nil), but this
     *   can be overridden to match other strings if desired.  By default,
     *   we'll match the response without regard to case.
     */
    responseMatches(response)
    {
        /* do all of our work in lower-case */
        response = response.toLower();

        /* 
         *   check for a match the full response keyword or to the single
         *   response character 
         */
        return (response == responseKeyword.toLower()
                || (responseChar != nil
                    && response == responseChar.toLower()));
    }

    /*
     *   Carry out the option.  This is called when the player enters a
     *   response that matches this option.  This routine must perform the
     *   action of the option, then return true to indicate that we should
     *   ask for another option, or nil to indicate that the finishGame()
     *   routine should simply return.  
     */
    doOption()
    {
        /* tell finishGame() to ask for another option */
        return true;
    }
;

/*
 *   QUIT option for finishGame.  The language-specific code should modify
 *   this to specify the description and response keywords.  
 */
finishOptionQuit: FinishOption
    doOption()
    {
        /* 
         *   carry out the Quit action - this will signal a
         *   QuittingException, so this call will never return 
         */
        QuitAction.terminateGame();
    }
;

/*
 *   RESTORE option for finishGame. 
 */
finishOptionRestore: FinishOption
    doOption()
    {
        /* try carrying out the Restore action */
        RestoreAction.execSystemAction();

        /* 
         *   if we got back from restoring, it means that the restore
         *   didn't succeed; show a blank line and go back to asking for
         *   another option 
         */
        "\b";
        return true;
    }
;

/*
 *   RESTART option for finishGame 
 */
finishOptionRestart: FinishOption
    doOption()
    {
        /* 
         *   carry out the restart - this will not return, since we'll
         *   reset the game state and re-enter the game at the restart
         *   entrypoint 
         */
        RestartAction.doRestartGame();
    }
;

/*
 *   UNDO option for finishGame 
 */
finishOptionUndo: FinishOption
    doOption()
    {
        /* try performing the undo */
        if (UndoAction.performUndo(nil))
        {
            /* 
             *   Success - terminate the current command with no further
             *   processing.
             */
            throw new TerminateCommandException();
        }
        else
        {
            /* 
             *   failure - show a blank line and tell the caller to ask
             *   for another option, since we couldn't carry out this
             *   option 
             */
            "<.p>";
            return true;
        }
    }
;

/*
 *   CREDITS option for finishGame 
 */
finishOptionCredits: FinishOption
    doOption()
    {
        /* show a blank line before the credits */
        "\b";

        /* run the Credits action */
        CreditsAction.execSystemAction();

        /* show a paragraph break after the credits */
        "<.p>";

        /* 
         *   this option has now had its full effect, so tell the caller
         *   to go back and ask for a new option 
         */
        return true;
    }
;

/*
 *   AMUSING option for finishGame 
 */
finishOptionAmusing: FinishOption
    /*
     *   The game must modify this object to define a doOption method.  We
     *   have no built-in way to show a list of amusing things to try, so
     *   if a game wants to offer this option, it must provide a suitable
     *   definition here.  (We never offer this option by default, so a
     *   game need not provide a definition if the game doesn't explicitly
     *   offer this option via the 'extra' argument to finishGame()).  
     */
;

/* ------------------------------------------------------------------------ */
/*
 *   Utility functions 
 */

/*
 *   nilToList - convert a 'nil' value to an empty list.  This can be
 *   useful for mix-in classes that will be used in different inheritance
 *   contexts, since the classes might or might not inherit a base class
 *   definition for list-valued methods such as preconditions.  This
 *   provides a usable default for list-valued methods that return nothing
 *   from superclasses. 
 */
nilToList(val)
{
    return (val != nil ? val : []);
}
