#charset "us-ascii"

/* Copyright (c) 2000, 2002 Michael J. Roberts.  All Rights Reserved. */
/*
 *   TADS 3 Library - module ID's
 *   
 *   This module defines a framework for "module ID's."  The module ID
 *   mechanism allows library modules to identify themselves.  The module
 *   ID framework is modular, in that several libraries can be linked
 *   together without any prior knowledge of one another, and the module
 *   ID framework will be able to find all of their ID's.  
 */

/* include the library header */
#include "adv3.h"


/* ------------------------------------------------------------------------ */
/*
 *   Module ID.  Each library add-in can define one of these, so that the
 *   "credits" command and the like can automatically show the version of
 *   each library module included in the finished game, without the game's
 *   author having to compile a list of the module versions manually.
 *   
 *   An easy way to implement a CREDITS command is to define a ModuleID
 *   object for the game itself, and override the showCredit() method to
 *   display the text of the game's credits.  This module object for the
 *   game should usually have a listingOrder of 1, because the author
 *   usually will want the game's own credits to be the first item
 *   displayed in the CREDITS listing.  
 */
class ModuleID: object
    /* my name */
    name = ''

    /* the "byline" for the module, in plain text and HTML versions */
    byline = ''
    htmlByline = ''

    /* my version number string */
    version = ''

    /*
     *   Show my library credit.  By default, we show our name and byline,
     *   then start a new line.  Typically, the game's module ID object
     *   will override this to display the full credits for the game (most
     *   authors like to show their name here along with notes of thanks
     *   to important contributors).  Libraries should normally show only
     *   a simple one-liner author credit here, since it wouldn't be
     *   polite to upstage the game's own credits.  
     */
    showCredit()
    {
        /* show the credit */
        libMessages.showCredit(name, htmlByline);
        "\n";
    }

    /*
     *   Show version information.  By default, we show our name and
     *   version number, then start a new line.  The main game's module ID
     *   should generally override this to show an appropriate version
     *   message for the game, and any library add-ins that want to
     *   display their version information can override this to do so.  
     */
    showVersion()
    {
        libMessages.showVersion(name, version);
        "\n";
    }

    /*
     *   Show the "about this game" information.  By default, we show
     *   nothing here.  Typically, only the game's module ID object will
     *   override this; in the game's module ID object, this method should
     *   display any desired background information about the game that
     *   the author wants the player to see on typing the ABOUT command.
     *   
     *   The ABOUT command conventionally displays information about the
     *   game and its author - the kind of thing you'd find in an author's
     *   notes section in a book - along with any special instructions to
     *   the player, such as notes on unusual command syntax.  Information
     *   that players will find especially helpful include:
     *   
     *   - A list of any unusual command phrasings that the game uses.
     *   Ideally, you will disclose here every verb that's required to
     *   complete the game, beyond the basic set common to most games
     *   (LOOK, INVENTORY, NORTH, SOUTH, TAKE, DROP, PUT IN, etc).  By
     *   disclosing every necessary verb and phrasing, you can be certain
     *   to avoid "guess the verb" puzzles.  (Note that it's possible to
     *   disclose every *required* verb without disclosing every
     *   *accepted* verb - some verbs might be so suggestive of a
     *   particular puzzle solution that you wouldn't want to disclose
     *   them, but as long as you disclose less suggestive alternatives
     *   that can be used to solve the same puzzles, you have a valid
     *   defense against accusations of using "guess the verb" puzzles.)
     *   
     *   - A quick overview of the NPC conversation system, if any.
     *   Conversation systems have been slowly evolving as authors
     *   experiment with different styles, and at least three or four
     *   different conventions have emerged.  The default that experienced
     *   players will expect is the traditional ASK/TELL system, so it's
     *   especially important to mention your system if you're using
     *   something else.
     *   
     *   - An indication of the "cruelty" level of the game.  In
     *   particular, many experienced players find it helpful to know from
     *   the outset how careful they have to be about saving positions
     *   throughout play, so it's helpful to point out whether or not it's
     *   possible for the player character to be killed; whether it's
     *   possible to get into situations where the game becomes
     *   "unwinnable"; and if the game can become unwinnable, whether or
     *   not this will become immediately clear.  The kindest games never
     *   kill the PC and are always winnable, no matter what actions the
     *   player takes; it's never necessary to save these games except to
     *   suspend a session for later resumption.  The cruelest games kill
     *   the PC without warning (although if they offer an UNDO command
     *   from a "death" prompt, then even this doesn't constitute true
     *   cruelty), and can become unwinnable in ways that aren't readily
     *   and immediately apparent to the player, which means that the
     *   player could proceed for quite some time (and thus invest
     *   substantial effort) after the game is already lost.
     *   
     *   - A description of any special status line displays or other
     *   on-screen information whose meaning might not be immediately
     *   apparent.  
     */
    showAbout() { }

    /* 
     *   My listing order.  When we compile a list of modules, we'll sort
     *   the modules first by ascending listing order; any modules with
     *   the same listing order will be sorted alphabetically by name with
     *   respect to the other modules with the same listing order.
     *   
     *   The value 1 is reserved for the game's own ID object.  Note that
     *   the TADS 3 library defines a module ID with listing order 50,
     *   which is chosen so that the main library credit will appear after
     *   the game credits but before any extension credits using the
     *   default order value 100 that we define here.  Extensions are
     *   free, however, to use a number lower than 5 if they wish to
     *   appear before the main library credit.  
     */
    listingOrder = 100

    /* 
     *   get a list of all of the modules that are part of the game,
     *   sorted in listing order 
     */
    getModuleList()
    {
        local lst;
        
        /* compile a list of all of the modules */
        lst = new Vector(10);
        forEachInstance(ModuleID, { obj: lst.append(obj) });
        lst = lst.toList();

        /* 
         *   sort the list by listing order (and alphabetically by name
         *   where listing orders are the same) 
         */
        lst = lst.sort(SortAsc, new function(a, b) {

            /* if the listings order differ, sort by listing order */
            if (a.listingOrder != b.listingOrder)
                return a.listingOrder - b.listingOrder;

            /* the listing orders are the same; sort by name */
            if (a.name < b.name)
                return -1;
            else if (a.name > b.name)
                return 1;
            else
                return 0;
        });

        /* return the sorted list */
        return lst;
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Base class for the game's module ID.  This merely sets the listing
 *   order to 1 so that the game's credit is listed first.  
 */
class GameID: ModuleID
    /* always list the game's credits before any library credits */
    listingOrder = 1

    /* 
     *   show a blank line after the game's credits, to make it stand
     *   apart from the list of library credits 
     */
    showCredit()
    {
        inherited();
        "\b";
    }

    /* 
     *   show a blank line after the game's version information, to make
     *   it stand apart from the list of library and VM version numbers 
     */
    showVersion()
    {
        inherited();
        "\b";
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   The main TADS 3 library ID.
 */
ModuleID
    name = 'TADS 3 Library'
    byline = 'by Michael J.\ Roberts'
    htmlByline = 'by <a href="mjr_@hotmail.com">Michael J.\ Roberts</a>'
    version = '3.0'

    /*
     *   We use a listing order of 50 so that, if all of the other credits
     *   use the defaults, we appear after the game's own credits
     *   (conventionally at listing order 1) and before any extension
     *   credits (which inherit the default order 100), but so that
     *   there's room for extensions that want to appear before us, or
     *   after us but before any default-ordered extensions.  
     */
    listingOrder = 50
;


/* ------------------------------------------------------------------------ */
/*
 *   An ID module not for the library but for the T3 VM itself.  This
 *   doesn't display any credit information, but displays version number
 *   information for the VM so that the "version" command shows what
 *   version of the interpreter is in use.  
 */
ModuleID
    showCredit() { }
    showVersion()
    {
        local vsn = t3GetVMVsn();

        /* 
         *   show the version information - note that we must decompose
         *   the version number into the standard 3-part dotted string 
         */
        libMessages.showVersion('T3 VM (' + t3GetVMID() + ')',
                                '' + (vsn >> 16) + '.'
                                + ((vsn >> 8) & 0xFF) + '.'
                                + (vsn & 0xFF));
        "\n";
    }

    /*
     *   Use a very high listing order so that we're the last thing shown.
     */
    listingOrder = 10000
;

