#charset "us-ascii"

/* 
 *   Copyright (c) 2000, 2002 Michael J. Roberts.  All Rights Reserved. 
 *   
 *   Portions based on work by Kevin Forchione, used by permission.  
 */

/*
 *   TADS 3 Library - objects
 *   
 *   This module defines the basic physical simulation objects (apart from
 *   Thing, the base class for most game objects, which is so large that
 *   it's defined in its own separate module for convenience).  We define
 *   such basic classes as containers, surfaces, fixed-in-place objects,
 *   openables, and lockables.  
 */

/* include the library header */
#include "adv3.h"


/* ------------------------------------------------------------------------ */
/*
 *   Intangible - this is an object that represents something that can be
 *   sensed but which has no tangible existence, such as a ray of light, a
 *   sound, or an odor. 
 */
class Intangible: Thing
    /*
     *   The base intangible object has no presence in any sense,
     *   including sight.  Subclasses should override these as appropriate
     *   for the senses in which the object can be sensed.  
     */
    sightPresence = nil
    soundPresence = nil
    smellPresence = nil
    touchPresence = nil

    /* intangibles aren't included in regular room/inventory/contents lists */
    isListed = nil
    isListedInInventory = nil
    isListedInContents = nil

    /* hide intangibles from 'all' for all actions by default */
    hideFromAll(action) { return true; }

    /* 
     *   our default message to use when trying to manipulate the object
     *   with a command that's meaningless with intangibles; this can be a
     *   library message property or simply a single-quoted string giving
     *   the actual text 
     */
    notWithIntangibleMsg = &notWithIntangible

    /*
     *   Essentially all verbs are meaningless on intangibles.  Each
     *   subclass should re-enable verbs that are meaningful for that
     *   specific type of intangible; to re-enable an action, just define
     *   a verify() handler for the action.
     *   
     *   Note that the verbs we handle via the Default handlers have no
     *   preconditions; since these verbs don't do anything anyway,
     *   there's no need to apply any preconditions to them.  
     */
    dobjFor(Default)
    {
        preCond = []
        verify() { illogical(notWithIntangibleMsg, self); }
    }
    iobjFor(Default)
    {
        preCond = []
        verify() { illogical(notWithIntangibleMsg, self); }
    }
;

/*
 *   A sensory emanation.  This is an intangible object that represents a
 *   sound, odor, or the like. 
 */
class SensoryEmanation: Intangible
    /* 
     *   The description shown when the *source* is examined (with "listen
     *   to", "smell", or whatever verb is appropriate to the type of
     *   sense the subclass involves). 
     */
    sourceDesc = ""

    /* our description, with and without being able to see the source */
    descWithSource = ""
    descWithoutSource = ""

    /* 
     *   Our "I am here" message, with and without being able to see the
     *   source.  These are displayed in room descriptions, inventory
     *   descriptions, and by the daemon that schedules background
     *   messages for sensory emanations.
     *   
     *   If different messages are desired as the emanation is mentioned
     *   repeatedly while the emanation remains continuously within sense
     *   range of the player character ("A phone is ringing", "The phone
     *   is still ringing", etc), these can check the displayCount
     *   property of self to determine which iteration of the message is
     *   being shown.  displayCount is set to 1 the first time a message
     *   is displayed for the object when the object can first be sensed.
     *   Note that displayCount resets to 1 when the object comes back
     *   into sense range after leaving.  
     */
    hereWithSource = ""
    hereWithoutSource = ""

    /* 
     *   A message to display when the emanation ceases to be within sense
     *   range.  In most cases, this displays nothing at all, but some
     *   emanations might want to note explicitly when the noise/etc
     *   stops.
     */
    noLongerHere = ""

    /*
     *   The schedule for displaying messages about the emanation.  This
     *   is a list of intervals between messages, in game clock times.
     *   When the player character can repeatedly sense this emanation for
     *   multiple consecutive turns, we'll use this schedule to display
     *   messages periodically about the noise/odor/etc.
     *   
     *   Human sensory perception tends to be "edge-sensitive," which
     *   means that we tend to perceive sensory input most acutely when
     *   something changes.  When a sound or odor is continually present
     *   without variation for an extended period, it tends to fade into
     *   the background of our awareness, so that even though it remains
     *   audible, we gradually stop noticing it.  This message display
     *   schedule mechanism is meant to approximate this perceptual model
     *   by allowing the sensory emanation to specify how noticeable the
     *   emanation remains during continuous exposure.  Typically, a
     *   continuous emanation would have relatively frequent messages
     *   (every two turns, say) for a couple of iterations, then would
     *   switch to infrequent messages.  Emanations that are analogous to
     *   white noise would probably not be mentioned at all after the
     *   first couple of messages, because the human senses are especially
     *   given to treating such input as background.
     *   
     *   We use this list by applying each interval in the list once and
     *   then moving to the next entry in the list.  The first entry in
     *   the list is the interval between first sensing the emanation and
     *   displaying the first "still here" message.  When we reach the end
     *   of the list, we simply repeat the last interval in the list
     *   indefinitely.  If the last entry in the list is nil, though, we
     *   simply never produce another message.  
     */
    displaySchedule = [nil]

    /*
     *   Show our "I am here" description.  This is the description shown
     *   as part of our room's description.  We show our hereWithSource or
     *   hereWithoutSource message, according to whether or not we can see
     *   the source object.  
     */
    emanationHereDesc()
    {
        local actor;
        
        /* note that we're mentioning the emanation */
        noteDisplay();

        /* 
         *   get the actor driving the description - if there's a command
         *   active, use the command's actor; otherwise use the player
         *   character
         */
        if ((actor = gActor) == nil)
            actor = gPlayerChar;

        /* our display varies according to our source's visibility */
        if (canSeeSource(actor))
            hereWithSource;
        else
            hereWithoutSource;
    }

    /*
     *   Show a message describing that we cannot see the source of this
     *   emanation because the given obstructor is in the way.  This
     *   should be overridden for each subclass. 
     */
    cannotSeeSource(obs) { }

    /* 
     *   Get the source of the noise/odor/whatever, as perceived by the
     *   current actor.  This is the object we appear to be coming from.
     *   By default, an emanation is generated by its direct container,
     *   and by default this is apparent to actors, so we'll simply return
     *   our direct container.
     *   
     *   If the source is not apparent, this should simply return nil.  
     */
    getSource() { return location; }

    /* determine if our source is apparent and visible */
    canSeeSource(actor)
    {
        local src;
        
        /* get our source */
        src = getSource();

        /* 
         *   return true if we have an apparent source, and the apparent
         *   source is visible to the current actor 
         */
        return src != nil && actor.canSee(src);
    }

    /*
     *   Note that we're displaying a message about the emanation.  This
     *   method should be called any time a message about the emanation is
     *   displayed, either by an explicit action or by our background
     *   daemon.
     *   
     *   We'll adjust our next display time so that we wait the full
     *   interval at the current point in the display schedule before we
     *   show any background message about this object.  Note we do not
     *   advance through the schedule list; instead, we merely delay any
     *   further message by the interval at the current point in the
     *   schedule list.  
     */
    noteDisplay()
    {
        /* calculate our next display time */
        calcNextDisplayTime();

        /* count the display */
        if (displayCount == nil)
            displayCount = 1;
        else
            ++displayCount;
    }

    /*
     *   Note an indirect message about the emanation.  This can be used
     *   when we don't actually display a message ourselves, but another
     *   object (usually our source object) describes the emanation; for
     *   example, if our source object mentions the noise it's making when
     *   it is examined, it should call this method to let us know we have
     *   been described indirectly.  This method advances our next display
     *   time, just as noteDisplay() does, but this method doesn't count
     *   the display as a direct display. 
     */
    noteIndirectDisplay()
    {
        /* calculate our next display time */
        calcNextDisplayTime();
    }
    
    /*
     *   Begin the emanation.  This is called from the sense change daemon
     *   when the item first becomes noticeable to the player character -
     *   for example, when the player character first enters the room
     *   containing the emanation, or when the emanation is first
     *   activated.  
     */
    startEmanation()
    {
        /* 
         *   if we've already initialized our scheduling, we must have
         *   been explicitly mentioned, such as by a room description - in
         *   this case, act as though we're continuing our emanation 
         */
        if (scheduleIndex != nil)
        {
            continueEmanation();
            return;
        }

        /* show our message */
        emanationHereDesc;
    }

    /*
     *   Continue the emanation.  This is called on each turn in which the
     *   emanation remains continuously within sense range of the player
     *   character.  
     */
    continueEmanation()
    {
        /* 
         *   if we are not to run again, our next display time will be set
         *   to zero - do nothing in this case 
         */
        if (nextDisplayTime == 0)
            return;

        /* if we haven't yet reached our next display time, do nothing */
        if (Schedulable.gameClockTime < nextDisplayTime)
            return;

        /* 
         *   Advance to the next schedule interval, if we have one.  If
         *   we're already on the last schedule entry, simply repeat it
         *   forever. 
         */
        if (scheduleIndex < displaySchedule.length())
            ++scheduleIndex;

        /* show our description */
        emanationHereDesc;
    }

    /*
     *   End the emanation.  This is called when the player character can
     *   no longer sense the emanation. 
     */
    endEmanation()
    {
        /* show our "no longer here" message */
        noLongerHere;

        /* uninitialize the display scheduling */
        scheduleIndex = nil;
        nextDisplayTime = nil;
    }

    /*
     *   Calculate our next display time.  The caller must set our
     *   scheduleIndex to the correct index prior to calling this.  
     */
    calcNextDisplayTime()
    {
        local delta;

        /* if our scheduling isn't initialized, set it up now */
        if (scheduleIndex == nil)
        {
            /* start at the first display schedule interval */
            scheduleIndex = 1;
        }
        
        /* get the next display interval from the schedule list */
        delta = displaySchedule[scheduleIndex];

        /* 
         *   if the current display interval is nil, it means that we're
         *   never to display another message 
         */
        if (delta == nil)
        {
            /* 
             *   we're not to display again - simply set the next display
             *   time to zero and return 
             */
            nextDisplayTime = 0;
            return;
        }

        /* 
         *   our next display time is the current game clock time plus the
         *   interval 
         */
        nextDisplayTime = Schedulable.gameClockTime + delta;
    }

    /*
     *   Internal counters that keep track of our display scheduling.
     *   scheduleIndex is the index in the displaySchedule list of the
     *   interval we're waiting to expire; nextDisplayTime is the game
     *   clock time of our next display.  noiseList and odorList are lists
     *   of senseInfo entries for the sound and smell senses,
     *   respectively, indicating which objects were within sense range on
     *   the last turn.  displayCount is the number of times in a row
     *   we've displayed a message already.  
     */
    scheduleIndex = nil
    nextDisplayTime = nil
    noiseList = nil
    odorList = nil
    displayCount = nil

    /*
     *   Class method implementing the sensory change daemon.  This runs
     *   on each turn to check for changes in the set of objects the
     *   player can hear and smell, and to generate "still here" messages
     *   for objects continuously within sense range for multiple turns.  
     */
    noteSenseChanges()
    {
#ifdef SENSE_CACHE
        /* emanations don't change anything, so turn on caching */
        libGlobal.enableSenseCache();
#endif

        /* note sound changes */
        noteSenseChangesFor(sound, &noiseList, Noise);

        /* note odor changes */
        noteSenseChangesFor(smell, &odorList, Odor);

#ifdef SENSE_CACHE
        /* done with sense caching */
        libGlobal.disableSenseCache();
#endif
    }

    /*
     *   Note sense changes for a particular sense.  'listProp' is the
     *   property of SensoryEmanation giving the list of SenseInfo entries
     *   for the sense on the previous turn.  'typ' is a subclass of ours
     *   (such as Noise) giving the type of sensory emanation used for
     *   this sense. 
     */
    noteSenseChangesFor(sense, listProp, sub)
    {
        local newInfo;
        local oldInfo;

        /* get the old table of SenseInfo entries for the sense */
        oldInfo = self.(listProp);

        /* 
         *   Get the new table of items we can reach in the given sense,
         *   and reduce it to include only emanations of the subclass of
         *   interest.  
         */
        newInfo = gPlayerChar.senseInfoTable(sense);
        newInfo.forEachAssoc(new function(obj, info)
        {
            /* remove this item if it's not of the subclass of interest */
            if (!obj.ofKind(sub))
                newInfo.removeElement(obj);
        });

        /* run through the new list and note each change */
        newInfo.forEachAssoc(new function(obj, info)
        {
            /* treat this as a new command visually */
            "<.commandsep>";
        
            /* 
             *   Check to see whether the item is starting anew or was
             *   already here on the last turn.  If the item was in our
             *   list from the previous turn, it was already here.  
             */
            if (oldInfo == nil || oldInfo[obj] == nil)
            {
                /* 
                 *   the item wasn't in sense range on the last turn, so
                 *   it is becoming newly noticeable 
                 */
                obj.startEmanation();
            }
            else
            {
                /* the item was already here - continue its emanation */
                obj.continueEmanation();
            }
        });

        /* run through the old list and note each item no longer sensed */
        if (oldInfo != nil)
        {
            oldInfo.forEachAssoc(new function(obj, info)
            {
                /* if this item isn't in the new list, note its departure */
                if (newInfo[obj] == nil)
                {
                    /* treat this as a new command visually */
                    "<.commandsep>";
                    
                    /* note the departure */
                    obj.endEmanation();
                }
            });
        }

        /* store the current list for comparison the next time we run */
        self.(listProp) = newInfo;
    }

    /* 
     *   Examine the sensory emanation.  We'll show our descWithSource or
     *   descWithoutSource, according to whether or not we can see the
     *   source object. 
     */
    dobjFor(Examine)
    {
        verify() { inherited(); }
        action()
        {
            /* note that we're displaying a message about us */
            noteDisplay();
            
            /* display our sound description */
            if (canSeeSource(gActor))
            {
                /* we can see the source */
                descWithSource;
            }            
            else
            {
                local src;
            
                /* show the unseen-source version of the description */
                descWithoutSource;

                /* 
                 *   If we have a source, find out what's keeping us from
                 *   seeing the source; in other words, find the opaque
                 *   visual obstructor on the sense path to the source.  
                 */
                if ((src = getSource()) != nil)
                {
                    local obs;
                    
                    /* get the visual obstructor */
                    obs = gActor.findVisualObstructor(src);
                    
                    /* 
                     *   If we found an obstructor, and we can see it, add
                     *   a message describing the obstruction.  If we
                     *   can't see the obstructor, we can't localize the
                     *   sensory emanation at all.  
                     */
                    if (obs != nil && gActor.canSee(obs))
                        cannotSeeSource(obs);
                }
            }
        }
    }
;

/*
 *   Noise - this is an intangible object representing a sound.
 *   
 *   A Noise object is generally placed directly within the object that is
 *   generating the noise.  This will ensure that the noise is
 *   automatically in scope whenever the object is in scope (or, more
 *   precisely, whenever the object's contents are in scope) and with the
 *   same sense attributes.
 *   
 *   By default, when a noise is specifically examined via "listen to",
 *   and the container is visible, we'll mention that the noise is coming
 *   from the container.
 */
class Noise: SensoryEmanation
    /* by default, a noise has a definite presence in the sound sense */
    soundPresence = true

    /* 
     *   By default, a noise is listed in a room description (i.e., on
     *   LOOK or entry to a room).  Set this to nil to omit the noise from
     *   the room description, while still allowing it to be heard in an
     *   explicit LISTEN command.  
     */
    isSoundListedInRoom = true

    /* show our description as part of a room description */
    soundHereDesc() { emanationHereDesc(); }

    /* explain that we can't see the source because of the obstructor */
    cannotSeeSource(obs) { obs.cannotSeeSoundSource(self); }

    /* include in LISTEN ALL (intangibles are hidden from ALL by default) */
    hideFromAll(action)
    {
        /* if the command is LISTEN TO, include me in ALL */
        if (action.ofKind(ListenToAction))
            return nil;

        /* use inherited behavior */
        return inherited(action);
    }

    /* treat "listen to" the same as "examine" */
    dobjFor(ListenTo) asDobjFor(Examine)

    /* "examine" requires that the object is audible */
    dobjFor(Examine)
    {
        preCond = [objAudible]
    }
;

/*
 *   Odor - this is an intangible object representing an odor. 
 */
class Odor: SensoryEmanation
    /* by default, an odor has a definite presence in the smell sense */
    smellPresence = true

    /* 
     *   By default, an odor is listed in a room description (i.e., on
     *   LOOK or entry to a room).  Set this to nil to omit the odor from
     *   the room description, while still allowing it to be listed in an
     *   explicit SMELL command.  
     */
    isSmellListedInRoom = true

    /* mention the odor as part of a room description */
    smellHereDesc() { emanationHereDesc(); }

    /* explain that we can't see the source because of the obstructor */
    cannotSeeSource(obs) { obs.cannotSeeSmellSource(self); }

    /* include in SMELL ALL (intangibles are hidden from ALL by default) */
    hideFromAll(action)
    {
        /* if the command is SMELL, include me in ALL */
        if (action.ofKind(SmellAction))
            return nil;

        /* use inherited behavior */
        return inherited(action);
    }

    /* handle "smell" using our "examine" handler */
    dobjFor(Smell) asDobjFor(Examine)

    /* "examine" requires that the object is smellable */
    dobjFor(Examine)
    {
        preCond = [objSmellable]
    }
;


/* ------------------------------------------------------------------------ */
/*
 *   Sensory Event.  This is an object representing a transient event,
 *   such as a sound, visual display, or odor, to which some objects
 *   observing the event might react.
 *   
 *   A sensory event differs from a sensory emanation in that an emanation
 *   is ongoing and passive, while an event is isolated in time and
 *   actively notifies observers.  
 */
class SensoryEvent: object
    /* 
     *   Trigger the event.  This routine must be called at the time when
     *   the event is to occur.  We'll notify every interested observer
     *   capable of sensing the event that the event is occurring, so
     *   observers can take appropriate action in response to the event.
     *   
     *   'source' is the source object - this is the physical object in
     *   the simulation that is causing the event.  For example, if the
     *   event is the sound of a phone ringing, the phone would probably
     *   be the source object.  The source is used to determine which
     *   observers are capable of detecting the event: an observer must be
     *   able to sense the source object in the appropriate sense to be
     *   notified of the event.  
     */
    triggerEvent(source)
    {
        /* 
         *   Run through all objects connected to the source object by
         *   containment, and notify any that are interested and can
         *   detect the event.  Containment is the only way sense
         *   information can propagate, so we can limit our search
         *   accordingly.
         *   
         *   Connection by containment is no guarantee of a sense
         *   connection: it's a necessary, but not sufficient, condition.
         *   Because it's a necessary condition, though, we can use it to
         *   limit the number of objects we have to test with a more
         *   expensive sense path calculation.  
         */
        source.connectionTable().forEachAssoc(new function(cur, val)
        {
            /* 
             *   If this object defines the observer notification method,
             *   then it might be interested in the event.  If the object
             *   doesn't define this method, then there's no way it could
             *   be interested.  (We make this test before checking the
             *   sense path because checking to see if an object defines a
             *   property is fast and simple, while the sense path
             *   calculation could be expensive.) 
             */
            if (cur.propDefined(notifyProp, PropDefAny))
            {
                local info;
                
                /* 
                 *   This object might be interested in the event, so
                 *   check to see if the object can sense the event.  If
                 *   this object can sense the source object at all (i.e.,
                 *   the sense path isn't 'opaque'), then notify the
                 *   object of the event.  
                 */
                info = cur.senseObj(sense, source);
                if (info.trans != opaque)
                {
                    /* 
                     *   this observer object can sense the source of the
                     *   event, so notify it of the event 
                     */
                    cur.(notifyProp)(self, source, info);
                }
            }
        });
    }

    /* the sense in which the event is observable */
    sense = nil

    /* 
     *   the notification property - this is the property we'll invoke on
     *   each observer to notify it of the event 
     */
    notifyProp = nil
;

/*
 *   Visual event 
 */
class SightEvent: SensoryEvent
    sense = sight
    notifyProp = &notifySightEvent
;

/* 
 *   Visual event observer.  This is a mix-in that can be added to any
 *   other classes.  
 */
class SightObserver: object
    /*
     *   Receive notification of a sight event.  This routine is called
     *   whenever a SightEvent occurs within view of this object.
     *   
     *   'event' is the SightEvent object; 'source' is the physical
     *   simulation object that is making the visual display; and 'info'
     *   is a SenseInfo object describing the viewing conditions from this
     *   object to the source object.  
     */
    notifySightEvent(event, source, info) { }
;

/*
 *   Sound event 
 */
class SoundEvent: SensoryEvent
    sense = sound
    notifyProp = &notifySoundEvent
;

/* 
 *   Sound event observer.  This is a mix-in that can be added to any
 *   other classes.  
 */
class SoundObserver: object
    /*
     *   Receive notification of a sound event.  This routine is called
     *   whenever a SoundEvent occurs within hearing range of this object.
     */
    notifySoundEvent(event, source, info) { }
;

/*
 *   Smell event 
 */
class SmellEvent: SensoryEvent
    sense = smell
    notifyProp = &notifySmellEvent
;

/* 
 *   Smell event observer.  This is a mix-in that can be added to any
 *   other classes.  
 */
class SmellObserver: object
    /*
     *   Receive notification of a smell event.  This routine is called
     *   whenever a SmellEvent occurs within smelling range of this
     *   object.  
     */
    notifySmellEvent(event, source, info) { }
;


/* ------------------------------------------------------------------------ */
/*
 *   A readable object.  Any ordinary object will show its normal full
 *   description when read, but an object that is explicitly readable will
 *   have elevated logicalness for the "read" action, and can optionally
 *   show a separate description when read. 
 */
class Readable: Thing
    /* 
     *   Show my special reading desription.  By default, we set this to
     *   nil to indicate that we should use our default "examine"
     *   description; objects can override this to show a special message
     *   for reading the object as desired.  
     */
    readDesc = nil

    /* our reading description when obscured */
    obscuredReadDesc() { mainReport(&obscuredReadDesc, self); }

    /* our reading description in dim light */
    dimReadDesc() { mainReport(&dimReadDesc, self); }

    /* "Read" action */
    dobjFor(Read)
    {
        verify() { }
        action()
        {
            /* 
             *   if we have a special reading description defined, show
             *   it; otherwise, use the same handling as "examine"
             */
            if (propType(&readDesc) != TypeNil)
            {
                local info;
                
                /* 
                 *   Reading requires a transparent sight path and plenty
                 *   of light; in the absence of either of these, we can't
                 *   make out the details. 
                 */
                info = gActor.bestVisualInfo(self);
                if (info.trans != transparent)
                    obscuredReadDesc;
                else if (info.ambient < 3)
                    dimReadDesc;
                else
                    readDesc;
            }
            else
            {
                /* 
                 *   we have no special reading description, so use the
                 *   default "examine" handling 
                 */
                actionDobjExamine();
            }
        }
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   A fixed-in-place object.  These objects cannot be removed from their
 *   containers.  This class is meant primarily for permanent features of
 *   rooms that obviously cannot be moved to a new container, such as
 *   walls, floors, doors, and heavy furniture.  
 */
class Fixed: Thing
    /* 
     *   fixed objects are not listed in room or container contents
     *   listings - this type of item is a fixture of its location, so it
     *   should be described in its container's description 
     */
    isListed = nil
    isListedInContents = nil
    isListedInInventory = nil

    /* 
     *   fixed items aren't normally listed in a room's description, but
     *   their listable contents usually are 
     */
    contentsListed = true

    /* 
     *   hide fixed items from "all" for certain commands, since it's
     *   fairly obvious for many commands that a fixed item shouldn't be
     *   included 
     */
    hideFromAll(action)
    {
        return (action.ofKind(TakeAction)
                || action.ofKind(DropAction)
                || action.ofKind(PutInAction)
                || action.ofKind(PutOnAction));
    }

    /*
     *   Are my contents within a fixed item that is within the given
     *   location?  Since we're fixed in place, our contents are certainly
     *   within a fixed item, so we merely need to check if we're fixed in
     *   place within the given location.  We are if we're in the given
     *   location or we ourselves are fixed in place in the given
     *   location.  
     */
    contentsInFixedIn(loc)
    {
        return isDirectlyIn(loc) || isInFixedIn(loc);
    }

    /* a fixed item can't be moved by an actor action */
    verifyMoveTo(newLoc)
    {
        /* it's never possible to do this */
        illogical(cannotMoveMsg);
    }

    /* 
     *   a fixed item can't be taken - this would be caught by
     *   verifyMoveTo anyway, but provide a more explicit message when a
     *   fixed item is explicitly taken 
     */
    dobjFor(Take) { verify() { illogical(cannotTakeMsg); }}
    dobjFor(TakeFrom) { verify() { illogical(cannotTakeMsg); }}

    /* fixed objects can't be put anywhere */
    dobjFor(PutIn) { verify() { illogical(cannotPutMsg); }}
    dobjFor(PutOn) { verify() { illogical(cannotPutMsg); }}
    dobjFor(PutUnder) { verify() { illogical(cannotPutMsg); }}

    /* fixed objects can't be pushed, pulled, or moved */
    dobjFor(Push) { verify() { illogical(cannotMoveMsg); }}
    dobjFor(Pull) { verify() { illogical(cannotMoveMsg); }}
    dobjFor(Move) { verify() { illogical(cannotMoveMsg); }}
    dobjFor(MoveWith) { verify() { illogical(cannotMoveMsg); }}
    dobjFor(MoveTo) { verify() { illogical(cannotMoveMsg); }}
    dobjFor(PushTravel) { verify() { illogical(cannotMoveMsg); }}

    /*
     *   The messages to use for illogical messages.  These can be
     *   overridden with new properties (of playerActionMessages and the
     *   like), or simply with single-quoted strings to display.  
     */
    cannotTakeMsg = &cannotTakeFixed
    cannotMoveMsg = &cannotMoveFixed
    cannotPutMsg = &cannotPutFixed
;

/*
 *   A component object.  These objects cannot be removed from their
 *   containers because they are permanent features of other objects,
 *   which may themselves be portable: the hands of a watch, a tuning dial
 *   on a radio.  This class behaves essentially the same way as Fixed,
 *   but its messages are more suitable for objects that are component
 *   parts of other objects rather than fixed features of rooms.  
 */
class Component: Fixed
    /* a component cannot be removed from its container by an actor action */
    verifyMoveTo(newLoc)
    {
        /* it's never possible to do this */
        illogical(&cannotMoveComponent, location);
    }

    /* a component cannot be taken separately */
    dobjFor(Take) { verify() { illogical(&cannotTakeComponent, location); }}
    dobjFor(TakeFrom)
        { verify() { illogical(&cannotTakeComponent, location); }}

    /* a component cannot be separately put somewhere */
    dobjFor(PutIn) { verify() { illogical(&cannotPutComponent, location); }}
    dobjFor(PutOn) { verify() { illogical(&cannotPutComponent, location); }}
    dobjFor(PutUnder)
        { verify() { illogical(&cannotPutComponent, location); }}
;

/* ------------------------------------------------------------------------ */
/*
 *   Decoration.  This is an object that is included for scenery value but
 *   which has no other purpose, and which the author wants to make clear
 *   is not important.  We use the catch-all action routine to respond to
 *   any command on this object with a flat "that's not important"
 *   message, so that the player can plainly see that there's no point
 *   wasting any time trying to manipulate this object.
 *   
 *   We use the "default" catch-all verb verify handling to report our
 *   "that's not important" message, so a decoration can be made
 *   responsive to specific verbs simply by defining an action handler for
 *   those verbs.  
 */
class Decoration: Fixed
    /* don't include decorations in 'all' */
    hideFromAll(action) { return true; }

    /* 
     *   The message I show for most actions.  This can be either a
     *   property pointer, which is taken as a player (or NPC) action
     *   messages property that takes 'self' as its parameter; or it can
     *   return a simple single-quoted string giving the actual text of
     *   the message.  By default, we use a library message property for a
     *   message that says "That's not important."  
     */
    notImportantMsg = &notImportant

    dobjFor(Default)
    {
        verify() { illogical(notImportantMsg, self); }
    }
    iobjFor(Default)
    {
        verify() { illogical(notImportantMsg, self); }
    }

    /*
     *   The catch-all Default verifier makes all actions illogical, but
     *   we can override this to allow specific actions by explicitly
     *   defining them here so that they hide the Default verify handlers.
     */
    dobjFor(Examine)
    {
        verify() { inherited(); }
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Distant item.  This is an object that's too far away to manipulate,
 *   but can be seen.  This is useful for scenery objects that are at a
 *   great distance within a large location.
 *   
 *   A Distant item is essentially just like a decoration, but the default
 *   message is different.  
 */
class Distant: Fixed
    /* don't include in 'all' */
    hideFromAll(action) { return true; }

    /* 
     *   my "that's too distant" message - this can be a property (of a
     *   library message object for actor actions), or simply a
     *   single-quoted string giving the actual text of the message 
     */
    tooDistantMsg = &tooDistant

    dobjFor(Default)
    {
        verify() { illogical(tooDistantMsg, self); }
    }
    iobjFor(Default)
    {
        verify() { illogical(tooDistantMsg, self); }
    }

    /* allow examining a Distant item */
    dobjFor(Examine)
    {
        verify() { inherited(); }
    }
;

/*
 *   Out Of Reach - this is a special mix-in that can be used to create an
 *   object that places its *contents* out of reach under customizable
 *   conditions, and can optionally place itself out of reach as well.
 */
class OutOfReach: object
    checkTouchViaPath(obj, dest, op)
    {
        /* check how we're traversing the object */
        if (op == PathTo)
        {
            /* 
             *   we're reaching from outside for this object itself -
             *   check to see if the source can reach me
             */
            if (!canObjReachSelf(obj))
                return new CheckStatusFailure(
                    cannotReachFromOutsideMsg(dest), dest);
        }
        else if (op == PathIn)
        {
            /* 
             *   we're reaching in to touch one of my contents - check to
             *   see if the source object is within reach of my contents 
             */
            if (!canObjReachContents(obj))
                return new CheckStatusFailure(
                    cannotReachFromOutsideMsg(dest), dest);
        }
        else if (op == PathOut)
        {
            local ok;
            
            /*
             *   We're reaching out.  If we're reaching for the object
             *   itself, check to see if we're reachable from within;
             *   otherwise, check to see if we can reach objects outside
             *   us from within.  
             */
            if (dest == self)
                ok = canReachSelfFromInside(obj);
            else
                ok = canReachFromInside(obj, dest);

            /* if we can't reach the object, say so */
            if (!ok)
                return new CheckStatusFailure(
                    cannotReachFromInsideMsg(dest), dest);
        }
        
        /* if we didn't find a problem, allow the operation */
        return checkStatusSuccess;
    }

    /* 
     *   The message to use to indicate that we can't reach an object,
     *   because the actor is outside me and the target is inside, or vice
     *   versa.  Each of these can return a property ID giving an actor
     *   action message property, or can simply return a string with the
     *   message text.  
     */
    cannotReachFromOutsideMsg(dest) { return &tooDistant; }
    cannotReachFromInsideMsg(dest) { return  &tooDistant; }

    /*
     *   Determine if the given object can reach my contents.  'obj' is
     *   the object (usually an actor) attempting to reach my contents
     *   from outside of me.
     *   
     *   By default, we'll return nil, so that nothing within me can be
     *   reached from anyone outside.  This can be overridden to allow my
     *   contents to become reachable from some external locations but not
     *   others; for example, a high shelf could allow an actor standing
     *   on a chair to reach my contents. 
     */
    canObjReachContents(obj) { return nil; }

    /*
     *   Determine if the given object can reach me.  'obj' is the object
     *   (usually an actor) attempting to reach this object.
     *   
     *   By default, make this object subject to the same rules as its
     *   contents.  
     */
    canObjReachSelf(obj) { return canObjReachContents(obj); }

    /*
     *   Determine if the given object outside of me is reachable from
     *   within me.  'obj' (usually an actor) is attempting to reach
     *   'dest'.
     *   
     *   By default, we return nil, so nothing outside of me is reachable
     *   from within me.  This can be overridden as needed.  This should
     *   usually behave symmetrically with canObjReachContents().  
     */
    canReachFromInside(obj, dest) { return nil; }

    /* 
     *   Determine if we can reach this object itself from within.  This
     *   is used when 'obj' tries to touch this object when 'obj' is
     *   located within this object.
     *   
     *   By default, we we use the same rules as we use to reach an
     *   external object from within.  
     */
    canReachSelfFromInside(obj) { return canReachFromInside(obj, self); }

    /*
     *   We cannot implicitly remove this obstruction, so simply return
     *   nil when asked.  
     */
    tryImplicitRemoveObstructor(sense, obj) { return nil; }
;

/* ------------------------------------------------------------------------ */
/*
 *   A Fill Medium - this is the class of object returned from
 *   Thing.fillMedium().  
 */
class FillMedium: Thing
    /*
     *   Get the transparency sensing through this medium. 
     */
    senseThru(sense)
    {
        /* 
         *   if I have a meterial, use its transparency; otherwise, we're
         *   transparent 
         */
        return (material != nil ? material.senseThru(sense) : transparent);
    }

    /* my material */
    material = nil
;

/* ------------------------------------------------------------------------ */
/*
 *   MultiLoc: this class can be multiply inherited by any object that
 *   must exist in more than one place at a time.  To use this class, put
 *   it BEFORE Thing (or any subclass of Thing) in the object's superclass
 *   list, to ensure that we override the default containment
 *   implementation for the object.
 */
class MultiLoc: object
    /*
     *   We can be in any number of locations.  Our location must be given
     *   as a list.
     */
    locationList = []

    /*
     *   Initialize my location's contents list - add myself to my
     *   container during initialization
     */
    initializeLocation()
    {
        /*
         *   Add myself to each of my container's contents lists
         */
        locationList.forEach({loc: loc.addToContents(self)});
    }

    /*
     *   Determine if I'm in a given object, directly or indirectly
     */
    isIn(obj)
    {
        /* first, check to see if I'm directly in the given object */
        if (isDirectlyIn(obj))
            return true;

        /*
         *   Look at each object in my location list.  For each location
         *   object, if the location is within the object, I'm within the
         *   object.
         */
        return locationList.indexWhich({loc: loc.isIn(obj)}) != nil;
    }

    /*
     *   Determine if I'm directly in the given object
     */
    isDirectlyIn(obj)
    {
        /*
         *   we're directly in the given object only if the object is in
         *   my list of immediate locations
         */
        return (locationList.indexOf(obj) != nil);
    }

    /*
     *   Note that we don't need to override any of the contents
     *   management methods, since we provide special handling for our
     *   location relationships, not for our contents relationships.
     */

    /* save my location for later restoration */
    saveLocation()
    {
        /* return my list of locations */
        return locationList;
    }

    /* restore a previously saved location */
    restoreLocation(oldLoc)
    {
        /* remove myself from each current location not in the saved list */
        foreach (local cur in locationList)
        {
            /* 
             *   if this present location isn't in the saved list, remove
             *   myself from the location 
             */
            if (oldLoc.indexOf(cur) == nil)
                cur.removeFromContents(self);
        }

        /* add myself to each saved location not in the current list */
        foreach (local cur in oldLoc)
        {
            /* if I'm not already in this location, add me to it */
            if (locationList.indexOf(cur) == nil)
                cur.addToContents(self);
        }

        /* set my own list to the original list */
        locationList = oldLoc;
    }

    /*
     *   Basic routine to move this object into a given single container.
     *   Removes the object from all of its other containers.  Performs no
     *   notifications.  
     */
    baseMoveInto(newContainer)
    {
        /* remove myself from all of my current contents */
        locationList.forEach({loc: loc.removeFromContents(self)});

        /* set my location list to include only the new location */
        locationList = [newContainer];

        /* add myself to my new container's contents */
        newContainer.addToContents(self);
    }

    /*
     *   Add this object to a new location - base version that performs no
     *   notifications.  
     */
    baseMoveIntoAdd(newContainer)
    {
        /* add the new container to my list of locations */
        locationList += newContainer;

        /* add myself to my new container's contents */
        newContainer.addToContents(self);
    }

    /*
     *   Add this object to a new location.
     */
    moveIntoAdd(newContainer)
    {
        /* notify my new container that I'm about to be added */
        if (newContainer != nil)
            newContainer.sendNotifyInsert(self, newContainer);

        /* perform base move-into-add operation */
        baseMoveIntoAdd(newContainer);
        
        /* note that I've been moved */
        moved = true;
    }

    /*
     *   Base routine to move myself out of a given container.  Performs
     *   no notifications. 
     */
    baseMoveOutOf(cont)
    {
        /* remove myself from this container's contents list */
        cont.removeFromContents(self);

        /* remove this container from my location list */
        locationList -= cont;
    }

    /*
     *   Remove myself from a given container, leaving myself in any other
     *   containers.
     */
    moveOutOf(cont)
    {
        /* if I'm not actually directly in this container, do nothing */
        if (!isDirectlyIn(cont))
            return;

        /* 
         *   notify this container (and only this container) that we're
         *   being removed from it 
         */
        cont.sendNotifyRemove(obj, nil);

        /* perform base operation */
        baseMoveOutOf(cont);

        /* note that I've been moved */
        moved = true;
    }

    /*
     *   Call a function on each container.  We'll invoke the function as
     *   follows for each container 'cont':
     *   
     *   (func)(cont, args...)  
     */
    forEachContainer(func, [args])
    {
        /* call the function for each location in our list */
        foreach(local cur in locationList)
            (func)(cur, args...);
    }

    /* 
     *   Call a function on each connected container.  By default, we
     *   don't connect our containers for sense purposes, so we do nothing
     *   here. 
     */
    forEachConnectedContainer(func, ...)
    {
        /* we do not connect our containers, so do nothing */
    }

    /*
     *   Add the direct containment connections for this item to a lookup
     *   table. 
     *   
     *   By default, a MultiLoc object does not add any of its containers
     *   to the list, because we don't provide any sense connection among
     *   our containers.  The SenseConnector class provides that behavior.
     */
    addDirectConnections(tab)
    {
        /* if I'm not already in the table, add me */
        if (tab[self] == nil)
        {
            /* add myself */
            tab[self] = true;

            /* add my contents */
            foreach (local cur in contents)
                cur.addDirectConnections(tab);

            /* note that we do not by default add connections to containers */
        }
    }

    /*
     *   Transmit ambient energy to my location or locations. 
     */
    shineOnLoc(sense, ambient, fill)
    {
        /* 
         *   by default, we don't transmit sense information to our
         *   containers, so we do nothing here
         */
    }

    /*
     *   Build a sense path to my location or locations. 
     */
    sensePathToLoc(sense, trans, obs, fill)
    {
        /* 
         *   by default, we don't transmit sense information to our
         *   containers, so we do nothing here 
         */
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Multi-Location item with automatic initialization.  This is a
 *   subclass of MultiLoc for objects with computed initial locations.
 *   Each instance must provide one of the following:
 *   
 *   - Override initializeLocation() with a method that initializes the
 *   location list by calling moveIntoAdd() for each location.  If this
 *   method isn't overridden, the default implementation will initialize
 *   the location list using initialLocationClass and/or isInitiallyIn(),
 *   as described below.
 *   
 *   - Define the method isInitiallyIn(loc) to return true if the 'loc' is
 *   one of the object's initial containers, nil if not.  The default
 *   implementation of this method simply returns true, so if this isn't
 *   overridden, every object matching the initialLocationClass() will be
 *   part of the contents list.
 *   
 *   - Define the property initialLocationClass as a class object.  We
 *   will add each instance of the class that passes the isInitiallyIn()
 *   test to the location list.  If this is nil, we'll test every object
 *   instance with the isInitiallyIn() method.  
 */
class AutoMultiLoc: MultiLoc
    /* initialize the location */
    initializeLocation()
    {
        /* get the list of locations */
        locationList = buildLocationList();

        /* add ourselves into each of our containers */
        foreach (local loc in locationList)
            loc.addToContents(self);
    }

    /*
     *   build my list of locations, and return the list 
     */
    buildLocationList()
    {
        local lst;

        /* we have nothing in our list yet */
        lst = new Vector(16);

        /*
         *   if initialLocationClass is defined, loop over all objects of
         *   that class; otherwise, loop over all objects
         */
        if (initialLocationClass != nil)
        {
            /* loop over all instances of the given class */
            for (local obj = firstObj(initialLocationClass) ; obj != nil ;
                 obj = nextObj(obj, initialLocationClass))
            {
                /* if the object passes the test, include it */
                if (isInitiallyIn(obj))
                    lst.append(obj);
            }
        }
        else
        {
            /* loop over all objects */
            for (local obj = firstObj() ; obj != nil ; obj = nextObj(obj))
            {
                /* if the object passes the test, include it */
                if (isInitiallyIn(obj))
                    lst.append(obj);
            }
        }

        /* return the list of locations */
        return lst.toList();
    }

    /*
     *   Class of our locations.  If this is nil, we'll test every object
     *   in the entire game with our isInitiallyIn() method; otherwise,
     *   we'll test only objects of the given class.
     */
    initialLocationClass = nil

    /*
     *   Test an object for inclusion in our initial location list.  By
     *   default, we'll simply return true to include every object.  We
     *   return true by default so that an instance can merely specify a
     *   value for initialLocationClass in order to place this object in
     *   every instance of the given class.
     */
    isInitiallyIn(obj) { return true; }
;

/*
 *   Dynamic Multi Location Item.  This is almost exactly the same as a
 *   regular multi-location item with automatic initialization, but the
 *   library will re-initialize the location of these items, by calling
 *   the object's reInitializeLocation(), at the start of every turn.  
 */
class DynamicMultiLoc: AutoMultiLoc
    reInitializeLocation()
    {
        local newList;

        /* build the new location list */
        newList = buildLocationList();

        /*
         *   Update any containers that are not in the intersection of the
         *   two lists.  Note that we don't simply move ourselves out of
         *   the old list and into the new list, because the two lists
         *   could have common members; to avoid unnecessary work that
         *   might result from removing ourselves from a container and
         *   then adding ourselves right back in to the same container, we
         *   only notify containers when we're actually moving out or
         *   moving in. 
         */

        /* 
         *   For each item in the old list, if it's not in the new list,
         *   notify the old container that we're being removed.
         */
        foreach (local loc in locationList)
        {
            /* if it's not in the new list, remove me from the container */
            if (newList.indexOf(loc) == nil)
                loc.removeFromContents(self);
        }

        /* 
         *   for each item in the new list, if we weren't already in this
         *   location, add ourselves to the location 
         */
        foreach (local loc in newList)
        {
            /* if it's not in the old list, add me to the new container */
            if (!isDirectlyIn(loc) == nil)
                loc.addToContents(self);
        }
        
        /* make the new location list current */
        locationList = newList;
    }
;


/* ------------------------------------------------------------------------ */
/*
 *   Openable: a mix-in class that can be combined with an object's other
 *   superclasses to make the object respond to the verbs "open" and
 *   "close."  
 */
class Openable: object
    /*
     *   Get the master object, which holds our state.  By default, this
     *   is simply 'self', but some objects might want to override this.
     *   For example, doors are usually implemented with two separate
     *   objects, representing the two sides of the door, which share
     *   common state; in such cases, one of the pair can be designated as
     *   the master, which holds the common state of the door, and this
     *   method can be overridden so that all state operations on the lock
     *   are performed on the master side of the door.  
     */
    masterObject()
    {
        /* 
         *   inherit from the next superclass, if possible; otherwise, use
         *   'self' as the default master object 
         */
        if (canInherit())
            return inherited();
        else
            return self;
    }

    /*
     *   Flag: door is open.  Travel is only possible when the door is
     *   open.  Return the master's status; if we're the master, return
     *   our initial status.  
     */
    isOpen()
    {
        if (masterObject == self)
            return initiallyOpen;
        else
            return masterObject.isOpen;
    }

    /*
     *   Make the object open or closed.  By default, we'll simply set the
     *   isOpen flag to the new status.  Objects can override this to
     *   apply side effects of opening or closing the object.  
     */
    makeOpen(stat)
    {
        /* 
         *   if the next superclass in our defining class defines its own
         *   makeOpen, invoke it rather than this default handling 
         */
        if (canInherit())
        {
            /* 
             *   ignore this default definition, and use the definition
             *   obtained from the next superclass instead 
             */
            inherited(stat);
        }
        else
        {
            /* note the new open status */
            masterObject.isOpen = stat;
        }
    }

    /*
     *   Open status name.  This is an adjective describing whether the
     *   object is opened or closed.  In English, this will return "open"
     *   or "closed."  
     */
    openDesc = (masterObject.isOpen
                ? libMessages.openMsg : libMessages.closedMsg)

    /*
     *   Describe our contents using a special version of the contents
     *   lister, so that we add our open/closed status to the listing.  
     */
    descContentsLister = openableContentsLister


    /* -------------------------------------------------------------------- */
    /*
     *   Action handlers 
     */
    dobjFor(Open)
    {
        verify()
        {
            /* it makes no sense to open something that's already open */
            if (masterObject.isOpen)
                illogicalNow(&alreadyOpen);
        }
        action()
        {
            local trans;
            
            /* 
             *   note the effect we have currently, while still closed, on
             *   sensing from outside into our contents 
             */
            trans = transSensingIn(sight);
            
            /* make it open */
            makeOpen(true);

            /* 
             *   make the default report - if we make a non-default
             *   report, the default will be ignored, so we don't need to
             *   worry about whether or not we'll make a non-default
             *   report now 
             */
            defaultReport(&okayOpen);

            /*
             *   If the actor is outside me, and we have any listable
             *   contents, and our sight transparency is now better than
             *   it was before we were open, reveal the new contents.
             *   Otherwise, just show our default 'opened' message.
             *   
             *   As a special case, if we're running as an implied command
             *   within a LookIn action on this same object, don't bother
             *   showing this result, because it would be redundant with
             *   the explicit examination of the contents that we'll get
             *   anyway with the LookIn action.  
             */
            if (!gActor.isIn(self)
                && transparencyCompare(transSensingIn(sight), trans) > 0
                && !(gAction.isImplicit
                     && gAction.parentAction.ofKind(LookInAction)
                     && gAction.parentAction.getDobj() == self))
            {
                /* show my contents list, if I have any */
                openingLister.showList(gActor, self, contents, ListRecurse,
                                       0, gActor.visibleInfoTable(), nil);
            }
        }
    }
    dobjFor(Close)
    {
        verify()
        {
            /* it makes no sense to close something that's already closed */
            if (!masterObject.isOpen)
                illogicalNow(&alreadyClosed);
        }
        action()
        {
            /* make it closed */
            makeOpen(nil);

            /* show the default report */
            defaultReport(&okayClose);
        }
    }

    dobjFor(LookIn)
    {
        /* 
         *   to look in an openable object, we must be open, unless the
         *   object is transparent 
         */
        preCond
        {
            local lst;

            /* get the inherited preconditions */
            lst = nilToList(inherited());

            /* if I'm not transparent looking in, we must be open */
            if (transSensingIn(sight) != transparent)
                lst += objOpen;

            /* return the result */
            return lst;
        }
    }

    /* must be open to put anything into me */
    iobjFor(PutIn)
    {
        preCond { return nilToList(inherited()) + objOpen; }
    }

    /* must be open to pour anything into me */
    iobjFor(PourInto)
    {
        preCond { return nilToList(inherited()) + objOpen; }
    }

    /* can't lock an openable that isn't closed */
    dobjFor(Lock)
    {
        preCond { return nilToList(inherited()) + objClosed; }
    }
    dobjFor(LockWith)
    {
        preCond { return nilToList(inherited()) + objClosed; }
    }

    /* must be open to get out of a nested room */
    dobjFor(GetOutOf)
    {
        preCond()
        {
            return nilToList(inherited())
                + new ObjectPreCondition(self, objOpen);
        }
    }

    /* must be open to get into a nested room */
    dobjFor(Board)
    {
        preCond()
        {
            return nilToList(inherited())
                + new ObjectPreCondition(self, objOpen);
        }
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Lockable: a mix-in class that can be combined with an object's other
 *   superclasses to make the object respond to the verbs "lock" and
 *   "unlock."  A Lockable requires no key.
 */
class Lockable: object
    /* 
     *   get the master object - this works the same way it does for
     *   Openable 
     */
    masterObject()
    {
        /* 
         *   inherit from the next superclass, if possible; otherwise, use
         *   'self' as the default master object 
         */
        if (canInherit())
            return inherited();
        else
            return self;
    }

    /*
     *   Current locked state.  Returns the master's status; if we're the
     *   master, returns our initial status.  
     */
    isLocked()
    {
        if (masterObject == self)
            return initiallyLocked;
        else
            return masterObject.isLocked;
    }

    /* description of the object as locked or unlocked */
    lockedDesc = (masterObject.isLocked
                  ? libMessages.lockedMsg : libMessages.unlockedMsg)
    
    /*
     *   Make the object locked or unlocked.  By default, we'll simply set
     *   the isLocked flag to the new status.  Objects can override this
     *   to apply side effects of locking or unlocking.  
     */
    makeLocked(stat)
    {
        /* 
         *   if the next superclass in our defining class defines its own
         *   makeLocked, invoke it rather than this default handling 
         */
        if (canInherit())
        {
            /* 
             *   ignore this default definition, and use the definition
             *   obtained from the next superclass instead 
             */
            inherited(stat);
        }
        else
        {
            /* note the new status */
            masterObject.isLocked = stat;
        }
    }

    /*
     *   Action handling 
     */

    /* "lock" */
    dobjFor(Lock)
    {
        preCond = (nilToList(inherited()) + [touchObj])
        verify()
        {
            /* if we're already locked, there's no point in locking us */
            if (masterObject.isLocked)
                illogicalNow(&alreadyLocked);
        }
        action()
        {
            /* make it locked */
            makeLocked(true);

            /* make the default report */
            defaultReport(&okayLock);
        }
    }

    /* "unlock" */
    dobjFor(Unlock)
    {
        preCond = (nilToList(inherited()) + [touchObj])
        verify()
        {
            /* if we're already unlocked, there's no point in doing this */
            if (!masterObject.isLocked)
                illogicalNow(&alreadyUnlocked);
        }
        action()
        {
            /* make it unlocked */
            makeLocked(nil);

            /* make the default report */
            defaultReport(&okayUnlock);
        }
    }

    /* "lock with" */
    dobjFor(LockWith)
    {
        preCond = (nilToList(inherited()) + [touchObj])
        verify() { illogical(&noKeyNeeded); }
    }

    /* "unlock with" */
    dobjFor(UnlockWith)
    {
        preCond = (nilToList(inherited()) + [touchObj])
        verify() { illogical(&noKeyNeeded); }
    }

    /* 
     *   a locked object can't be opened - apply a precondition for "open"
     *   that ensures that we unlock this object before we can open it 
     */
    dobjFor(Open)
    {
        preCond = (nilToList(inherited()) + objUnlocked)
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   LockableWithKey: a mix-in class that can be combined with an object's
 *   other superclasses to make the object respond to the verbs "lock" and
 *   "unlock," with a key as an indirect object.  A LockableWithKey cannot
 *   be locked or unlocked except with the keys listed in the keyList
 *   property.
 */
class LockableWithKey: Lockable
    /*
     *   Determine if the key fits this lock.  Returns true if so, nil if
     *   not.  By default, we'll return true if the key is in my keyList.
     *   This can be overridden to use other key selection criteria.  
     */
    keyFitsLock(key) { return keyList.indexOf(key) != nil; }

    /*
     *   Determine if the key is plausibly of the right type for this
     *   lock.  This doesn't check to see if the key actually fits the
     *   lock - rather, this checks to see if the key is generally the
     *   kind of object that might plausibly be used with this lock.
     *   
     *   The point of this routine is to make this class concerned only
     *   with the abstract notion of objects that serve to lock and unlock
     *   other objects, without requiring that the key objects resemble
     *   little notched metal sticks or that the lock objects resemble
     *   cylinders with pins - or, more specifically, without requiring
     *   that all of the kinds of keys in a game remotely resemble one
     *   another.
     *   
     *   For example, one kind of "key" in a game might be a plastic card
     *   with a magnetic stripe, and the corresponding lock would be a
     *   card slot; another kind of key might the traditional notched
     *   metal stick.  Clearly, no one would ever think to use a plastic
     *   card with a conventional door lock, nor would one try to put a
     *   house key into a card slot (not with the expectation that it
     *   would actually work, anyway).  This routine is meant to
     *   facilitate this kind of distinction: the card slot can use this
     *   routine to indicate that only plastic card objects are plausible
     *   as keys, and door locks can indicate that only metal keys are
     *   plausible.
     *   
     *   This routine can be used for disambiguation and other purposes
     *   when we must programmatically select a key that is not specified
     *   or is only vaguely specified.  For example, the keyring searcher
     *   uses it so that, when we're searching for a key on a keyring to
     *   open this lock, we implicitly try only the kinds of keys that
     *   would be plausibly useful for this kind of lock.
     *   
     *   By default, we'll simply return true.  Subclasses specific to a
     *   game (such as the "card reader" base class or the "door lock"
     *   base class) can override this to discriminate among the
     *   game-specific key classes.  
     */
    keyIsPlausible(key) { return true; }

    /* the list of objects that can serve as keys for this object */
    keyList = []

    /* 
     *   The list of keys which the player knows will fit this lock.  This
     *   is used to make key disambiguation automatic once the player
     *   knows the correct key for a lock.  
     */
    knownKeyList = []

    /*
     *   Flag: remember my keys after they're successfully used.  If this
     *   is true, whenever a key is successfully used to lock or unlock
     *   this object, we'll add the key to our known key list;
     *   subsequently, whenever we try to use a key in this lock, we will
     *   automatically disambiguate the key based on the keys known to
     *   work previously.
     *   
     *   Some authors might prefer not to assume that the player should
     *   remember which keys operate which locks, so this property can be
     *   changed to nil to eliminate this memory feature.  By default we
     *   set this to true, since it shouldn't generally give away any
     *   secrets or puzzles for the game to assume that a key that was
     *   used successfully once with a given lock is the one to be used
     *   subsequently with the same lock.  
     */
    rememberKnownKeys = true

    /*
     *   Determine if the player knows that the given key operates this
     *   lock.  Returns true if the key is in our known key list, nil if
     *   not.  
     */
    isKeyKnown(key) { return masterObject.knownKeyList.indexOf(key) != nil; }

    /*
     *   Action handling 
     */

    /* "lock" */
    dobjFor(Lock)
    {
        preCond
        {
            /* 
             *   remove any objClosed from our precondition - since we
             *   won't actually do any locking but will instead merely ask
             *   for an indirect object, we don't want to apply the normal
             *   closed precondition here 
             */
            return inherited() - objClosed;
        }
        verify()
        {
            /* if we're already locked, there's no point in locking us */
            if (masterObject.isLocked)
                illogicalNow(&alreadyLocked);
        }
        action()
        {
            /* ask for an indirect object to use as the key */
            askForIobj(LockWith);
        }
    }

    /* "unlock" */
    dobjFor(Unlock)
    {
        verify()
        {
            /* if we're not locked, there's no point in unlocking us */
            if (!masterObject.isLocked)
                illogicalNow(&alreadyUnlocked);
        }
        action()
        {
            /* ask for an indirect object to use as the key */
            askForIobj(UnlockWith);
        }
    }

    /* 
     *   perform the action processing for LockWith or UnlockWith - these
     *   are highly symmetrical, in that the only thing that varies is the
     *   new lock state we establish 
     */
    lockOrUnlockAction(lock)
    {
        /* 
         *   If it's a keyring, let the keyring's action handler do the
         *   work.  Otherwise, if it's my key, lock/unlock; it's not a
         *   key, fail.  
         */
        if (gIobj.ofKind(Keyring))
        {
            /* 
             *   do nothing - let the indirect object action handler do
             *   the work 
             */
        }
        else if (keyFitsLock(gIobj))
        {
            /* remember the key that fits this door, if appropriate */
            if (rememberKnownKeys
                && masterObject.knownKeyList.indexOf(gIobj) == nil)
                masterObject.knownKeyList += gIobj;
            
            /* set my new state and issue a default report */
            makeLocked(lock);
            defaultReport(lock ? &okayLock : &okayUnlock);
        }
        else
        {
            /* the key doesn't work in this lock */
            reportFailure(&keyDoesNotFitLock);
        }
    }

    /* "lock with" */
    dobjFor(LockWith)
    {
        verify()
        {
            /* if we're already locked, there's no point in locking us */
            if (masterObject.isLocked)
                illogicalNow(&alreadyLocked);
        }
        action()
        {
            /* perform the generic lock/unlock action processing */
            lockOrUnlockAction(true);
        }
    }

    /* "unlock with" */
    dobjFor(UnlockWith)
    {
        verify()
        {
            /* if we're not locked, there's no point in unlocking us */
            if (!masterObject.isLocked)
                illogicalNow(&alreadyUnlocked);
        }
        action()
        {
            /* perform the generic lock/unlock action processing */
            lockOrUnlockAction(nil);
        }
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   The common base class for containers and surfaces: things that can
 *   have their contents directly manipulated by actors.
 */
class BaseContainer: Thing
    /*
     *   A container can limit the cumulative amount of bulk of its
     *   contents, and the maximum bulk of any one object, using
     *   bulkCapacity and maxSingleBulk.  We count the cumulative and
     *   single-item limits separately, since we want to allow modelling
     *   some objects as so large that they won't fit in this container at
     *   all, even if the container is carrying nothing else, without
     *   limiting the number of small items we can carry.
     *   
     *   By default, we set bulkCapacity to a very large number, making
     *   the total capacity of the object essentially unlimited.  However,
     *   we set maxSingleBulk to a relatively low number - this way, if an
     *   author wants to designate certain objects as especially large and
     *   thus unable to fit in ordinary containers, the author merely
     *   needs to set the bulk of those large items to something greater
     *   than 10.  On the other hand, if an author doesn't want to worry
     *   about bulk and limited carrying capacities and simply uses
     *   library defaults for everything, we will be able to contain
     *   anything and everything.
     *   
     *   In a game that models bulk realistically, a container's bulk
     *   should generally be equal to or slightly greater than its
     *   bulkCapacity, because a container shouldn't be smaller on the
     *   outside than on the inside.  If bulkCapacity exceeds bulk, the
     *   player can work around a holding bulk limit by piling objects
     *   into the container, thus "hiding" the bulks of the contents
     *   behind the smaller bulk of the container.  
     */
    bulkCapacity = 10000
    maxSingleBulk = 10

    /* 
     *   receive notification that we're about to insert an object into
     *   this container 
     */
    notifyInsert(obj, newCont)
    {
        /* if I'm the new direct container, check our bulk limit */
        if (newCont == self)
        {
            /* 
             *   do a 'what if' test to see what would happen to our
             *   contained bulk if we moved this item into me 
             */
            obj.whatIf({: checkBulkInserted(obj)}, &moveInto, self);
        }

        /* inherit base class handling */
        inherited(obj, newCont);
    }

    /*
     *   Check to see if a proposed insertion - already tentatively in
     *   effect when this routine is called - would overflow our bulk
     *   limits.  Reports failure and exits if the inserted object would
     *   exceed our capacity. 
     */
    checkBulkInserted(insertedObj)
    {
        local objBulk;

        /* get the bulk of the inserted object itself */
        objBulk = insertedObj.getBulk();

        /*
         *   Check the object itself to see if it fits by itself.  If it
         *   doesn't, we can report the simple fact that the object is too
         *   big for the container.  
         */
        if (objBulk > maxSingleBulk || objBulk > bulkCapacity)
        {
            reportFailure(&tooLargeForContainer, insertedObj, self);
            exit;
        }
            
        /* 
         *   If our contained bulk is over our maximum, don't allow it.
         *   Note that we merely need to check our current bulk within,
         *   since this routine is called with the insertion already
         *   tentatively in effect. 
         */
        if (getBulkWithin() > bulkCapacity)
        {
            reportFailure(tooFullMsg, insertedObj, self);
            exit;
        }
    }

    /* 
     *   the message property to use when we're too full to hold a new
     *   object (i.e., the object's bulk would push us over our bulk
     *   capacity limit) 
     */
    tooFullMsg = &containerTooFull
    

    /*
     *   Check a bulk change of one of my direct contents. 
     */
    checkBulkChangeWithin(obj)
    {
        local objBulk;
        
        /* get the object's new bulk */
        objBulk = obj.getBulk();
        
        /* 
         *   if this change would cause the object to exceed our
         *   single-item bulk limit, don't allow it 
         */
        if (objBulk > maxSingleBulk || objBulk > bulkCapacity)
        {
            reportFailure(&becomingTooLargeForContainer, obj, self);
            exit;
        }

        /* 
         *   If our total carrying capacity is exceeded with this change,
         *   don't allow it.  Note that 'obj' is already among our
         *   contents when this routine is called, so we can simply check
         *   our current total bulk within.  
         */
        if (getBulkWithin() > bulkCapacity)
        {
            reportFailure(&containerBecomingTooFull, obj, self);
            exit;
        }
    }
;


/* ------------------------------------------------------------------------ */
/*
 *   Container: an object that can have other objects placed within it.  
 */
class Container: BaseContainer
    /* 
     *   My current open/closed state.  By default, this state never
     *   changes, but is fixed in the object's definition; for example, a
     *   box without a lid would always be open, while a hollow glass cube
     *   would always be closed.  Our default state is open. 
     */
    isOpen = true

    /* the material that the container is made of */
    material = adventium

    /* prepositional phrase for objects being put into me */
    putDestMessage = &putDestContainer

    /*
     *   Determine if I can move an object via a path through this
     *   container. 
     */
    checkMoveViaPath(obj, dest, op)
    {
        /* 
         *   if we're moving the object in or out of me, we must consider
         *   our openness and whether or not the object fits through our
         *   opening 
         */
        if (op is in (PathIn, PathOut))
        {
            /* if we're closed, we can't move anything in or out */
            if (!isOpen)
                return new CheckStatusFailure(&cannotMoveThroughClosed,
                                              obj, self);

            /* if it doesn't fit through our opening, don't allow it */
            if (!canFitObjThruOpening(obj))
                return new CheckStatusFailure(op == PathIn
                                              ? &cannotFitIntoOpening
                                              : &cannotFitOutOfOpening,
                                              obj, self);
        }
        
        /* in any other cases, allow the operation */
        return checkStatusSuccess;
    }

    /*
     *   Determine if an actor can touch an object via a path through this
     *   container.  
     */
    checkTouchViaPath(obj, dest, op)
    {
        /* 
         *   if we're reaching from inside directly to me, allow it -
         *   treat this as touching our interior, which we allow from
         *   within regardless of our open/closed status 
         */
        if (op == PathOut && dest == self)
            return checkStatusSuccess;

        /* 
         *   if we're reaching in or out of me, consider our openness and
         *   whether or not the actor's hand fits through our opening 
         */
        if (op is in (PathIn, PathOut))
        {
            /* if we're closed, we can't reach into/out of the container */
            if (!isOpen)
                return new CheckStatusFailure(&cannotTouchThroughClosed,
                                              obj, self);

            /* 
             *   if the object's "hand" doesn't fit through our opening,
             *   don't allow it 
             */
            if (!canObjReachThruOpening(obj))
                return new CheckStatusFailure(op == PathIn
                                              ? &cannotReachIntoOpening
                                              : &cannotReachOutOfOpening,
                                              obj, self);
        }
        
        /* in any other cases, allow the operation */
        return checkStatusSuccess;
    }

    /*
     *   Determine if the given object fits through our opening.  This is
     *   only called when we're open; this determines if the object can be
     *   moved in or out of this container.  By default, we'll return
     *   true; some objects might want to override this to disallow
     *   objects over a certain size from being moved in or out of this
     *   container.
     *   
     *   Note that this method doesn't care whether or not the object can
     *   actually fit inside the container once through the opening; we
     *   only care about whether or not the object can fit through the
     *   opening itself.  This allows for things like narrow-mouthed
     *   bottles which have greater capacity within than in their
     *   openings.  
     */
    canFitObjThruOpening(obj) { return true; }

    /*
     *   Determine if the given object can "reach" through our opening,
     *   for the purposes of touching an object on the other side of the
     *   opening.  This is used to determine if the object, which is
     *   usually an actor, can its "hand" (or whatever appendange 'obj'
     *   uses to reach things) through our opening.  This is only called
     *   when we're open.  By default, we'll simply return true.
     *   
     *   This differs from canFitObjThruOpening() in that we don't care if
     *   all of 'obj' is able to fit through the opening; we only care
     *   whether obj's hand (or whatever it uses for reaching) can fit.  
     */
    canObjReachThruOpening(obj) { return true; }

    /*
     *   Determine how a sense passes to my contents.  If I'm open, the
     *   sense passes through directly, since there's nothing in the way.
     *   If I'm closed, the sense must pass through my material.  
     */
    transSensingIn(sense)
    {
        if (isOpen)
        {
            /* I'm open, so the sense passes through without interference */
            return transparent;
        }
        else
        {
            /* I'm closed, so the sense must pass through my material */
            return material.senseThru(sense);
        }
    }

    /*
     *   Get my fill medium.  If I'm open, inherit my parent's medium,
     *   assuming that the medium behaves like fog or smoke and naturally
     *   disperses to fill any nested open containers.  If I'm closed, I
     *   am by default filled with no medium.  
     */
    fillMedium()
    {
        if (isOpen && location != nil)
        {
            /* I'm open, so return my location's medium */
            return location.fillMedium();
        }
        else
        {
            /* 
             *   I'm closed, so we're cut off from the parent - assume
             *   we're filled with nothing 
             */
            return nil;
        }
    }

    /*
     *   If we're obstructing a sense path, it must be because we're
     *   closed.  Try implicitly opening.  
     */
    tryImplicitRemoveObstructor(sense, obj)
    {
        /* 
         *   try opening me, returning true if we attempt the command, nil
         *   if not 
         */
        return tryImplicitAction(Open, self);
    }

    /*
     *   Try putting an object into me when I'm serving as a bag of
     *   holding.  For a container, this simply does a "put obj in bag".  
     */
    tryPuttingObjInBag(target)
    {
        /* if the object won't fit all by itself, don't even try */
        if (target.getBulk() > maxSingleBulk)
            return nil;

        /* if we can't fit the object with other contents, don't try */
        if (target.whatIf({: getBulkWithin() > bulkCapacity},
                          &moveInto, self))
            return nil;

        /* we're a container, so use "put in" to get the object */
        return tryImplicitActionMsg(&announceMoveToBag, PutIn, target, self);
    }

    /*
     *   Display a message explaining why we are obstructing a sense path
     *   to the given object.
     */
    cannotReachObject(obj)
    {
        /* 
         *   We must be obstructing by containment.  Show an appropriate
         *   message depending on whether the object is inside me or not -
         *   if not, then the actor trying to reach the object must be
         *   inside me. 
         */
        if (obj.isIn(self))
            libMessages.cannotReachContents(obj, self);
        else
            libMessages.cannotReachOutside(obj, self);
    }

    /* explain why we can't see the source of a sound */
    cannotSeeSoundSource(obj)
    {
        /* we must be obstructing by containment */
        if (obj.isIn(self))
            libMessages.soundIsFromWithin(obj, self);
        else
            libMessages.soundIsFromWithout(obj, self);
    }

    /* explain why we can't see the source of an odor */
    cannotSeeSmellSource(obj)
    {
        /* we must be obstructing by containment */
        if (obj.isIn(self))
            libMessages.smellIsFromWithin(obj, self);
        else
            libMessages.smellIsFromWithout(obj, self);
    }

    /* -------------------------------------------------------------------- */
    /* 
     *   Show our status for "examine".  This shows our open/closed status,
     *   and lists our contents. 
     */
    examineStatus()
    {
        /* show any special container-specific status */
        examineContainerStatus();

        /* show my contents */
        examineContainerContents();
    }

    /*
     *   mention my open/closed status for Examine processing 
     */
    examineContainerStatus()
    {
        /*
         *   By default, show nothing extra.  This can be overridden by
         *   subclasses as needed to show any extra status before our
         *   contents list.  
         */
    }

    /* 
     *   show my contents as part of Examine processing 
     */
    examineContainerContents()
    {
        /* show my contents, if I have any */
        descContentsLister.showList(gActor, self, contents, ListRecurse, 0,
                                    gActor.visibleInfoTable(), nil);
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Look in" 
     */
    dobjFor(LookIn)
    {
        verify() { }
        check()
        {
            /* 
             *   if I'm closed, and I can't see my contents when closed, we
             *   can't go on 
             */
            if (!isOpen && transSensingIn(sight) == opaque)
            {
                /* we can't see anything because we're closed */
                mainReport(&cannotLookInClosed);
                exit;
            }
        }
        action()
        {
            /* show my contents, if I have any */
            lookInLister.showList(gActor, self, contents, ListRecurse,
                                  0, gActor.visibleInfoTable(), nil);

            /* examine my special contents */
            examineSpecialContents();

            /* 
             *   Anything that the an overriding caller (a routine that
             *   called us with 'inherited') wants to add is an addendum to
             *   our description, so add a transcript marker to indicate
             *   tht the main description is now finished.
             *   
             *   The important thing about this is that any message that an
             *   overriding caller wants to add is not considered part of
             *   the description, in the sense that we don't want it to
             *   suppress any default description we've already generated.
             *   One of the transformations we apply to the transcript is
             *   to suppress any default descriptive text if there's any
             *   more specific descriptive text following (for example, we
             *   suppress "It's an ordinary <thing>" if we also are going
             *   to say "it's open" or "it contains three coins").  If we
             *   have an overriding caller who's going to add anything,
             *   then we must assume that what the caller's adding is
             *   something about the act of examining the object, rather
             *   than a description of the object, so we don't want it to
             *   suppress a default description.  
             */
            gTranscript.endDescription();
        }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   Put In processing.  A container can accept new contents. 
     */

    iobjFor(PutIn)
    {
        verify()
        {
            /* we can't put myself in myself */
            if (gDobj == self)
                illogical(&cannotPutInSelf);

            /* 
             *   if we haven't resolved the direct object yet, we can at
             *   least check to see if all of the potential direct objects
             *   are already in me, and rule out this indirect object as
             *   illogical if so 
             */
            if (gDobj == nil
                && gTentativeDobj.indexWhich({x: !x.obj_.isIn(self)}) == nil)
            {
                /* 
                 *   all of the potential direct objects are already in me
                 *   - it's illogical 
                 */
                illogicalNow(&alreadyPutIn);
            }

            /* 
             *   if I'm not held by the actor, give myself a slightly
             *   lower ranking than fully logical, so that objects being
             *   held are preferred 
             */
            if (!isIn(gActor))
                logicalRank(60, 'not indirectly held');
            else if (!isHeldBy(gActor))
                logicalRank(70, 'not held');
        }

        action()
        {
            /* move the direct object into me */
            gDobj.moveInto(self);
            
            /* issue our default acknowledgment of the command */
            defaultReport(&okayPutIn);
        }
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   OpenableContainer: an object that can contain things, and which can
 *   be opened and closed.  
 */
class OpenableContainer: Openable, Container
;

/* ------------------------------------------------------------------------ */
/*
 *   LockableContainer: an object that can contain things, and that can be
 *   opened and closed as well as locked and unlocked.  
 */
class LockableContainer: Lockable, OpenableContainer
;

/* ------------------------------------------------------------------------ */
/*
 *   KeyedContainer: an openable container that can be locked and
 *   unlocked, but only with a specified key.  
 */
class KeyedContainer: LockableWithKey, OpenableContainer
;


/* ------------------------------------------------------------------------ */
/*
 *   Surface: an object that can have other objects placed on top of it.
 *   A surface is essentially the same as a regular container, but the
 *   contents of a surface behave as though they are on the surface's top
 *   rather than contained within the object.  
 */
class Surface: BaseContainer
    /* my contents lister */
    contentsLister = surfaceContentsLister
    descContentsLister = surfaceDescContentsLister
    lookInLister = surfaceLookInLister

    /* 
     *   we're a surface, so taking something from me that's not among my
     *   contents shows the message as "that's not on the iobj" 
     */
    takeFromNotInMessage = &takeFromNotOn

    /* 
     *   my message indicating that another object x cannot be put into me
     *   because I'm already in x 
     */
    circularlyInMessage = &circularlyOn

    /* message phrase for objects put into me */
    putDestMessage = &putDestSurface

    /* message when we're too full for another object */
    tooFullMsg = &surfaceTooFull

    /* -------------------------------------------------------------------- */
    /*
     *   Status for "Examine".  We'll what's on our surface.
     */
    examineStatus()
    {
        /* show my contents */
        examineSurfaceContents();
    }

    /*
     *   As part of Examine action processing, show the items on the
     *   surface.  This is separated from the main Examine action
     *   processing so that subclasses can override the action processing
     *   and insert the standard contents list at any desired point,
     *   without having to inherit all of the basic surface Examine
     *   behavior. 
     */
    examineSurfaceContents()
    {
        /* show my contents, if I have any */
        descContentsLister.showList(gActor, self, contents, ListRecurse,
                                    0, gActor.visibleInfoTable(), nil);
    }

    /* -------------------------------------------------------------------- */
    /*
     *   Put On processing 
     */
    iobjFor(PutOn)
    {
        verify()
        {
            /* we can't put myself on myself, but otherwise allow it */
            if (gDobj == self)
                illogical(&cannotPutOnSelf);

            /* 
             *   if we haven't resolved the direct object yet, we can at
             *   least check to see if all of the potential direct objects
             *   are already on me, and rule out this indirect object as
             *   illogical if so 
             */
            if (gDobj == nil
                && gTentativeDobj.indexWhich({x: !x.obj_.isIn(self)}) == nil)
            {
                /* 
                 *   all of the potential direct objects are already on me
                 *   - it's illogical 
                 */
                illogicalNow(&alreadyPutOn);
            }

            /* 
             *   if I'm not held by the actor, give myself a slightly
             *   lower ranking than fully logical, so that objects being
             *   held are preferred 
             */
            if (!isIn(gActor))
                logicalRank(60, 'not indirectly held');
            else if (!isHeldBy(gActor))
                logicalRank(70, 'not held');
        }

        action()
        {
            /* move the direct object onto me */
            gDobj.moveInto(self);
            
            /* issue our default acknowledgment */
            defaultReport(&okayPutOn);
        }
    }

    /*
     *   Looking "in" a surface simply shows the surface's contents. 
     */
    dobjFor(LookIn)
    {
        verify() { }
        action()
        {
            /* show my contents, if I have any */
            lookInLister.showList(gActor, self, contents, ListRecurse,
                                  0, gActor.visibleInfoTable(), nil);

            /* examine my special contents */
            examineSpecialContents();

            /* anything that an overriding caller adds is an addendum */
            gTranscript.endDescription();
        }
    }
;


/* ------------------------------------------------------------------------ */
/*
 *   Food - something you can eat.  By default, when an actor eats a food
 *   item, the item disappears.  
 */
class Food: Thing
    dobjFor(Taste)
    {
        /* tasting food is perfectly logical */
        verify() { }
    }

    dobjFor(Eat)
    {
        verify() { }
        action()
        {
            /* describe the consumption */
            defaultReport(&okayEat);

            /* the object disappears */
            moveInto(nil);
        }
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   OnOffControl - a generic control that can be turned on and off.  We
 *   keep track of an internal on/off state, and recognize the commands
 *   "turn on" and "turn off".  
 */
class OnOffControl: Thing
    /*
     *   The current on/off setting.  We'll start in the 'off' position by
     *   default.  
     */
    isOn = nil

    /*
     *   On/off status name.  This returns the appropriate name ('on' or
     *   'off' in English) for our current status. 
     */
    onDesc = (isOn ? libMessages.onMsg : libMessages.offMsg)

    /*
     *   Change our on/off setting.  Subclasses can override this to apply
     *   any side effects of changing the value. 
     */
    makeOn(val)
    {
        /* remember the new value */
        isOn = val;
    }

    dobjFor(TurnOn)
    {
        verify()
        {
            /* if it's already on, complain */
            if (isOn)
                illogicalNow(&alreadySwitchedOn);
        }
        action()
        {
            /* set to 'on' and generate a default report */
            makeOn(true);
            defaultReport(&okayTurnOn);
        }
    }

    dobjFor(TurnOff)
    {
        verify()
        {
            /* if it's already off, complain */
            if (!isOn)
                illogicalNow(&alreadySwitchedOff);
        }
        action()
        {
            /* set to 'off' and generate a default report */
            makeOn(nil);
            defaultReport(&okayTurnOff);
        }
    }
;

/*
 *   Switch - a simple extension of the generic on/off control that can be
 *   used with a "switch" command without specifying "on" or "off", and
 *   treats "flip" synonymously.  
 */
class Switch: OnOffControl
    /* "switch" with no specific new setting - reverse our setting */
    dobjFor(Switch)
    {
        verify() { }
        action()
        {
            /* reverse our setting and generate a report */
            makeOn(!isOn);
            defaultReport(isOn ? &okayTurnOn : &okayTurnOff);
        }
    }

    /* "flip" is the same as "switch" for our purposes */
    dobjFor(Flip) asDobjFor(Switch)
;

/* ------------------------------------------------------------------------ */
/*
 *   Dial - something you can turn to different settings.  Note that dials
 *   are usually used as components of larger objects; since our base
 *   class is Thing, component dials should be created to inherit multiply
 *   from Dial and Component, in that order.
 */
class Dial: Thing
    /*
     *   The dial's current setting.  This is an arbitrary string value.  
     */
    curSetting = '1'

    /*
     *   Change our setting.  Subclasses can override this routine to
     *   apply any side effects of changing the value.  
     */
    makeSetting(val)
    {
        /* remember the new value */
        curSetting = val;
    }

    /*
     *   Is the given text a valid setting?  Returns true if so, nil if
     *   not.  This should not display any messages; simply indicate
     *   whether or not the setting is valid.  
     */
    isValidSetting(val)
    {
        /* 
         *   by default, allow anything; subclasses should override to
         *   enforce the valid range of values on the dial 
         */
        return true;
    }

    /* "turn" with no destination */
    dobjFor(Turn)
    {
        verify() { illogical(&mustSpecifyTurnTo); }
    }

    /*
     *   "turn <self> to <literal>" action
     */
    dobjFor(TurnTo)
    {
        verify()
        {
            local txt;
            
            /* 
             *   If we already know our literal text, and it's not valid,
             *   reduce the logicalness.  Don't actually make it illogical,
             *   as it's probably still more logical to turn a dial to an
             *   invalid setting than to turn something that isn't
             *   dial-like at all.  
             */
            if ((txt = gAction.getLiteral()) != nil
                && !isValidSetting(txt))
                logicalRank(50, 'invalid setting');
        }
        check()
        {
            /* if the setting is not valid, don't allow it */
            if (!isValidSetting(gAction.getLiteral()))
            {
                /* there is no such setting */
                reportFailure(&turnToInvalid);
                exit;
            }
        }
        action()
        {
            /* set the new value */
            makeSetting(gAction.getLiteral());

            /* remark on the change */
            defaultReport(&okayTurnTo, curSetting);
        }
    }

    /* treat "set <self> to <literal>" the same as "turn to" */
    dobjFor(SetTo) asDobjFor(TurnTo)
;

/*
 *   Numbered Dial - something you can turn to a range of numeric values. 
 */
class NumberedDial: Dial
    /*
     *   The range of settings - the dial can be set to values from the
     *   minimum to the maximum, inclusive. 
     */
    minSetting = 1
    maxSetting = 10

    /* 
     *   Check a setting for validity.  A setting is valid only if it's a
     *   number within the allowed range for the dial. 
     */
    isValidSetting(val)
    {
        local num;
        
        /* if it doesn't look like a number, it's not valid */
        if (rexMatch('<digit>+', val) != val.length())
            return nil;

        /* get the numeric value */
        num = toInteger(val);

        /* it's valid if it's within range */
        return num >= minSetting && num <= maxSetting;
    }
;


/* ------------------------------------------------------------------------ */
/*
 *   Button - something you can push to activate, as a control for a
 *   mechanical device.  
 */
class Button: Thing
    dobjFor(Push)
    {
        verify() { }
        action()
        {
            /* 
             *   individual buttons should override this to carry out any
             *   special action for the button; by default, we'll just
             *   show a simple acknowledgment  
             */
            defaultReport(&okayPushButton);
        }
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Lever - something you can push, pull, or move, generally as a control
 *   for a mechanical device.  Our basic lever has two states, "pushed"
 *   and "pulled".  
 */
class Lever: Thing
    /*
     *   The current state.  We have two states: "pushed" and "pulled".
     *   We start in the pushed state, so the lever can initially be
     *   pulled, since "pull" is the verb most people would first think to
     *   apply to a lever.  
     */
    isPulled = nil

    /* 
     *   Set the state.  This can be overridden to apply side effects as
     *   needed. 
     */
    makePulled(pulled)
    {
        /* note the new state */
        isPulled = pulled;
    }

    /*
     *   Action handlers.  We handle push and pull, and we treat "move" as
     *   equivalent to whichever of push or pull is appropriate to reverse
     *   the current state.  
     */
    dobjFor(Push)
    {
        verify()
        {
            /* if it's already pushed, pushing it again makes no sense */
            if (!isPulled)
                illogicalNow(&alreadyPushed);
        }
        action()
        {
            /* set the new state to pushed (i.e., not pulled) */
            makePulled(nil);

            /* make the default report */
            defaultReport(&okayPushLever);
        }
    }
    dobjFor(Pull)
    {
        verify()
        {
            /* if it's already pulled, pulling it again makes no sense */
            if (isPulled)
                illogicalNow(&alreadyPulled);
        }
        action()
        {
            /* set the new state to pulled */
            makePulled(true);

            /* make the default report */
            defaultReport(&okayPullLever);
        }
    }
    dobjFor(Move)
    {
        verify() { }
        check()
        {
            /* run the check for pushing or pulling, as appropriate */
            if (isPulled)
                checkDobjPush();
            else
                checkDobjPull();
        }
        action()
        {
            /* if we're pulled, push the lever; otherwise pull it */
            if (isPulled)
                actionDobjPush();
            else
                actionDobjPull();
        }
    }
;

/*
 *   A spring-loaded lever is a lever that bounces back to its starting
 *   position after being pulled.  This is essentially equivalent in terms
 *   of functionality to a button, but can at least provide superficial
 *   variety.  
 */
class SpringLever: Lever
    dobjFor(Pull)
    {
        action()
        {
            /*
             *   Individual objects should override this to perform the
             *   appropriate action when the lever is pulled.  By default,
             *   we'll do nothing except show a default report. 
             */
            defaultReport(&okayPullSpringLever);
        }
    }
;
    

/* ------------------------------------------------------------------------ */
/*
 *   An item that can be worn
 */
class Wearable: Thing
    /* is the item currently being worn? */
    isWorn()
    {
        /* it's being worn if the wearer is non-nil */
        return wornBy != nil;
    }

    /* 
     *   make the item worn by the given actor; if actor is nil, the item
     *   isn't being worn by anyone 
     */
    makeWornBy(actor)
    {
        /* remember who's wearing the item */
        wornBy = actor;
    }

    /*
     *   An item being worn is not considered to be held in the wearer's
     *   hands. 
     */
    isHeldBy(actor)
    {
        if (isWornBy(actor))
        {
            /* it's being worn by the actor, so it's not also being held */
            return nil;
        }
        else
        {
            /* 
             *   it's not being worn by this actor, so use the default
             *   interpretation of being held 
             */
            return inherited(actor);
        }
    }

    /*
     *   A wearable is not considered held by an actor when it is being
     *   worn, so we must do a what-if test for removing the item if the
     *   actor is currently wearing the item.  If the actor isn't wearing
     *   the item, we can use the default test of moving the item into the
     *   actor's inventory.  
     */
    whatIfHeldBy(func, newLoc)
    {
        /*
         *   If the article is being worn, and it's already in the same
         *   location we're moving it to, simply test with the article no
         *   longer being worn.  Otherwise, inherit the default handling.  
         */
        if (location == newLoc && wornBy != nil)
            return whatIf(func, &wornBy, nil);
        else
            return inherited(func, newLoc);
    }

    /*
     *   Try making the current command's actor hold me.  If I'm already
     *   directly in the actor's inventory and I'm being worn, we'll try a
     *   'doff' command; otherwise, we'll use the default handling.  
     */
    tryHolding()
    {
        /*   
         *   Try an implicit 'take' command.  If the actor is carrying the
         *   object indirectly, make the command "take from" instead,
         *   since what we really want to do is take the object out of its
         *   container.  
         */
        if (location == gActor && isWornBy(gActor))
            return tryImplicitAction(Doff, self);
        else
            return inherited();
    }

    /* 
     *   The object wearing this object, if any; if I'm not being worn,
     *   this is nil.  The wearer should always be a container (direct or
     *   indirect) of this object - in order to wear something, you must
     *   be carrying it.  In most cases, the wearer should be the direct
     *   container of the object.
     *   
     *   The reason we keep track of who's wearing the object (rather than
     *   simply keeping track of whether it's being worn) is to allow for
     *   cases where an actor is carrying another actor.  Since this
     *   object will be (indirectly) inside both actors in such cases, we
     *   would have to inspect intermediate containers to determine
     *   whether or not the outer actor was wearing the object if we
     *   didn't keep track of the wearer directly.  
     */
    wornBy = nil

    /* am I worn by the given object? */
    isWornBy(actor)
    {
        return wornBy == actor;
    }

    /*
     *   An article of clothing that is being worn by an actor does not
     *   typically encumber the actor at all, so by default we'll return
     *   zero if we're being worn by the actor, and our normal bulk
     *   otherwise.  
     */
    getEncumberingBulk(actor)
    {
        /* 
         *   if we're being worn by the actor, we create no encumbrance at
         *   all; otherwise, return our normal bulk 
         */
        return isWornBy(actor) ? 0 : getBulk();
    }

    /*
     *   An article of clothing typically encumbers an actor with the same
     *   weight whether or not the actor is wearing the item.  However,
     *   this might not apply to all objects; a suit of armor, for
     *   example, might be slightly less encumbering in terms of weight
     *   when worn than it is when held because the distribution of weight
     *   is more manageable when worn.  By default, we simply return our
     *   normal weight, whether worn or not; subclasses can override as
     *   needed to differentiate.  
     */
    getEncumberingWeight(actor)
    {
        return getWeight();
    }

    /* get my state */
    getState = (isWorn() ? wornState : unwornState)


    /* -------------------------------------------------------------------- */
    /*
     *   Action processing 
     */

    dobjFor(Wear)
    {
        preCond = [objHeld]
        verify()
        {
            /* make sure the actor isn't already wearing the item */
            if (isWornBy(gActor))
                illogicalNow(&alreadyWearing);
        }
        action()
        {
            /* make the item worn and describe what happened */
            makeWornBy(gActor);
            defaultReport(&okayWear);
        }
    }

    dobjFor(Doff)
    {
        preCond = [roomToHoldObj]
        verify()
        {
            /* make sure the actor is actually wearing the item */
            if (!isWornBy(gActor))
                illogicalNow(&notWearing);
        }
        action()
        {
            /* un-wear the item and describe what happened */
            makeWornBy(nil);
            defaultReport(&okayDoff);
        }
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   An item that can provide light.
 *   
 *   Any Thing can provide light, but this class should be used for
 *   objects that explicitly serve as light sources from the player's
 *   perspective.  Objects of this class display a "providing light"
 *   status message in inventory listings, and can be turned on and off
 *   via the isLit property.  
 */
class LightSource: Thing
    /* is the light source currently turned on? */
    isLit = true

    /*
     *   Turn the light source on or off.  Note that we don't have to make
     *   any special check for a change to the light level, because the
     *   main action handler always checks for a change in light/dark
     *   status over the course of the turn.  
     */
    makeLit(lit)
    {
        /* change the status */
        isLit = lit;
    }

    /* 
     *   We can distinguish light sources according to their isLit status.
     *   Give the lit/unlit distinction higher priority than the normal
     *   ownership/containment distinction. 
     */
    distinguishers = [basicDistinguisher, litUnlitDistinguisher,
                      ownershipAndLocationDistinguisher]

    /* the brightness that the object has when it is on and off */
    brightnessOn = 3
    brightnessOff = 0

    /* 
     *   return the appropriate on/off brightness, depending on whether or
     *   not we're currently lit 
     */
    brightness { return isLit ? brightnessOn : brightnessOff; }

    /* get our current state: lit or unlit */
    getState = (brightness > 1 ? lightSourceStateOn : lightSourceStateOff)

    /* get our set of possible states */
    allStates = [lightSourceStateOn, lightSourceStateOff]
;

/*
 *   A Flashlight is a special kind of light source that can be switched
 *   on and off. 
 */
class Flashlight: LightSource, Switch
    /* our switch status - start in the 'off' position */
    isOn = nil

    /* 
     *   Change the on/off status.  Note that switching the flashlight on
     *   or off should always be done via makeOn - the makeLit inherited
     *   from the LightSource should never be called directly on a
     *   Flashlight object, because it doesn't keep the switch on/off and
     *   flashlight lit/unlit status in sync.  This routine is the one to
     *   call because it keeps everything properly synchronized.  
     */
    makeOn(stat)
    {
        /* inherit the default handling */
        inherited(stat);

        /* 
         *   Set the 'lit' status to track the on/off status.  Note that
         *   we don't simply do this by deriving isLit from isOn because
         *   we want to invoke the side effects of changing the status by
         *   calling makeLit explicitly. 
         */
        makeLit(stat);
    }

    /* initialize */
    initializeThing()
    {
        /* inherit default handling */
        inherited();

        /* 
         *   Make sure our initial isLit setting (for the LightSource)
         *   matches our initial isOn steting (for the Switch).  The
         *   switch status drives the light source status, so initialize
         *   the latter from the former.  
         */
        isLit = isOn;
    }

    /* treat 'light' and 'extinguish' as 'turn on' and 'turn off' */
    dobjFor(Light) asDobjFor(TurnOn)
    dobjFor(Extinguish) asDobjFor(TurnOff)
;

