/**
 * \file ControlCharacter.C
 * This file is part of LyX, the document processor.
 * Licence details can be found in the file COPYING.
 *
 * \author Angus Leeming
 *
 * Full author contact details are available in file CREDITS.
 */

#include <config.h>

#include "ControlCharacter.h"
#include "ButtonController.h"

#include "buffer.h"
#include "bufferparams.h"
#include "bufferview_funcs.h"
#include "funcrequest.h"
#include "language.h"
#include "LColor.h"

using bv_funcs::font2string;

using std::string;

namespace lyx {
namespace frontend {

ControlCharacter::ControlCharacter(Dialog & parent)
	: Dialog::Controller(parent),
	  font_(0), toggleall_(false)
{}


bool ControlCharacter::initialiseParams(string const &)
{
	// Do this the first time only.
	if (!font_.get())
		font_.reset(new LyXFont(LyXFont::ALL_IGNORE));

	// so that the user can press Ok
	if (getFamily()   != LyXFont::IGNORE_FAMILY ||
	    getSeries()   != LyXFont::IGNORE_SERIES ||
	    getShape()    != LyXFont::IGNORE_SHAPE  ||
	    getSize()     != LyXFont::IGNORE_SIZE ||
	    getBar()      != IGNORE ||
	    getColor()    != LColor::ignore ||
	    font_->language() != ignore_language)
		dialog().bc().valid();

	return true;
}


void ControlCharacter::clearParams()
{}


void ControlCharacter::dispatchParams()
{
	// Nothing to dispatch. (Can be called from the Toolbar.)
	if (!font_.get())
		return;

	string data;
	if (font2string(*font_.get(), toggleall_, data)) {
		kernel().dispatch(FuncRequest(getLfun(), data));
	}
}


LyXFont::FONT_FAMILY ControlCharacter::getFamily() const
{
	if (!font_.get())
		return LyXFont::IGNORE_FAMILY;
	return font_->family();
}


void ControlCharacter::setFamily(LyXFont::FONT_FAMILY val)
{
	font_->setFamily(val);
}


LyXFont::FONT_SERIES ControlCharacter::getSeries() const
{
	if (!font_.get())
		return LyXFont::IGNORE_SERIES;
	return font_->series();
}


void ControlCharacter::setSeries(LyXFont::FONT_SERIES val)
{
	font_->setSeries(val);
}


LyXFont::FONT_SHAPE ControlCharacter::getShape() const
{
	if (!font_.get())
		return LyXFont::IGNORE_SHAPE;
	return font_->shape();
}


void ControlCharacter::setShape(LyXFont::FONT_SHAPE val)
{
	font_->setShape(val);
}


LyXFont::FONT_SIZE ControlCharacter::getSize() const
{
	if (!font_.get())
		return LyXFont::IGNORE_SIZE;
	return font_->size();
}


void ControlCharacter::setSize(LyXFont::FONT_SIZE val)
{
	font_->setSize(val);
}


FONT_STATE ControlCharacter::getBar() const
{
	if (!font_.get())
		return IGNORE;

	if (font_->emph() == LyXFont::TOGGLE)
		return EMPH_TOGGLE;

	if (font_->underbar() == LyXFont::TOGGLE)
		return UNDERBAR_TOGGLE;

	if (font_->noun() == LyXFont::TOGGLE)
		return NOUN_TOGGLE;

	if (font_->emph() == LyXFont::IGNORE &&
	    font_->underbar() == LyXFont::IGNORE &&
	    font_->noun() == LyXFont::IGNORE)
		return IGNORE;

	return INHERIT;
}


void ControlCharacter::setBar(FONT_STATE val)
{
	switch (val) {
	case IGNORE:
		font_->setEmph(LyXFont::IGNORE);
		font_->setUnderbar(LyXFont::IGNORE);
		font_->setNoun(LyXFont::IGNORE);
		break;

	case EMPH_TOGGLE:
		font_->setEmph(LyXFont::TOGGLE);
		break;

	case UNDERBAR_TOGGLE:
		font_->setUnderbar(LyXFont::TOGGLE);
		break;

	case NOUN_TOGGLE:
		font_->setNoun(LyXFont::TOGGLE);
		break;

	case INHERIT:
		font_->setEmph(LyXFont::INHERIT);
		font_->setUnderbar(LyXFont::INHERIT);
		font_->setNoun(LyXFont::INHERIT);
		break;
	}
}


LColor_color ControlCharacter::getColor() const
{
	if (!font_.get())
		return LColor::ignore;

	return font_->color();
}


void ControlCharacter::setColor(LColor_color val)
{
	switch (val) {
	case LColor::ignore:
	case LColor::none:
	case LColor::black:
	case LColor::white:
	case LColor::red:
	case LColor::green:
	case LColor::blue:
	case LColor::cyan:
	case LColor::magenta:
	case LColor::yellow:
	case LColor::inherit:
		font_->setColor(val);
		break;
	default:
		break;
	}
}


string ControlCharacter::getLanguage() const
{
	if (font_.get() && font_->language())
		return font_->language()->lang();
	return "ignore";
}


void ControlCharacter::setLanguage(string const & val)
{
	if (val == "ignore")
		font_->setLanguage(ignore_language);

	else if (val == "reset")
		font_->setLanguage(kernel().buffer().params().language);

	else
		font_->setLanguage(languages.getLanguage(val));
}


bool ControlCharacter::getToggleAll() const
{
	return toggleall_;
}


void ControlCharacter::setToggleAll(bool t)
{
	toggleall_ = t;
}

} // namespace frontend
} // namespace lyx
