// fragment.h             -*-c++-*-
//
// Copyright 2004 Daniel Burrows
//
// Fragments are pieces of text that live in a vs_text_layout widget.
// See vs_text_layout.h for details.

#ifndef FRAGMENT_H
#define FRAGMENT_H

#include "fragment_contents.h"

#include <string>
#include <vector>

/** A fragment represents a logical unit of text.
 */
class fragment
{
public:
  /** Return all the lines of this fragment, given a particular
   *  formatting width.
   *
   *  \param w the width to which this should be formatted, if
   *  applicable (an advisory value only; some fragments will ignore
   *  the width).
   *
   *  \return the lines of this fragment; the caller is responsible
   *  for deleting it.
   */
  virtual fragment_contents layout(size_t w)=0;

  /** Set the attributes of this fragment.
   *
   *  \param attr attributes assigned to this fragment
   */
  virtual void set_attr(int attr)=0;

  /** Nothing to do in the base class */
  virtual ~fragment();
};

// Factory methods to avoid cluttering the .h file:

/** Create a fragment from a string of text.  The text will simply be
 *  formatted as a single line without clipping.
 *
 *  \param s the text to use
 *
 *  \return the new fragment
 */
fragment *text_fragment(const chstring &s);

/** Create a fragment from a string of text.  The text will simply be
 *  formatted as a single line without clipping.
 *
 *  \param s the text to use
 *  \param attr attributes to assign to it
 *
 *  \return the new fragment
 */
fragment *text_fragment(const std::string &s,
			int attr=A_NORMAL);

/** Create a fragment from a string of text.  The text will simply be
 *  formatted as a single line without clipping.
 *
 *  \param s the text to use
 *  \param attr attributes to assign to it
 *
 *  \return the new fragment
 */
inline fragment *text_fragment(const char *s,
			       int attr=A_NORMAL)
{
  return text_fragment(std::string(s), attr);
}

/** Create a fragment which simply produces a newline wherever it occurs. */
fragment *newline_fragment();

/** Create a fragment which simply assigns a particular attribute to
 *  the text of its child fragment.  It's not clear that this is
 *  really necessary, but it might make it easier to support some
 *  layout models.
 *
 *  (presently this actually just sets the attributes of f and returns
 *  it, but this could also return a separate node)
 *
 *  \param f the child of this fragment
 *  \param attr the text attribute which should be assigned to f
 *
 *  \return the new fragment
 */
fragment *attr_fragment(fragment *f,
			int attr);

/** Create a fragment from a sequence of other fragments.
 *
 *  The fragment will simply "shove" the two sequences together,
 *  respecting the value of final_nl on each.
 *
 *  \todo can this be made more efficient?  It should be possible to
 *  just "plug" the sequences together if they're lists -- but that only
 *  works if they aren't cached elsewhere.
 *
 *  \param fragments the fragments in the sequence
 *
 *  \return the new fragment
 */
fragment *sequence_fragment(const std::vector<fragment *> &fragments);

/** Create a fragment from a sequence of other fragments.
 *
 *  The fragment will simply "shove" the two sequences together,
 *  respecting the value of final_nl on each.
 *
 *  \param f the first fragment in the sequence; the sequence should
 *           be terminated with a NULL pointer.
 *
 *  \return the new fragment
 */
fragment *sequence_fragment(fragment *f, ...);

/** Join fragments into a single fragment, placing text between them.
 *
 *  This is useful for creating lists, for instance.  The new fragment
 *  takes ownership of all pointers in fragments.
 *
 *  \param fragments the list of fragments to join
 *  \param between a string to place between adjacent items in the input list
 *  \param attr the attribute with which to display between
 */
fragment *join_fragments(const std::vector<fragment *> &fragments,
			 const std::string &between, int attr=A_NORMAL);

/** Create a flowbox.
 *
 *  Each line of the fragment placed inside the flowbox will be
 *  reflowed (word-wrapped) to the current width, possibly to
 *  several lines.
 *
 *  The contents of a flowbox always have final_nl=\b true. (ie, a
 *  flowbox is always followed by a line break)
 *
 *  \param contents the contents of the flowbox
 *
 *  \return the new flowbox
 */
fragment *flowbox(fragment *contents);

/** Create a fillbox.
 *
 *  Each line of the fragment placed inside the fillbox will be
 *  reflowed (word-wrapped) and expanded to the current width,
 *  possibly to several lines.
 *
 *  The contents of a fillbox always have final_nl=\b true.
 *
 *  \param contents the contents of the fillbox
 *
 *  \return the new fillbox
 */
fragment *fillbox(fragment *contents);

/** Create a clipbox.
 *
 *  Each line of the fragment placed inside the clipbox will be
 *  clipped to the current width.  The whole layout widget
 *  implicitly uses one of these, but there may be other uses for
 *  clipboxes as well.
 *
 *  \param contents the contents of the clipbox
 *
 *  \return the new clipbox
 */
fragment *clipbox(fragment *contents);

/** Create an indentbox.
 *
 *  Each line of the indentbox will be indented by the specified
 *  amount.  (this effectively decreases the width of each line) If
 *  desired, the first line can be indented a different amount
 *  (typically less) than the remaining lines, although it is
 *  formatted to the same width; this supports things like bulletted
 *  lists.
 *
 *  \param firstindent the number of spaces of indentation to use for the first line
 *  \param restindent the number of spaces of indentation to use for later lines
 *  \param contents the contents of the indentbox
 *  \return the new indentbox
 */
fragment *indentbox(size_t firstindent, size_t restindent, fragment *contents);

/** A printf-alike for fragments.
 *
 *  Formatting codes:
 *
 *  - %F: substitutes a fragment into the sequence being built
 *  - %s: substitutes a const char * into the sequence being built
 *  - %n: substitutes a newline fragment into the sequence being built
 *  - %%: inserts a literal %
 *  - %B/%b: enable/disable the bold character attribute
 *  - %R/%r: enable/disable the reverse video character attribute
 *  - %D/%d: enable/disable the dim character attribute
 *
 *  Note: if you use a parameter index multiple times, you are virtually
 *  GUARANTEED to segfault!
 *
 *  \param format the format string
 *  \return the formatted fragment, or NULL if there is an error in the format.
 */
fragment *fragf(const char *format, ...);

#endif
