\begin{code}
module ColourPrinter (colourPrinters, escapedPrinter, escapedPrinters) where
import External (getTermNColors)
import Printer (Printer, Printers(..), Printable(..), Color(..),
                invisiblePrinter, (<>), Doc, unsafeBoth, simplePrinter, hcat,
                unsafeText, unsafeChar, backslash, unsafePackedString)
import Char (isAscii, isPrint, isSpace, ord, intToDigit)
import System ( getEnv )
import System.IO.Unsafe ( unsafePerformIO )
import FastPackedString ( packString, unpackPS, anyPS )

colourPrinters :: IO (Printers)
escapedPrinters, realColourPrinters :: Printers
realColourPrinters  = Printers { colorP = colorPrinter,
                                 invisibleP = invisiblePrinter,
                                 defP = escapedPrinter
                               }
escapedPrinters = Printers { colorP = \_ -> escapedPrinter,
                             invisibleP = invisiblePrinter,
                             defP = escapedPrinter
                           }

escapedPrinter :: Printer
escapedPrinter
 = \p -> case p of
         S s -> escape s
         PS ps -> if anyPS (not . no_escape) ps
                  then escape $ unpackPS ps
                  else unsafePackedString ps
         Both s _ -> escape s

colorPrinter :: Color -> Printer
colorPrinter Blue  = make_blue  . simplePrinter
colorPrinter Red   = make_red   . simplePrinter
colorPrinter Green = make_green . simplePrinter

colourPrinters = do num_colors <- getTermNColors
                    if num_colors > 4 then return realColourPrinters
                                      else return escapedPrinters
-- escape assumes the input is in ['\0'..'\255']
escape :: String -> Doc
escape x = hcat (map escape' x)
    where escape' '\x1B' = make_blue (unsafeText "^[")
          escape' c
           | no_escape c = unsafeChar c
           | otherwise
              = let (q, r) = quotRem (ord c) 16
                in make_blue $
                   backslash <> unsafeText [intToDigit q, intToDigit r]

no_escape :: Char -> Bool
no_escape c = (isAscii c || trustIsPrint) && (isPrint c || isSpace c)

make_blue :: Doc -> Doc
make_blue x = unsafeBoth "\x1B[01;34m" (packString "\x1B[01;34m")
           <> x
           <> reset_colour

make_red :: Doc -> Doc
make_red x = unsafeBoth "\x1B[01;31m" (packString "\x1B[01;31m")
          <> x
          <> reset_colour

make_green :: Doc -> Doc
make_green x = unsafeBoth "\x1B[01;32m" (packString "\x1B[01;32m")
            <> x
            <> reset_colour

reset_colour :: Doc
reset_colour = unsafeBoth "\x1B[00m" (packString "\x1B[00m")

trustIsPrint :: Bool
trustIsPrint =
    unsafePerformIO $ do
        n <- getEnv "DARCS_USE_ISPRINT" `catch` \_ -> return "0"
        return $ n /= "0"

\end{code}
