//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: TransformMatrix.cpp,v 1.2 2003/05/04 18:09:01 delpinux Exp $

#include <TransformMatrix.hpp>
#include <cmath>

#include <sstream>

//! Applies the TransformMatrix to the vector \a v.
TinyVector<3>& TransformMatrix::operator()(TinyVector<3>& v) const
{
  v = __matrix*v;
  v += __translate;
  return (v);
}

//! Applies the inverse TransformMatrix to the vector \a v.
TinyVector<3>& TransformMatrix::inverse(TinyVector<3>& v) const
{
  // As 'matrix' is composed of transformMatrixs, and
  // as transformMatrixs are othogonal matrices
  // the inverse of 'matrix' is its transposed
  TinyVector<3> Temporary = v;
  Temporary -= __translate;
  v = __invMatrix*Temporary;
  return v;
}

/*! Constructs the TransformMatrix for a set of given angles given in \a v by
  builting the invert of the TransformMatrix matrix since it is more usefull than the
  TransformMatrix matrix.
  \remark The angles contained in r are expressed using degrees.
 */
TransformMatrix::TransformMatrix(const real_t m[12])
{
  __type = matrix;

  for (size_t i=0;i<3;++i) {
    __translate[i] = m[9+i];
    for (size_t j=0; j<3; ++j) {
      __matrix(i,j) = m[3*i+j];
    }
  }

  __invMatrix = __matrix.invert();
}

/*! Default constructor: the TransformMatrix is the identity.
  \warning this might not be pertinent and may be removed.
*/
TransformMatrix::TransformMatrix()
{
  __type = matrix;

  __translate = 0;
  
  __matrix(0,0) = 1;
  __matrix(1,0) = 0;
  __matrix(2,0) = 0;
  __matrix(0,1) = 0;
  __matrix(1,1) = 1;
  __matrix(2,1) = 0;
  __matrix(0,2) = 0;
  __matrix(1,2) = 0;
  __matrix(2,2) = 1;

  __invMatrix = __matrix;
}

//! Copy constructor.
TransformMatrix::TransformMatrix(const TransformMatrix& r)
{
  __type = matrix;

  __translate = r.__translate;
  __matrix    = r.__matrix;
  __invMatrix = r.__invMatrix;
}

//! prints transformMatrix information to a string.
std::string TransformMatrix::povWrite() const
{
  std::stringstream povs;
  povs << "transformMatrix <";
  for (size_t i=0; i<3; ++i) {
    for(size_t j=0; j<3; ++j) {
      povs << __matrix(i,j) << ',';
    }
  }
  for (size_t i=0; i<2; ++i)
    povs << __translate << ',';
  povs << __translate[2];
  povs << ">";
  povs << std::ends;
  return povs.str();
}

