//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: BoundaryExpression.cpp,v 1.4 2004/04/16 16:25:41 delpinux Exp $

#include <MeshExpression.hpp>

#include <Mesh.hpp>
#include <SurfaceMeshOfTriangles.hpp>

#include <BoundaryExpression.hpp>

#include <Boundary.hpp>

#include <BoundaryPOVRay.hpp>
#include <BoundarySurfaceMesh.hpp>
#include <BoundaryReferences.hpp>
#include <BoundaryList.hpp>

ReferenceCounting<Boundary>
BoundaryExpression::boundary()
{
  return __boundary;
}

BoundaryExpression::BoundaryExpression(const BoundaryExpression& e)
  : Expression(Expression::boundary),
    __boundaryType(e.__boundaryType)
{
  ;
}

BoundaryExpression::BoundaryExpression(const BoundaryExpression::BoundaryType& t)
  : Expression(Expression::boundary),
    __boundaryType(t)
{
  ;
}

BoundaryExpression::~BoundaryExpression()
{
  ;
}

std::ostream& BoundaryExpressionSurfaceMesh::put(std::ostream& os) const
{
  os << "Surface Mesh";
  return os;
}

void BoundaryExpressionSurfaceMesh::execute()
{
  (*__surfaceMeshExpression).execute();

  Mesh* m = (*__surfaceMeshExpression).mesh();

  assert((*m).type() == Mesh::surfaceMeshTriangles);
  __boundary
    = new BoundarySurfaceMesh(dynamic_cast<SurfaceMeshOfTriangles*>(m));
}

BoundaryExpressionSurfaceMesh::
BoundaryExpressionSurfaceMesh(ReferenceCounting<MeshExpression> m)
  : BoundaryExpression(BoundaryExpression::surfaceMesh),
    __surfaceMeshExpression(m)
{
  ;
}

BoundaryExpressionSurfaceMesh::
BoundaryExpressionSurfaceMesh(const BoundaryExpressionSurfaceMesh& m)
  : BoundaryExpression(m),
    __surfaceMeshExpression(m.__surfaceMeshExpression)
{
  ;
}

BoundaryExpressionSurfaceMesh::
~BoundaryExpressionSurfaceMesh()
{
  ;
}

std::ostream& BoundaryExpressionPOVRay::put(std::ostream& os) const
{
  os << (*__povrayReference);
  return os;
}

void BoundaryExpressionPOVRay::execute()
{
  (*__povrayReference).execute();

  BoundaryPOVRay* b = new BoundaryPOVRay();

  TinyVector<3> ref;
  for (size_t i=0; i<3; ++i)
    ref[i] = (*__povrayReference).value(i);

  (*b).addPOVRef(ref);
  __boundary = b;
}


void BoundaryExpressionReferences::execute()
{
  for (BoundaryExpressionReferences::ReferencesSet::iterator i
	 = __references.begin();
       i != __references.end(); ++i)
    {
      ReferenceCounting<RealExpression> ref = *i;
      (*ref).execute();
    }

  BoundaryReferences* b= new BoundaryReferences();

  for (BoundaryExpressionReferences::ReferencesSet::iterator i
	 = __references.begin();
       i != __references.end(); ++i) {
    ReferenceCounting<RealExpression> ref = *i;
    const double value = (*ref).realValue();
    if ((value<0) or (value != int(value))) {
      fferr(0) << value << " is not a correct boundary reference\n";
      fferr(0) << "Boundary references must be positive integers.\n";
      std::exit(0);
    }
    b->add(static_cast<size_t>(value));
  }
  __boundary = b;
}

std::ostream& BoundaryExpressionReferences::put(std::ostream& os) const
{
  for (BoundaryExpressionReferences::ReferencesSet::const_iterator i
	 = __references.begin();
       i != __references.end(); ++i) {
    os << (**i) << ',';
  }
  os << "\b ";
  return os;
}

void
BoundaryExpressionList::execute()
{
  BoundaryList* b = new BoundaryList();

  for(List::iterator i = __boundaries.begin();
      i != __boundaries.end(); ++i) {
    (**i).execute();
    b->add((**i).boundary());
  }
  __boundary = b;
}

std::ostream&
BoundaryExpressionList::put(std::ostream& os) const
{
  for (List::const_iterator i = __boundaries.begin();
       i != __boundaries.end(); ++i) {
    os << (**i) << ',';
  }
  os << "\b ";
  return os;
}
