//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: PDE.hpp,v 1.2 2003/09/20 21:31:09 delpinux Exp $

// This class describes a PDE

#ifndef PDE_HPP
#define PDE_HPP

#include <VectorialPDEOperator.hpp>
#include <UserFunction.hpp>

/*!
  \class PDE

  This class describles a scalar PDE. But the unknown \a u can be part of a
  PDE system so be a vector function.

  \author Stphane Del Pino
*/
class PDE
{
private:
  //! The PDEOperator applied to u
  ConstReferenceCounting<VectorialPDEOperator> __vectorPDEOperator;

  //! The second member of the equation.
  ConstReferenceCounting<UserFunction> __secondMember;

public:
  ConstReferenceCounting<PDE> operator * (ConstReferenceCounting<UserFunction> u) const
  {
    PDE* newPDE = new PDE();
    (*newPDE).__secondMember = new MulUserFunction(__secondMember, u);
    (*newPDE).__vectorPDEOperator = (*__vectorPDEOperator) * u;
    return newPDE;
  }

  //! read-only access to the second member function
  ConstReferenceCounting<UserFunction> secondMember() const
  {
    return __secondMember;
  }

  // Access to the ith component of the vectorial PDE operator
  ConstReferenceCounting<PDEOperatorSum> operator [] (const size_t i) const
  {
    assert (i<(*__vectorPDEOperator).size());
    return (*__vectorPDEOperator)[i];
  }

  PDE()
  {
    ;
  }

  PDE(const VectorialPDEOperator& A,
      const UserFunction& f)
    : __vectorPDEOperator(&A),
      __secondMember(&f)
  {
    ;
  }

  PDE(const PDE& pde)
    : __vectorPDEOperator(pde.__vectorPDEOperator),
      __secondMember(pde.__secondMember)
  {
    ;
  }

  ~PDE()
  {
    ;
  }

  const PDE& operator = (const PDE& pde)
  {
    __vectorPDEOperator=pde.__vectorPDEOperator;
    __secondMember = pde.__secondMember;
    return *this;
  }

  friend std::ostream& operator << (std::ostream& os,
				    const PDE& pde)
  {
    os << *(pde.__vectorPDEOperator) << " = " << "pde.secondMember" << '\n';
    return os;
  }

};

#endif // PDE_HPP
