//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: QuadratureFormula.hpp,v 1.3 2004/03/27 16:16:06 delpinux Exp $

#ifndef QUADRATURE_FORMULA_HPP
#define QUADRATURE_FORMULA_HPP

#include <TinyVector.hpp>
#include <StaticBase.hpp>

class QuadratureFormulaQ1Hexahedron
  : public StaticBase<QuadratureFormulaQ1Hexahedron>
{
public:
  enum {
    numberOfQuadraturePoints = 8
  };
private:
  TinyVector<numberOfQuadraturePoints,TinyVector<3> > __integrationVertices;

public:
  const TinyVector<3>& operator[](const size_t& i) const
  {
    return __integrationVertices[i];
  }

  const TinyVector<numberOfQuadraturePoints,TinyVector<3> >& vertices() const
  {
    return __integrationVertices;
  }

  size_t numberOfVertices() const
  {
    return numberOfQuadraturePoints;
  }

  real_t weight(const size_t& i) const
  {
    return 1./numberOfQuadraturePoints;
  }

  QuadratureFormulaQ1Hexahedron()
  {
    this->__setQuadratureVertices();
  }

private:
  void __setQuadratureVertices()
  {
    const real_t xm =  0;
    const real_t ym =  0;
    const real_t zm =  0;

    static const real_t minus = 0.5 - std::sqrt(3.)/6.;
    static const real_t plus  = 0.5 + std::sqrt(3.)/6.;

    // Computes integration vertices.
    const real_t Xm = xm + minus;
    const real_t Xp = xm + plus;
    const real_t Ym = ym + minus;
    const real_t Yp = ym + plus;
    const real_t Zm = zm + minus;
    const real_t Zp = zm + plus;

    __integrationVertices[0][0] = Xm;
    __integrationVertices[0][1] = Ym;
    __integrationVertices[0][2] = Zm;

    __integrationVertices[1][0] = Xp;
    __integrationVertices[1][1] = Ym;
    __integrationVertices[1][2] = Zm;

    __integrationVertices[2][0] = Xp;
    __integrationVertices[2][1] = Yp;
    __integrationVertices[2][2] = Zm;

    __integrationVertices[3][0] = Xm;
    __integrationVertices[3][1] = Yp;
    __integrationVertices[3][2] = Zm;

    __integrationVertices[4][0] = Xm;
    __integrationVertices[4][1] = Ym;
    __integrationVertices[4][2] = Zp;

    __integrationVertices[5][0] = Xp;
    __integrationVertices[5][1] = Ym;
    __integrationVertices[5][2] = Zp;

    __integrationVertices[6][0] = Xp;
    __integrationVertices[6][1] = Yp;
    __integrationVertices[6][2] = Zp;

    __integrationVertices[7][0] = Xm;
    __integrationVertices[7][1] = Yp;
    __integrationVertices[7][2] = Zp;
  }
};

class QuadratureFormulaP1Tetrahedron
  : public StaticBase<QuadratureFormulaP1Tetrahedron>
{
public:
  enum {
    numberOfQuadraturePoints = 4
  };

private:
  TinyVector<numberOfQuadraturePoints, TinyVector<3> > __integrationVertices;

public:
  const TinyVector<3>& operator[](const size_t& i) const
  {
    return __integrationVertices[i];
  }

  const TinyVector<numberOfQuadraturePoints,TinyVector<3> >& vertices() const
  {
    return __integrationVertices;
  }

  size_t numberOfVertices() const
  {
    return numberOfQuadraturePoints;
  }

  real_t weight(const size_t& i) const
  {
    return 1./numberOfQuadraturePoints;
  }

  QuadratureFormulaP1Tetrahedron()
  {
    this->__setQuadratureVertices();
  }
private:
  void __setQuadratureVertices()
  {
    TinyVector<4, real_t> mu;
    mu[0] = 0.585410196624969;
    mu[1] = 0.138196601125011;
    mu[2] = 0.138196601125011;
    mu[3] = 0.138196601125011;

    TinyVector<4, TinyVector<3,real_t> > vertex;
    vertex[0] = TinyVector<3, real_t>(0,0,0);
    vertex[1] = TinyVector<3, real_t>(1,0,0);
    vertex[2] = TinyVector<3, real_t>(0,1,0);
    vertex[3] = TinyVector<3, real_t>(0,0,1);

    for (size_t i=0; i<4; ++i) {
      __integrationVertices[i] = 0;
      for (size_t j=0; j<4; ++j) {
	__integrationVertices[i] += mu[(i-j)%4]*vertex[j];
      }
    }
  }
};

class QuadratureFormulaP1Triangle3D
  : public StaticBase<QuadratureFormulaP1Triangle3D>
{
public:
  enum {
    numberOfQuadraturePoints = 3
  };

private:
  TinyVector<3,TinyVector<3> > __integrationVertices;

public:
  const TinyVector<3>& operator[](const size_t& i) const
  {
    return __integrationVertices[i];
  }

  const TinyVector<numberOfQuadraturePoints, TinyVector<3> >& vertices() const
  {
    return __integrationVertices;
  }

  size_t numberOfVertices() const
  {
    return numberOfQuadraturePoints;
  }

  real_t weight(const size_t& i) const
  {
    return 1./numberOfQuadraturePoints;
  }

  QuadratureFormulaP1Triangle3D()
  {
    this->__setQuadratureVertices();
  }
private:
  void __setQuadratureVertices()
  {
    __integrationVertices[0][0] = 0;
    __integrationVertices[0][1] = 0.5;
    __integrationVertices[0][2] = 0;

    __integrationVertices[1][0] = 0.5;
    __integrationVertices[1][1] = 0;
    __integrationVertices[1][2] = 0;

    __integrationVertices[2][0] = 0.5;
    __integrationVertices[2][1] = 0.5;
    __integrationVertices[2][2] = 0;
  }
};

class QuadratureFormulaQ1Quadrangle3D
  : public StaticBase<QuadratureFormulaQ1Quadrangle3D>
{
public:
  enum {
    numberOfQuadraturePoints = 4
  };

private:
  TinyVector<numberOfQuadraturePoints, TinyVector<3> > __integrationVertices;

  void setQuadratureVertices();

public:
  const TinyVector<3>& operator[](const size_t& i) const
  {
    return __integrationVertices[i];
  }

  const TinyVector<numberOfQuadraturePoints, TinyVector<3> >& vertices() const
  {
    return __integrationVertices;
  }

  size_t numberOfVertices() const
  {
    return numberOfQuadraturePoints;
  }

  real_t weight(const size_t& i) const
  {
    return 1./numberOfQuadraturePoints;
  }

  QuadratureFormulaQ1Quadrangle3D()
  {
    this->__setQuadratureVertices();
  }

private:
  void __setQuadratureVertices()
  {
    const real_t xm =  0;
    const real_t ym =  0;

    static const real_t minus = 0.5 - std::sqrt(3.)/6.;
    static const real_t plus  = 0.5 + std::sqrt(3.)/6.;

    // Computes integration vertices.
    const real_t Xm = xm + minus;
    const real_t Xp = xm + plus;
    const real_t Ym = ym + minus;
    const real_t Yp = ym + plus;

    __integrationVertices[0][0] = Xm;
    __integrationVertices[0][1] = Ym;
    __integrationVertices[0][2] = 0;

    __integrationVertices[1][0] = Xp;
    __integrationVertices[1][1] = Ym;
    __integrationVertices[1][2] = 0;

    __integrationVertices[2][0] = Xp;
    __integrationVertices[2][1] = Yp;
    __integrationVertices[2][2] = 0;

    __integrationVertices[3][0] = Xm;
    __integrationVertices[3][1] = Yp;
    __integrationVertices[3][2] = 0;
  }
};

#endif // QUADRATURE_FORMULA_HPP

