/* GNOME DB library
 * Copyright (C) 1999-2000 Rodrigo Moya
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "config.h"
#include "e-hpaned.h"
#include "gnome-db-designer.h"
#include "gnome-db-util.h"
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-stock.h>

static void gnome_db_designer_init (GnomeDbDesigner *designer);
static void gnome_db_designer_class_init (GnomeDbDesignerClass *klass);

static void gnome_db_designer_destroy (GtkObject *object);

static void table_add_field_cb (GtkWidget *w, gpointer user_data);

struct _GnomeDbDesignerPrivate {
	GtkWidget*      object_tree;
	GtkCTreeNode*   tables_node;
	GtkCTreeNode*   views_node;
	GtkWidget*      detail_container;
	GtkWidget*      detail;
	GdaXmlDatabase* xmldb;
	gchar*          filename;
};

static GnomeUIInfo table_detail_popup_menu[] = {
	{ GNOME_APP_UI_ITEM, N_("Add field..."), N_("Add a new field to this table"),
	  table_add_field_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_NEW, 0, 0, NULL },
	{ GNOME_APP_UI_ITEM, N_("Edit field..."), N_("Edit the selected field properties"),
	  NULL, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_PROP, 0, 0, NULL },
	GNOMEUIINFO_END
};

/*
 * Private functions
 */
static void
add_object_nodes (GnomeDbDesigner *designer)
{
	gchar*       ctree_row[2] = { "", "" };
	
	ctree_row[1] = _("Views");
	designer->priv->views_node =
		gtk_ctree_insert_node(GTK_CTREE(designer->priv->object_tree),
				      NULL, NULL,
				      ctree_row, 0,
				      NULL,
				      NULL,
				      NULL,
				      NULL,
				      FALSE, TRUE);
	gtk_ctree_expand(GTK_CTREE(designer->priv->object_tree),
			 designer->priv->views_node);
	ctree_row[1] = _("Tables");
	designer->priv->tables_node =
		gtk_ctree_insert_node(GTK_CTREE(designer->priv->object_tree),
				      NULL, NULL,
				      ctree_row, 0,
				      NULL,
				      NULL,
				      NULL,
				      NULL,
				      FALSE, TRUE);
	gtk_ctree_expand(GTK_CTREE(designer->priv->object_tree),
			 designer->priv->tables_node);
}

static const gchar *
get_current_name_from_tree (GnomeDbDesigner *designer)
{
	GList *selection;
	gint row;
	xmlNodePtr xmlnode;

	g_return_val_if_fail(GNOME_DB_IS_DESIGNER(designer), NULL);
	g_return_val_if_fail(designer->priv != NULL, NULL);
	g_return_val_if_fail(GTK_IS_CTREE(designer->priv->object_tree), NULL);

	selection = GTK_CLIST(designer->priv->object_tree)->selection;
	if (selection) {
		row = GPOINTER_TO_INT(selection->data);
		xmlnode = (xmlNodePtr) gtk_ctree_node_get_row_data(designer->priv->object_tree, row);
		if (xmlnode)
			return (const gchar *) xmlGetProp(xmlnode, "name");
	}
	return NULL;
}

static GtkWidget *
show_table_detail (GnomeDbDesigner *designer, xmlNodePtr xmlnode)
{
	GtkWidget* container;
	GtkWidget* frame;
	GtkWidget* table;
	GtkWidget* label;
	GtkWidget* entry;
	GtkWidget* scroll;
	GtkWidget* clist;
	gint       field_count;
	gint       cnt;
	const gchar*clist_titles[] = { N_("Name"), N_("GDA Type"), N_("Size"), N_("Scale") };

	g_return_val_if_fail(GNOME_DB_IS_DESIGNER(designer), NULL);
	g_return_val_if_fail(xmlnode != NULL, NULL);

	/* create container */
	container = gnome_db_new_table_widget(1, 4, FALSE);
	
	/* detail pane */
	frame = gnome_db_new_frame_widget(NULL);
	gtk_table_attach(GTK_TABLE(container), frame, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
	table = gnome_db_new_table_widget(3, 3, FALSE);
	gtk_container_add(GTK_CONTAINER(frame), table);
	
	label = gnome_db_new_label_widget(_("Table name"));
	gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
	entry = gnome_db_new_entry_widget(0, FALSE);
	gtk_entry_set_text(GTK_ENTRY(entry),
	                   gda_xml_database_table_get_name(designer->priv->xmldb, xmlnode));
	gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
	
	label = gnome_db_new_label_widget(_("Owner"));
	gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 3, 3);
	entry = gnome_db_new_entry_widget(0, FALSE);
	gtk_entry_set_text(GTK_ENTRY(entry),
	                   gda_xml_database_table_get_owner(designer->priv->xmldb, xmlnode));
	gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 1, 2, GTK_FILL, GTK_FILL, 3, 3);
	
	/* create description area */
	scroll = gnome_db_new_scrolled_window_widget();
	gtk_table_attach(GTK_TABLE(container), scroll, 0, 1, 1, 4,
	                 GTK_FILL | GTK_EXPAND | GTK_SHRINK,
	                 GTK_FILL | GTK_EXPAND | GTK_SHRINK,
	                 3, 3);
	clist = gnome_db_new_clist_widget(clist_titles,
	                                  sizeof(clist_titles) / sizeof(clist_titles[0]));
	gtk_object_set_data(GTK_OBJECT(clist),
			    "GNOME_DB_DesignerWidget",
			    (gpointer) designer);
	
	gtk_container_add(GTK_CONTAINER(scroll), clist);
	gnome_db_new_popup_menu(clist, table_detail_popup_menu, (gpointer) clist);
	field_count = gda_xml_database_table_field_count(designer->priv->xmldb, xmlnode);
	gtk_clist_freeze(GTK_CLIST(clist));
	for (cnt = 0; cnt < field_count; cnt++) {
		gchar*     row[4];
		xmlNodePtr field;
		gint       row_number;
		
		field = gda_xml_database_table_get_field(designer->priv->xmldb, xmlnode, cnt);
		if (field) {
			row[0] = (gchar *) gda_xml_database_field_get_name(designer->priv->xmldb, field);
			row[1] = (gchar *) gda_xml_database_field_get_gdatype(designer->priv->xmldb, field);
			row[2] = g_strdup_printf("%d",
						 gda_xml_database_field_get_size(designer->priv->xmldb, field));
			row[3] = g_strdup_printf("%d",
						 gda_xml_database_field_get_scale(designer->priv->xmldb, field));
			row_number = gtk_clist_append(GTK_CLIST(clist), row);
			gtk_clist_set_row_data(GTK_CLIST(clist), row_number, (gpointer) field);
			g_free((gpointer) row[2]);
			g_free((gpointer) row[3]);
		}
	}
	gtk_clist_thaw(GTK_CLIST(clist));
	
	return container;
}

/*
 * Callbacks
 */
static void
select_tree_row_cb (GtkCTree *ctree, GtkCTreeNode *row, gint column, GnomeDbDesigner *designer)
{
	xmlNodePtr xmlnode;

	g_return_if_fail(GTK_IS_CTREE(ctree));
	g_return_if_fail(row != NULL);
	g_return_if_fail(GNOME_DB_IS_DESIGNER(designer));

	/* clear the current detail */
	if (GTK_IS_WIDGET(designer->priv->detail))
		gtk_widget_destroy(designer->priv->detail);
	designer->priv->detail = NULL;
	
	xmlnode = (xmlNodePtr) gtk_ctree_node_get_row_data(ctree, row);
	if (xmlnode) {
		if (gda_xml_database_table_find (designer->priv->xmldb,
						 xmlGetProp (xmlnode, "name"))) {
			designer->priv->detail = show_table_detail(designer, xmlnode);
		}
		
		if (GTK_IS_WIDGET(designer->priv->detail)) {
			gtk_widget_show(designer->priv->detail);
			gtk_box_pack_start(GTK_BOX(designer->priv->detail_container),
					   designer->priv->detail, TRUE, TRUE, 0);
		}
	}
}

static void
table_add_field_cb (GtkWidget *w, gpointer user_data)
{
	GnomeDbDesigner *designer;
	GtkCList *clist = (GtkCList *) user_data;
	gchar *table_name;

	g_return_if_fail(GTK_IS_CLIST(clist));

	designer = gtk_object_get_data (GTK_OBJECT (clist),
					"GNOME_DB_DesignerWidget");
	table_name = get_current_name_from_tree(designer);
	if (GNOME_DB_IS_DESIGNER(designer) && table_name) {
		GtkWidget *dialog;
		GtkWidget *table;
		GtkWidget *label;
		GtkWidget *name_entry;

		/* create dialog */
		dialog = gnome_dialog_new(_("Add field"),
					  GNOME_STOCK_BUTTON_OK,
					  GNOME_STOCK_BUTTON_CANCEL,
					  NULL);
		gnome_dialog_set_default(GNOME_DIALOG(dialog), 0);
		table = gnome_db_new_table_widget(3, 4, FALSE);
		gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), table, TRUE, TRUE, 0);

		label = gnome_db_new_label_widget(_("Name"));
		gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
		gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
		name_entry = gnome_db_new_entry_widget(0, TRUE);
		gtk_table_attach(GTK_TABLE(table), name_entry, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 3, 3);

		/* run the dialog */
	run_dialog:
		if (!gnome_dialog_run(GNOME_DIALOG(dialog))) {
			gchar *name_str;
			GdaXmlDatabaseTable *xml_table;
			GdaXmlDatabaseField *xml_field;

			/* check data entered by user */
			name_str = gtk_entry_get_text(GTK_ENTRY(name_entry));
			if (!name_str || strlen(name_str) <= 0) {
				gnome_db_show_error(_("You must specify a name for the new field"));
				goto run_dialog;
			}

			xml_table = gda_xml_database_table_find(designer->priv->xmldb, table_name);
			if (!xml_table) {
				gnome_db_show_error(_("Could not find table %s"), table_name);
				goto run_dialog;
			}
			xml_field = gda_xml_database_table_get_field(designer->priv->xmldb,
								     xml_table,
								     name_str);
			if (xml_field) {
				gnome_db_show_error(_("There is already a field named %s"), name_str);
				goto run_dialog;
			}

			/* add the field to the database */
			xml_field = gda_xml_database_table_add_field(designer->priv->xmldb,
								     xml_table,
								     name_str);
			if (GTK_IS_WIDGET(designer->priv->detail))
				gtk_widget_destroy(designer->priv->detail);
			designer->priv->detail = show_table_detail(designer, xml_table);
		}
		gnome_dialog_close(GNOME_DIALOG(dialog));
	}
}

static void
xmldb_changed_cb (GdaXmlDatabase *xmldb, gpointer user_data)
{
	GnomeDbDesigner *designer = (GnomeDbDesigner *) user_data;

	g_return_if_fail (GNOME_DB_IS_DESIGNER (designer));
	gnome_db_designer_refresh (designer);
}

/*
 * GnomeDbDesigner widget interface
 */
static void
gnome_db_designer_class_init (GnomeDbDesignerClass *klass)
{
	GtkObjectClass* object_class = GTK_OBJECT_CLASS(klass);
	
	object_class->destroy = gnome_db_designer_destroy;
}

static void
gnome_db_designer_init (GnomeDbDesigner *designer)
{
	GtkWidget* pane;
	GtkWidget* frame;
	GtkWidget* scroll;

	/* allocate private structure */
	designer->priv = g_new0(GnomeDbDesignerPrivate, 1);

	/* create main container */
	pane = e_hpaned_new ();
	gtk_container_set_border_width (GTK_CONTAINER (pane), 5);
	gtk_widget_show (pane);
	gtk_box_pack_start (GTK_BOX (designer), pane, 1, 1, 0);

	frame = gnome_db_new_frame_widget (NULL);
	e_paned_pack1(E_PANED(pane), frame, FALSE, FALSE);
	scroll = gnome_db_new_scrolled_window_widget();
	gtk_container_add(GTK_CONTAINER(frame), scroll);
	designer->priv->object_tree = gnome_db_new_ctree_widget(NULL, 2);
	add_object_nodes(designer);
	gtk_container_add(GTK_CONTAINER(scroll), designer->priv->object_tree);
	gtk_signal_connect_after(GTK_OBJECT(designer->priv->object_tree),
	                         "tree_select_row",
	                         GTK_SIGNAL_FUNC(select_tree_row_cb),
	                         (gpointer) designer);

	/* create detail pane */
	designer->priv->detail_container = gtk_vbox_new(FALSE, 0);
	gtk_widget_show(designer->priv->detail_container);
	e_paned_pack2(E_PANED(pane), designer->priv->detail_container, TRUE, FALSE);

	e_paned_set_position(E_PANED(pane), 130);

	/* initialize fields */
	designer->priv->detail = NULL;
	designer->priv->xmldb = NULL;
}

GtkType
gnome_db_designer_get_type (void)
{
	static guint db_designer_type = 0;
	
	if (!db_designer_type) {
		GtkTypeInfo db_designer_info = {
			"GnomeDbDesigner",
			sizeof (GnomeDbDesigner),
			sizeof (GnomeDbDesignerClass),
			(GtkClassInitFunc) gnome_db_designer_class_init,
			(GtkObjectInitFunc) gnome_db_designer_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};
		db_designer_type = gtk_type_unique(gtk_vbox_get_type(), &db_designer_info);
	}
	return (db_designer_type);
}

/**
 * gnome_db_designer_new
 *
 * Create a new #GnomeDbDesigner widget. This widget lets you visually
 * manage a XML database
 */
GtkWidget *
gnome_db_designer_new (void)
{
	GnomeDbDesigner* designer;
	
	designer = GNOME_DB_DESIGNER (gtk_type_new (gnome_db_designer_get_type ()));
	designer->priv->xmldb = gda_xml_database_new ();
	
	return GTK_WIDGET (designer);
}

/**
 * gnome_db_designer_new_from_file
 */
GtkWidget *
gnome_db_designer_new_from_file (const gchar *filename)
{
	GnomeDbDesigner* designer;
	
	designer = GNOME_DB_DESIGNER (gnome_db_designer_new ());
	gnome_db_designer_load_file (designer, filename);
	
	return GTK_WIDGET (designer);
}

/**
 * gnome_db_designer_new_from_xml
 * @xmldb: XML database
 *
 * Create a new #GnomeDbDesigner widget from the given XML database
 */
GtkWidget *
gnome_db_designer_new_from_xml (GdaXmlDatabase *xmldb)
{
	GnomeDbDesigner* designer;
	
	designer = GNOME_DB_DESIGNER(gtk_type_new(gnome_db_designer_get_type()));
	gtk_object_ref (GTK_OBJECT (xmldb));
	designer->priv->xmldb = xmldb;
	gtk_signal_connect (GTK_OBJECT (designer->priv->xmldb),
			    "changed",
			    GTK_SIGNAL_FUNC (xmldb_changed_cb),
			    designer);
	gnome_db_designer_refresh(designer);

	return GTK_WIDGET(designer);
}

static void
gnome_db_designer_destroy (GtkObject *object)
{
	GtkObjectClass *parent_class;
	GnomeDbDesigner *designer;
	
	g_return_if_fail(GNOME_DB_IS_DESIGNER(object));

	/* free memory */
	designer = GNOME_DB_DESIGNER(object);

	g_free((gpointer) designer->priv->filename);
	gda_xml_database_free(designer->priv->xmldb);

	g_free((gpointer) designer->priv);

	/* call parent class destructor */
	parent_class = gtk_type_class(gtk_vbox_get_type());
	if (parent_class && parent_class->destroy)
		parent_class->destroy(object);
}

/**
 * gnome_db_designer_load_file
 */
void
gnome_db_designer_load_file (GnomeDbDesigner *designer, const gchar *filename)
{
	GdaXmlDatabase *xmldb;
	
	g_return_if_fail(GNOME_DB_IS_DESIGNER(designer));
	g_return_if_fail(filename != NULL);
	
	xmldb = gda_xml_database_new_from_file(filename);
	if (GDA_IS_XML_DATABASE(xmldb)) {
		/* free the previous database, if any */
		if (GDA_IS_XML_DATABASE(designer->priv->xmldb)) {
			gtk_signal_disconnect_by_data (
				GTK_OBJECT (designer->priv->xmldb),
				designer);
			gda_xml_database_free(designer->priv->xmldb);
		}
		if (designer->priv->filename)
			g_free((gpointer) designer->priv->filename);

		designer->priv->xmldb = xmldb;
		gtk_signal_connect (GTK_OBJECT (designer->priv->xmldb),
				    "changed",
				    GTK_SIGNAL_FUNC (xmldb_changed_cb),
				    designer);
		designer->priv->filename = g_strdup(filename);
		gnome_db_designer_refresh(designer);
	}
	else
		gnome_db_show_error(_("File %s is not a valid XML file"), filename);
}

/**
 * gnome_db_designer_save
 * @designer: the #GnomeDbDesigner widget
 *
 * Saves the database being edited by the given #GnomeDbDesigner widget
 * to disk. The current file name will be used, or, if no filename has
 * been specified, a dialog box will pop up to ask the user about a
 * file name
 *
 * Returns: TRUE if successful, FALSE otherwise
 */
gboolean
gnome_db_designer_save (GnomeDbDesigner *designer)
{
	g_return_val_if_fail(GNOME_DB_IS_DESIGNER(designer), FALSE);
	g_return_val_if_fail(designer->priv != NULL, FALSE);

	if (!designer->priv->filename) {
		gchar *filename = gnome_db_select_file(_("Select file"));
		if (!filename) {
			return TRUE;
		}
		designer->priv->filename = filename;
	}

	gda_xml_database_save(designer->priv->xmldb, designer->priv->filename);
	return TRUE;
}

/**
 * gnome_db_designer_refresh
 */
void
gnome_db_designer_refresh (GnomeDbDesigner *designer)
{
	GList *list;
	GList *l;
	GnomePixmap *pixmap;
	
	g_return_if_fail(GNOME_DB_IS_DESIGNER(designer));

	gtk_ctree_remove_node(GTK_CTREE(designer->priv->object_tree), designer->priv->tables_node);
	gtk_ctree_remove_node(GTK_CTREE(designer->priv->object_tree), designer->priv->views_node);
	add_object_nodes(designer);

	/* get pixmaps */
	pixmap = gnome_db_get_pixmap(GNOME_STOCK_MENU_BOOK_RED);
	
	/* refresh list of tables */
	if ((list = gda_xml_database_get_tables(designer->priv->xmldb))) {
		for (l = list; l != NULL; l = g_list_next (l)) {
			gchar *ctree_row[2] = { "", "" };
			GtkCTreeNode* new_node;
			GdaXmlDatabaseTable *xml_table;

			xml_table = gda_xml_database_table_find (designer->priv->xmldb,
								 (const gchar *) l->data);
			ctree_row[1] = (gchar *) l->data;
			new_node = gtk_ctree_insert_node (
				GTK_CTREE (designer->priv->object_tree),
				designer->priv->tables_node, NULL,
				ctree_row, 0,
				GNOME_PIXMAP (pixmap)->pixmap,
				GNOME_PIXMAP (pixmap)->mask,
				GNOME_PIXMAP (pixmap)->pixmap,
				GNOME_PIXMAP (pixmap)->mask,
				TRUE, FALSE);
			gtk_ctree_node_set_row_data (
				GTK_CTREE (designer->priv->object_tree),
				new_node,
				(gpointer) xml_table);
		}
		g_list_foreach (list, (GFunc) g_free, NULL);
		g_list_free (list);
	}
}

/**
 * gnome_db_designer_get_database
 */
GdaXmlDatabase *
gnome_db_designer_get_database (GnomeDbDesigner *designer)
{
	g_return_val_if_fail(GNOME_DB_IS_DESIGNER(designer), NULL);
	return designer->priv->xmldb;
}

/**
 * gnome_db_designer_get_filename
 * @designer: the #GnomeDbDesigner widget
 *
 * Return the complete path of the file being edited by the given
 * #GnomeDbDesigner widget
 *
 * Returns: the complete path name
 */
const gchar *
gnome_db_designer_get_filename (GnomeDbDesigner *designer)
{
	g_return_val_if_fail(GNOME_DB_IS_DESIGNER(designer), NULL);
	g_return_val_if_fail(designer->priv != NULL, NULL);

	return (const gchar *) designer->priv->filename;
}

/**
 * gnome_db_designer_set_filename
 * @designer: the #GnomeDbDesigner widget
 * @filename: complete path name
 *
 * Associate a file name with the given #GnomeDbDesigner widget. This
 * file name, which should be a full path, is the one that will be
 * used when saving the file to disk by calling
 * #gnome_db_designer_save
 */
void
gnome_db_designer_set_filename (GnomeDbDesigner *designer, const gchar *filename)
{
	g_return_if_fail(GNOME_DB_IS_DESIGNER(designer));
	g_return_if_fail(designer->priv != NULL);

	if (designer->priv->filename)
		g_free((gpointer) designer->priv->filename);
	designer->priv->filename = g_strdup(filename);
}
