# GNU Enterprise Forms - Curses UI Driver - Entry Widget
#
# Copyright 2000-2005 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: entry.py 6851 2005-01-03 20:59:28Z jcater $

import curses

from _base import UIHelper

# =============================================================================
# Entry class
# =============================================================================

class UIEntry (UIHelper):

  # ---------------------------------------------------------------------------
  # Initialization
  # ---------------------------------------------------------------------------

  def __init__ (self, event):

    UIHelper.__init__ (self, event)

    self.__style = event.object.style
    if self.__style in ['default', 'password', 'label', 'dropdown', 'listbox']:
      self.__length = event.object ['Char:width']
    self.__value = {}
    self.__selection = {}
    self.__enabled = {}

    if self.__style == 'checkbox':
      self._setCursor (1, 0)

  # ---------------------------------------------------------------------------
  # Initialization per row
  # ---------------------------------------------------------------------------

  def _init (self, index):

    self.__value [index] = None
    self.__selection [index] = None
    self.__enabled [index] = True

  # ---------------------------------------------------------------------------
  # Focus has changed to this entry
  # ---------------------------------------------------------------------------

  def _getFocus (self, index):

    self.__repaint (index)

  # ---------------------------------------------------------------------------
  # Focus has changed away from this entry
  # ---------------------------------------------------------------------------

  def _loseFocus (self, index):

    self.__repaint (index)

  # ---------------------------------------------------------------------------
  # Set value for entry
  # ---------------------------------------------------------------------------

  def setValue (self, value, index = 0, enabled = True):

    self.__value [index] = value
    self.__enabled [index] = enabled

    self.__repaint (index)

  # ---------------------------------------------------------------------------
  # Set cursor position
  # ---------------------------------------------------------------------------

  def setCursorPosition (self, position, index = 0):

    if self.__style != 'checkbox':
      self._setCursor (position, 0)

  # ---------------------------------------------------------------------------
  # Set start and end of selection area
  # ---------------------------------------------------------------------------

  def setSelectedArea (self, selection1, selection2, index = 0):

    if selection1 == selection2:
      self.__selection [index] = None
    else:
      self.__selection [index] = (selection1, selection2)

    self.__repaint (index)

  # ---------------------------------------------------------------------------
  # Update entry representation on screen
  # ---------------------------------------------------------------------------

  def __repaint (self, index):

    value = self.__value [index]

    if self.__style in ['default', 'label', 'dropdown', 'listbox']:
      text = value or ''
      text += ' ' * (self.__length - len (text))

    elif self.__style == 'password':
      text = '*' * len (value or '')
      text += ' ' * (self.__length - len (text))

    elif self.__style == 'checkbox':
      if self.__value [index]:
        text = '[X]'
      else:
        text = '[ ]'

    if self.__style == 'label':
      attr = self._uiDriver.attr ['background']
    elif not self.__enabled [index]:
      attr = self._uiDriver.attr ['disabled']
    elif index == self._focusIndex:
      attr = self._uiDriver.attr ['focusentry']
    else:
      attr = self._uiDriver.attr ['entry']

    self._setText (index, text, attr, self.__selection [index])

# =============================================================================
# Configuration data
# =============================================================================

configuration = {
  'baseClass'  : UIEntry,
  'provides'   : 'GFEntry',
  'container'  : 0,
  }
