#include <stdio.h>
#include <ctype.h>

#include <algorithm>
#include <sigc++/slot.h>

#include <glib.h>

#include "gql++/object.h"

namespace GQL
{

using namespace std;

SQLType::SQLType()
{
  type_ = VOID;
  length_ = 0;
  decimals_ = 0;
}

SQLType::SQLType(TypeCode type, ...)
{
  va_list args;
  
  type_ = type;
  length_ = 0;
  decimals_ = 0;

  va_start(args, type);
  
  switch (type)
  {
    case SQLType::VOID:
    case SQLType::BOOLEAN:
    case SQLType::SMALLINT:
    case SQLType::INTEGER:
    case SQLType::INTERVAL:
    case SQLType::DATE:
    case SQLType::TIME:
    case SQLType::TIME_WITH_TZ:
      break;
    case SQLType::NUMERIC:
      length_ = va_arg(args, int);
      decimals_ = va_arg(args, int);
      break;
    case SQLType::FLOAT:
      decimals_ = va_arg(args, int);
      break;
    case SQLType::CHARACTER_VARYING:
    case SQLType::CHARACTER:
    case SQLType::BLOB:
      length_ = va_arg(args, int);
      break;
    default:
      g_assert_not_reached();
  }

  va_end(args);
}

namespace
{

enum SQLTypeWord
{
  TW_BOOL,
  TW_CHAR,
  TW_VARCHAR,
  TW_DATE,
  TW_DECIMAL,
  TW_REAL,
  TW_SMALLINT,
  TW_INT,
  TW_INTERVAL,
  TW_MONEY,
  TW_TIME,
  TW_WITH,
  TW_WITHOUT,
  TW_ZONE,
  TW_TIMESTAMP,
  TW_BLOB
};

#include "type_words.cc"

enum Token
{
  TOK_EMPTY,
  TOK_IDENT,
  TOK_INTEGER,
  TOK_COMMA,
  TOK_PAREN_OPEN,
  TOK_PAREN_CLOSE,
  TOK_INVALID
};

Token scan_token(string& token, string::iterator& start, string::iterator end)
{
  int pos1, pos2;
  Token result = TOK_INVALID;  
  bool is_identifier = true;
  
  for (pos1 = 0; pos1 < end - start && isspace(*(start + pos1)); pos1++)
    ;
  if (pos1 == end - start || isspace(*(start + pos1)))
    return TOK_EMPTY;

  for (pos2 = pos1; pos2 < end - start && isalpha(*(start + pos2)); ++pos2)
    ;
  if (pos1 == pos2)
    is_identifier = false;
  for (; pos2 < end - start && isdigit(*(start + pos2)); ++pos2)
    ;
  if (pos2 == pos1)
  {
    // no alphanumerics at the start
    if (*(start + pos1) == '(')
      result = TOK_PAREN_OPEN;
    else if (*(start + pos1) == ')')
      result = TOK_PAREN_CLOSE;
    else if (*(start + pos1) == ',')
      result = TOK_COMMA;
    ++pos2;
  }
  else if (is_identifier)
    result = TOK_IDENT;
  else
  {
    if (pos2 < end - start && strchr(" \t\n\r", *(start + pos2)) == NULL)
      result = TOK_INVALID;
    result = TOK_INTEGER;
  }
  
  token.assign(start + pos1, start + pos2);
  start += pos2;
  
  return result;
}

Token scan_int_1_1(int& arg1, int& arg2, 
                   string::iterator& start, 
                   string::iterator end)
{
  Token tok_code;
  string token;
  
  if ((tok_code = scan_token(token, start, end)) != TOK_INTEGER)
    return tok_code;
  arg1 = atoi(token.c_str());
  if ((tok_code = scan_token(token, start, end)) == TOK_COMMA)
  {
    if ((tok_code = scan_token(token, start, end)) != TOK_INTEGER)
      return tok_code;
    arg2 = atoi(token.c_str());
    return scan_token(token, start, end);
  }
  else
    arg2 = -1;
  
  return tok_code;
}

}

SQLType::SQLType(const string& str)
{
  SQLType type;
  string::iterator start;
  string val = str;
  string token;
  enum State { STATE_INIT, STATE_HAD_CHAR, STATE_HAD_VARCHAR, 
               STATE_HAD_DECIMAL, STATE_HAD_TIMEX, STATE_HAD_TIMEX_WITHX,
               STATE_DONE, STATE_INVALID };
  State state = STATE_INIT;
  bool have_timestamp = false;
  bool have_with = false;
  type_ = VOID;
  
  transform(val.begin(), val.end(), val.begin(), &tolower);
  
  start = val.begin();
  do
  {
    Token tok_code = scan_token(token, start, val.end());
    Type *tw;
        
    switch (state)
    {
      case STATE_INIT:
      {
        if (tok_code != TOK_IDENT
            || ((tw = lookup_type(token.c_str(), token.length())) == NULL))
        {
          state = STATE_INVALID;
          break;
        }
        switch (tw->type_word)
        {
          case TW_BOOL:
            type.type_ = SQLType::BOOLEAN;
            state = STATE_DONE;
            break;
          case TW_DATE:
            type.type_ = SQLType::DATE;
            state = STATE_DONE;
            break;
          case TW_DECIMAL:
            state = STATE_HAD_DECIMAL;
            break;
          case TW_REAL:
            type.type_ = SQLType::FLOAT;
            state = STATE_DONE;
            break;
          case TW_SMALLINT:
            type.type_ = SQLType::SMALLINT;
            state = STATE_DONE;
            break;
          case TW_MONEY:
            type.type_ = SQLType::NUMERIC;
            type.decimals_ = 2;
            type.length_ = 9;
            state = STATE_DONE;
            break;
          case TW_BLOB:
            type.type_ = SQLType::BLOB;
            state = STATE_DONE;
            break;
          case TW_CHAR:
            state = STATE_HAD_CHAR;
            break;
          case TW_VARCHAR:
            state = STATE_HAD_VARCHAR;
            break;
          case TW_TIME:
            state = STATE_HAD_TIMEX;
            break;
          case TW_TIMESTAMP:
            have_timestamp = true;
            state = STATE_HAD_TIMEX;
            break;
          default:
            state = STATE_INVALID;
            break;
        }
        break;
      }
      case STATE_HAD_CHAR:
      {
        state = STATE_INVALID;
        break;
      }
      case STATE_HAD_VARCHAR:
      {
        state = STATE_INVALID;
        break;
      }
      case STATE_HAD_DECIMAL:
      {
        if (tok_code == TOK_EMPTY)
        {
          type.type_ = DECIMAL;
          type.decimals_ = -1;
          type.length_ = -1;
          state = STATE_DONE;
        }
        else if (tok_code == TOK_PAREN_OPEN)
        {
          if (scan_int_1_1(type.length_, type.decimals_, start, val.end()) !=
              TOK_PAREN_CLOSE)
            state = STATE_INVALID;
          else
          {
            type.type_ = DECIMAL;
            state = STATE_DONE;
          }
        }
        break;
      }
      case STATE_HAD_TIMEX:
      {
        if (tok_code == TOK_EMPTY)
        {
          type.type_ = have_timestamp ? TIMESTAMP_WITH_TZ : TIME;
          state = STATE_DONE;
        }
        else if (tok_code == TOK_IDENT && 
                 (tw = lookup_type(token.c_str(), token.size())) != NULL)
        {
          if (tw->type_word == TW_WITH)
          {
            have_with = true;
            state = STATE_HAD_TIMEX_WITHX;
          }
          else if (tw->type_word == TW_WITHOUT)
            state = STATE_HAD_TIMEX_WITHX;
          else
            state = STATE_INVALID;
        }
        else 
          state = STATE_INVALID;
        break;
      }
      case STATE_HAD_TIMEX_WITHX:
      {
        if (tok_code == TOK_IDENT && 
            (tw = lookup_type(token.c_str(), token.size())) != NULL && 
            tw->type_word == TW_TIME &&
            scan_token(token, start, val.end()) == TOK_IDENT &&
            (tw = lookup_type(token.c_str(), token.size())) != NULL && 
            tw->type_word == TW_ZONE)
        {
          if (have_timestamp)
            if (!have_with)
              state = STATE_INVALID;
            else
            {
              type.type_ =  TIMESTAMP_WITH_TZ;
              state = STATE_DONE;
            }
          else
          {
            type.type_ = have_with ? TIME_WITH_TZ : TIME;
            state = STATE_DONE;
          }
        }
        else
          state = STATE_INVALID;
        break;
      }
      default:
        g_assert_not_reached();
    }
  } while (state != STATE_INVALID && state != STATE_DONE);

  if (state == STATE_DONE && scan_token(token, start, val.end()) != TOK_EMPTY)
    state = STATE_INVALID;
  
  if (state == STATE_DONE)
    *this = type;
}


string SQLType::as_string() const
{
  // Note: this static stuff is not thread safe, we should initialize
  // the array with the values in its definition, but haven't done so
  // yet since that would make us depend on the order of
  // SQLType::TypeCode
  static const char *types[SQLType::MAX_TYPE] = { 0 };
  static bool initialized = false;
  char buf[64];
  string result;
  
  if (!initialized)
  {
    types[SQLType::VOID] = "VOID";
    types[SQLType::BOOLEAN] = "BOOLEAN";
    types[SQLType::DATE] = "DATE";
    types[SQLType::DECIMAL] = "DECIMAL";
    types[SQLType::FLOAT] = "FLOAT";
    types[SQLType::SMALLINT] = "SMALLINT";
    types[SQLType::INTEGER] = "INTEGER";
    types[SQLType::INTERVAL] = "INTERVAL";
    types[SQLType::NUMERIC] = "NUMERIC";
    types[SQLType::TIME] = "TIME";
    types[SQLType::TIME_WITH_TZ] = "TIME WITH TIME ZONE";
    types[SQLType::TIMESTAMP_WITH_TZ] = "TIMESTAMP";
    types[SQLType::CHARACTER] = "CHARACTER";
    types[SQLType::CHARACTER_VARYING] = "CHARACTER VARYING";
    types[SQLType::BLOB] = "BLOB";
    
    initialized = true;
  }
  
  result = types[type_];
  buf[0] = '\0';
  
  switch (type_)
  {
    case SQLType::CHARACTER:
    case SQLType::CHARACTER_VARYING:
      sprintf(buf, "(%i)", length_);
      break;
    case SQLType::NUMERIC:
      if (length_ >= 0)
      {
        if (decimals_ >= 0)
          sprintf(buf, "(%i, %i)", length_, decimals_);
        else
          sprintf(buf, "(%i)", length_);
      }
      break;
    default:
      break;
  }
  result += buf;
  
  return result;
}

Blob::~Blob()
{
}

BasicSQLObject::BasicSQLObject()
{ 
  type_ = VOID; 
}

BasicSQLObject::~BasicSQLObject()
{ 
}

string BasicSQLObject::output() const
{
  if (is_null())
    return("NULL");
  
  switch (type_)
  {
    case VOID:
      return("VOID");
      
    case STRING:
      return("'" + value_ + "'");

    case TYPE:
    case INT:
      return(value_);

    default:
      assert(0);
  }
}

bool BasicSQLObject::input(const string& str)
{
  int c = str[0];
  
  set_null(false);
  
  if (c == '\'')
  {
    type_ = STRING;
    value_ = str.substr(0, str.length() - 1);
  }
  else if (isdigit(c) || c == '-' || c == '+')
  {
    type_ = INT;
    value_ = str;
  }
  else if (str == "NULL")
    set_null(true);
  else
    throw SQLException("invalid input for BasicSQLObject");
  
  return true;
}

string BasicSQLObject::to_string() const
{
  return is_null() ? string() : value_;
}

long BasicSQLObject::to_int() const
{
  if (is_null())
    return 0;
  
  switch (type_)
  {
    case STRING:
    case INT:
    {
      char *endptr;
      long l = strtol(value_.c_str(), &endptr, 10);
      if (*endptr == '\0')
        return l;
      else
	break;
    }
    case FLOAT:
    {
      char *endptr;
      double d = strtod(value_.c_str(), &endptr);
      if (*endptr == '\0')
        return (long)d;
      else
	break;
    }
    case TYPE:
    case VOID:
      break;
    default:
      g_assert_not_reached();
  }
  throw SQLException("object type mismatch");
}

double BasicSQLObject::to_real() const
{
  if (is_null())
    return 0.0;
  
  switch (type_)
  {
    case STRING:
    case INT:
    {
      char *endptr;
      long l = strtol(value_.c_str(), &endptr, 10);
      if (*endptr == '\0')
        return l;
      else
	break;
    }
    case FLOAT:
    {
      char *endptr;
      double d = strtod(value_.c_str(), &endptr);
      if (*endptr == '\0')
        return d;
      else
	break;
    }
    case TYPE:
    case VOID:
      break;
    default:
      g_assert_not_reached();
  }
  throw SQLException("object type mismatch");
}

bool BasicSQLObject::to_boolean() const
{
  if (is_null())
    return false;
  
  switch (type_)
  {
    case STRING:
    {
      string val = value_;
      transform(val.begin(), val.end(), val.begin(), SigC::slot(&tolower));
      if (val == "1" || val == "t" || val == "true")
        return true;
      else
        return false;
    }
    case INT:
    {
      char *endptr;
      long l = strtol(value_.c_str(), &endptr, 10);
      if (*endptr == '\0')
        return l;
      else
	break;
    }
    case FLOAT:
    {
      char *endptr;
      double d = strtod(value_.c_str(), &endptr);
      if (*endptr == '\0')
        return (bool)d;
      else
	break;
    }
    case TYPE:
    case VOID:
      break;
    default:
      g_assert_not_reached();
  }
  throw SQLException("object type mismatch");
}

SQLType BasicSQLObject::to_type() const
{
  if (is_null())
    return SQLType();

  switch (type_)
  {
    case STRING:
      return SQLType(value_);
    default:
      throw SQLException("object type mismatch");
  }
}

Blob *BasicSQLObject::to_blob() const
{
  throw SQLException("BLOBs not supported by driver");
}

void BasicSQLObject::from_string(const string& s)
{
  type_ = STRING;
  value_ = s;
  set_null(false);
}

void BasicSQLObject::from_int(long l)
{
  char buffer[64];

  sprintf(buffer, "%li", l);
  value_ = buffer;
  type_ = INT;
  set_null(false);
}

void BasicSQLObject::from_real(double d)
{
  char buffer[64];

  sprintf(buffer, "%f", d);
  value_ = buffer;
  type_ = FLOAT;
  set_null(false);
}

void BasicSQLObject::from_boolean(bool b)
{
  value_ = b ? "t" : "f";
  type_ = STRING;
  set_null(false);
}

void BasicSQLObject::from_blob(const Blob *blob)
{
  throw SQLException("BLOBs not supported by driver");
}

void BasicSQLObject::from_type(const SQLType& type)
{
  type_ = TYPE;
  value_ = type.as_string();
  set_null(false);
}

}
