/* *************************************************************************
  Module:        proutines.c
  Author:        Matt Simpson
                 Arlington, TX
                 matthewsimpson@home.com
  Date:          August, 2000
  Description:
                 Routines for parsing printer info. 
  Changes:

****************************************************************************
                 COPYRIGHT (C) 1999, 2000 Matt Simpson
                 GNU General Public License
                 See lexgui.c for full notice.
**************************************************************************** */
#include <stdio.h>
#include <string.h>
#include <gtk/gtk.h>
#include <fcntl.h>  /* for open */
#include <unistd.h> /* for read & close */
#include <errno.h>  /* for extern int errno */
#include <malloc.h>
#include "lexgui.h"
#define BUFSIZE 256
#define QCHUNK 2048
/* -------------------------------------------------------------------------
        chopSpace() -- gets rid of blank spaces at the end of a string
   ------------------------------------------------------------------------- */
gchar *chopSpace(char *string, int last)
{
  static gchar STRING[MAXLINELENGTH];
  gchar *ret_ptr;
  gint i;

  for(i = 0; i < MAXLINELENGTH; i++)
      STRING[i] = 0;
  strcpy(STRING, string);
  STRING[last] = 0;
  for(i = last; i >= 0; i--)
  {
    if(STRING[i] == 0 ||
       STRING[i] == '\0' ||
       STRING[i] == ' ' ||
       STRING[i] == '\n' ||
       STRING[i] == 13) /* 13 = control-M */
      STRING[i] = 0;
    else
    {
      break;
    }
  }
  ret_ptr = STRING;
  return(ret_ptr);
} 
/* -------------------------------------------------------------------------
        flush_dev() flushes fp and checks for error, issuing message in
                    message label sent to it.
   ------------------------------------------------------------------------- */
int flush_dev(msgbox_struct *msgbox)
{
  extern FILE *fp;
  gchar message[50];

  fflush(fp);
  if (ferror(fp))
  {
    sprintf(message, "%s", "Flush Error during output.");
    put_msg(msgbox, message, RED, 0);
    return(1);
  }
  else
    return(0);
}
/* -------------------------------------------------------------------------
        clear_junk() Clears junk in output device by reading it. The
                     junk occurs whenever the printer is powered up.
                     We let the other functions test for errors, this
                     is just a quick & dirty read. 
   ------------------------------------------------------------------------- */
void clear_junk(void)
{
  int fd, numread;
  gchar junk[1024] = "\0";
  extern gchar *output;
  int terminate = 0;

  fd = open(output, O_RDONLY);
  if(fd >= 0)
  {
    while(1)
    {
      numread = read(fd, junk, sizeof(junk));
      if(numread < 0)
      {
        close(fd);
        return;
      }
      if(terminate && numread == 0) /* This happens when numread is 0 twice */
        break;
      if(numread == 0)
        terminate = 1; /* if the next numread is also 0, we are at the end */
    }
    close(fd);
  }
}
/* -------------------------------------------------------------------------
        put_text() Puts text in text box.
   ------------------------------------------------------------------------- */
void put_text(io_struct *io)
{
  gint i;
  static GdkColor red = {0, 0xc8c8, 0x0000, 0x0000};
  GdkFont *font;
  gchar *sptr1, *sptr2;

  /* The following must be in the same order as the commands were sent in
     send_commands() */
  static gchar *info[] = {"@PJL INFO ID", "@PJL INFO STATUS", 
    "@PJL INFO CONFIG", "@PJL INFO PAGECOUNT", "@PJL INFO VARIABLES"};
  gint infonum = 5; /* number of things in *info[] */

  font = 
  gdk_font_load("-adobe-helvetica-bold-r-normal--12-120-75-75-p-70-iso8859-1");

  gtk_text_freeze(GTK_TEXT ((io->aj_ptr)->text));

  sptr1 = io->query;
  for(i = 0; i < infonum; i++)
  {
    sptr2 = strstr(io->query, info[i]);
    if(sptr2)
    {
      gtk_text_insert(GTK_TEXT((io->aj_ptr)->text), NULL, NULL, NULL,
                      sptr1, sptr2 - sptr1);
      gtk_text_insert(GTK_TEXT((io->aj_ptr)->text), NULL, NULL, NULL, "\n", 1);
      gtk_text_insert(GTK_TEXT((io->aj_ptr)->text), font, &red, NULL,
                      sptr2, strlen(info[i]));
      sptr1 = sptr2 + strlen(info[i]);
    }
  }
  gtk_text_insert(GTK_TEXT((io->aj_ptr)->text), NULL, NULL, NULL, sptr1, -1);

  gtk_adjustment_set_value(GTK_TEXT((io->aj_ptr)->text)->vadj, 0);
  gtk_adjustment_set_value(GTK_TEXT((io->aj_ptr)->text)->vadj, 0);
  gtk_text_thaw(GTK_TEXT ((io->aj_ptr)->text));
}
/* -------------------------------------------------------------------------
        report_alloc_err()
   ------------------------------------------------------------------------- */
void report_alloc_err(io_struct *io, gint messagenum)
{
  gchar message[50];

  close_io(io);
  if(messagenum == 7)
    sprintf(message, "%s", "Error: Cannot Initialize Memory.");
  else
    sprintf(message, "%s", "Error: Cannot Allocate Memory.");
  put_msg(&(io->msgbox), message, RED, messagenum);
}
/* -------------------------------------------------------------------------
        rtimer() Timer callback that reads the data from the printer and 
                 automatically waits if it is not yet ready. Also see note
                 in qqtimer(). The data is stored into 1 big string,
                 io->query. For the 'set choices dynamically' window, this
                 string is parsed into a list of smaller strings. The theory is
                 to have the query window post the data coming back 'as-is',
                 to accomodate for unseen variations in PJL from different
                 printers. This way if the 'set choices dynamically' window
                 gets jumbled up, one can look at the query window and ajust
                 the build_choices() and other routines accordingly.
   ------------------------------------------------------------------------- */
gint rtimer(io_struct *io)
{
  int i, numread;
  unsigned char buff[BUFSIZE]; /* unsigned to test for extended ASCII set */
  gchar message[50];
  extern int errno; /* in errno.h */
  extern int open_file_errno;
  int rTIMELIMIT = 10; /* 1 sec */

  open_file_errno = 0;
  /* note io->count is used by more than 1 timer (the timers are only
     used one at a time.) */
  if(io->count >= rTIMELIMIT)
    io->count = 0;
  if(!io->count)
  {
    put_query_message(&(io->msgbox), "  Reading Data  ");
    for(i = 0; i < BUFSIZE; i++)
      buff[i] = 0;
    numread = read(io->fd, buff, sizeof(buff) - 1);
    buff[BUFSIZE - 1] = 0;
    if(numread < 0)
    {
      /* One thing that would cause this is the printer being powered off.
         In this case open_output_dev() will catch this error first. Of
         course if the printer is powered off after open_output_dev()
         then this check will catch it. Interestingly, my "other
         operating system" completely locks up if the printer loses
         power during a job, requiring a reboot :( 
         Another thing that would cause this is the kernel not being
         configured for printer readback. If this is the case and for
         some reason this test passes, the test for PUP_START below
         will catch it. */
      open_file_errno = errno;
      sprintf(message, "%s", "Error reading device.");
      put_msg(&(io->msgbox), message, RED, 3);
      close_io(io);
      return(FALSE); /* returning FALSE will automatically remove the timer */
    }

    if(io->qlength <= io->j + numread)
    {
      io->qlength += QCHUNK;
      io->query = (gchar *) realloc(
                                     io->query,
                                     io->qlength * sizeof(gchar)
                                   );
      if(io->query == NULL)
      {
        io->qlength -= QCHUNK;
        report_alloc_err(io, 8);
        return(FALSE);/*returning FALSE removes the timer*/
      }
    }
    for(i = 0; i < numread; i++)
    {
      if(buff[i] == 4 ||    /* Skip ctrl-D */
         buff[i] > 127)     /* Skip extended ASCII set */ 
        continue;
      /* The above buff[i] > 127 may occur if the connection is disabled
         while on a read, resulting in garbage in the buffer. */

      if(buff[i] == 12 || /* Form Feed */
         buff[i] == 13 || /* ctrl-M */
         buff[i] == '\n') /* carriage return */
      {
        if(io->j != 0)
          if(io->query[io->j - 1] == '\n')
            continue;
        io->query[io->j] = '\n';
        io->j++;
        io->query[io->j] = 0;
        continue;
      }
      else
        io->query[io->j] = buff[i];
      io->j++;
      /*Set the terminator. This gets over-written next round but not the last*/
      io->query[io->j] = 0;
    }

    if(!io->start_read)
    {
    /* At the initial read, test for the string PUP_START, which was sent
       to the printer in send_commands(). This serves two purposes--
       to wait for the printer to start writing out the data, and to
       make sure we indeed have bi-directional communication. If we don't
       ever see PUP_START, the timer catches this fact and reports an error.
       Note qqtimer() also has a timer that may have already given an
       initial delay after sending the query commands. But all printers are
       different and thus require different amounts of time to wait for the
       data. The following assures enough time to wait. */
      if(!strstr(io->query, "PUP_START"))
      {
        /* start count and keep looping until count reaches rTIMELIMIT */
        io->count++;
        /* the following increments only once per second because when
           io->count is incremented above, rTIMELIMIT will not let it
           get back to this point for 1 second (that is if the increment
           in rtimer_call() is 100 ms). */
        io->timeout_count++;
        if(io->timeout_count >= QUERY_TIMEOUT)
        {
          close_io(io);
          sprintf(message, "%s", "Timeout. Could not start.");
          put_msg(&(io->msgbox), message, RED, 4);
          return(FALSE);/*returning FALSE will automatically remove the timer*/
        }
        else
          return(TRUE);
      }
      else
        io->start_read = 1;
    }
    if(io->terminate && numread == 0)
    { /* This happens when numread is 0 twice */
      clear_msgbox(&(io->msgbox));
      if(close_io(io) < 0)
      {
        open_file_errno = errno;
        sprintf(message, "%s", "Error closing device.");
        put_msg(&(io->msgbox), message, RED, 5);
      }
      /* ----------------------------------------------------------
          Uncomment to print results in shell window. Note
          a '|' will be printed wherever there's a carriage return.
         ---------------------------------------------------------- */ 
      /*
      printf("******* Query Results *******\n");
      for(i = 0; i < io->j; i++)
      {
        if(io->query[i] == '\n')
          printf("|\n");
        else
          printf("%c", io->query[i]);
          printf("zero at i=%d\n", i); 
      }
      */
      /* ---------------------------------------------------------- */ 
      if(io->command == 3)
        build_choices(io);
      else
        put_text(io);
      /* Read is finished. Exit timer normally. */
      return(FALSE);/* returning FALSE will automatically remove the timer */
    }
    if(numread == 0)
    {
      /* Check for expected end string because maybe the printer has not
         finished writing out the data. */
      if(!strstr(io->query, "PUP_END"))
      {
        /* start count and keep looping until count reaches rTIMELIMIT */
        io->count++;
        io->timeout_count++; /* 1 sec increment */
        if(io->timeout_count >= QUERY_TIMEOUT)
        {
          close_io(io);
          sprintf(message, "%s", "Timeout. Could not finish.");
          put_msg(&(io->msgbox), message, RED, 6);
          if(io->command == 3)
            build_choices(io); /* Build choices from what we have anyway */
          else
            put_text(io); /* Display what we have anyway. */
          return(FALSE);/*returning FALSE will automatically remove the timer*/
        }
        else
          return(TRUE);
      }
      else
        io->terminate = 1;/* if the next numread is also 0, 
                                we are at the end */
    }
    io->timeout_count = 0;
    return(TRUE); /* Still reading. All is well for next timer iteration */
  }
  else 
  {
    /*we must be in a wait period (either PUP_START or PUP_END was not found)*/
    put_query_message(&(io->msgbox), "Waiting For Data");
    io->count++; /* keep incrementing. rTIMELIMIT will catch and reset
                       it at the appropriate time. */
    return(TRUE);
  }
}
/* -------------------------------------------------------------------------
        rtimer_call() Calls rtimer() with timeout.
   ------------------------------------------------------------------------- */
void rtimer_call(io_struct *io)
{
  if(io->clock[2])
  {
    gtk_timeout_remove(io->clock[2]);
    io->clock[2] = 0;
  }
  io->count = 0;
  io->j = 0;
  io->query[0] = 0;
  /* call rtimer every 100 ms */
  io->clock[2] = gtk_timeout_add(100, (GtkFunction)rtimer, io);
} 
/* -------------------------------------------------------------------------
        get_settings() Reads recently queried device.
                       The output device was already determined
                       to be a device and not a file before this
                       function is called.
   ------------------------------------------------------------------------- */
void get_settings(io_struct *io)
{
  extern gchar *output;
  gchar message[50];
  extern int errno; /* in errno.h */
  extern int open_file_errno;

  open_file_errno = 0;

  /* Instead of open_output_dev() which uses fopen, I am doing a low-level
     i/o call. I could have read the device with fgets but wanted to do 
     it this way instead. */
  io->fd = open(output, O_RDONLY);
  if(io->fd < 0)
  {
    /* Note this should never happen because the open error will be caught
       in send_commands() which causes qqtimer() to not execute and
       thus not call this function. It is here if I need to do something 
       else later. */
    open_file_errno = errno;
    close_io(io);
    sprintf(message, "%s", "Cannot open output device for reading."); 
    put_msg(&(io->msgbox), message, RED, 2);
    return;
  }

  if(io->query == NULL)
  {
    io->query = (gchar *) malloc(QCHUNK * sizeof(gchar));
    if(io->query == NULL)
    {
      report_alloc_err(io, 7);
      return;
    }
    io->qlength = QCHUNK;
  }

  rtimer_call(io);
}
