//  (C) Copyright John Maddock 2006.
//  Use, modification and distribution are subject to the
//  Boost Software License, Version 1.0. (See accompanying file
//  LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

#include <boost/math/concepts/real_concept.hpp>
#include <boost/test/included/test_exec_monitor.hpp>
#include <boost/test/floating_point_comparison.hpp>
#include <boost/math/special_functions/sqrt1pm1.hpp>
#include <boost/math/special_functions/powm1.hpp>
#include <boost/math/tools/test.hpp>
#include <boost/array.hpp>
#include "functor.hpp"

#include "handle_test_result.hpp"

//
// DESCRIPTION:
// ~~~~~~~~~~~~
//
// This file tests the functions powm1 and sqrt1pm1.  
// The accuracy tests
// use values generated with NTL::RR at 1000-bit precision
// and our generic versions of these functions.
//
// Note that when this file is first run on a new platform many of
// these tests will fail: the default accuracy is 1 epsilon which
// is too tight for most platforms.  In this situation you will 
// need to cast a human eye over the error rates reported and make
// a judgement as to whether they are acceptable.  Either way please
// report the results to the Boost mailing list.  Acceptable rates of
// error are marked up below as a series of regular expressions that
// identify the compiler/stdlib/platform/data-type/test-data/test-function
// along with the maximum expected peek and RMS mean errors for that
// test.
//

void expected_results()
{
   //
   // Define the max and mean errors expected for
   // various compilers and platforms.
   //
   add_expected_result(
      ".*",                          // compiler
      ".*",                          // stdlib
      ".*",                          // platform
      ".*",                          // test type(s)
      ".*",                          // test data group
      ".*", 5, 2);                   // test function

   //
   // Finish off by printing out the compiler/stdlib/platform names,
   // we do this to make it easier to mark up expected error rates.
   //
   std::cout << "Tests run with " << BOOST_COMPILER << ", " 
      << BOOST_STDLIB << ", " << BOOST_PLATFORM << std::endl;
}

template <class T>
void test_powm1_sqrtp1m1(T, const char* type_name)
{
#define SC_(x) static_cast<T>(BOOST_JOIN(x, L))
   static const boost::array<boost::array<T, 2>, 141> sqrtp1m1_data = {
      SC_(-0.990433037281036376953125), SC_(-0.902189148255607021082179865003660033379), 
      SC_(-0.928576648235321044921875), SC_(-0.7327485233629588435419837213946140663952), 
      SC_(-0.804919183254241943359375), SC_(-0.5583204592175928547330219789723812512248), 
      SC_(-0.780276477336883544921875), SC_(-0.5312532425038905348288090678272629719499), 
      SC_(-0.775070965290069580078125), SC_(-0.525733160857803152349794525146692520785), 
      SC_(-0.74602639675140380859375), SC_(-0.4960420620244223284705423670972348730775), 
      SC_(-0.72904598712921142578125), SC_(-0.4794675678972648545670296583048773096015), 
      SC_(-0.7162272930145263671875), SC_(-0.4672967927771847619430106324762494277417), 
      SC_(-0.68477380275726318359375), SC_(-0.4385499156267435234127335149836228483716), 
      SC_(-0.62323606014251708984375), SC_(-0.3861890031471553130754715740017444288273), 
      SC_(-0.576151371002197265625), SC_(-0.3489634196162225368137418186622322770468), 
      SC_(-0.5579319000244140625), SC_(-0.3351179804088653673227401220715512851566), 
      SC_(-0.44300353527069091796875), SC_(-0.2536780421766293214167684432407843582845), 
      SC_(-0.40594112873077392578125), SC_(-0.2292478535422518369418742953348121057781), 
      SC_(-0.38366591930389404296875), SC_(-0.2149305249240001453212315877592832894955), 
      SC_(-0.277411997318267822265625), SC_(-0.1499482352928545459530860365223688354153), 
      SC_(-0.202522933483123779296875), SC_(-0.1069842854031759981174720343292165991554), 
      SC_(-0.156477451324462890625), SC_(-0.08156516362044655434273010203540784115386), 
      SC_(-0.029248714447021484375), SC_(-0.01473288619127324715091701268111937769847), 
      SC_(0.1431564604442703013402649929484277542953e-29), SC_(0.7157823022213515067013249647418825993242e-30), 
      SC_(0.1791466932348087634896446282571611213266e-29), SC_(0.8957334661740438174482231412854044374117e-30), 
      SC_(0.6013619202535540063110633226832922483532e-29), SC_(0.3006809601267770031555316613411940789777e-29), 
      SC_(0.115805324961653822428570241697281798758e-28), SC_(0.5790266248082691121428512084847326346288e-29), 
      SC_(0.1422457400834001098175711728787848259007e-28), SC_(0.711228700417000549087855864391394898182e-29), 
      SC_(0.4970121018327539153628705477876439795096e-28), SC_(0.2485060509163769576814352738907342268877e-28), 
      SC_(0.9660079415057497591758174164417478444323e-28), SC_(0.483003970752874879587908708209209280428e-28), 
      SC_(0.1232929313253182131376331095427391968754e-27), SC_(0.6164646566265910656881655476946945507336e-28), 
      SC_(0.3296523285617759312781860549364832953326e-27), SC_(0.1648261642808879656390930274546578154505e-27), 
      SC_(0.528364435768055252017009628713605422886e-27), SC_(0.26418221788402762600850481432190658932e-27), 
      SC_(0.886586057273120049620324386849842094685e-27), SC_(0.4432930286365600248101621933266666927236e-27), 
      SC_(0.2499669674831043259218157022821422146034e-26), SC_(0.1249834837415521629609078510629667512608e-26), 
      SC_(0.4131050397232622964314362671638736040881e-26), SC_(0.2065525198616311482157181333686170847381e-26), 
      SC_(0.7679738097881433551381658732998641759182e-26), SC_(0.3839869048940716775690829359127023723085e-26), 
      SC_(0.199929739820949207249437007767740538737e-25), SC_(0.9996486991047460362471850338422150855758e-26), 
      SC_(0.5151477415246978459754129800826163591626e-25), SC_(0.2575738707623489229877064867240932346063e-25), 
      SC_(0.101200734533556026342258477595279955025e-24), SC_(0.5060036726677801317112923751744139374608e-25), 
      SC_(0.2064292695896540981798546456623054911033e-24), SC_(0.1032146347948270490899273175045223276369e-24), 
      SC_(0.4063294332896333395257434433879773416284e-24), SC_(0.2031647166448166697628717010560376261299e-24), 
      SC_(0.8138195767936862452966745688936976428456e-24), SC_(0.406909788396843122648337201659060874841e-24), 
      SC_(0.9575550627132253801929510132578249716542e-24), SC_(0.478777531356612690096475392014950219861e-24), 
      SC_(0.2855160956298500804375620841706273850616e-23), SC_(0.1427580478149250402187809401860126128887e-23), 
      SC_(0.65201444297915461398563707001320281266e-23), SC_(0.3260072214895773069928180036030590895584e-23), 
      SC_(0.1310988374636350038320977491775043421995e-22), SC_(0.6554941873181750191604865975243736714241e-23), 
      SC_(0.2590288837798696209228010176465529547374e-22), SC_(0.1295144418899348104613996701237435743037e-22), 
      SC_(0.2937779542193655202274099291941187976629e-22), SC_(0.1468889771096827601137038857784795823862e-22), 
      SC_(0.7863513178004503049754083414326234074965e-22), SC_(0.3931756589002251524876964413613741224383e-22), 
      SC_(0.1903818607087388763706780167350761726053e-21), SC_(0.9519093035436943818533447771092722109619e-22), 
      SC_(0.3812242142377350870566942975497647799754e-21), SC_(0.1906121071188675435283289822871922426703e-21), 
      SC_(0.5493133580141330277178034419485741501887e-21), SC_(0.2746566790070665138588640028286254797086e-21), 
      SC_(0.9672153634284186955666772243312215295852e-21), SC_(0.4836076817142093477832216739707042687854e-21), 
      SC_(0.1702169477623814384559878647986894129041e-20), SC_(0.8510847388119071922795771513771277983779e-21), 
      SC_(0.4817114569977399785676754474621208412799e-20), SC_(0.2408557284988699892835476663213068137742e-20), 
      SC_(0.7538352992756463183303278501219690799218e-20), SC_(0.3769176496378231591644535904879420358976e-20), 
      SC_(0.2596305715949999708394617609422128090557e-19), SC_(0.1298152857974999854188882800497720744566e-19), 
      SC_(0.4444587480324321591032923385589104015025e-19), SC_(0.2222293740162160795491768745456732380306e-19), 
      SC_(0.9715574921498573937069095571295029856174e-19), SC_(0.4857787460749286968416557290578449901693e-19), 
      SC_(0.2036598542733453787268262970278076551267e-18), SC_(0.1018299271366726893582284814835737930738e-18), 
      SC_(0.4248971931658660264162106698360155121463e-18), SC_(0.2124485965829330131855381318229788476899e-18), 
      SC_(0.6521097487613458963613731825259556273977e-18), SC_(0.3260548743806729481275307007092796054895e-18), 
      SC_(0.1436126164096190058281493628911107407475e-17), SC_(0.718063082048095028882939519555349077068e-18), 
      SC_(0.3118908901459261162419055180006211003274e-17), SC_(0.1559454450729630579993578498052878596544e-17), 
      SC_(0.3593346613595175715618300349429858897565e-17), SC_(0.1796673306797587856195132689035439820049e-17), 
      SC_(0.9445874854124767215374919304693435151421e-17), SC_(0.4722937427062383596534390682373393588305e-17), 
      SC_(0.2566182432094081539023303073498993853718e-16), SC_(0.1283091216047040761280036193264127770924e-16), 
      SC_(0.3363765695149349330660137891158001366421e-16), SC_(0.1681882847574674651186419380749519252198e-16), 
      SC_(0.1073581901339262605326457800103412409953e-15), SC_(0.53679095066963128825600266401137964529e-16), 
      SC_(0.186668406231853462907965823802669547149e-15), SC_(0.9333420311592672709834617625880158673234e-16), 
      SC_(0.3727540802657755688795382376099496468669e-15), SC_(0.1863770401328877670715685744569461355954e-15), 
      SC_(0.6211646767866855090717281839829411183018e-15), SC_(0.3105823383933427063051696310530511222847e-15), 
      SC_(0.1561186859754253464932505224282976996619e-14), SC_(0.7805934298771264278032012284733416682125e-15), 
      SC_(0.3092010764722992466335682593125966377556e-14), SC_(0.1546005382361495038101520151206746652194e-14), 
      SC_(0.6192850577371690132255643845837767003104e-14), SC_(0.3096425288685840272203037716324248888074e-14), 
      SC_(0.1047879028014987723427253740737796761096e-13), SC_(0.5239395140074924891505551783318071383455e-14), 
      SC_(0.1978473638988408750405412206418986897916e-13), SC_(0.989236819494199482255280888212449451182e-14), 
      SC_(0.4041816252346730475863978426787070930004e-13), SC_(0.2020908126173344817583717046094650765545e-13), 
      SC_(0.9410302262901834580155480125540634617209e-13), SC_(0.4705151131450806597841891098742774312111e-13), 
      SC_(0.1334530223958893535574077304772799834609e-12), SC_(0.6672651119794245056505554066901540498883e-13), 
      SC_(0.266297021326439287136622624529991298914e-12), SC_(0.1331485106632107793053653966876952769505e-12), 
      SC_(0.5920415525016708979677559909760020673275e-12), SC_(0.29602077625079163483389193486561174812e-12), 
      SC_(0.155163989296047688526414276566356420517e-11), SC_(0.7758199464799374943373933162927267207972e-12), 
      SC_(0.326923297461201300961874949280172586441e-11), SC_(0.1634616487304670519279090616362400067267e-11), 
      SC_(0.3753785910581841633870681107509881258011e-11), SC_(0.1876892955289159453352533516401529801122e-11), 
      SC_(0.9579165585749116473834874341264367103577e-11), SC_(0.4789582792863088185252592080679746767978e-11), 
      SC_(0.1858167439361402273334533674642443656921e-10), SC_(0.9290837196763851538764282981715353513818e-11), 
      SC_(0.5449485307451595872407779097557067871094e-10), SC_(0.2724742653688676823559737485578304836506e-10), 
      SC_(0.6089519166696533147842274047434329986572e-10), SC_(0.3044759583301913769320592805847923970217e-10), 
      SC_(0.1337744776064297980155970435589551925659e-9), SC_(0.6688723880097794765058899684143827289977e-10), 
      SC_(0.2554458866654840676346793770790100097656e-9), SC_(0.1277229433245854586915920539477320784002e-9), 
      SC_(0.9285605062636648199259070679545402526855e-9), SC_(0.4642802530240543332889135775318577029962e-9), 
      SC_(0.1698227447555211711005540564656257629395e-8), SC_(0.8491137234171087978551371134983090509552e-9), 
      SC_(0.339355921141759608872234821319580078125e-8), SC_(0.1696779604269267531629088042873986404561e-8), 
      SC_(0.6313728651008432279922999441623687744141e-8), SC_(0.3156864320521319970871232105329124668304e-8), 
      SC_(0.8383264749056706932606175541877746582031e-8), SC_(0.4191632365743462521529019590756957227221e-8), 
      SC_(0.1962631124285962869180366396903991699219e-7), SC_(0.9813155573280803193195787399755096477409e-8), 
      SC_(0.5256384838503436185419559478759765625e-7), SC_(0.2628192384714742037346966666026188341227e-7), 
      SC_(0.116242290459922514855861663818359375e-6), SC_(0.5812114354092759417537613189484537667262e-7), 
      SC_(0.1776920584006802528165280818939208984375e-6), SC_(0.8884602525353202473263890111106254261705e-7), 
      SC_(0.246631174150024889968335628509521484375e-6), SC_(0.1233155794716463747702014149912298889881e-6), 
      SC_(0.7932688959044753573834896087646484375e-6), SC_(0.3966343692928262265327186740527956086072e-6), 
      SC_(0.1372093493046122603118419647216796875e-5), SC_(0.6860465111931535414103361330991297079332e-6), 
      SC_(0.214747751670074649155139923095703125e-5), SC_(0.1073738181893531617762304794194699568343e-5), 
      SC_(0.527022712049074470996856689453125e-5), SC_(0.2635110088342783512028120843144783486949e-5), 
      SC_(0.9233162927557714283466339111328125e-5), SC_(0.461657080741584719962950298693491037714e-5), 
      SC_(0.269396477960981428623199462890625e-4), SC_(0.1346973318119308516416495011442959614762e-4), 
      SC_(0.3208058114978484809398651123046875e-4), SC_(0.1604016193149502975581143111396377659995e-4), 
      SC_(0.00010957030463032424449920654296875), SC_(0.5478365169091582645735757270110530879622e-4), 
      SC_(0.000126518702018074691295623779296875), SC_(0.6325735026285620681070344590408428362373e-4), 
      SC_(0.00028976381872780621051788330078125), SC_(0.0001448714155003885621472683658275189243597), 
      SC_(0.000687857042066752910614013671875), SC_(0.0003438693979519525361574726383087724411954), 
      SC_(0.00145484809763729572296142578125), SC_(0.0007271596682270997885618336103424497281234), 
      SC_(0.0073254108428955078125), SC_(0.003656022172385267560713803761753901548099), 
      SC_(0.09376299381256103515625), SC_(0.04583124537975104196281645139037475336575), 
      SC_(0.0944411754608154296875), SC_(0.04615542605332570615786127417302399776704), 
      SC_(0.264718532562255859375), SC_(0.1245970534205822199491885900676517027912), 
      SC_(0.27952671051025390625), SC_(0.1311616641799057655376985599143742875242), 
      SC_(0.31148135662078857421875), SC_(0.1451992650280511659331021017070743457453), 
      SC_(0.3574702739715576171875), SC_(0.1651052630434546351309425398611544399279), 
      SC_(0.362719058990478515625), SC_(0.1673555837834839069815597770182209283592), 
      SC_(0.45167791843414306640625), SC_(0.2048559741455171397876451754340651729329), 
      SC_(0.58441460132598876953125), SC_(0.2587353182166570039327003862804562417757), 
      SC_(0.59585726261138916015625), SC_(0.263272441958340655449252054139087309968), 
      SC_(0.5962116718292236328125), SC_(0.2634127084326893190571263246997038824839), 
      SC_(0.6005609035491943359375), SC_(0.265132761234643838648662687260938123223), 
      SC_(0.62944734096527099609375), SC_(0.2764980771490691893425893413189776581002), 
      SC_(0.67001712322235107421875), SC_(0.2922914234886615056408835783570718269519), 
      SC_(0.6982586383819580078125), SC_(0.3031725282486421443925178739003813209589), 
      SC_(0.75686132907867431640625), SC_(0.3254664571684469135448954017993676523574), 
      SC_(0.81158387660980224609375), SC_(0.3459509190939327287182180832834475919551), 
      SC_(0.826751708984375), SC_(0.3515737896927326105907938446356122609441), 
      SC_(0.83147108554840087890625), SC_(0.3533185454830658156956304096558121255029), 
      SC_(0.8679864406585693359375), SC_(0.3667430046129994134441282712127036765511), 
      SC_(0.91433393955230712890625), SC_(0.3835945719582406373102286771719657103388), 
      SC_(0.91501367092132568359375), SC_(0.3838401898056457601350544781643172326518), 
      SC_(0.918984889984130859375), SC_(0.3852743013512272831269535677278325535267), 
      SC_(0.92977702617645263671875), SC_(0.3891641465919182859360893492188078577616), 
      SC_(0.93538987636566162109375), SC_(0.3911829054317989358513931994433199352401), 
      SC_(0.93773555755615234375), SC_(0.3920257029078710024155036194055376074007), 
      SC_(0.94118559360504150390625), SC_(0.3932643660142325996680401116017235829116), 
      SC_(0.96221935749053955078125), SC_(0.4007924034240546775728864480529865192659), 
      SC_(0.98576259613037109375), SC_(0.4091708896121758495875515950932191750449), 
      SC_(0.99292266368865966796875), SC_(0.4117091285702801135545007937655927942821), 
   };
#undef SC_

#define SC_(x) static_cast<T>(BOOST_JOIN(x, L))
   static const boost::array<boost::array<T, 3>, 1400> powm1_data = {
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.1431564604442703013402649929484277542953e-29), SC_(-0.4876113153308343652049349438365788782568e-28), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.1791466932348087634896446282571611213266e-29), SC_(-0.6101991796549119337733033929476086235147e-28), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.6013619202535540063110633226832922483532e-29), SC_(-0.2048324441766037485142714404837079817647e-27), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.115805324961653822428570241697281798758e-28), SC_(-0.3944494481885382670819636026287935618847e-27), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.1422457400834001098175711728787848259007e-28), SC_(-0.4845092719324132218645425927278781231393e-27), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.4970121018327539153628705477876439795096e-28), SC_(-0.1692894082166527210362866376720838911671e-26), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.9660079415057497591758174164417478444323e-28), SC_(-0.3290360780895537432240111567751307160722e-26), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.1232929313253182131376331095427391968754e-27), SC_(-0.4199533030361316903341770129910084958631e-26), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.3296523285617759312781860549364832953326e-27), SC_(-0.1122842832471787971733666733700489713877e-25), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.528364435768055252017009628713605422886e-27), SC_(-0.1799684601724219422633144864504456459061e-25), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.886586057273120049620324386849842094685e-27), SC_(-0.301983851933262353667369307173490332305e-25), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.2499669674831043259218157022821422146034e-26), SC_(-0.851423131205077246153257242575159396693e-25), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.4131050397232622964314362671638736040881e-26), SC_(-0.1407094665264327834604146533729082291379e-24), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.7679738097881433551381658732998641759182e-26), SC_(-0.261582829282236901328884539002832986633e-24), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.199929739820949207249437007767740538737e-25), SC_(-0.6809891995464351323188379271234568515929e-24), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.5151477415246978459754129800826163591626e-25), SC_(-0.1754666656712664684348626881941041760604e-23), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.101200734533556026342258477595279955025e-24), SC_(-0.34470413088736644015953540038270992073e-23), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.2064292695896540981798546456623054911033e-24), SC_(-0.7031275246329503001381414764473488822476e-23), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.4063294332896333395257434433879773416284e-24), SC_(-0.1384015983694437753942599100031834846519e-22), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.8138195767936862452966745688936976428456e-24), SC_(-0.2771985511871706510319651593329877641942e-22), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.9575550627132253801929510132578249716542e-24), SC_(-0.3261569070527988349552625022548697814845e-22), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.2855160956298500804375620841706273850616e-23), SC_(-0.9725085302202836634236273890970136659468e-22), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.65201444297915461398563707001320281266e-23), SC_(-0.2220854156138726384270360939328805807496e-21), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.1310988374636350038320977491775043421995e-22), SC_(-0.4465413322989495361133815710858355990476e-21), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.2590288837798696209228010176465529547374e-22), SC_(-0.8822893101478278733146155237640398711321e-21), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.2937779542193655202274099291941187976629e-22), SC_(-0.1000649598540977920935091326953457942208e-20), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.7863513178004503049754083414326234074965e-22), SC_(-0.2678424705352926224139081344679444879469e-20), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.1903818607087388763706780167350761726053e-21), SC_(-0.6484677619663470968358777401078973025876e-20), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.3812242142377350870566942975497647799754e-21), SC_(-0.1298504028134944266344521819289603363198e-19), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.5493133580141330277178034419485741501887e-21), SC_(-0.1871039617763820057590094637996686252126e-19), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.9672153634284186955666772243312215295852e-21), SC_(-0.3294473432116758998094638207256180094583e-19), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.1702169477623814384559878647986894129041e-20), SC_(-0.5797831933845967032001409153808505427486e-19), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.4817114569977399785676754474621208412799e-20), SC_(-0.1640777904312887822627372613029956055284e-18), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.7538352992756463183303278501219690799218e-20), SC_(-0.2567670510166787584153650238170661635392e-18), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.2596305715949999708394617609422128090557e-19), SC_(-0.8843387446340099351857154955251648734277e-18), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.4444587480324321591032923385589104015025e-19), SC_(-0.1513889866135372642811595930728137815273e-17), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.9715574921498573937069095571295029856174e-19), SC_(-0.3309263341636914772803140564669851761043e-17), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.2036598542733453787268262970278076551267e-18), SC_(-0.6936945012060519151079315126759585887524e-17), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.4248971931658660264162106698360155121463e-18), SC_(-0.1447260421199383628860906433743869048343e-16), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.6521097487613458963613731825259556273977e-18), SC_(-0.2221178781221441480704197986591689852107e-16), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.1436126164096190058281493628911107407475e-17), SC_(-0.4891650475869226303643170206367209825168e-16), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.3118908901459261162419055180006211003274e-17), SC_(-0.1062344840825147041561189485134686674784e-15), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.3593346613595175715618300349429858897565e-17), SC_(-0.1223945090048411651974452688903537147646e-15), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.9445874854124767215374919304693435151421e-17), SC_(-0.3217399653341720695630524182811530497325e-15), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.2566182432094081539023303073498993853718e-16), SC_(-0.8740783246589107366187111286071499874759e-15), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.3363765695149349330660137891158001366421e-16), SC_(-0.114574655589157486912055633512820155344e-14), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.1073581901339262605326457800103412409953e-15), SC_(-0.3656773025840533327467103325113205753957e-14), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.186668406231853462907965823802669547149e-15), SC_(-0.6358192065586750730853832514556047648021e-14), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.3727540802657755688795382376099496468669e-15), SC_(-0.1269653544166010514919541179082417894329e-13), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.6211646767866855090717281839829411183018e-15), SC_(-0.2115775453968538439661164062909478793222e-13), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.1561186859754253464932505224282976996619e-14), SC_(-0.5317625036268033276607060773206517931366e-13), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.3092010764722992466335682593125966377556e-14), SC_(-0.1053182951942663933832876385506518241085e-12), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.6192850577371690132255643845837767003104e-14), SC_(-0.2109373203492102058861790697529436159834e-12), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.1047879028014987723427253740737796761096e-13), SC_(-0.3569225374615943825992302718383009611711e-12), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.1978473638988408750405412206418986897916e-13), SC_(-0.673896330253118868079779235488630600479e-12), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.4041816252346730475863978426787070930004e-13), SC_(-0.1376700243226680691549667180741589712713e-11), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.9410302262901834580155480125540634617209e-13), SC_(-0.3205283121576780981640095118048982901968e-11), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.1334530223958893535574077304772799834609e-12), SC_(-0.4545600218337299481258164553284154943185e-11), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.266297021326439287136622624529991298914e-12), SC_(-0.9070456229087110845090862250063546365371e-11), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.5920415525016708979677559909760020673275e-12), SC_(-0.2016577940297158071169597347971748413736e-10), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.155163989296047688526414276566356420517e-11), SC_(-0.5285106706035087367329172863247919698404e-10), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.326923297461201300961874949280172586441e-11), SC_(-0.1113547363379608255313626394695583529731e-9), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.3753785910581841633870681107509881258011e-11), SC_(-0.1278593002042478917504524915640565559874e-9), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.9579165585749116473834874341264367103577e-11), SC_(-0.3262800376442197422488996705022038458178e-9), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.1858167439361402273334533674642443656921e-10), SC_(-0.6329183231503080899469654537111874935801e-9), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.5449485307451595872407779097557067871094e-10), SC_(-0.1856172392031744263095629658180062804824e-8), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.6089519166696533147842274047434329986572e-10), SC_(-0.2074177049579756457337028786392419850624e-8), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.1337744776064297980155970435589551925659e-9), SC_(-0.455654943076412885664049644579104536681e-8), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.2554458866654840676346793770790100097656e-9), SC_(-0.8700851073320290645945712060227430593679e-8), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.9285605062636648199259070679545402526855e-9), SC_(-0.3162809427238687518730427707811416855358e-7), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.1698227447555211711005540564656257629395e-8), SC_(-0.5784404650086212080717891696990416782702e-7), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.339355921141759608872234821319580078125e-8), SC_(-0.1155894585519911283131451474518816660213e-6), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.6313728651008432279922999441623687744141e-8), SC_(-0.2150545767594101004999522536841406461754e-6), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.8383264749056706932606175541877746582031e-8), SC_(-0.2855459093319640788691378100921481385269e-6), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.1962631124285962869180366396903991699219e-7), SC_(-0.6684999204190548245454426331521639746678e-6), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.5256384838503436185419559478759765625e-7), SC_(-0.179039801806545165134802090058682340943e-5), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.116242290459922514855861663818359375e-6), SC_(-0.3959370311448686121052770212054786570022e-5), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.1776920584006802528165280818939208984375e-6), SC_(-0.6052426544092270726332077550989562797738e-5), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.246631174150024889968335628509521484375e-6), SC_(-0.8400574149975703424776922592399345686356e-5), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.7932688959044753573834896087646484375e-6), SC_(-0.27019504593353575174331138594035344317e-4), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.1372093493046122603118419647216796875e-5), SC_(-0.4673436860116692184822065995212276990381e-4), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.214747751670074649155139923095703125e-5), SC_(-0.7314347094958011015415152898676611708704e-4), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.527022712049074470996856689453125e-5), SC_(-0.0001794953387273520847063999705353424942151), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.9233162927557714283466339111328125e-5), SC_(-0.0003144452455611145988190362789271573355057), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.269396477960981428623199462890625e-4), SC_(-0.0009171819454041827299228772649419689423581), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.3208058114978484809398651123046875e-4), SC_(-0.001092113543483231235223797564026187686493), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.00010957030463032424449920654296875), SC_(-0.003725165092629997722835852852785971357824), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.000126518702018074691295623779296875), SC_(-0.00430013520740725016534237907337289449972), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.00028976381872780621051788330078125), SC_(-0.009821222484048957677148666590881994977724), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.000687857042066752910614013671875), SC_(-0.02315705443914839871460170590017136367078), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.00145484809763729572296142578125), SC_(-0.04834642336924823485713723180889556615121), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.002847635187208652496337890625), SC_(-0.09243901554235025144966433539780526547712), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.0056468211114406585693359375), SC_(-0.1749726801434218290890659901859536145181), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.011621631681919097900390625), SC_(-0.3268919046371966783822069227177497837546), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.0257236398756504058837890625), SC_(-0.5836311906673689148844925144089516409523), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.0560617186129093170166015625), SC_(-0.851851787129424067479440220671140566588), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.106835305690765380859375), SC_(-0.9737204139282653351703557652385731401992), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.2401093542575836181640625), SC_(-0.9997193681670538982377759944125290364389), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.438671648502349853515625), SC_(-0.9999996757599823494298976392514104009862), 
      SC_(0.161179845478123719842988847972264920827e-14), SC_(0.903765499591827392578125), SC_(-0.9999999999999572557406571086231285055005), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.1431564604442703013402649929484277542953e-29), SC_(-0.4844007908534772131394346887325243880064e-28), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.1791466932348087634896446282571611213266e-29), SC_(-0.6061815136558852969659818106156943003957e-28), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.6013619202535540063110633226832922483532e-29), SC_(-0.2034837888950098477207926024848071370102e-27), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.115805324961653822428570241697281798758e-28), SC_(-0.391852318940973320303604982480182149622e-27), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.1422457400834001098175711728787848259007e-28), SC_(-0.4813191718913792878925970621589443986986e-27), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.4970121018327539153628705477876439795096e-28), SC_(-0.1681747749590792958677137312544793427331e-26), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.9660079415057497591758174164417478444323e-28), SC_(-0.3268696427558630548790076209679634511913e-26), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.1232929313253182131376331095427391968754e-27), SC_(-0.417188251618426766144157233436814113815e-26), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.3296523285617759312781860549364832953326e-27), SC_(-0.11154498243841273107122578840758587454e-25), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.528364435768055252017009628713605422886e-27), SC_(-0.1787835140311622475639860991274546364897e-25), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.886586057273120049620324386849842094685e-27), SC_(-0.2999955335372043527184121145958580830903e-25), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.2499669674831043259218157022821422146034e-26), SC_(-0.8458172014052994532431758589389027360406e-25), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.4131050397232622964314362671638736040881e-26), SC_(-0.139783008972484357758582011811719467464e-24), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.7679738097881433551381658732998641759182e-26), SC_(-0.2598605188069413080179985074671226218721e-24), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.199929739820949207249437007767740538737e-25), SC_(-0.6765054387615232771938258210126025383256e-24), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.5151477415246978459754129800826163591626e-25), SC_(-0.1743113601904746001017522277567702601782e-23), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.101200734533556026342258477595279955025e-24), SC_(-0.3424345341514720877425029223076741866833e-23), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.2064292695896540981798546456623054911033e-24), SC_(-0.6984980009579182199231442594111782966722e-23), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.4063294332896333395257434433879773416284e-24), SC_(-0.1374903362528766565146765802153324723796e-22), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.8138195767936862452966745688936976428456e-24), SC_(-0.2753734238660982596336253868089307184809e-22), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.9575550627132253801929510132578249716542e-24), SC_(-0.3240094287219449814641214477324994087466e-22), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.2855160956298500804375620841706273850616e-23), SC_(-0.9661053514126047301784928416981811109473e-22), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.65201444297915461398563707001320281266e-23), SC_(-0.2206231635280927768363223999023196596939e-21), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.1310988374636350038320977491775043421995e-22), SC_(-0.4436012203031383617142073222753070828535e-21), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.2590288837798696209228010176465529547374e-22), SC_(-0.8764801516289811807308128013848969166913e-21), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.2937779542193655202274099291941187976629e-22), SC_(-0.9940611336543627968065635102501795200551e-21), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.7863513178004503049754083414326234074965e-22), SC_(-0.2660789454063773222600366980456030660059e-20), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.1903818607087388763706780167350761726053e-21), SC_(-0.6441981284342428898077697521581402414226e-20), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.3812242142377350870566942975497647799754e-21), SC_(-0.1289954433744490880354316722541691455113e-19), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.5493133580141330277178034419485741501887e-21), SC_(-0.1858720341524588254633292074950762494054e-19), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.9672153634284186955666772243312215295852e-21), SC_(-0.327278200031182352483851821159619105861e-19), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.1702169477623814384559878647986894129041e-20), SC_(-0.5759657919515339462877754431293255193149e-19), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.4817114569977399785676754474621208412799e-20), SC_(-0.1629974714439967767492007802627804358342e-18), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.7538352992756463183303278501219690799218e-20), SC_(-0.2550764485299244254665139001625061474125e-18), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.2596305715949999708394617609422128090557e-19), SC_(-0.8785160922536063934682803376716121644709e-18), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.4444587480324321591032923385589104015025e-19), SC_(-0.1503922130936379389738401096793841960612e-17), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.9715574921498573937069095571295029856174e-19), SC_(-0.3287474530290037904749700610945658689136e-17), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.2036598542733453787268262970278076551267e-18), SC_(-0.6891270863288581151471557672725046281868e-17), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.4248971931658660264162106698360155121463e-18), SC_(-0.1437731386779379837932143059395401138972e-16), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.6521097487613458963613731825259556273977e-18), SC_(-0.2206554122971131139129524597311766415841e-16), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.1436126164096190058281493628911107407475e-17), SC_(-0.4859442930445883228137348922949175216834e-16), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.3118908901459261162419055180006211003274e-17), SC_(-0.1055350162876483889457554761503780038655e-15), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.3593346613595175715618300349429858897565e-17), SC_(-0.1215886405709071742924307747054308864747e-15), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.9445874854124767215374919304693435151421e-17), SC_(-0.3196215689771298575136390344854601244426e-15), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.2566182432094081539023303073498993853718e-16), SC_(-0.8683232288106103751410423217086764148434e-15), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.3363765695149349330660137891158001366421e-16), SC_(-0.1138202745387419932577321617647410529503e-14), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.1073581901339262605326457800103412409953e-15), SC_(-0.3632696145467823089954720178273131694505e-14), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.186668406231853462907965823802669547149e-15), SC_(-0.6316328534908726298725443019958683850532e-14), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.3727540802657755688795382376099496468669e-15), SC_(-0.1261293906780355188892580207761067957886e-13), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.6211646767866855090717281839829411183018e-15), SC_(-0.2101844791020430642773541142203342494795e-13), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.1561186859754253464932505224282976996619e-14), SC_(-0.5282612794337671118039133316909578883138e-13), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.3092010764722992466335682593125966377556e-14), SC_(-0.1046248597591077806008202786356426820985e-12), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.6192850577371690132255643845837767003104e-14), SC_(-0.20954846941635251434268850248704701934e-12), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.1047879028014987723427253740737796761096e-13), SC_(-0.3545724924421034074396236737574831845485e-12), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.1978473638988408750405412206418986897916e-13), SC_(-0.6694592702517320235988690147574107350722e-12), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.4041816252346730475863978426787070930004e-13), SC_(-0.136763579027020869461040423925528219022e-11), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.9410302262901834580155480125540634617209e-13), SC_(-0.3184178935527112257177727440556197379289e-11), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.1334530223958893535574077304772799834609e-12), SC_(-0.4515671132800559230661721186901784591853e-11), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.266297021326439287136622624529991298914e-12), SC_(-0.9010734641772444722067928174106284922519e-11), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.5920415525016708979677559909760020673275e-12), SC_(-0.2003300412409264445197987894935007034666e-10), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.155163989296047688526414276566356420517e-11), SC_(-0.5250308570899088268956597325889034576883e-10), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.326923297461201300961874949280172586441e-11), SC_(-0.1106215558406617834692768957077652441882e-9), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.3753785910581841633870681107509881258011e-11), SC_(-0.1270174505587799408364463356223313843969e-9), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.9579165585749116473834874341264367103577e-11), SC_(-0.3241317485987752496947990132937528973582e-9), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.1858167439361402273334533674642443656921e-10), SC_(-0.6287510700449849979080586601206737290248e-9), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.5449485307451595872407779097557067871094e-10), SC_(-0.1843951004410181545727179659293501080432e-8), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.6089519166696533147842274047434329986572e-10), SC_(-0.2060520278352719993567570837393709141609e-8), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.1337744776064297980155970435589551925659e-9), SC_(-0.4526548253624255940192936818697766683679e-8), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.2554458866654840676346793770790100097656e-9), SC_(-0.864356303601324975300150098008619669461e-8), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.9285605062636648199259070679545402526855e-9), SC_(-0.3141984896296551029496096338645100058567e-7), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.1698227447555211711005540564656257629395e-8), SC_(-0.5746319044607867904532351843764246487694e-7), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.339355921141759608872234821319580078125e-8), SC_(-0.1148283958919762100905636417765440193453e-6), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.6313728651008432279922999441623687744141e-8), SC_(-0.2136386171883463991491417449073061348682e-6), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.8383264749056706932606175541877746582031e-8), SC_(-0.2836658217038255820052319695862381911414e-6), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.1962631124285962869180366396903991699219e-7), SC_(-0.6640983928550264487076716288393362934515e-6), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.5256384838503436185419559478759765625e-7), SC_(-0.1778609711747170689000833844616637008016e-5), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.116242290459922514855861663818359375e-6), SC_(-0.3933301124986543207818818644487273191085e-5), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.1776920584006802528165280818939208984375e-6), SC_(-0.6012576350834688181085732977929129409074e-5), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.246631174150024889968335628509521484375e-6), SC_(-0.8345263422871632821588559198783594782703e-5), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.7932688959044753573834896087646484375e-6), SC_(-0.2684160548823891069028421052079627257037e-4), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.1372093493046122603118419647216796875e-5), SC_(-0.4642666788425309547022984501278551973067e-4), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.214747751670074649155139923095703125e-5), SC_(-0.7266189809699178326523971745169713532905e-4), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.527022712049074470996856689453125e-5), SC_(-0.0001783136130811065049523382962320407639286), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.9233162927557714283466339111328125e-5), SC_(-0.0003123752024084363268668477179963218575383), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.269396477960981428623199462890625e-4), SC_(-0.000911145799218773746646444219615156988864), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.3208058114978484809398651123046875e-4), SC_(-0.001084926764902528738172824319788974827948), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.00010957030463032424449920654296875), SC_(-0.003700683345993665188517190883989775516663), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.000126518702018074691295623779296875), SC_(-0.004271882870878222814910007056691176648148), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.00028976381872780621051788330078125), SC_(-0.009756874208170178110266341097702108886193), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.000687857042066752910614013671875), SC_(-0.02300635158962086528635239434937191719585), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.00145484809763729572296142578125), SC_(-0.04803587273567413116934075718056596923777), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.002847635187208652496337890625), SC_(-0.09185923466664269951387331370012753321941), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.0056468211114406585693359375), SC_(-0.1739272081185672090101525836675528469431), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.011621631681919097900390625), SC_(-0.3251352635416613049576932478851735637997), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.0257236398756504058837890625), SC_(-0.5812222341617614641543552054608546168083), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.0560617186129093170166015625), SC_(-0.8499773932245513164849346383273868845555), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.106835305690765380859375), SC_(-0.9730831614940445343420828088736870749082), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.2401093542575836181640625), SC_(-0.9997038422742669808353938658563990853002), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.438671648502349853515625), SC_(-0.999999642239583643658188237392236744113), 
      SC_(0.201701245224235325981254618454840965569e-14), SC_(0.903765499591827392578125), SC_(-0.9999999999999476516969892665713129740181), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.1431564604442703013402649929484277542953e-29), SC_(-0.4670646585263469940709901808955382437758e-28), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.1791466932348087634896446282571611213266e-29), SC_(-0.5844869930575957226200600044822824983155e-28), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.6013619202535540063110633226832922483532e-29), SC_(-0.19620134436288115149103308253280183024e-27), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.115805324961653822428570241697281798758e-28), SC_(-0.3778283871429173295224218281033374232068e-27), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.1422457400834001098175711728787848259007e-28), SC_(-0.4640933270681455745331795406744730255659e-27), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.4970121018327539153628705477876439795096e-28), SC_(-0.1621559983430480636456924332764721998048e-26), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.9660079415057497591758174164417478444323e-28), SC_(-0.3151713642073222703755018012599557417151e-26), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.1232929313253182131376331095427391968754e-27), SC_(-0.4022575767063603482081249297839543231515e-26), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.3296523285617759312781860549364832953326e-27), SC_(-0.1075529192285806320476237467015488238885e-25), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.528364435768055252017009628713605422886e-27), SC_(-0.1723850631704759160263389734502030547382e-25), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.886586057273120049620324386849842094685e-27), SC_(-0.2892590476248142013731511886947366714432e-25), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.2499669674831043259218157022821422146034e-26), SC_(-0.8155464025028249421637501209664265047502e-25), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.4131050397232622964314362671638736040881e-26), SC_(-0.1347803401362882902734793654342734095988e-24), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.7679738097881433551381658732998641759182e-26), SC_(-0.2505604176805653145243612785884078114724e-24), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.199929739820949207249437007767740538737e-25), SC_(-0.6522941079217671379314515550175652218872e-24), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.5151477415246978459754129800826163591626e-25), SC_(-0.1680729624350543500140339548813665482187e-23), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.101200734533556026342258477595279955025e-24), SC_(-0.330179206518811807797546994456889233539e-23), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.2064292695896540981798546456623054911033e-24), SC_(-0.6734995822858371523491495726201894721699e-23), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.4063294332896333395257434433879773416284e-24), SC_(-0.1325697194661413074389975129129737205398e-22), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.8138195767936862452966745688936976428456e-24), SC_(-0.2655181341851992399243727184578853247614e-22), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.9575550627132253801929510132578249716542e-24), SC_(-0.312413513856354742046541652296217654716e-22), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.2855160956298500804375620841706273850616e-23), SC_(-0.9315295816568866899504021320688046754633e-22), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.65201444297915461398563707001320281266e-23), SC_(-0.2127273210159156512760646198518192723583e-21), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.1310988374636350038320977491775043421995e-22), SC_(-0.4277252564301193321055067703470477963339e-21), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.2590288837798696209228010176465529547374e-22), SC_(-0.8451119619446267401173718362869071620453e-21), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.2937779542193655202274099291941187976629e-22), SC_(-0.9584848594622303420066613446868755698132e-21), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.7863513178004503049754083414326234074965e-22), SC_(-0.2565562941346879248671683738197120292887e-20), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.1903818607087388763706780167350761726053e-21), SC_(-0.6211430380828240395740546833358999026185e-20), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.3812242142377350870566942975497647799754e-21), SC_(-0.1243788487731146761472488262362637259264e-19), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.5493133580141330277178034419485741501887e-21), SC_(-0.1792198935267128535562926214858722343728e-19), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.9672153634284186955666772243312215295852e-21), SC_(-0.3155652997001798429320322164486809845992e-19), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.1702169477623814384559878647986894129041e-20), SC_(-0.55535265635450219027295966056920134225e-19), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.4817114569977399785676754474621208412799e-20), SC_(-0.157163984407441759959674443782140863952e-18), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.7538352992756463183303278501219690799218e-20), SC_(-0.2459475636297617006441590683912433787683e-18), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.2596305715949999708394617609422128090557e-19), SC_(-0.8470750386583229571630557052498880681824e-18), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.4444587480324321591032923385589104015025e-19), SC_(-0.1450098533692296614683964680209407156037e-17), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.9715574921498573937069095571295029856174e-19), SC_(-0.3169819698680942798209909967313405111447e-17), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.2036598542733453787268262970278076551267e-18), SC_(-0.6644640416262319536369758321461137730053e-17), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.4248971931658660264162106698360155121463e-18), SC_(-0.1386276678111046853303166140297811374299e-16), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.6521097487613458963613731825259556273977e-18), SC_(-0.2127584156395719014286056179015852384457e-16), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.1436126164096190058281493628911107407475e-17), SC_(-0.4685529205965962627641463132289456691771e-16), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.3118908901459261162419055180006211003274e-17), SC_(-0.1017580426698205682533290718321643209762e-15), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.3593346613595175715618300349429858897565e-17), SC_(-0.1172371266960037029314016045477824196056e-15), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.9445874854124767215374919304693435151421e-17), SC_(-0.3081826904306483808166787058542751288317e-15), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.2566182432094081539023303073498993853718e-16), SC_(-0.8372469657622804501545876592687226293867e-15), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.3363765695149349330660137891158001366421e-16), SC_(-0.1097467813112902235711533181290032884344e-14), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.1073581901339262605326457800103412409953e-15), SC_(-0.3502686239887104556183603267483924538092e-14), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.186668406231853462907965823802669547149e-15), SC_(-0.6090274594926766310097318898918904189533e-14), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.3727540802657755688795382376099496468669e-15), SC_(-0.1216153687184881360673588742979606365464e-13), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.6211646767866855090717281839829411183018e-15), SC_(-0.2026622247795390792057156419945858437211e-13), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.1561186859754253464932505224282976996619e-14), SC_(-0.5093554320105476807018586816442371125346e-13), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.3092010764722992466335682593125966377556e-14), SC_(-0.1008804595687293085666255201604454630459e-12), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.6192850577371690132255643845837767003104e-14), SC_(-0.2020489771295036962841043702748443936045e-12), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.1047879028014987723427253740737796761096e-13), SC_(-0.3418827616146511291456502884207649745993e-12), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.1978473638988408750405412206418986897916e-13), SC_(-0.6455001134628760079521534669131105956391e-12), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.4041816252346730475863978426787070930004e-13), SC_(-0.1318689720232516020580123962164582013965e-11), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.9410302262901834580155480125540634617209e-13), SC_(-0.3070220931283945435837713059354241443233e-11), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.1334530223958893535574077304772799834609e-12), SC_(-0.4354060595035063074489098732513294021707e-11), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.266297021326439287136622624529991298914e-12), SC_(-0.8688251088765620667457021296035152000077e-11), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.5920415525016708979677559909760020673275e-12), SC_(-0.1931604656134805361799836893722292796705e-10), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.155163989296047688526414276566356420517e-11), SC_(-0.5062406226683881116608414737604687082455e-10), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.326923297461201300961874949280172586441e-11), SC_(-0.1066625409785978045275661888117913045346e-9), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.3753785910581841633870681107509881258011e-11), SC_(-0.1224716459849947744876770763335625029259e-9), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.9579165585749116473834874341264367103577e-11), SC_(-0.3125314560510405517739269322326717366916e-9), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.1858167439361402273334533674642443656921e-10), SC_(-0.6062488116803458186700114397560238184659e-9), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.5449485307451595872407779097557067871094e-10), SC_(-0.1777958175347169005832208106622024055763e-8), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.6089519166696533147842274047434329986572e-10), SC_(-0.1986776690713578859161025588821554119682e-8), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.1337744776064297980155970435589551925659e-9), SC_(-0.4364548436901294166447367781125164853821e-8), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.2554458866654840676346793770790100097656e-9), SC_(-0.8334220122510207334577043840076605398572e-8), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.9285605062636648199259070679545402526855e-9), SC_(-0.3029536968598131272157030834689682539192e-7), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.1698227447555211711005540564656257629395e-8), SC_(-0.5540665076917325215577133620146511883601e-7), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.339355921141759608872234821319580078125e-8), SC_(-0.1107188234191793003548407513913256517037e-6), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.6313728651008432279922999441623687744141e-8), SC_(-0.2059927441298561653785769202991913439107e-6), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.8383264749056706932606175541877746582031e-8), SC_(-0.2735137582842921999867218511345265044419e-6), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.1962631124285962869180366396903991699219e-7), SC_(-0.6403311031475707881066461509462911410011e-6), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.5256384838503436185419559478759765625e-7), SC_(-0.1714955424917509853565377296478446588717e-5), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.116242290459922514855861663818359375e-6), SC_(-0.3792533189075770307948598804199129382162e-5), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.1776920584006802528165280818939208984375e-6), SC_(-0.5797393966603564614109548422983248771233e-5), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.246631174150024889968335628509521484375e-6), SC_(-0.8046597500633838071232896567874258574425e-5), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.7932688959044753573834896087646484375e-6), SC_(-0.2588098855334784114746356115187656688766e-4), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.1372093493046122603118419647216796875e-5), SC_(-0.4476514945146166310348073528504896865758e-4), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.214747751670074649155139923095703125e-5), SC_(-0.7006150566189193979269085691512524345184e-4), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.527022712049074470996856689453125e-5), SC_(-0.0001719325280577979557725849711147940153461), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.9233162927557714283466339111328125e-5), SC_(-0.0003011973465610182074617306088919186693139), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.269396477960981428623199462890625e-4), SC_(-0.0008785512954806844971925787811026104679626), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.3208058114978484809398651123046875e-4), SC_(-0.001046118833759536454554604741205345872518), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.00010957030463032424449920654296875), SC_(-0.00356847685564822123148350977400303322428), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.000126518702018074691295623779296875), SC_(-0.004119312560332169043533898018737790564652), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.00028976381872780621051788330078125), SC_(-0.009409335282965604709458452119979719540133), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.000687857042066752910614013671875), SC_(-0.02219218723675143425941273152091920012992), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.00145484809763729572296142578125), SC_(-0.04635721512666774012012900306815754937453), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.002847635187208652496337890625), SC_(-0.08872213856761157161353738758915939189135), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.0056468211114406585693359375), SC_(-0.1682589533286847094444390355424124915849), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.011621631681919097900390625), SC_(-0.315570268386307973049039982116169159971), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.0257236398756504058837890625), SC_(-0.5679715214637885742711302281747963865236), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.0560617186129093170166015625), SC_(-0.8394386222531893115849360215766369975108), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.106835305690765380859375), SC_(-0.9693654408932809842438913488064211267056), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.2401093542575836181640625), SC_(-0.9996039015988134850627985522402102650313), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.438671648502349853515625), SC_(-0.999999391442516368767686856850429328651), 
      SC_(0.6770733299921779385632802927830198314041e-14), SC_(0.903765499591827392578125), SC_(-0.9999999999998436069674179642725048390556), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.1431564604442703013402649929484277542953e-29), SC_(-0.4576836344104429456517604264584932284747e-28), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.1791466932348087634896446282571611213266e-29), SC_(-0.5727475336975031881646903362900562654118e-28), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.6013619202535540063110633226832922483532e-29), SC_(-0.1922606275703748803999379921456053425121e-27), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.115805324961653822428570241697281798758e-28), SC_(-0.3702396793553404218237800202711211418232e-27), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.1422457400834001098175711728787848259007e-28), SC_(-0.4547719823382898701812602276896112982996e-27), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.4970121018327539153628705477876439795096e-28), SC_(-0.1588990845448724714046351014391003029543e-26), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.9660079415057497591758174164417478444323e-28), SC_(-0.3088411268102136096173518347405241169365e-26), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.1232929313253182131376331095427391968754e-27), SC_(-0.3941782070537865987806008706345989318215e-26), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.3296523285617759312781860549364832953326e-27), SC_(-0.1053927118341642116888565560193377342088e-25), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.528364435768055252017009628713605422886e-27), SC_(-0.1689226979383767837562270129142188112561e-25), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.886586057273120049620324386849842094685e-27), SC_(-0.2834492607955699817292337275058110941043e-25), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.2499669674831043259218157022821422146034e-26), SC_(-0.7991660998405411529548763861428929797149e-25), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.4131050397232622964314362671638736040881e-26), SC_(-0.1320732682179000408587410618703226772025e-24), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.7679738097881433551381658732998641759182e-26), SC_(-0.2455278953566357627103257555388002069096e-24), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.199929739820949207249437007767740538737e-25), SC_(-0.6391927382390703388925725094314601383285e-24), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.5151477415246978459754129800826163591626e-25), SC_(-0.1646972060273454609443143683096363086915e-23), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.101200734533556026342258477595279955025e-24), SC_(-0.3235475356304688137060009654456926802713e-23), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.2064292695896540981798546456623054911033e-24), SC_(-0.6599722992680869704727736254365743724253e-23), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.4063294332896333395257434433879773416284e-24), SC_(-0.1299070480080302189776038836003092355401e-22), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.8138195767936862452966745688936976428456e-24), SC_(-0.2601851851501339074899141114386754841731e-22), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.9575550627132253801929510132578249716542e-24), SC_(-0.3061386680633381331917827751652934185889e-22), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.2855160956298500804375620841706273850616e-23), SC_(-0.9128197492799882445346136216771188720859e-22), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.65201444297915461398563707001320281266e-23), SC_(-0.2084546789049532018546149488447629509599e-21), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.1310988374636350038320977491775043421995e-22), SC_(-0.4191343667699763810630900480617325436685e-21), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.2590288837798696209228010176465529547374e-22), SC_(-0.8281378330934832948361688337452449936706e-21), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.2937779542193655202274099291941187976629e-22), SC_(-0.9392336285733129065505426348368290979349e-21), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.7863513178004503049754083414326234074965e-22), SC_(-0.2514033442412874267581007499883858640708e-20), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.1903818607087388763706780167350761726053e-21), SC_(-0.6086673396686778493371550510019707589098e-20), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.3812242142377350870566942975497647799754e-21), SC_(-0.1218806914868614442363178428163220534513e-19), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.5493133580141330277178034419485741501887e-21), SC_(-0.1756202502813247862463984994792399363846e-19), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.9672153634284186955666772243312215295852e-21), SC_(-0.3092271500830152539390293064404827736998e-19), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.1702169477623814384559878647986894129041e-20), SC_(-0.5441983620464496005891242970018046790907e-19), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.4817114569977399785676754474621208412799e-20), SC_(-0.1540073355346077842799391738372805101797e-18), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.7538352992756463183303278501219690799218e-20), SC_(-0.2410076907801688874670912850628612782649e-18), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.2596305715949999708394617609422128090557e-19), SC_(-0.8300614812834058274447270567991076964986e-18), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.4444587480324321591032923385589104015025e-19), SC_(-0.1420973210106639000468002738423441514335e-17), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.9715574921498573937069095571295029856174e-19), SC_(-0.3106153663382499810004103501514106185175e-17), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.2036598542733453787268262970278076551267e-18), SC_(-0.6511182380316850294413660631165069138149e-17), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.4248971931658660264162106698360155121463e-18), SC_(-0.1358433220655484535433282563890822329602e-16), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.6521097487613458963613731825259556273977e-18), SC_(-0.2084851489910661607334823468566561947389e-16), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.1436126164096190058281493628911107407475e-17), SC_(-0.4591420046399868101410450579900532388592e-16), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.3118908901459261162419055180006211003274e-17), SC_(-0.9971422574887292722937153254757672122027e-16), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.3593346613595175715618300349429858897565e-17), SC_(-0.1148824113632603900131783308508965417968e-15), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.9445874854124767215374919304693435151421e-17), SC_(-0.3019928210019576647967558032888200642401e-15), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.2566182432094081539023303073498993853718e-16), SC_(-0.820430805872203077560259601545938909032e-15), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.3363765695149349330660137891158001366421e-16), SC_(-0.1075425100539178680519852196810206870717e-14), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.1073581901339262605326457800103412409953e-15), SC_(-0.3432334558407927537802268484739968889925e-14), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.186668406231853462907965823802669547149e-15), SC_(-0.5967951032643659578568659958577023318101e-14), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.3727540802657755688795382376099496468669e-15), SC_(-0.1191727161092919798123625157343681290628e-13), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.6211646767866855090717281839829411183018e-15), SC_(-0.1985917407826593212205439443077791764114e-13), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.1561186859754253464932505224282976996619e-14), SC_(-0.4991249949521462427180932679900775040103e-13), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.3092010764722992466335682593125966377556e-14), SC_(-0.9885426896157956603114149340902946160351e-13), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.6192850577371690132255643845837767003104e-14), SC_(-0.1979908102516547879748989765228991496763e-12), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.1047879028014987723427253740737796761096e-13), SC_(-0.3350160240592180797771261476764337516904e-12), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.1978473638988408750405412206418986897916e-13), SC_(-0.6325352016018118342307089629651851902982e-12), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.4041816252346730475863978426787070930004e-13), SC_(-0.1292203751232165767835988707455380182517e-11), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.9410302262901834580155480125540634617209e-13), SC_(-0.300855534372191830458549151625414202144e-11), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.1334530223958893535574077304772799834609e-12), SC_(-0.426660900412917518009780811303798217014e-11), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.266297021326439287136622624529991298914e-12), SC_(-0.8513746999234309491732296073981503017467e-11), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.5920415525016708979677559909760020673275e-12), SC_(-0.1892808250689370669879461004467640065353e-10), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.155163989296047688526414276566356420517e-11), SC_(-0.4960727467589400330882774486696882456676e-10), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.326923297461201300961874949280172586441e-11), SC_(-0.1045202168894907835176851266302666007759e-9), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.3753785910581841633870681107509881258011e-11), SC_(-0.1200117949912250367534712217121551184824e-9), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.9579165585749116473834874341264367103577e-11), SC_(-0.306254241382315870578416840778207876678e-9), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.1858167439361402273334533674642443656921e-10), SC_(-0.594072264778111368274375677089333414961e-9), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.5449485307451595872407779097557067871094e-10), SC_(-0.1742247769515788478898057219060667252918e-8), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.6089519166696533147842274047434329986572e-10), SC_(-0.1946872151395269773560274878675690506499e-8), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.1337744776064297980155970435589551925659e-9), SC_(-0.4276886197195062037700785013461254705044e-8), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.2554458866654840676346793770790100097656e-9), SC_(-0.8166826768699312855473201998535874187492e-8), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.9285605062636648199259070679545402526855e-9), SC_(-0.2968688521982070868659209377155196789766e-7), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.1698227447555211711005540564656257629395e-8), SC_(-0.5429380460655229791611910734387908997677e-7), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.339355921141759608872234821319580078125e-8), SC_(-0.1084950287533348777373165891522186913259e-6), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.6313728651008432279922999441623687744141e-8), SC_(-0.2018553668522742749769063428064977349517e-6), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.8383264749056706932606175541877746582031e-8), SC_(-0.2680202173541270856078281970646947949939e-6), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.1962631124285962869180366396903991699219e-7), SC_(-0.6274700152306449635938038051873467096003e-6), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.5256384838503436185419559478759765625e-7), SC_(-0.1680510462347347402690704488039448237192e-5), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.116242290459922514855861663818359375e-6), SC_(-0.3716360054269949707624089241905458968568e-5), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.1776920584006802528165280818939208984375e-6), SC_(-0.5680953261582804705578830765241198560168e-5), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.246631174150024889968335628509521484375e-6), SC_(-0.7884981702423820096156321424607597246034e-5), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.7932688959044753573834896087646484375e-6), SC_(-0.2536117380268745579986453976758213516184e-4), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.1372093493046122603118419647216796875e-5), SC_(-0.4386605822686713834736918688546026975306e-4), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.214747751670074649155139923095703125e-5), SC_(-0.6865436405912565016099095734914674324923e-4), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.527022712049074470996856689453125e-5), SC_(-0.0001684795429426290375748066540229315936542), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.9233162927557714283466339111328125e-5), SC_(-0.0002951486710691193140380235522757858793913), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.269396477960981428623199462890625e-4), SC_(-0.000860913133371783332027113437255719030992), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.3208058114978484809398651123046875e-4), SC_(-0.001025118242070136625968938047128000303572), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.00010957030463032424449920654296875), SC_(-0.003496929231056143015211140903172753003707), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.000126518702018074691295623779296875), SC_(-0.004036743119796422185275261877051587365852), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.00028976381872780621051788330078125), SC_(-0.009221222234750118900172217847005146800635), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.000687857042066752910614013671875), SC_(-0.02175133923710351978467930052642747145844), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.00145484809763729572296142578125), SC_(-0.04544761667106517358284474532983548098311), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.002847635187208652496337890625), SC_(-0.08702005945634039017497404179236497904558), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.0056468211114406585693359375), SC_(-0.1651755152288914603917240984520648158053), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.011621631681919097900390625), SC_(-0.3103379996752338368650923165157389362038), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.0257236398756504058837890625), SC_(-0.5606272348374307625044520965986379186451), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.0560617186129093170166015625), SC_(-0.833430364008775042020638226729314813071), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.106835305690765380859375), SC_(-0.967143887842835777817854037909635743789), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.2401093542575836181640625), SC_(-0.9995364074628132999593163736445072059694), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.438671648502349853515625), SC_(-0.9999991887694171862610058365254266630964), 
      SC_(0.1303852045862058384173565173114184290171e-13), SC_(0.903765499591827392578125), SC_(-0.9999999999997172371310746311970351796287), 
      SC_(1.2698681354522705078125), SC_(0.1431564604442703013402649929484277542953e-29), SC_(0.3420194870024062814047728978212532998563e-30), 
      SC_(1.2698681354522705078125), SC_(0.1791466932348087634896446282571611213266e-29), SC_(0.4280048551647399999228434903587663134634e-30), 
      SC_(1.2698681354522705078125), SC_(0.6013619202535540063110633226832922483532e-29), SC_(0.1436732193780183251915891734035523793972e-29), 
      SC_(1.2698681354522705078125), SC_(0.115805324961653822428570241697281798758e-28), SC_(0.2766740509831951084519809145718404791547e-29), 
      SC_(1.2698681354522705078125), SC_(0.1422457400834001098175711728787848259007e-28), SC_(0.3398436570771565748173304184306615543909e-29), 
      SC_(1.2698681354522705078125), SC_(0.4970121018327539153628705477876439795096e-28), SC_(0.1187426844553768113983924271844288423442e-28), 
      SC_(1.2698681354522705078125), SC_(0.9660079415057497591758174164417478444323e-28), SC_(0.2307919178559647802149642900401165721481e-28), 
      SC_(1.2698681354522705078125), SC_(0.1232929313253182131376331095427391968754e-27), SC_(0.2945629208213355182163252959258097705325e-28), 
      SC_(1.2698681354522705078125), SC_(0.3296523285617759312781860549364832953326e-27), SC_(0.7875824811115599105986865868056157469008e-28), 
      SC_(1.2698681354522705078125), SC_(0.528364435768055252017009628713605422886e-27), SC_(0.1262331666422106497789462114589822777359e-27), 
      SC_(1.2698681354522705078125), SC_(0.886586057273120049620324386849842094685e-27), SC_(0.2118169920875373173985427015088627653426e-27), 
      SC_(1.2698681354522705078125), SC_(0.2499669674831043259218157022821422146034e-26), SC_(0.5972037428195599891261787402460995389719e-27), 
      SC_(1.2698681354522705078125), SC_(0.4131050397232622964314362671638736040881e-26), SC_(0.9869619109454157638442734233027413870915e-27), 
      SC_(1.2698681354522705078125), SC_(0.7679738097881433551381658732998641759182e-26), SC_(0.1834789765267188977663823170091590188669e-26), 
      SC_(1.2698681354522705078125), SC_(0.199929739820949207249437007767740538737e-25), SC_(0.4776582687073725066644404893622921417791e-26), 
      SC_(1.2698681354522705078125), SC_(0.5151477415246978459754129800826163591626e-25), SC_(0.1230755257149676211409917633859091287298e-25), 
      SC_(1.2698681354522705078125), SC_(0.101200734533556026342258477595279955025e-24), SC_(0.2417817764005694971846160775891732628672e-25), 
      SC_(1.2698681354522705078125), SC_(0.2064292695896540981798546456623054911033e-24), SC_(0.4931864944706428865614203337067832232701e-25), 
      SC_(1.2698681354522705078125), SC_(0.4063294332896333395257434433879773416284e-24), SC_(0.9707741019609786198310523485010740172696e-25), 
      SC_(1.2698681354522705078125), SC_(0.8138195767936862452966745688936976428456e-24), SC_(0.1944321292267828828344050037733790054678e-24), 
      SC_(1.2698681354522705078125), SC_(0.9575550627132253801929510132578249716542e-24), SC_(0.228772414677875118955602612841516306209e-24), 
      SC_(1.2698681354522705078125), SC_(0.2855160956298500804375620841706273850616e-23), SC_(0.6821352543587544946351638809580690112947e-24), 
      SC_(1.2698681354522705078125), SC_(0.65201444297915461398563707001320281266e-23), SC_(0.1557747688185563600528634178272482438289e-23), 
      SC_(1.2698681354522705078125), SC_(0.1310988374636350038320977491775043421995e-22), SC_(0.3132122504061177888115197841327782195659e-23), 
      SC_(1.2698681354522705078125), SC_(0.2590288837798696209228010176465529547374e-22), SC_(0.6188538447671767027554645970431141067489e-23), 
      SC_(1.2698681354522705078125), SC_(0.2937779542193655202274099291941187976629e-22), SC_(0.7018739139183943070584537523519395160206e-23), 
      SC_(1.2698681354522705078125), SC_(0.7863513178004503049754083414326234074965e-22), SC_(0.1878696032879880597606438295234445400993e-22), 
      SC_(1.2698681354522705078125), SC_(0.1903818607087388763706780167350761726053e-21), SC_(0.4548471381039414088284351477845900938907e-22), 
      SC_(1.2698681354522705078125), SC_(0.3812242142377350870566942975497647799754e-21), SC_(0.9107944537176083904466648983578898062406e-22), 
      SC_(1.2698681354522705078125), SC_(0.5493133580141330277178034419485741501887e-21), SC_(0.1312381378587531773768084733888620567011e-21), 
      SC_(1.2698681354522705078125), SC_(0.9672153634284186955666772243312215295852e-21), SC_(0.2310803867279284395325040960542387104948e-21), 
      SC_(1.2698681354522705078125), SC_(0.1702169477623814384559878647986894129041e-20), SC_(0.4066705265842243511990384452435092146391e-21), 
      SC_(1.2698681354522705078125), SC_(0.4817114569977399785676754474621208412799e-20), SC_(0.1150871605055410326357052222776027126075e-20), 
      SC_(1.2698681354522705078125), SC_(0.7538352992756463183303278501219690799218e-20), SC_(0.1801011016495003147161825515613996028105e-20), 
      SC_(1.2698681354522705078125), SC_(0.2596305715949999708394617609422128090557e-19), SC_(0.6202913555663948280929759905533895728393e-20), 
      SC_(1.2698681354522705078125), SC_(0.4444587480324321591032923385589104015025e-19), SC_(0.1061870016372484253118587189306114048478e-19), 
      SC_(1.2698681354522705078125), SC_(0.9715574921498573937069095571295029856174e-19), SC_(0.2321177780082074129440532042729073364871e-19), 
      SC_(1.2698681354522705078125), SC_(0.2036598542733453787268262970278076551267e-18), SC_(0.4865699994634248908023627956961028830466e-19), 
      SC_(1.2698681354522705078125), SC_(0.4248971931658660264162106698360155121463e-18), SC_(0.1015134906132475864793915005929116755576e-18), 
      SC_(1.2698681354522705078125), SC_(0.6521097487613458963613731825259556273977e-18), SC_(0.1557975386150659035553775662378897074586e-18), 
      SC_(1.2698681354522705078125), SC_(0.1436126164096190058281493628911107407475e-17), SC_(0.3431093031991568741685817578089247870823e-18), 
      SC_(1.2698681354522705078125), SC_(0.3118908901459261162419055180006211003274e-17), SC_(0.7451480842526166040425102554556006578373e-18), 
      SC_(1.2698681354522705078125), SC_(0.3593346613595175715618300349429858897565e-17), SC_(0.8584974520811751074564412604859386909514e-18), 
      SC_(1.2698681354522705078125), SC_(0.9445874854124767215374919304693435151421e-17), SC_(0.2256742910428662195907300458159668294607e-17), 
      SC_(1.2698681354522705078125), SC_(0.2566182432094081539023303073498993853718e-16), SC_(0.6130945095007296482544550987552049128862e-17), 
      SC_(1.2698681354522705078125), SC_(0.3363765695149349330660137891158001366421e-16), SC_(0.8036475712523944301253096611257704330887e-17), 
      SC_(1.2698681354522705078125), SC_(0.1073581901339262605326457800103412409953e-15), SC_(0.2564927422846314270714133018927527673521e-16), 
      SC_(1.2698681354522705078125), SC_(0.186668406231853462907965823802669547149e-15), SC_(0.4459752102059645897424792488546670241155e-16), 
      SC_(1.2698681354522705078125), SC_(0.3727540802657755688795382376099496468669e-15), SC_(0.8905581970586991466939320975391393663305e-16), 
      SC_(1.2698681354522705078125), SC_(0.6211646767866855090717281839829411183018e-15), SC_(0.1484043566313956188983726548784674614331e-15), 
      SC_(1.2698681354522705078125), SC_(0.1561186859754253464932505224282976996619e-14), SC_(0.3729879372757421060398162969122566665345e-15), 
      SC_(1.2698681354522705078125), SC_(0.3092010764722992466335682593125966377556e-14), SC_(0.7387217679695032934124755312004784940608e-15), 
      SC_(1.2698681354522705078125), SC_(0.6192850577371690132255643845837767003104e-14), SC_(0.1479552910837560805834157616713680398401e-14), 
      SC_(1.2698681354522705078125), SC_(0.1047879028014987723427253740737796761096e-13), SC_(0.2503519900464339012932954652447103903128e-14), 
      SC_(1.2698681354522705078125), SC_(0.1978473638988408750405412206418986897916e-13), SC_(0.4726832005727230498618201285266448046373e-14), 
      SC_(1.2698681354522705078125), SC_(0.4041816252346730475863978426787070930004e-13), SC_(0.9656427079124197097947552889241995946754e-14), 
      SC_(1.2698681354522705078125), SC_(0.9410302262901834580155480125540634617209e-13), SC_(0.2248244153639328280064179719350034279301e-13), 
      SC_(1.2698681354522705078125), SC_(0.1334530223958893535574077304772799834609e-12), SC_(0.3188367057771179968157617218069382264218e-13), 
      SC_(1.2698681354522705078125), SC_(0.266297021326439287136622624529991298914e-12), SC_(0.6362183749282957838259047728566148154723e-13), 
      SC_(1.2698681354522705078125), SC_(0.5920415525016708979677559909760020673275e-12), SC_(0.1414464617540412529087267058049917265071e-12), 
      SC_(1.2698681354522705078125), SC_(0.155163989296047688526414276566356420517e-11), SC_(0.370707042180260653184099429109179552233e-12), 
      SC_(1.2698681354522705078125), SC_(0.326923297461201300961874949280172586441e-11), SC_(0.7810624692721233057497812838799967569936e-12), 
      SC_(1.2698681354522705078125), SC_(0.3753785910581841633870681107509881258011e-11), SC_(0.896828496227341088445752699754610938274e-12), 
      SC_(1.2698681354522705078125), SC_(0.9579165585749116473834874341264367103577e-11), SC_(0.2288587807623386713791234899514841493899e-11), 
      SC_(1.2698681354522705078125), SC_(0.1858167439361402273334533674642443656921e-10), SC_(0.4439404777151353666181995924897026455737e-11), 
      SC_(1.2698681354522705078125), SC_(0.5449485307451595872407779097557067871094e-10), SC_(0.1301953235997653634700263262629394678664e-10), 
      SC_(1.2698681354522705078125), SC_(0.6089519166696533147842274047434329986572e-10), SC_(0.1454865686841154894708369394823364698654e-10), 
      SC_(1.2698681354522705078125), SC_(0.1337744776064297980155970435589551925659e-9), SC_(0.31960470427744776254708958163118342765e-10), 
      SC_(1.2698681354522705078125), SC_(0.2554458866654840676346793770790100097656e-9), SC_(0.6102935965707334929910468209231817062867e-10), 
      SC_(1.2698681354522705078125), SC_(0.9285605062636648199259070679545402526855e-9), SC_(0.2218452363607120861626064151860511000808e-9), 
      SC_(1.2698681354522705078125), SC_(0.1698227447555211711005540564656257629395e-8), SC_(0.4057287241815955603623730534846927438842e-9), 
      SC_(1.2698681354522705078125), SC_(0.339355921141759608872234821319580078125e-8), SC_(0.8107656318617117866294966834817726714903e-9), 
      SC_(1.2698681354522705078125), SC_(0.6313728651008432279922999441623687744141e-8), SC_(0.1508432263006119427013132176270683105493e-8), 
      SC_(1.2698681354522705078125), SC_(0.8383264749056706932606175541877746582031e-8), SC_(0.2002871475622493393270773060560290521481e-8), 
      SC_(1.2698681354522705078125), SC_(0.1962631124285962869180366396903991699219e-7), SC_(0.4688982179315023500581635814164190019958e-8), 
      SC_(1.2698681354522705078125), SC_(0.5256384838503436185419559478759765625e-7), SC_(0.125581901903791838140659620318980086674e-7), 
      SC_(1.2698681354522705078125), SC_(0.116242290459922514855861663818359375e-6), SC_(0.2777180225032743072377212027192763421518e-7), 
      SC_(1.2698681354522705078125), SC_(0.1776920584006802528165280818939208984375e-6), SC_(0.4245295515132007171579175941366890236852e-7), 
      SC_(1.2698681354522705078125), SC_(0.246631174150024889968335628509521484375e-6), SC_(0.5892341140943752460189213003787636411203e-7), 
      SC_(1.2698681354522705078125), SC_(0.7932688959044753573834896087646484375e-6), SC_(0.1895223210326633627186407689532055005047e-6), 
      SC_(1.2698681354522705078125), SC_(0.1372093493046122603118419647216796875e-5), SC_(0.3278111152459676807338808062624539138807e-6), 
      SC_(1.2698681354522705078125), SC_(0.214747751670074649155139923095703125e-5), SC_(0.5130605665632870980170027253254440754451e-6), 
      SC_(1.2698681354522705078125), SC_(0.527022712049074470996856689453125e-5), SC_(0.1259126905864534707564519232075067997608e-5), 
      SC_(1.2698681354522705078125), SC_(0.9233162927557714283466339111328125e-5), SC_(0.2205925685198983967324676018114026228766e-5), 
      SC_(1.2698681354522705078125), SC_(0.269396477960981428623199462890625e-4), SC_(0.6436254530234825694705316236264512301357e-5), 
      SC_(1.2698681354522705078125), SC_(0.3208058114978484809398651123046875e-4), SC_(0.7664499332850079936680739794066881971046e-5), 
      SC_(1.2698681354522705078125), SC_(0.00010957030463032424449920654296875), SC_(0.2617811992332665457636119426168275512229e-4), 
      SC_(1.2698681354522705078125), SC_(0.000126518702018074691295623779296875), SC_(0.3022742768393338821381257470272750370786e-4), 
      SC_(1.2698681354522705078125), SC_(0.00028976381872780621051788330078125), SC_(0.6923075831766976300466789565154105776674e-4), 
      SC_(1.2698681354522705078125), SC_(0.000687857042066752910614013671875), SC_(0.000164351538248818105860618367502154175576), 
      SC_(1.2698681354522705078125), SC_(0.00145484809763729572296142578125), SC_(0.0003476426314174918948018704195274956806275), 
      SC_(1.2698681354522705078125), SC_(0.002847635187208652496337890625), SC_(0.000680568731680074922669161880274103682441), 
      SC_(1.2698681354522705078125), SC_(0.0056468211114406585693359375), SC_(0.001350009781592489152232064712193415932619), 
      SC_(1.2698681354522705078125), SC_(0.011621631681919097900390625), SC_(0.002780417854006973908325787047211816372794), 
      SC_(1.2698681354522705078125), SC_(0.0257236398756504058837890625), SC_(0.006164637283266366505602149296090563266474), 
      SC_(1.2698681354522705078125), SC_(0.0560617186129093170166015625), SC_(0.0134839767907499950210480287606334307119), 
      SC_(1.2698681354522705078125), SC_(0.106835305690765380859375), SC_(0.02585288579558250741731425521987916267727), 
      SC_(1.2698681354522705078125), SC_(0.2401093542575836181640625), SC_(0.05904256744106580356642467885497221866553), 
      SC_(1.2698681354522705078125), SC_(0.438671648502349853515625), SC_(0.1104933632279540298689424673420546499493), 
      SC_(1.2698681354522705078125), SC_(0.903765499591827392578125), SC_(0.2410048139472866141620003722996635906314), 
      SC_(1.35477006435394287109375), SC_(0.1431564604442703013402649929484277542953e-29), SC_(0.4346684599696975402671455475334038730311e-30), 
      SC_(1.35477006435394287109375), SC_(0.1791466932348087634896446282571611213266e-29), SC_(0.543946232083267508305187631764097119092e-30), 
      SC_(1.35477006435394287109375), SC_(0.6013619202535540063110633226832922483532e-29), SC_(0.182592569661074201344820342946185653938e-29), 
      SC_(1.35477006435394287109375), SC_(0.115805324961653822428570241697281798758e-28), SC_(0.3516217298273326120488390423622761830313e-29), 
      SC_(1.35477006435394287109375), SC_(0.1422457400834001098175711728787848259007e-28), SC_(0.4319032238392848362014276761582776730152e-29), 
      SC_(1.35477006435394287109375), SC_(0.4970121018327539153628705477876439795096e-28), SC_(0.1509086521275416486211375995904547917102e-28), 
      SC_(1.35477006435394287109375), SC_(0.9660079415057497591758174164417478444323e-28), SC_(0.2933106776667358781322526450618145182053e-28), 
      SC_(1.35477006435394287109375), SC_(0.1232929313253182131376331095427391968754e-27), SC_(0.3743564797425857105137717585913322179135e-28), 
      SC_(1.35477006435394287109375), SC_(0.3296523285617759312781860549364832953326e-27), SC_(0.100092912004591899463442542292872840124e-27), 
      SC_(1.35477006435394287109375), SC_(0.528364435768055252017009628713605422886e-27), SC_(0.1604282160129719838241108522731077492994e-27), 
      SC_(1.35477006435394287109375), SC_(0.886586057273120049620324386849842094685e-27), SC_(0.269195672308156214888914671036944147799e-27), 
      SC_(1.35477006435394287109375), SC_(0.2499669674831043259218157022821422146034e-26), SC_(0.7589790671128956425988310244562285412391e-27), 
      SC_(1.35477006435394287109375), SC_(0.4131050397232622964314362671638736040881e-26), SC_(0.1254318043803087156788686077889298988149e-26), 
      SC_(1.35477006435394287109375), SC_(0.7679738097881433551381658732998641759182e-26), SC_(0.2331812285395425091460001937193325749895e-26), 
      SC_(1.35477006435394287109375), SC_(0.199929739820949207249437007767740538737e-25), SC_(0.6070501592482794530202847647513877332323e-26), 
      SC_(1.35477006435394287109375), SC_(0.5151477415246978459754129800826163591626e-25), SC_(0.1564152080671049736488993542676328625855e-25), 
      SC_(1.35477006435394287109375), SC_(0.101200734533556026342258477595279955025e-24), SC_(0.3072775569540396031565668649210987090127e-25), 
      SC_(1.35477006435394287109375), SC_(0.2064292695896540981798546456623054911033e-24), SC_(0.626784794949124855640879021859677666643e-25), 
      SC_(1.35477006435394287109375), SC_(0.4063294332896333395257434433879773416284e-24), SC_(0.1233745151704984504651169396130041455861e-24), 
      SC_(1.35477006435394287109375), SC_(0.8138195767936862452966745688936976428456e-24), SC_(0.2471014588096857154338036760034812976784e-24), 
      SC_(1.35477006435394287109375), SC_(0.9575550627132253801929510132578249716542e-24), SC_(0.2907441153225324701651770229136798814231e-24), 
      SC_(1.35477006435394287109375), SC_(0.2855160956298500804375620841706273850616e-23), SC_(0.8669175054960294708313833109432533870185e-24), 
      SC_(1.35477006435394287109375), SC_(0.65201444297915461398563707001320281266e-23), SC_(0.1979722835617877432909824292401135860431e-23), 
      SC_(1.35477006435394287109375), SC_(0.1310988374636350038320977491775043421995e-22), SC_(0.3980576888202649500521161897841410231419e-23), 
      SC_(1.35477006435394287109375), SC_(0.2590288837798696209228010176465529547374e-22), SC_(0.7864939217611961289331896158796651127429e-23), 
      SC_(1.35477006435394287109375), SC_(0.2937779542193655202274099291941187976629e-22), SC_(0.892003130960973877376705794115785453666e-23), 
      SC_(1.35477006435394287109375), SC_(0.7863513178004503049754083414326234074965e-22), SC_(0.238761223379454015302604155102700720118e-22), 
      SC_(1.35477006435394287109375), SC_(0.1903818607087388763706780167350761726053e-21), SC_(0.578059767219853091440898725349310555188e-22), 
      SC_(1.35477006435394287109375), SC_(0.3812242142377350870566942975497647799754e-21), SC_(0.1157517736828806740751659171592593909548e-21), 
      SC_(1.35477006435394287109375), SC_(0.5493133580141330277178034419485741501887e-21), SC_(0.1667889738456737745364769967835454077299e-21), 
      SC_(1.35477006435394287109375), SC_(0.9672153634284186955666772243312215295852e-21), SC_(0.2936772892929463971966445448499847054145e-21), 
      SC_(1.35477006435394287109375), SC_(0.1702169477623814384559878647986894129041e-20), SC_(0.5168326900162478077819960709525085950495e-21), 
      SC_(1.35477006435394287109375), SC_(0.4817114569977399785676754474621208412799e-20), SC_(0.1462628906255185640976980233954852620135e-20), 
      SC_(1.35477006435394287109375), SC_(0.7538352992756463183303278501219690799218e-20), SC_(0.2288883279106359382950009176243908980249e-20), 
      SC_(1.35477006435394287109375), SC_(0.2596305715949999708394617609422128090557e-19), SC_(0.7883208369781105252744306785950621655756e-20), 
      SC_(1.35477006435394287109375), SC_(0.4444587480324321591032923385589104015025e-19), SC_(0.1349517855692760347142895808847553614241e-19), 
      SC_(1.35477006435394287109375), SC_(0.9715574921498573937069095571295029856174e-19), SC_(0.2949956974168135667156808197073231787888e-19), 
      SC_(1.35477006435394287109375), SC_(0.2036598542733453787268262970278076551267e-18), SC_(0.6183759708777513872953328570501725677214e-19), 
      SC_(1.35477006435394287109375), SC_(0.4248971931658660264162106698360155121463e-18), SC_(0.1290122765159817800402213740283310109813e-18), 
      SC_(1.35477006435394287109375), SC_(0.6521097487613458963613731825259556273977e-18), SC_(0.1980012214228124779356815068874369496388e-18), 
      SC_(1.35477006435394287109375), SC_(0.1436126164096190058281493628911107407475e-17), SC_(0.4360534942905293245505752689260940008432e-18), 
      SC_(1.35477006435394287109375), SC_(0.3118908901459261162419055180006211003274e-17), SC_(0.9469997545174276650633607857032679082187e-18), 
      SC_(1.35477006435394287109375), SC_(0.3593346613595175715618300349429858897565e-17), SC_(0.109105410529256834805498048512462946672e-17), 
      SC_(1.35477006435394287109375), SC_(0.9445874854124767215374919304693435151421e-17), SC_(0.286806747188840236538677410101145183657e-17), 
      SC_(1.35477006435394287109375), SC_(0.2566182432094081539023303073498993853718e-16), SC_(0.7791744517138707380833772533614205899588e-17), 
      SC_(1.35477006435394287109375), SC_(0.3363765695149349330660137891158001366421e-16), SC_(0.1021346050238969028275187596396231325334e-16), 
      SC_(1.35477006435394287109375), SC_(0.1073581901339262605326457800103412409953e-15), SC_(0.3259735468858851290566465051497394384048e-16), 
      SC_(1.35477006435394287109375), SC_(0.186668406231853462907965823802669547149e-15), SC_(0.5667845405648623853390446357365525628399e-16), 
      SC_(1.35477006435394287109375), SC_(0.3727540802657755688795382376099496468669e-15), SC_(0.1131799721184225271458928249504868843214e-15), 
      SC_(1.35477006435394287109375), SC_(0.6211646767866855090717281839829411183018e-15), SC_(0.1886053151974602783674544556158370539306e-15), 
      SC_(1.35477006435394287109375), SC_(0.1561186859754253464932505224282976996619e-14), SC_(0.4740258916351755510976682204889287367117e-15), 
      SC_(1.35477006435394287109375), SC_(0.3092010764722992466335682593125966377556e-14), SC_(0.938832626303357894697051759656700794569e-15), 
      SC_(1.35477006435394287109375), SC_(0.6192850577371690132255643845837767003104e-14), SC_(0.1880346031841516855769337546314235790654e-14), 
      SC_(1.35477006435394287109375), SC_(0.1047879028014987723427253740737796761096e-13), SC_(0.3181693385882041955286922106493303892486e-14), 
      SC_(1.35477006435394287109375), SC_(0.1978473638988408750405412206418986897916e-13), SC_(0.6007274048833591601597783001452504589672e-14), 
      SC_(1.35477006435394287109375), SC_(0.4041816252346730475863978426787070930004e-13), SC_(0.122722372461282431021848075783061887515e-13), 
      SC_(1.35477006435394287109375), SC_(0.9410302262901834580155480125540634617209e-13), SC_(0.2857266503915347020988103314525614536204e-13), 
      SC_(1.35477006435394287109375), SC_(0.1334530223958893535574077304772799834609e-12), SC_(0.4052057416277482631320308942466563186577e-13), 
      SC_(1.35477006435394287109375), SC_(0.266297021326439287136622624529991298914e-12), SC_(0.8085622946758080825746284367765171804202e-13), 
      SC_(1.35477006435394287109375), SC_(0.5920415525016708979677559909760020673275e-12), SC_(0.1797626101297568840672137193060265187326e-12), 
      SC_(1.35477006435394287109375), SC_(0.155163989296047688526414276566356420517e-11), SC_(0.4711271294412824829106304681371916647916e-12), 
      SC_(1.35477006435394287109375), SC_(0.326923297461201300961874949280172586441e-11), SC_(0.9926429152742702703054554070735643184272e-12), 
      SC_(1.35477006435394287109375), SC_(0.3753785910581841633870681107509881258011e-11), SC_(0.1139768569120691065544692978325902161875e-11), 
      SC_(1.35477006435394287109375), SC_(0.9579165585749116473834874341264367103577e-11), SC_(0.2908538769425127863215161715077435412554e-11), 
      SC_(1.35477006435394287109375), SC_(0.1858167439361402273334533674642443656921e-10), SC_(0.5641986234702706533668562124484149537871e-11), 
      SC_(1.35477006435394287109375), SC_(0.5449485307451595872407779097557067871094e-10), SC_(0.1654636737235620470978205504080948724594e-10), 
      SC_(1.35477006435394287109375), SC_(0.6089519166696533147842274047434329986572e-10), SC_(0.184897133524674878963951844506509164495e-10), 
      SC_(1.35477006435394287109375), SC_(0.1337744776064297980155970435589551925659e-9), SC_(0.4061817817034875814852773093914474371976e-10), 
      SC_(1.35477006435394287109375), SC_(0.2554458866654840676346793770790100097656e-9), SC_(0.775614805103473668893076112107054063527e-10), 
      SC_(1.35477006435394287109375), SC_(0.9285605062636648199259070679545402526855e-9), SC_(0.2819404475708815591004595352083698435008e-9), 
      SC_(1.35477006435394287109375), SC_(0.1698227447555211711005540564656257629395e-8), SC_(0.5156357646777448789484766496043850538637e-9), 
      SC_(1.35477006435394287109375), SC_(0.339355921141759608872234821319580078125e-8), SC_(0.1030392308125733705668616120623767224539e-8), 
      SC_(1.35477006435394287109375), SC_(0.6313728651008432279922999441623687744141e-8), SC_(0.1917048454197643427385877634015244798617e-8), 
      SC_(1.35477006435394287109375), SC_(0.8383264749056706932606175541877746582031e-8), SC_(0.2545425313897712019064828317034450623532e-8), 
      SC_(1.35477006435394287109375), SC_(0.1962631124285962869180366396903991699219e-7), SC_(0.5959171162631144193665816152742671537119e-8), 
      SC_(1.35477006435394287109375), SC_(0.5256384838503436185419559478759765625e-7), SC_(0.1596005317470424257548009841588541970833e-7), 
      SC_(1.35477006435394287109375), SC_(0.116242290459922514855861663818359375e-6), SC_(0.3529485020265880596919475204028927512801e-7), 
      SC_(1.35477006435394287109375), SC_(0.1776920584006802528165280818939208984375e-6), SC_(0.5395295134970362732906233403283727078332e-7), 
      SC_(1.35477006435394287109375), SC_(0.246631174150024889968335628509521484375e-6), SC_(0.7488505676206374863406026808157919943437e-7), 
      SC_(1.35477006435394287109375), SC_(0.7932688959044753573834896087646484375e-6), SC_(0.2408616487025528515561978599530612181557e-6), 
      SC_(1.35477006435394287109375), SC_(0.1372093493046122603118419647216796875e-5), SC_(0.4166112293783892260959556502868348983591e-6), 
      SC_(1.35477006435394287109375), SC_(0.214747751670074649155139923095703125e-5), SC_(0.6520425598876542684833903774124534019394e-6), 
      SC_(1.35477006435394287109375), SC_(0.527022712049074470996856689453125e-5), SC_(0.16002095413128507927344180962142231847e-5), 
      SC_(1.35477006435394287109375), SC_(0.9233162927557714283466339111328125e-5), SC_(0.2803485308045222086184242704224431441333e-5), 
      SC_(1.35477006435394287109375), SC_(0.269396477960981428623199462890625e-4), SC_(0.817976574373463519902797340314014762871e-5), 
      SC_(1.35477006435394287109375), SC_(0.3208058114978484809398651123046875e-4), SC_(0.9740730299091002262280512341108226291389e-5), 
      SC_(1.35477006435394287109375), SC_(0.00010957030463032424449920654296875), SC_(0.3326957629495933828880645428848388897065e-4), 
      SC_(1.35477006435394287109375), SC_(0.000126518702018074691295623779296875), SC_(0.3841583223020501061702565234652379174836e-4), 
      SC_(1.35477006435394287109375), SC_(0.00028976381872780621051788330078125), SC_(0.8798536461436966877060675684116694683454e-4), 
      SC_(1.35477006435394287109375), SC_(0.000687857042066752910614013671875), SC_(0.0002088770462671550614362972773327027840634), 
      SC_(1.35477006435394287109375), SC_(0.00145484809763729572296142578125), SC_(0.0004418356482830615743464365455165332342344), 
      SC_(1.35477006435394287109375), SC_(0.002847635187208652496337890625), SC_(0.0008650063454750985541660058177986793841704), 
      SC_(1.35477006435394287109375), SC_(0.0056468211114406585693359375), SC_(0.001716024840256529625913567378533352874622), 
      SC_(1.35477006435394287109375), SC_(0.011621631681919097900390625), SC_(0.003534929494144331508395673950178983206543), 
      SC_(1.35477006435394287109375), SC_(0.0257236398756504058837890625), SC_(0.007841095311478799412510932341963582641969), 
      SC_(1.35477006435394287109375), SC_(0.0560617186129093170166015625), SC_(0.01716781927650258688653495893317532969192), 
      SC_(1.35477006435394287109375), SC_(0.106835305690765380859375), SC_(0.03297045686500958223466855780868842534087), 
      SC_(1.35477006435394287109375), SC_(0.2401093542575836181640625), SC_(0.07562815633502792792709905060267641476341), 
      SC_(1.35477006435394287109375), SC_(0.438671648502349853515625), SC_(0.1424723457223528966681250808495522250368), 
      SC_(1.35477006435394287109375), SC_(0.903765499591827392578125), SC_(0.3157566460256508623029725507037422680168), 
      SC_(2.2103404998779296875), SC_(0.1431564604442703013402649929484277542953e-29), SC_(0.113544056434324064497545784338755528187e-29), 
      SC_(2.2103404998779296875), SC_(0.1791466932348087634896446282571611213266e-29), SC_(0.1420895863417514235587593434730661156564e-29), 
      SC_(2.2103404998779296875), SC_(0.6013619202535540063110633226832922483532e-29), SC_(0.4769681479887127754732602999884494452096e-29), 
      SC_(2.2103404998779296875), SC_(0.115805324961653822428570241697281798758e-28), SC_(0.9185059697644637550578685161263247637075e-29), 
      SC_(2.2103404998779296875), SC_(0.1422457400834001098175711728787848259007e-28), SC_(0.1128217216984021149540833352700168873153e-28), 
      SC_(2.2103404998779296875), SC_(0.4970121018327539153628705477876439795096e-28), SC_(0.3942034468015438861282111618526526448826e-28), 
      SC_(2.2103404998779296875), SC_(0.9660079415057497591758174164417478444323e-28), SC_(0.7661858911986258510727160044348824771726e-28), 
      SC_(2.2103404998779296875), SC_(0.1232929313253182131376331095427391968754e-27), SC_(0.9778936632625772683753326669131270882148e-28), 
      SC_(2.2103404998779296875), SC_(0.3296523285617759312781860549364832953326e-27), SC_(0.2614626156707461888850641570798646477338e-27), 
      SC_(2.2103404998779296875), SC_(0.528364435768055252017009628713605422886e-27), SC_(0.41907044311208384016092465904276084092e-27), 
      SC_(2.2103404998779296875), SC_(0.886586057273120049620324386849842094685e-27), SC_(0.7031926956596745919082619252778429002887e-27), 
      SC_(2.2103404998779296875), SC_(0.2499669674831043259218157022821422146034e-26), SC_(0.1982604443735002847795119421501924343532e-26), 
      SC_(2.2103404998779296875), SC_(0.4131050397232622964314362671638736040881e-26), SC_(0.327652847786787611152824517398240571351e-26), 
      SC_(2.2103404998779296875), SC_(0.7679738097881433551381658732998641759182e-26), SC_(0.6091157976947442312678548919893389284067e-26), 
      SC_(2.2103404998779296875), SC_(0.199929739820949207249437007767740538737e-25), SC_(0.1585735885804947118054859952359699228642e-25), 
      SC_(2.2103404998779296875), SC_(0.5151477415246978459754129800826163591626e-25), SC_(0.4085876673268639950654533295689160154488e-25), 
      SC_(2.2103404998779296875), SC_(0.101200734533556026342258477595279955025e-24), SC_(0.8026701608444971511150594786716526222479e-25), 
      SC_(2.2103404998779296875), SC_(0.2064292695896540981798546456623054911033e-24), SC_(0.1637286683621835613460510758630465292556e-24), 
      SC_(2.2103404998779296875), SC_(0.4063294332896333395257434433879773416284e-24), SC_(0.3222787987435994441362857955482547801035e-24), 
      SC_(2.2103404998779296875), SC_(0.8138195767936862452966745688936976428456e-24), SC_(0.6454782108194994386823177367288297364745e-24), 
      SC_(2.2103404998779296875), SC_(0.9575550627132253801929510132578249716542e-24), SC_(0.7594815193269524677636810539738959230936e-24), 
      SC_(2.2103404998779296875), SC_(0.2855160956298500804375620841706273850616e-23), SC_(0.226456113643043688630560107923008237966e-23), 
      SC_(2.2103404998779296875), SC_(0.65201444297915461398563707001320281266e-23), SC_(0.517143022954522727544618651691532472474e-23), 
      SC_(2.2103404998779296875), SC_(0.1310988374636350038320977491775043421995e-22), SC_(0.1039805940524776941504913486327592128234e-22), 
      SC_(2.2103404998779296875), SC_(0.2590288837798696209228010176465529547374e-22), SC_(0.2054478722563207934819083258028876432179e-22), 
      SC_(2.2103404998779296875), SC_(0.2937779542193655202274099291941187976629e-22), SC_(0.2330089784947528107264468588426750195853e-22), 
      SC_(2.2103404998779296875), SC_(0.7863513178004503049754083414326234074965e-22), SC_(0.6236918552502042747352326270568582326212e-22), 
      SC_(2.2103404998779296875), SC_(0.1903818607087388763706780167350761726053e-21), SC_(0.1510007209545383730260305774543135012135e-21), 
      SC_(2.2103404998779296875), SC_(0.3812242142377350870566942975497647799754e-21), SC_(0.3023666802127385946619807910284467485336e-21), 
      SC_(2.2103404998779296875), SC_(0.5493133580141330277178034419485741501887e-21), SC_(0.4356860090625489418374213914686629672833e-21), 
      SC_(2.2103404998779296875), SC_(0.9672153634284186955666772243312215295852e-21), SC_(0.7671435537623837195216868945727243097762e-21), 
      SC_(2.2103404998779296875), SC_(0.1702169477623814384559878647986894129041e-20), SC_(0.1350069892957022932985394824585295494688e-20), 
      SC_(2.2103404998779296875), SC_(0.4817114569977399785676754474621208412799e-20), SC_(0.382067792739988712387271268803897239573e-20), 
      SC_(2.2103404998779296875), SC_(0.7538352992756463183303278501219690799218e-20), SC_(0.5979018864919508687514281036079968537009e-20), 
      SC_(2.2103404998779296875), SC_(0.2596305715949999708394617609422128090557e-19), SC_(0.2059250988867151744066203492948950586334e-19), 
      SC_(2.2103404998779296875), SC_(0.4444587480324321591032923385589104015025e-19), SC_(0.3525209341772555369249150375724257906549e-19), 
      SC_(2.2103404998779296875), SC_(0.9715574921498573937069095571295029856174e-19), SC_(0.7705874982903644524067519026267133873655e-19), 
      SC_(2.2103404998779296875), SC_(0.2036598542733453787268262970278076551267e-18), SC_(0.1615321160865183505128353787808388201179e-18), 
      SC_(2.2103404998779296875), SC_(0.4248971931658660264162106698360155121463e-18), SC_(0.3370057539135106965125306224301456904353e-18), 
      SC_(2.2103404998779296875), SC_(0.6521097487613458963613731825259556273977e-18), SC_(0.5172186144093411922105579120765768699787e-18), 
      SC_(2.2103404998779296875), SC_(0.1436126164096190058281493628911107407475e-17), SC_(0.113905854976364504284929597288735620002e-17), 
      SC_(2.2103404998779296875), SC_(0.3118908901459261162419055180006211003274e-17), SC_(0.2473751916062967142882431444403477459773e-17), 
      SC_(2.2103404998779296875), SC_(0.3593346613595175715618300349429858897565e-17), SC_(0.2850050562971067301359785244992360103055e-17), 
      SC_(2.2103404998779296875), SC_(0.9445874854124767215374919304693435151421e-17), SC_(0.7491963297917875107344765057203193285057e-17), 
      SC_(2.2103404998779296875), SC_(0.2566182432094081539023303073498993853718e-16), SC_(0.2035358809418812864287988540884004777584e-16), 
      SC_(2.2103404998779296875), SC_(0.3363765695149349330660137891158001366421e-16), SC_(0.2667959243589747787005179773360369279401e-16), 
      SC_(2.2103404998779296875), SC_(0.1073581901339262605326457800103412409953e-15), SC_(0.8515078091078428277880415261612326730386e-16), 
      SC_(2.2103404998779296875), SC_(0.186668406231853462907965823802669547149e-15), SC_(0.1480554072510474250247512240729600257286e-15), 
      SC_(2.2103404998779296875), SC_(0.3727540802657755688795382376099496468669e-15), SC_(0.2956486224545779278361194238326617566546e-15), 
      SC_(2.2103404998779296875), SC_(0.6211646767866855090717281839829411183018e-15), SC_(0.4926746365284206858817364168219676908241e-15), 
      SC_(2.2103404998779296875), SC_(0.1561186859754253464932505224282976996619e-14), SC_(0.1238250012317603604269397453825897591703e-14), 
      SC_(2.2103404998779296875), SC_(0.3092010764722992466335682593125966377556e-14), SC_(0.2452417751009692619762264155551728281242e-14), 
      SC_(2.2103404998779296875), SC_(0.6192850577371690132255643845837767003104e-14), SC_(0.491183823115104569225085526533817924118e-14), 
      SC_(2.2103404998779296875), SC_(0.1047879028014987723427253740737796761096e-13), SC_(0.8311216631372152968929101293659046224335e-14), 
      SC_(2.2103404998779296875), SC_(0.1978473638988408750405412206418986897916e-13), SC_(0.1569219592479211069148075557178993819683e-13), 
      SC_(2.2103404998779296875), SC_(0.4041816252346730475863978426787070930004e-13), SC_(0.3205752721388975304877892859290364631272e-13), 
      SC_(2.2103404998779296875), SC_(0.9410302262901834580155480125540634617209e-13), SC_(0.7463749018990070051529885490739294989884e-13), 
      SC_(2.2103404998779296875), SC_(0.1334530223958893535574077304772799834609e-12), SC_(0.1058478077707826704240767632367503550663e-12), 
      SC_(2.2103404998779296875), SC_(0.266297021326439287136622624529991298914e-12), SC_(0.2112125706653359695225983010011261143797e-12), 
      SC_(2.2103404998779296875), SC_(0.5920415525016708979677559909760020673275e-12), SC_(0.4695757302193123935681588076652272081297e-12), 
      SC_(2.2103404998779296875), SC_(0.155163989296047688526414276566356420517e-11), SC_(0.1230677868294645738601754834515720019433e-11), 
      SC_(2.2103404998779296875), SC_(0.326923297461201300961874949280172586441e-11), SC_(0.2592980939978512831393456948513604570032e-11), 
      SC_(2.2103404998779296875), SC_(0.3753785910581841633870681107509881258011e-11), SC_(0.2977302442037073373659741633226979002019e-11), 
      SC_(2.2103404998779296875), SC_(0.9579165585749116473834874341264367103577e-11), SC_(0.7597682385347750350558637534305099370148e-11), 
      SC_(2.2103404998779296875), SC_(0.1858167439361402273334533674642443656921e-10), SC_(0.1473799142183762080678986023548495818435e-10), 
      SC_(2.2103404998779296875), SC_(0.5449485307451595872407779097557067871094e-10), SC_(0.4322240612686516303198646898232044277138e-10), 
      SC_(2.2103404998779296875), SC_(0.6089519166696533147842274047434329986572e-10), SC_(0.4829881276696936992411088540931931106568e-10), 
      SC_(2.2103404998779296875), SC_(0.1337744776064297980155970435589551925659e-9), SC_(0.1061027688759849881043035142977131281485e-9), 
      SC_(2.2103404998779296875), SC_(0.2554458866654840676346793770790100097656e-9), SC_(0.202606030383735803246335572381604218027e-9), 
      SC_(2.2103404998779296875), SC_(0.9285605062636648199259070679545402526855e-9), SC_(0.7364845864274135340389459936374528632136e-9), 
      SC_(2.2103404998779296875), SC_(0.1698227447555211711005540564656257629395e-8), SC_(0.1346943286212731290152899320206693483058e-8), 
      SC_(2.2103404998779296875), SC_(0.339355921141759608872234821319580078125e-8), SC_(0.2691589872627375759762663884560486790864e-8), 
      SC_(2.2103404998779296875), SC_(0.6313728651008432279922999441623687744141e-8), SC_(0.5007712273908649713033224252630498129911e-8), 
      SC_(2.2103404998779296875), SC_(0.8383264749056706932606175541877746582031e-8), SC_(0.6649157753560806770698669762163438399867e-8), 
      SC_(2.2103404998779296875), SC_(0.1962631124285962869180366396903991699219e-7), SC_(0.1556654168303510693017729921289761484476e-7), 
      SC_(2.2103404998779296875), SC_(0.5256384838503436185419559478759765625e-7), SC_(0.416908372372826741357131178708640289097e-7), 
      SC_(2.2103404998779296875), SC_(0.116242290459922514855861663818359375e-6), SC_(0.9219717891539001165302722466972755173166e-7), 
      SC_(2.2103404998779296875), SC_(0.1776920584006802528165280818939208984375e-6), SC_(0.140935857635182293325780947427104819961e-6), 
      SC_(2.2103404998779296875), SC_(0.246631174150024889968335628509521484375e-6), SC_(0.1956146904456601163353269942965359902827e-6), 
      SC_(2.2103404998779296875), SC_(0.7932688959044753573834896087646484375e-6), SC_(0.629178706570496148987844682096273195374e-6), 
      SC_(2.2103404998779296875), SC_(0.1372093493046122603118419647216796875e-5), SC_(0.108827184813522739421660086353558290147e-5), 
      SC_(2.2103404998779296875), SC_(0.214747751670074649155139923095703125e-5), SC_(0.1703265889973953905010651792171315917245e-5), 
      SC_(2.2103404998779296875), SC_(0.527022712049074470996856689453125e-5), SC_(0.4180071331858123189933278770825483662709e-5), 
      SC_(2.2103404998779296875), SC_(0.9233162927557714283466339111328125e-5), SC_(0.7323278376758031314660215592026101660667e-5), 
      SC_(2.2103404998779296875), SC_(0.269396477960981428623199462890625e-4), SC_(0.2136731768612834393845491631839395961775e-4), 
      SC_(2.2103404998779296875), SC_(0.3208058114978484809398651123046875e-4), SC_(0.2544492681185144948683838084521582047796e-4), 
      SC_(2.2103404998779296875), SC_(0.00010957030463032424449920654296875), SC_(0.8690908832535010797435364557905581016139e-4), 
      SC_(2.2103404998779296875), SC_(0.000126518702018074691295623779296875), SC_(0.00010035291032264936941113629988396983743), 
      SC_(2.2103404998779296875), SC_(0.00028976381872780621051788330078125), SC_(0.0002298515925041686812616062307466566814354), 
      SC_(2.2103404998779296875), SC_(0.000687857042066752910614013671875), SC_(0.0005457203088721067508285680739202453850977), 
      SC_(2.2103404998779296875), SC_(0.00145484809763729572296142578125), SC_(0.001154573794984565689401328437672284137464), 
      SC_(2.2103404998779296875), SC_(0.002847635187208652496337890625), SC_(0.002261144638932774280741629860724815198317), 
      SC_(2.2103404998779296875), SC_(0.0056468211114406585693359375), SC_(0.004488801451427910647070198885794977682362), 
      SC_(2.2103404998779296875), SC_(0.011621631681919097900390625), SC_(0.00926027081123263749767139964130364819499), 
      SC_(2.2103404998779296875), SC_(0.0257236398756504058837890625), SC_(0.02061217301579647881319524823726294980894), 
      SC_(2.2103404998779296875), SC_(0.0560617186129093170166015625), SC_(0.04546855212175825897801442555081432132684), 
      SC_(2.2103404998779296875), SC_(0.106835305690765380859375), SC_(0.08842974526620618171455331479918566244557), 
      SC_(2.2103404998779296875), SC_(0.2401093542575836181640625), SC_(0.2097840978949737299216830878053929952957), 
      SC_(2.2103404998779296875), SC_(0.438671648502349853515625), SC_(0.416134414130719918038219956020285288407), 
      SC_(2.2103404998779296875), SC_(0.903765499591827392578125), SC_(1.047907441370060274169644114866504275322), 
      SC_(3.081670284271240234375), SC_(0.1431564604442703013402649929484277542953e-29), SC_(0.1611185520717568314823155973617711167757e-29), 
      SC_(3.081670284271240234375), SC_(0.1791466932348087634896446282571611213266e-29), SC_(0.2016245423563825681157746100335493201015e-29), 
      SC_(3.081670284271240234375), SC_(0.6013619202535540063110633226832922483532e-29), SC_(0.6768158528204367266258854475192690611878e-29), 
      SC_(3.081670284271240234375), SC_(0.115805324961653822428570241697281798758e-28), SC_(0.1303356217534067036250250716172227240104e-28), 
      SC_(3.081670284271240234375), SC_(0.1422457400834001098175711728787848259007e-28), SC_(0.1600935620333729507106200488860234636498e-28), 
      SC_(3.081670284271240234375), SC_(0.4970121018327539153628705477876439795096e-28), SC_(0.5593730800616404164427053831623290303e-28), 
      SC_(3.081670284271240234375), SC_(0.9660079415057497591758174164417478444323e-28), SC_(0.1087214648519581934497528988754812776161e-27), 
      SC_(3.081670284271240234375), SC_(0.1232929313253182131376331095427391968754e-27), SC_(0.1387627111914451504334411009396474302627e-27), 
      SC_(3.081670284271240234375), SC_(0.3296523285617759312781860549364832953326e-27), SC_(0.3710143831450268844517355879522607084369e-27), 
      SC_(3.081670284271240234375), SC_(0.528364435768055252017009628713605422886e-27), SC_(0.5946592462049591428384073261891171039041e-27), 
      SC_(3.081670284271240234375), SC_(0.886586057273120049620324386849842094685e-27), SC_(0.9978275614774745463815833774434083935213e-27), 
      SC_(3.081670284271240234375), SC_(0.2499669674831043259218157022821422146034e-26), SC_(0.2813307603558986394305127835522469606386e-26), 
      SC_(3.081670284271240234375), SC_(0.4131050397232622964314362671638736040881e-26), SC_(0.4649380520250322071349650654192689223679e-26), 
      SC_(3.081670284271240234375), SC_(0.7679738097881433551381658732998641759182e-26), SC_(0.8643328277193990796566355800772368371045e-26), 
      SC_(3.081670284271240234375), SC_(0.199929739820949207249437007767740538737e-25), SC_(0.2250152741697216913904384523516261928793e-25), 
      SC_(3.081670284271240234375), SC_(0.5151477415246978459754129800826163591626e-25), SC_(0.579784230204589050279913597684573640108e-25), 
      SC_(3.081670284271240234375), SC_(0.101200734533556026342258477595279955025e-24), SC_(0.1138985678050647200131155222333664297409e-24), 
      SC_(3.081670284271240234375), SC_(0.2064292695896540981798546456623054911033e-24), SC_(0.2323303113132160111598615290786783476886e-24), 
      SC_(3.081670284271240234375), SC_(0.4063294332896333395257434433879773416284e-24), SC_(0.4573122983943089982242669378258844610491e-24), 
      SC_(3.081670284271240234375), SC_(0.8138195767936862452966745688936976428456e-24), SC_(0.9159309433449788336954133179744170085861e-24), 
      SC_(3.081670284271240234375), SC_(0.9575550627132253801929510132578249716542e-24), SC_(0.1077701172231728538372225932972842404746e-23), 
      SC_(3.081670284271240234375), SC_(0.2855160956298500804375620841706273850616e-23), SC_(0.3213402998251057022555087473420489435704e-23), 
      SC_(3.081670284271240234375), SC_(0.65201444297915461398563707001320281266e-23), SC_(0.7338238362184864815106492247439099374513e-23), 
      SC_(3.081670284271240234375), SC_(0.1310988374636350038320977491775043421995e-22), SC_(0.1475480380339123385829505774877819111234e-22), 
      SC_(3.081670284271240234375), SC_(0.2590288837798696209228010176465529547374e-22), SC_(0.2915296911495156299456268705511236745575e-22), 
      SC_(3.081670284271240234375), SC_(0.2937779542193655202274099291941187976629e-22), SC_(0.3306387882707776095613692598267905692144e-22), 
      SC_(3.081670284271240234375), SC_(0.7863513178004503049754083414326234074965e-22), SC_(0.8850161938241560734432334736419307545243e-22), 
      SC_(3.081670284271240234375), SC_(0.1903818607087388763706780167350761726053e-21), SC_(0.2142694059557310026468132635557608861119e-21), 
      SC_(3.081670284271240234375), SC_(0.3812242142377350870566942975497647799754e-21), SC_(0.4290570835717772210461778143386231860226e-21), 
      SC_(3.081670284271240234375), SC_(0.5493133580141330277178034419485741501887e-21), SC_(0.6182366663875839410089349666830403828667e-21), 
      SC_(3.081670284271240234375), SC_(0.9672153634284186955666772243312215295852e-21), SC_(0.1088573567783975509940118455542324120655e-20), 
      SC_(3.081670284271240234375), SC_(0.1702169477623814384559878647986894129041e-20), SC_(0.1915743660917430126387769015482828415204e-20), 
      SC_(3.081670284271240234375), SC_(0.4817114569977399785676754474621208412799e-20), SC_(0.5421526365417942386504300766346803457766e-20), 
      SC_(3.081670284271240234375), SC_(0.7538352992756463183303278501219690799218e-20), SC_(0.8484203335493463512606833846920578585603e-20), 
      SC_(3.081670284271240234375), SC_(0.2596305715949999708394617609422128090557e-19), SC_(0.2922068737878133969259824029588766352176e-19), 
      SC_(3.081670284271240234375), SC_(0.4444587480324321591032923385589104015025e-19), SC_(0.5002257649873140104824664051662958782155e-19), 
      SC_(3.081670284271240234375), SC_(0.9715574921498573937069095571295029856174e-19), SC_(0.1093460510995174418291578793832011515439e-18), 
      SC_(3.081670284271240234375), SC_(0.2036598542733453787268262970278076551267e-18), SC_(0.2292134126104661541032908482925310173531e-18), 
      SC_(3.081670284271240234375), SC_(0.4248971931658660264162106698360155121463e-18), SC_(0.4782097875973148530365392653363141428528e-18), 
      SC_(3.081670284271240234375), SC_(0.6521097487613458963613731825259556273977e-18), SC_(0.7339311001839622378170319237223033369594e-18), 
      SC_(3.081670284271240234375), SC_(0.1436126164096190058281493628911107407475e-17), SC_(0.1616319427243881800616876839070988687325e-17), 
      SC_(3.081670284271240234375), SC_(0.3118908901459261162419055180006211003274e-17), SC_(0.3510243859671668114015447789294114453729e-17), 
      SC_(3.081670284271240234375), SC_(0.3593346613595175715618300349429858897565e-17), SC_(0.4044210101854173772052568584500977262538e-17), 
      SC_(3.081670284271240234375), SC_(0.9445874854124767215374919304693435151421e-17), SC_(0.1063106530312731394370945221114577589865e-16), 
      SC_(3.081670284271240234375), SC_(0.2566182432094081539023303073498993853718e-16), SC_(0.2888165832878597073070174805437777192811e-16), 
      SC_(3.081670284271240234375), SC_(0.3363765695149349330660137891158001366421e-16), SC_(0.3785823263785517775913841283360266404166e-16), 
      SC_(3.081670284271240234375), SC_(0.1073581901339262605326457800103412409953e-15), SC_(0.1208286101356655978989267825821491117805e-15), 
      SC_(3.081670284271240234375), SC_(0.186668406231853462907965823802669547149e-15), SC_(0.2100900178467809780355311483309898859117e-15), 
      SC_(3.081670284271240234375), SC_(0.3727540802657755688795382376099496468669e-15), SC_(0.4195241870669270679568235822116939980401e-15), 
      SC_(3.081670284271240234375), SC_(0.6211646767866855090717281839829411183018e-15), SC_(0.6991032958722281494972268237207947405754e-15), 
      SC_(3.081670284271240234375), SC_(0.1561186859754253464932505224282976996619e-14), SC_(0.1757071707252635757021776909227155254672e-14), 
      SC_(3.081670284271240234375), SC_(0.3092010764722992466335682593125966377556e-14), SC_(0.3479970766645161024115324864896348377545e-14), 
      SC_(3.081670284271240234375), SC_(0.6192850577371690132255643845837767003104e-14), SC_(0.6969878377310835469094076936692296574283e-14), 
      SC_(3.081670284271240234375), SC_(0.1047879028014987723427253740737796761096e-13), SC_(0.1179358243534255376613100037832741628033e-13), 
      SC_(3.081670284271240234375), SC_(0.1978473638988408750405412206418986897916e-13), SC_(0.2226716188963402971286477297381342039567e-13), 
      SC_(3.081670284271240234375), SC_(0.4041816252346730475863978426787070930004e-13), SC_(0.4548950011038629785664879661486185524825e-13), 
      SC_(3.081670284271240234375), SC_(0.9410302262901834580155480125540634617209e-13), SC_(0.1059102935662924931807407941027836281315e-12), 
      SC_(3.081670284271240234375), SC_(0.1334530223958893535574077304772799834609e-12), SC_(0.1501976066696445847907892536948793939421e-12), 
      SC_(3.081670284271240234375), SC_(0.266297021326439287136622624529991298914e-12), SC_(0.2997097746339349648220262840418873714391e-12), 
      SC_(3.081670284271240234375), SC_(0.5920415525016708979677559909760020673275e-12), SC_(0.6663260422155390476630130771377319671241e-12), 
      SC_(3.081670284271240234375), SC_(0.155163989296047688526414276566356420517e-11), SC_(0.174632686582879261942845237972252727791e-11), 
      SC_(3.081670284271240234375), SC_(0.326923297461201300961874949280172586441e-11), SC_(0.3679429357369118715402219339783099461878e-11), 
      SC_(3.081670284271240234375), SC_(0.3753785910581841633870681107509881258011e-11), SC_(0.422477999822460603402470434270331290893e-11), 
      SC_(3.081670284271240234375), SC_(0.9579165585749116473834874341264367103577e-11), SC_(0.1078108025617639564996840882401703556569e-10), 
      SC_(3.081670284271240234375), SC_(0.1858167439361402273334533674642443656921e-10), SC_(0.2091314959945325300456139698423630112915e-10), 
      SC_(3.081670284271240234375), SC_(0.5449485307451595872407779097557067871094e-10), SC_(0.6133241766211751141791377828628892970034e-10), 
      SC_(3.081670284271240234375), SC_(0.6089519166696533147842274047434329986572e-10), SC_(0.685358179393465879297183182258883516519e-10), 
      SC_(3.081670284271240234375), SC_(0.1337744776064297980155970435589551925659e-9), SC_(0.1505593954393473011955514013941002340912e-9), 
      SC_(3.081670284271240234375), SC_(0.2554458866654840676346793770790100097656e-9), SC_(0.2874971291579766817915476026737397913361e-9), 
      SC_(3.081670284271240234375), SC_(0.9285605062636648199259070679545402526855e-9), SC_(0.1045068618587674618710563098430916993829e-8), 
      SC_(3.081670284271240234375), SC_(0.1698227447555211711005540564656257629395e-8), SC_(0.1911307019263791285975489761534102504092e-8), 
      SC_(3.081670284271240234375), SC_(0.339355921141759608872234821319580078125e-8), SC_(0.3819355031974774192564251891675219233741e-8), 
      SC_(3.081670284271240234375), SC_(0.6313728651008432279922999441623687744141e-8), SC_(0.7105923259640368229860609131512930955636e-8), 
      SC_(3.081670284271240234375), SC_(0.8383264749056706932606175541877746582031e-8), SC_(0.9435127693032368181524002161033185967976e-8), 
      SC_(3.081670284271240234375), SC_(0.1962631124285962869180366396903991699219e-7), SC_(0.2208885910611401144114453165606788366465e-7), 
      SC_(3.081670284271240234375), SC_(0.5256384838503436185419559478759765625e-7), SC_(0.59159128182850440607462626611272020244e-7), 
      SC_(3.081670284271240234375), SC_(0.116242290459922514855861663818359375e-6), SC_(0.1308274226353751302833108364549200715007e-6), 
      SC_(3.081670284271240234375), SC_(0.1776920584006802528165280818939208984375e-6), SC_(0.19998741194136552159337658603684838832e-6), 
      SC_(3.081670284271240234375), SC_(0.246631174150024889968335628509521484375e-6), SC_(0.2775764577197436855139885183694135843601e-6), 
      SC_(3.081670284271240234375), SC_(0.7932688959044753573834896087646484375e-6), SC_(0.8928021311068074785199579060167240593811e-6), 
      SC_(3.081670284271240234375), SC_(0.1372093493046122603118419647216796875e-5), SC_(0.1544253657253189098543067799746046631157e-5), 
      SC_(3.081670284271240234375), SC_(0.214747751670074649155139923095703125e-5), SC_(0.2416928199749181643100601268885193514249e-5), 
      SC_(3.081670284271240234375), SC_(0.527022712049074470996856689453125e-5), SC_(0.5931509331960228699997899809120289644487e-5), 
      SC_(3.081670284271240234375), SC_(0.9233162927557714283466339111328125e-5), SC_(0.1039171803239835570149199230342972913351e-4), 
      SC_(3.081670284271240234375), SC_(0.269396477960981428623199462890625e-4), SC_(0.3032027220182921934725276295799376576054e-4), 
      SC_(3.081670284271240234375), SC_(0.3208058114978484809398651123046875e-4), SC_(0.3610643963209734621758397838332943183439e-4), 
      SC_(3.081670284271240234375), SC_(0.00010957030463032424449920654296875), SC_(0.0001233258865212769239899446236395683072946), 
      SC_(3.081670284271240234375), SC_(0.000126518702018074691295623779296875), SC_(0.0001424033633748521462833870852168266000127), 
      SC_(3.081670284271240234375), SC_(0.00028976381872780621051788330078125), SC_(0.0003261741754060267279249682887185749343808), 
      SC_(3.081670284271240234375), SC_(0.000687857042066752910614013671875), SC_(0.00077446341097905547224447459872652910626), 
      SC_(3.081670284271240234375), SC_(0.00145484809763729572296142578125), SC_(0.001638731690222058425783966330750126376051), 
      SC_(3.081670284271240234375), SC_(0.002847635187208652496337890625), SC_(0.003210074246402525822534006537661772827066), 
      SC_(3.081670284271240234375), SC_(0.0056468211114406585693359375), SC_(0.006375575647354314746464320490352208787519), 
      SC_(3.081670284271240234375), SC_(0.011621631681919097900390625), SC_(0.01316573314478029153750102000765303701693), 
      SC_(3.081670284271240234375), SC_(0.0257236398756504058837890625), SC_(0.02937439065090196216028526614430881237205), 
      SC_(3.081670284271240234375), SC_(0.0560617186129093170166015625), SC_(0.0651289594965291790274936548826001602713), 
      SC_(3.081670284271240234375), SC_(0.106835305690765380859375), SC_(0.1277676169007454599368609354599937296996), 
      SC_(3.081670284271240234375), SC_(0.2401093542575836181640625), SC_(0.31027402555021715485769456717179160097), 
      SC_(3.081670284271240234375), SC_(0.438671648502349853515625), SC_(0.6383875352447911147116499256317898261966), 
      SC_(3.081670284271240234375), SC_(0.903765499591827392578125), SC_(1.765337083764528955511064365585877254958), 
      SC_(6.323592662811279296875), SC_(0.1431564604442703013402649929484277542953e-29), SC_(0.2640216713937599924725935169870336878469e-29), 
      SC_(6.323592662811279296875), SC_(0.1791466932348087634896446282571611213266e-29), SC_(0.3303980080656743193848719859974506438038e-29), 
      SC_(6.323592662811279296875), SC_(0.6013619202535540063110633226832922483532e-29), SC_(0.1109084276079271114149147453706727026132e-28), 
      SC_(6.323592662811279296875), SC_(0.115805324961653822428570241697281798758e-28), SC_(0.2135783139495545999684618085405890945762e-28), 
      SC_(6.323592662811279296875), SC_(0.1422457400834001098175711728787848259007e-28), SC_(0.2623420412107904889320304484610549025392e-28), 
      SC_(6.323592662811279296875), SC_(0.4970121018327539153628705477876439795096e-28), SC_(0.9166332097173709132142553989377319218614e-28), 
      SC_(6.323592662811279296875), SC_(0.9660079415057497591758174164417478444323e-28), SC_(0.1781596377170008450538469724209077707655e-27), 
      SC_(6.323592662811279296875), SC_(0.1232929313253182131376331095427391968754e-27), SC_(0.2273876128155517374717470888031885279529e-27), 
      SC_(6.323592662811279296875), SC_(0.3296523285617759312781860549364832953326e-27), SC_(0.6079736708746525343688764872781137336946e-27), 
      SC_(6.323592662811279296875), SC_(0.528364435768055252017009628713605422886e-27), SC_(0.9744559274766997155824596884015128511179e-27), 
      SC_(6.323592662811279296875), SC_(0.886586057273120049620324386849842094685e-27), SC_(0.1635119588380558627010826035099821298601e-26), 
      SC_(6.323592662811279296875), SC_(0.2499669674831043259218157022821422146034e-26), SC_(0.4610109550299398554521105933337258419067e-26), 
      SC_(6.323592662811279296875), SC_(0.4131050397232622964314362671638736040881e-26), SC_(0.7618844634076498277644240575253280712652e-26), 
      SC_(6.323592662811279296875), SC_(0.7679738097881433551381658732998641759182e-26), SC_(0.1416364502291062812896800893478270568628e-25), 
      SC_(6.323592662811279296875), SC_(0.199929739820949207249437007767740538737e-25), SC_(0.3687279212201232042063109624304941208785e-25), 
      SC_(6.323592662811279296875), SC_(0.5151477415246978459754129800826163591626e-25), SC_(0.9500805434136804903505520643299761716077e-25), 
      SC_(6.323592662811279296875), SC_(0.101200734533556026342258477595279955025e-24), SC_(0.1866432502934594433541827585412325115359e-24), 
      SC_(6.323592662811279296875), SC_(0.2064292695896540981798546456623054911033e-24), SC_(0.3807149227671123852231868207099235585832e-24), 
      SC_(6.323592662811279296875), SC_(0.4063294332896333395257434433879773416284e-24), SC_(0.7493882971168561397190365843431779325436e-24), 
      SC_(6.323592662811279296875), SC_(0.8138195767936862452966745688936976428456e-24), SC_(0.1500917277580197805728854790038420077474e-23), 
      SC_(6.323592662811279296875), SC_(0.9575550627132253801929510132578249716542e-24), SC_(0.176600683842362435260535661406588759685e-23), 
      SC_(6.323592662811279296875), SC_(0.2855160956298500804375620841706273850616e-23), SC_(0.5265737679184896359661146696537048796582e-23), 
      SC_(6.323592662811279296875), SC_(0.65201444297915461398563707001320281266e-23), SC_(0.1202502090887074570233461613634947645288e-22), 
      SC_(6.323592662811279296875), SC_(0.1310988374636350038320977491775043421995e-22), SC_(0.2417839479790878125288977678942117907422e-22), 
      SC_(6.323592662811279296875), SC_(0.2590288837798696209228010176465529547374e-22), SC_(0.4777237340360520058118819832033238328801e-22), 
      SC_(6.323592662811279296875), SC_(0.2937779542193655202274099291941187976629e-22), SC_(0.5418110104910798251920517552801475271232e-22), 
      SC_(6.323592662811279296875), SC_(0.7863513178004503049754083414326234074965e-22), SC_(0.1450257910708703638231501372705993689616e-21), 
      SC_(6.323592662811279296875), SC_(0.1903818607087388763706780167350761726053e-21), SC_(0.3511188870651282430353895347101449393102e-21), 
      SC_(6.323592662811279296875), SC_(0.3812242142377350870566942975497647799754e-21), SC_(0.7030870552852379469894897807112131647926e-21), 
      SC_(6.323592662811279296875), SC_(0.5493133580141330277178034419485741501887e-21), SC_(0.1013091763038312657778669812906615972673e-20), 
      SC_(6.323592662811279296875), SC_(0.9672153634284186955666772243312215295852e-21), SC_(0.1783823210336399991452860718271760581388e-20), 
      SC_(6.323592662811279296875), SC_(0.1702169477623814384559878647986894129041e-20), SC_(0.3139289900595401379513738037289246532429e-20), 
      SC_(6.323592662811279296875), SC_(0.4817114569977399785676754474621208412799e-20), SC_(0.8884144216151370971376685999266889159817e-20), 
      SC_(6.323592662811279296875), SC_(0.7538352992756463183303278501219690799218e-20), SC_(0.13902890240000855616136881361620942629e-19), 
      SC_(6.323592662811279296875), SC_(0.2596305715949999708394617609422128090557e-19), SC_(0.4788334193559808055847315228140282886654e-19), 
      SC_(6.323592662811279296875), SC_(0.4444587480324321591032923385589104015025e-19), SC_(0.8197097159075328603909653177820246587535e-19), 
      SC_(6.323592662811279296875), SC_(0.9715574921498573937069095571295029856174e-19), SC_(0.1791831344086587158752905974689963984773e-18), 
      SC_(6.323592662811279296875), SC_(0.2036598542733453787268262970278076551267e-18), SC_(0.3756073247004505491197875029299517146387e-18), 
      SC_(6.323592662811279296875), SC_(0.4248971931658660264162106698360155121463e-18), SC_(0.7836325846701193893147797979687408120111e-18), 
      SC_(6.323592662811279296875), SC_(0.6521097487613458963613731825259556273977e-18), SC_(0.120267786215041498931028063321951852684e-17), 
      SC_(6.323592662811279296875), SC_(0.1436126164096190058281493628911107407475e-17), SC_(0.2648629541414183387075993336092813103229e-17), 
      SC_(6.323592662811279296875), SC_(0.3118908901459261162419055180006211003274e-17), SC_(0.5752164719165548416846878232210079425788e-17), 
      SC_(6.323592662811279296875), SC_(0.3593346613595175715618300349429858897565e-17), SC_(0.6627164264010534236716184141142009338674e-17), 
      SC_(6.323592662811279296875), SC_(0.9445874854124767215374919304693435151421e-17), SC_(0.1742090897625379557002226544777241404933e-16), 
      SC_(6.323592662811279296875), SC_(0.2566182432094081539023303073498993853718e-16), SC_(0.4732778197506075746341690759495359419571e-16), 
      SC_(6.323592662811279296875), SC_(0.3363765695149349330660137891158001366421e-16), SC_(0.6203751044515782530569432476712507272081e-16), 
      SC_(6.323592662811279296875), SC_(0.1073581901339262605326457800103412409953e-15), SC_(0.1979993687256799227419533949656782391203e-15), 
      SC_(6.323592662811279296875), SC_(0.186668406231853462907965823802669547149e-15), SC_(0.3442702093694849241267874103348331144276e-15), 
      SC_(6.323592662811279296875), SC_(0.3727540802657755688795382376099496468669e-15), SC_(0.687465693027008888543693450648884023717e-15), 
      SC_(6.323592662811279296875), SC_(0.6211646767866855090717281839829411183018e-15), SC_(0.1145606252536746689746494071388493007353e-14), 
      SC_(6.323592662811279296875), SC_(0.1561186859754253464932505224282976996619e-14), SC_(0.287927741990208979554706572189928219716e-14), 
      SC_(6.323592662811279296875), SC_(0.3092010764722992466335682593125966377556e-14), SC_(0.5702556821649460657410678877087527311379e-14), 
      SC_(6.323592662811279296875), SC_(0.6192850577371690132255643845837767003104e-14), SC_(0.1142139694607778851083907918355201497843e-13), 
      SC_(6.323592662811279296875), SC_(0.1047879028014987723427253740737796761096e-13), SC_(0.1932590199117776125622513918660019968254e-13), 
      SC_(6.323592662811279296875), SC_(0.1978473638988408750405412206418986897916e-13), SC_(0.3648874213242890170421744328311442014451e-13), 
      SC_(6.323592662811279296875), SC_(0.4041816252346730475863978426787070930004e-13), SC_(0.745427121555936380928065908178058767459e-13), 
      SC_(6.323592662811279296875), SC_(0.9410302262901834580155480125540634617209e-13), SC_(0.1735530289070848899327475101445921707334e-12), 
      SC_(6.323592662811279296875), SC_(0.1334530223958893535574077304772799834609e-12), SC_(0.2461257418363765792730654099699146351472e-12), 
      SC_(6.323592662811279296875), SC_(0.266297021326439287136622624529991298914e-12), SC_(0.4911282693048557581723249165619954121061e-12), 
      SC_(6.323592662811279296875), SC_(0.5920415525016708979677559909760020673275e-12), SC_(0.1091894838284186038251836896112937706401e-11), 
      SC_(6.323592662811279296875), SC_(0.155163989296047688526414276566356420517e-11), SC_(0.2861670068328674084615094708262254913112e-11), 
      SC_(6.323592662811279296875), SC_(0.326923297461201300961874949280172586441e-11), SC_(0.6029405529143244582479683065134643942967e-11), 
      SC_(6.323592662811279296875), SC_(0.3753785910581841633870681107509881258011e-11), SC_(0.6923060454930819217441054464640979660495e-11), 
      SC_(6.323592662811279296875), SC_(0.9579165585749116473834874341264367103577e-11), SC_(0.1766673540740391567943102645108241226625e-10), 
      SC_(6.323592662811279296875), SC_(0.1858167439361402273334533674642443656921e-10), SC_(0.3426994992440822149025126131105427520931e-10), 
      SC_(6.323592662811279296875), SC_(0.5449485307451595872407779097557067871094e-10), SC_(0.1005041766691009342634921330244156196593e-9), 
      SC_(6.323592662811279296875), SC_(0.6089519166696533147842274047434329986572e-10), SC_(0.1123082411702773503440362131299934516785e-9), 
      SC_(6.323592662811279296875), SC_(0.1337744776064297980155970435589551925659e-9), SC_(0.2467185976953044158569624193431277797317e-9), 
      SC_(6.323592662811279296875), SC_(0.2554458866654840676346793770790100097656e-9), SC_(0.4711156573351016254227307294740845189647e-9), 
      SC_(6.323592662811279296875), SC_(0.9285605062636648199259070679545402526855e-9), SC_(0.1712532541689447362551869993048849215144e-8), 
      SC_(6.323592662811279296875), SC_(0.1698227447555211711005540564656257629395e-8), SC_(0.3132019668696561736750747340448513321755e-8), 
      SC_(6.323592662811279296875), SC_(0.339355921141759608872234821319580078125e-8), SC_(0.6258698873913359974687081613348342619904e-8), 
      SC_(6.323592662811279296875), SC_(0.6313728651008432279922999441623687744141e-8), SC_(0.1164433093482480163520254315438795733052e-7), 
      SC_(6.323592662811279296875), SC_(0.8383264749056706932606175541877746582031e-8), SC_(0.1546115055530141876216042706262018282201e-7), 
      SC_(6.323592662811279296875), SC_(0.1962631124285962869180366396903991699219e-7), SC_(0.3619656126823783677718697037104997467652e-7), 
      SC_(6.323592662811279296875), SC_(0.5256384838503436185419559478759765625e-7), SC_(0.9694285354026179470000990982695592498024e-7), 
      SC_(6.323592662811279296875), SC_(0.116242290459922514855861663818359375e-6), SC_(0.2143842269388389813213009352719362814715e-6), 
      SC_(6.323592662811279296875), SC_(0.1776920584006802528165280818939208984375e-6), SC_(0.3277152969140139122822324864844921678809e-6), 
      SC_(6.323592662811279296875), SC_(0.246631174150024889968335628509521484375e-6), SC_(0.4548588965115780940122335443447594264831e-6), 
      SC_(6.323592662811279296875), SC_(0.7932688959044753573834896087646484375e-6), SC_(0.1463016983785711661450840189292630083083e-5), 
      SC_(6.323592662811279296875), SC_(0.1372093493046122603118419647216796875e-5), SC_(0.2530538088029879560464944967972563390119e-5), 
      SC_(6.323592662811279296875), SC_(0.214747751670074649155139923095703125e-5), SC_(0.3960573796416912678134897725452735748135e-5), 
      SC_(6.323592662811279296875), SC_(0.527022712049074470996856689453125e-5), SC_(0.971986126939888603885399069969989278817e-5), 
      SC_(6.323592662811279296875), SC_(0.9233162927557714283466339111328125e-5), SC_(0.1702875201527071042511818411550240394349e-4), 
      SC_(6.323592662811279296875), SC_(0.269396477960981428623199462890625e-4), SC_(0.4968569013812425366118713653269210769854e-4), 
      SC_(6.323592662811279296875), SC_(0.3208058114978484809398651123046875e-4), SC_(0.591675653295974588826092428631863663046e-4), 
      SC_(6.323592662811279296875), SC_(0.00010957030463032424449920654296875), SC_(0.0002020995632187329336952362792392842222475), 
      SC_(6.323592662811279296875), SC_(0.000126518702018074691295623779296875), SC_(0.0002333640865642182667238138781262256868632), 
      SC_(6.323592662811279296875), SC_(0.00028976381872780621051788330078125), SC_(0.0005345506118405693182416319246117093130363), 
      SC_(6.323592662811279296875), SC_(0.000687857042066752910614013671875), SC_(0.00126941116972756188231670540301804238022), 
      SC_(6.323592662811279296875), SC_(0.00145484809763729572296142578125), SC_(0.00268676106006709776150658499887655198484), 
      SC_(6.323592662811279296875), SC_(0.002847635187208652496337890625), SC_(0.005265673178025655617449193239271098286881), 
      SC_(6.323592662811279296875), SC_(0.0056468211114406585693359375), SC_(0.01046877983444355408069324209313930501994), 
      SC_(6.323592662811279296875), SC_(0.011621631681919097900390625), SC_(0.02166498029451568744434392290105678144036), 
      SC_(6.323592662811279296875), SC_(0.0257236398756504058837890625), SC_(0.04858515871765421540410821800310430346619), 
      SC_(6.323592662811279296875), SC_(0.0560617186129093170166015625), SC_(0.1089281600114160026489262189935621154398), 
      SC_(6.323592662811279296875), SC_(0.106835305690765380859375), SC_(0.2177866862158178437169020404646003493979), 
      SC_(6.323592662811279296875), SC_(0.2401093542575836181640625), SC_(0.5571086656816731263564355805234016885753), 
      SC_(6.323592662811279296875), SC_(0.438671648502349853515625), SC_(1.245743486034874630440378386980407164541), 
      SC_(6.323592662811279296875), SC_(0.903765499591827392578125), SC_(4.295214134949533685152193727106176234023), 
      SC_(8.14723682403564453125), SC_(0.1431564604442703013402649929484277542953e-29), SC_(0.3002962764187653347972894412214699705482e-29), 
      SC_(8.14723682403564453125), SC_(0.1791466932348087634896446282571611213266e-29), SC_(0.3757922258219752347267402157596665385415e-29), 
      SC_(8.14723682403564453125), SC_(0.6013619202535540063110633226832922483532e-29), SC_(0.1261464169145770321318890120398854612563e-28), 
      SC_(8.14723682403564453125), SC_(0.115805324961653822428570241697281798758e-28), SC_(0.2429223785467071694571189061291793443736e-28), 
      SC_(8.14723682403564453125), SC_(0.1422457400834001098175711728787848259007e-28), SC_(0.2983858775979273559126169516027652026393e-28), 
      SC_(8.14723682403564453125), SC_(0.4970121018327539153628705477876439795096e-28), SC_(0.1042571764154104837466580831424514639287e-27), 
      SC_(8.14723682403564453125), SC_(0.9660079415057497591758174164417478444323e-28), SC_(0.2026374408286396264005774213759479985747e-27), 
      SC_(8.14723682403564453125), SC_(0.1232929313253182131376331095427391968754e-27), SC_(0.2586289719014177795596914746737253162208e-27), 
      SC_(8.14723682403564453125), SC_(0.3296523285617759312781860549364832953326e-27), SC_(0.6915047108084518826259439071949255220125e-27), 
      SC_(8.14723682403564453125), SC_(0.528364435768055252017009628713605422886e-27), SC_(0.1108338891314068876466374481618204861507e-26), 
      SC_(8.14723682403564453125), SC_(0.886586057273120049620324386849842094685e-27), SC_(0.1859772803111157884456504072587687508341e-26), 
      SC_(8.14723682403564453125), SC_(0.2499669674831043259218157022821422146034e-26), SC_(0.5243504158311368244991840133692808224351e-26), 
      SC_(8.14723682403564453125), SC_(0.4131050397232622964314362671638736040881e-26), SC_(0.8665616962988200638037564294897244977957e-26), 
      SC_(8.14723682403564453125), SC_(0.7679738097881433551381658732998641759182e-26), SC_(0.1610962402610471480071712791338275473577e-25), 
      SC_(8.14723682403564453125), SC_(0.199929739820949207249437007767740538737e-25), SC_(0.4193883826638476182600418804157794933744e-25), 
      SC_(8.14723682403564453125), SC_(0.5151477415246978459754129800826163591626e-25), SC_(0.1080614511600233806563787463469263198849e-24), 
      SC_(8.14723682403564453125), SC_(0.101200734533556026342258477595279955025e-24), SC_(0.2122866383882235032870078232483926179772e-24), 
      SC_(8.14723682403564453125), SC_(0.2064292695896540981798546456623054911033e-24), SC_(0.4330223086631203680672072624772930549214e-24), 
      SC_(8.14723682403564453125), SC_(0.4063294332896333395257434433879773416284e-24), SC_(0.852348650124142569583864332670923091027e-24), 
      SC_(8.14723682403564453125), SC_(0.8138195767936862452966745688936976428456e-24), SC_(0.1707132097492570967003147340838531255859e-23), 
      SC_(8.14723682403564453125), SC_(0.9575550627132253801929510132578249716542e-24), SC_(0.2008642983392705299695765581848139274246e-23), 
      SC_(8.14723682403564453125), SC_(0.2855160956298500804375620841706273850616e-23), SC_(0.598921069361349484374992174068012361807e-23), 
      SC_(8.14723682403564453125), SC_(0.65201444297915461398563707001320281266e-23), SC_(0.1367716893741711175200464702736435160588e-22), 
      SC_(8.14723682403564453125), SC_(0.1310988374636350038320977491775043421995e-22), SC_(0.2750032559549373199320920778458197506507e-22), 
      SC_(8.14723682403564453125), SC_(0.2590288837798696209228010176465529547374e-22), SC_(0.5433594058040099707290986349056416146979e-22), 
      SC_(8.14723682403564453125), SC_(0.2937779542193655202274099291941187976629e-22), SC_(0.6162517952191302117861094956373595761412e-22), 
      SC_(8.14723682403564453125), SC_(0.7863513178004503049754083414326234074965e-22), SC_(0.1649512512111818004955831530316651421064e-21), 
      SC_(8.14723682403564453125), SC_(0.1903818607087388763706780167350761726053e-21), SC_(0.3993599987809599501685878670403590453519e-21), 
      SC_(8.14723682403564453125), SC_(0.3812242142377350870566942975497647799754e-21), SC_(0.7996859636022348345206395118114881895569e-21), 
      SC_(8.14723682403564453125), SC_(0.5493133580141330277178034419485741501887e-21), SC_(0.115228300201616594413399651589013021783e-20), 
      SC_(8.14723682403564453125), SC_(0.9672153634284186955666772243312215295852e-21), SC_(0.2028907191692179192545093848322480540521e-20), 
      SC_(8.14723682403564453125), SC_(0.1702169477623814384559878647986894129041e-20), SC_(0.3570604877892291096818904837199743491231e-20), 
      SC_(8.14723682403564453125), SC_(0.4817114569977399785676754474621208412799e-20), SC_(0.1010475925401865006098240977506391422236e-19), 
      SC_(8.14723682403564453125), SC_(0.7538352992756463183303278501219690799218e-20), SC_(0.1581304348423924557702183910898865913193e-19), 
      SC_(8.14723682403564453125), SC_(0.2596305715949999708394617609422128090557e-19), SC_(0.5446215535959395263786385569195301491108e-19), 
      SC_(8.14723682403564453125), SC_(0.4444587480324321591032923385589104015025e-19), SC_(0.9323317064537522990976000435543820405318e-19), 
      SC_(8.14723682403564453125), SC_(0.9715574921498573937069095571295029856174e-19), SC_(0.2038015583187277825165297954172993551461e-18), 
      SC_(8.14723682403564453125), SC_(0.2036598542733453787268262970278076551267e-18), SC_(0.4272129647832585254298530550491334440606e-18), 
      SC_(8.14723682403564453125), SC_(0.4248971931658660264162106698360155121463e-18), SC_(0.8912978469327704136743073992720868519783e-18), 
      SC_(8.14723682403564453125), SC_(0.6521097487613458963613731825259556273977e-18), SC_(0.136791681466336850858739043994763819133e-17), 
      SC_(8.14723682403564453125), SC_(0.1436126164096190058281493628911107407475e-17), SC_(0.3012531451303506521403267939793483351148e-17), 
      SC_(8.14723682403564453125), SC_(0.3118908901459261162419055180006211003274e-17), SC_(0.6542469174572584949884746185717932847323e-17), 
      SC_(8.14723682403564453125), SC_(0.3593346613595175715618300349429858897565e-17), SC_(0.7537687119364649787992623415722169434063e-17), 
      SC_(8.14723682403564453125), SC_(0.9445874854124767215374919304693435151421e-17), SC_(0.1981441171015517669713621273143494442641e-16), 
      SC_(8.14723682403564453125), SC_(0.2566182432094081539023303073498993853718e-16), SC_(0.5383026561131680690067572177474658478986e-16), 
      SC_(8.14723682403564453125), SC_(0.3363765695149349330660137891158001366421e-16), SC_(0.7056100087021666077931534148880848881114e-16), 
      SC_(8.14723682403564453125), SC_(0.1073581901339262605326457800103412409953e-15), SC_(0.2252030026463694583742748890551163061716e-15), 
      SC_(8.14723682403564453125), SC_(0.186668406231853462907965823802669547149e-15), SC_(0.3915703639394825250101464584673193502066e-15), 
      SC_(8.14723682403564453125), SC_(0.3727540802657755688795382376099496468669e-15), SC_(0.7819183428839392001729888591632596329746e-15), 
      SC_(8.14723682403564453125), SC_(0.6211646767866855090717281839829411183018e-15), SC_(0.1303003992296412497333639695669103250231e-14), 
      SC_(8.14723682403564453125), SC_(0.1561186859754253464932505224282976996619e-14), SC_(0.3274868625021751903175847444540694937743e-14), 
      SC_(8.14723682403564453125), SC_(0.3092010764722992466335682593125966377556e-14), SC_(0.6486045522580673933180535249562004618168e-14), 
      SC_(8.14723682403564453125), SC_(0.6192850577371690132255643845837767003104e-14), SC_(0.1299061155208216751337721180197078540544e-13), 
      SC_(8.14723682403564453125), SC_(0.1047879028014987723427253740737796761096e-13), SC_(0.2198113653227124650003647332454390917516e-13), 
      SC_(8.14723682403564453125), SC_(0.1978473638988408750405412206418986897916e-13), SC_(0.4150202267764266086007368292612694907912e-13), 
      SC_(8.14723682403564453125), SC_(0.4041816252346730475863978426787070930004e-13), SC_(0.847843238636878629688634386576373742241e-13), 
      SC_(8.14723682403564453125), SC_(0.9410302262901834580155480125540634617209e-13), SC_(0.1973979183862871363544627549465085816321e-12), 
      SC_(8.14723682403564453125), SC_(0.1334530223958893535574077304772799834609e-12), SC_(0.2799415798487301903063820714571160522817e-12), 
      SC_(8.14723682403564453125), SC_(0.266297021326439287136622624529991298914e-12), SC_(0.5586056240674623129969871288914642634269e-12), 
      SC_(8.14723682403564453125), SC_(0.5920415525016708979677559909760020673275e-12), SC_(0.1241913031027744437030775086591973516968e-11), 
      SC_(8.14723682403564453125), SC_(0.155163989296047688526414276566356420517e-11), SC_(0.3254842154895220360350562900122964558652e-11), 
      SC_(8.14723682403564453125), SC_(0.326923297461201300961874949280172586441e-11), SC_(0.6857800800453437785097236710724347410561e-11), 
      SC_(8.14723682403564453125), SC_(0.3753785910581841633870681107509881258011e-11), SC_(0.7874237236147777114390384507386571915503e-11), 
      SC_(8.14723682403564453125), SC_(0.9579165585749116473834874341264367103577e-11), SC_(0.2009401285629002547185238237069409269533e-10), 
      SC_(8.14723682403564453125), SC_(0.1858167439361402273334533674642443656921e-10), SC_(0.3897838499792479711219892220178230066437e-10), 
      SC_(8.14723682403564453125), SC_(0.5449485307451595872407779097557067871094e-10), SC_(0.1143126996323774119097065969850821243229e-9), 
      SC_(8.14723682403564453125), SC_(0.6089519166696533147842274047434329986572e-10), SC_(0.1277385544027438399433622551231764162295e-9), 
      SC_(8.14723682403564453125), SC_(0.1337744776064297980155970435589551925659e-9), SC_(0.2806158896779336233741305619995731444322e-9), 
      SC_(8.14723682403564453125), SC_(0.2554458866654840676346793770790100097656e-9), SC_(0.5358434287536001582491896939362610296213e-9), 
      SC_(8.14723682403564453125), SC_(0.9285605062636648199259070679545402526855e-9), SC_(0.1947821718046892961553014837249850920506e-8), 
      SC_(8.14723682403564453125), SC_(0.1698227447555211711005540564656257629395e-8), SC_(0.3562335771216184184970866941827809547094e-8), 
      SC_(8.14723682403564453125), SC_(0.339355921141759608872234821319580078125e-8), SC_(0.7118597340699191804540686314874729268662e-8), 
      SC_(8.14723682403564453125), SC_(0.6313728651008432279922999441623687744141e-8), SC_(0.132441750158373525162868935080838840289e-7), 
      SC_(8.14723682403564453125), SC_(0.8383264749056706932606175541877746582031e-8), SC_(0.1758539714307722824048198124784117315638e-7), 
      SC_(8.14723682403564453125), SC_(0.1962631124285962869180366396903991699219e-7), SC_(0.4116969844809739007223857306170195600183e-7), 
      SC_(8.14723682403564453125), SC_(0.5256384838503436185419559478759765625e-7), SC_(0.1102620780473938808288066281082898627349e-6), 
      SC_(8.14723682403564453125), SC_(0.116242290459922514855861663818359375e-6), SC_(0.2438390215504947278486570498249757266301e-6), 
      SC_(8.14723682403564453125), SC_(0.1776920584006802528165280818939208984375e-6), SC_(0.3727409385932641184830284812315176859141e-6), 
      SC_(8.14723682403564453125), SC_(0.246631174150024889968335628509521484375e-6), SC_(0.5173531266029305255369204117738691933193e-6), 
      SC_(8.14723682403564453125), SC_(0.7932688959044753573834896087646484375e-6), SC_(0.1664024753741567137349817967171097549819e-5), 
      SC_(8.14723682403564453125), SC_(0.1372093493046122603118419647216796875e-5), SC_(0.2878215614882762856182308078227200441801e-5), 
      SC_(8.14723682403564453125), SC_(0.214747751670074649155139923095703125e-5), SC_(0.4504728270448328808378965972337354594004e-5), 
      SC_(8.14723682403564453125), SC_(0.527022712049074470996856689453125e-5), SC_(0.1105530496816720602121909418016065775699e-4), 
      SC_(8.14723682403564453125), SC_(0.9233162927557714283466339111328125e-5), SC_(0.1936839796995393539450852194639748205793e-4), 
      SC_(8.14723682403564453125), SC_(0.269396477960981428623199462890625e-4), SC_(0.5651232562465381965657403175421220278403e-4), 
      SC_(8.14723682403564453125), SC_(0.3208058114978484809398651123046875e-4), SC_(0.6729702026750513326048907431986094438406e-4), 
      SC_(8.14723682403564453125), SC_(0.00010957030463032424449920654296875), SC_(0.00022986972439150582568813762789935818154), 
      SC_(8.14723682403564453125), SC_(0.000126518702018074691295623779296875), SC_(0.0002654308233220190960060128510217268500465), 
      SC_(8.14723682403564453125), SC_(0.00028976381872780621051788330078125), SC_(0.0006080161951350230359195132409744484724968), 
      SC_(8.14723682403564453125), SC_(0.000687857042066752910614013671875), SC_(0.001443944640669855964528390799484686036437), 
      SC_(8.14723682403564453125), SC_(0.00145484809763729572296142578125), SC_(0.003056465549723277550236744174463742737141), 
      SC_(8.14723682403564453125), SC_(0.002847635187208652496337890625), SC_(0.005991300521400921393361158561348919757934), 
      SC_(8.14723682403564453125), SC_(0.0056468211114406585693359375), SC_(0.01191564950603685046390396565923080993845), 
      SC_(8.14723682403564453125), SC_(0.011621631681919097900390625), SC_(0.0246780346873362021605808087853240101899), 
      SC_(8.14723682403564453125), SC_(0.0257236398756504058837890625), SC_(0.05544231479185505401377016952082859212136), 
      SC_(8.14723682403564453125), SC_(0.0560617186129093170166015625), SC_(0.1247935191691613843472500366450303784846), 
      SC_(8.14723682403564453125), SC_(0.106835305690765380859375), SC_(0.2512038389510235409424816149968257848692), 
      SC_(8.14723682403564453125), SC_(0.2401093542575836181640625), SC_(0.6547870159122512790726005830749938271159), 
      SC_(8.14723682403564453125), SC_(0.438671648502349853515625), SC_(1.509772797905131401789530365650239382294), 
      SC_(8.14723682403564453125), SC_(0.903765499591827392578125), SC_(5.657937524401060919087823696029901420293), 
      SC_(8.35008525848388671875), SC_(0.1431564604442703013402649929484277542953e-29), SC_(0.303816911746048393394900925176479831121e-29), 
      SC_(8.35008525848388671875), SC_(0.1791466932348087634896446282571611213266e-29), SC_(0.3801979660520079493591155051996548231306e-29), 
      SC_(8.35008525848388671875), SC_(0.6013619202535540063110633226832922483532e-29), SC_(0.1276253414523568970385340237418638328662e-28), 
      SC_(8.35008525848388671875), SC_(0.115805324961653822428570241697281798758e-28), SC_(0.2457703695970741261302951747977948193401e-28), 
      SC_(8.35008525848388671875), SC_(0.1422457400834001098175711728787848259007e-28), SC_(0.3018841156525633741398803896071070359265e-28), 
      SC_(8.35008525848388671875), SC_(0.4970121018327539153628705477876439795096e-28), SC_(0.1054794742833301654764119491149597015675e-27), 
      SC_(8.35008525848388671875), SC_(0.9660079415057497591758174164417478444323e-28), SC_(0.2050131363960954151037327999620809155486e-27), 
      SC_(8.35008525848388671875), SC_(0.1232929313253182131376331095427391968754e-27), SC_(0.261661105053362949521331571474625302723e-27), 
      SC_(8.35008525848388671875), SC_(0.3296523285617759312781860549364832953326e-27), SC_(0.6996118240330591461475652585969851842736e-27), 
      SC_(8.35008525848388671875), SC_(0.528364435768055252017009628713605422886e-27), SC_(0.1121332915422182033137202753286930185984e-26), 
      SC_(8.35008525848388671875), SC_(0.886586057273120049620324386849842094685e-27), SC_(0.1881576542769330443012072728830855135065e-26), 
      SC_(8.35008525848388671875), SC_(0.2499669674831043259218157022821422146034e-26), SC_(0.5304978333744577310115147769605291298771e-26), 
      SC_(8.35008525848388671875), SC_(0.4131050397232622964314362671638736040881e-26), SC_(0.8767211553425463654104292902836243726569e-26), 
      SC_(8.35008525848388671875), SC_(0.7679738097881433551381658732998641759182e-26), SC_(0.1629849120798232545767493770886803212811e-25), 
      SC_(8.35008525848388671875), SC_(0.199929739820949207249437007767740538737e-25), SC_(0.4243052386883939863234715495583651591012e-25), 
      SC_(8.35008525848388671875), SC_(0.5151477415246978459754129800826163591626e-25), SC_(0.1093283498608947757724519564692589494537e-24), 
      SC_(8.35008525848388671875), SC_(0.101200734533556026342258477595279955025e-24), SC_(0.214775459919854855860469177623499747264e-24), 
      SC_(8.35008525848388671875), SC_(0.2064292695896540981798546456623054911033e-24), SC_(0.4380990071009494974444657778777392512375e-24), 
      SC_(8.35008525848388671875), SC_(0.4063294332896333395257434433879773416284e-24), SC_(0.8623414772233518661182971579868599749488e-24), 
      SC_(8.35008525848388671875), SC_(0.8138195767936862452966745688936976428456e-24), SC_(0.1727146296944015109953344638919887600832e-23), 
      SC_(8.35008525848388671875), SC_(0.9575550627132253801929510132578249716542e-24), SC_(0.2032192058098413805068266190849302445531e-23), 
      SC_(8.35008525848388671875), SC_(0.2855160956298500804375620841706273850616e-23), SC_(0.6059427437563635612468332606231404035493e-23), 
      SC_(8.35008525848388671875), SC_(0.65201444297915461398563707001320281266e-23), SC_(0.1383751832540332977251232173344701356117e-22), 
      SC_(8.35008525848388671875), SC_(0.1310988374636350038320977491775043421995e-22), SC_(0.2782273591292393377146410912035666884325e-22), 
      SC_(8.35008525848388671875), SC_(0.2590288837798696209228010176465529547374e-22), SC_(0.5497296823265782289911172955748998204486e-22), 
      SC_(8.35008525848388671875), SC_(0.2937779542193655202274099291941187976629e-22), SC_(0.6234766528384919483230046708585808739178e-22), 
      SC_(8.35008525848388671875), SC_(0.7863513178004503049754083414326234074965e-22), SC_(0.1668851186877261729366775205077442586098e-21), 
      SC_(8.35008525848388671875), SC_(0.1903818607087388763706780167350761726053e-21), SC_(0.4040420445817919536244395122738304468065e-21), 
      SC_(8.35008525848388671875), SC_(0.3812242142377350870566942975497647799754e-21), SC_(0.8090613800668210377384738787292658254758e-21), 
      SC_(8.35008525848388671875), SC_(0.5493133580141330277178034419485741501887e-21), SC_(0.116579222128556739661293493199864289008e-20), 
      SC_(8.35008525848388671875), SC_(0.9672153634284186955666772243312215295852e-21), SC_(0.2052693841397049732704220591360221495998e-20), 
      SC_(8.35008525848388671875), SC_(0.1702169477623814384559878647986894129041e-20), SC_(0.3612466195064758274683701588091748448199e-20), 
      SC_(8.35008525848388671875), SC_(0.4817114569977399785676754474621208412799e-20), SC_(0.102232261655223366043584230403875938978e-19), 
      SC_(8.35008525848388671875), SC_(0.7538352992756463183303278501219690799218e-20), SC_(0.1599843359358859003043614240098451228971e-19), 
      SC_(8.35008525848388671875), SC_(0.2596305715949999708394617609422128090557e-19), SC_(0.5510066273785920985779323744923667519753e-19), 
      SC_(8.35008525848388671875), SC_(0.4444587480324321591032923385589104015025e-19), SC_(0.9432622447262610405053726165198520690847e-19), 
      SC_(8.35008525848388671875), SC_(0.9715574921498573937069095571295029856174e-19), SC_(0.2061909018514850029082537284682245312716e-18), 
      SC_(8.35008525848388671875), SC_(0.2036598542733453787268262970278076551267e-18), SC_(0.4322215552127710416548061649973866087213e-18), 
      SC_(8.35008525848388671875), SC_(0.4248971931658660264162106698360155121463e-18), SC_(0.901747309458462763450102224941690433692e-18), 
      SC_(8.35008525848388671875), SC_(0.6521097487613458963613731825259556273977e-18), SC_(0.1383954097309432722071591383044181578075e-17), 
      SC_(8.35008525848388671875), SC_(0.1436126164096190058281493628911107407475e-17), SC_(0.3047849986646316599823536826165457953048e-17), 
      SC_(8.35008525848388671875), SC_(0.3118908901459261162419055180006211003274e-17), SC_(0.6619172250542598263128134296831740065416e-17), 
      SC_(8.35008525848388671875), SC_(0.3593346613595175715618300349429858897565e-17), SC_(0.7626058003862181805512806919457213036816e-17), 
      SC_(8.35008525848388671875), SC_(0.9445874854124767215374919304693435151421e-17), SC_(0.2004671335134776866079507930597383925792e-16), 
      SC_(8.35008525848388671875), SC_(0.2566182432094081539023303073498993853718e-16), SC_(0.5446136479459122707777712170130656693809e-16), 
      SC_(8.35008525848388671875), SC_(0.3363765695149349330660137891158001366421e-16), SC_(0.7138824906441575420363373409374989236525e-16), 
      SC_(8.35008525848388671875), SC_(0.1073581901339262605326457800103412409953e-15), SC_(0.2278432539887516822635090434738477660544e-15), 
      SC_(8.35008525848388671875), SC_(0.186668406231853462907965823802669547149e-15), SC_(0.3961610850527872795280743881628279014308e-15), 
      SC_(8.35008525848388671875), SC_(0.3727540802657755688795382376099496468669e-15), SC_(0.7910854540244348513465934381335174491522e-15), 
      SC_(8.35008525848388671875), SC_(0.6211646767866855090717281839829411183018e-15), SC_(0.1318280245274229811349400509112621296791e-14), 
      SC_(8.35008525848388671875), SC_(0.1561186859754253464932505224282976996619e-14), SC_(0.3313262768002680373213799824392006451222e-14), 
      SC_(8.35008525848388671875), SC_(0.3092010764722992466335682593125966377556e-14), SC_(0.6562087094835536421935148657707169429622e-14), 
      SC_(8.35008525848388671875), SC_(0.6192850577371690132255643845837767003104e-14), SC_(0.1314291182865800526329078268768653499208e-13), 
      SC_(8.35008525848388671875), SC_(0.1047879028014987723427253740737796761096e-13), SC_(0.2223884057952832841780133364614796489439e-13), 
      SC_(8.35008525848388671875), SC_(0.1978473638988408750405412206418986897916e-13), SC_(0.4198858710972658766671444521370475155153e-13), 
      SC_(8.35008525848388671875), SC_(0.4041816252346730475863978426787070930004e-13), SC_(0.8577832448651965377422043802840370255599e-13), 
      SC_(8.35008525848388671875), SC_(0.9410302262901834580155480125540634617209e-13), SC_(0.1997121864594411126451917445231700374726e-12), 
      SC_(8.35008525848388671875), SC_(0.1334530223958893535574077304772799834609e-12), SC_(0.2832235793038836823300148051443354939542e-12), 
      SC_(8.35008525848388671875), SC_(0.266297021326439287136622624529991298914e-12), SC_(0.5651546453126307347005405388651202867825e-12), 
      SC_(8.35008525848388671875), SC_(0.5920415525016708979677559909760020673275e-12), SC_(0.1256473061350448036669401447786418232001e-11), 
      SC_(8.35008525848388671875), SC_(0.155163989296047688526414276566356420517e-11), SC_(0.3293001510089133195032192369813630570469e-11), 
      SC_(8.35008525848388671875), SC_(0.326923297461201300961874949280172586441e-11), SC_(0.6938200784274620573725127382346418124212e-11), 
      SC_(8.35008525848388671875), SC_(0.3753785910581841633870681107509881258011e-11), SC_(0.7966553791383517129221888809680799456275e-11), 
      SC_(8.35008525848388671875), SC_(0.9579165585749116473834874341264367103577e-11), SC_(0.2032959250573862910811995273561451963595e-10), 
      SC_(8.35008525848388671875), SC_(0.1858167439361402273334533674642443656921e-10), SC_(0.39435362623033519647896868270251518048e-10), 
      SC_(8.35008525848388671875), SC_(0.5449485307451595872407779097557067871094e-10), SC_(0.1156528871748459484879192119255551761764e-9), 
      SC_(8.35008525848388671875), SC_(0.6089519166696533147842274047434329986572e-10), SC_(0.1292361449579070679085354557462440019552e-9), 
      SC_(8.35008525848388671875), SC_(0.1337744776064297980155970435589551925659e-9), SC_(0.283905794656174652826326024378902159596e-9), 
      SC_(8.35008525848388671875), SC_(0.2554458866654840676346793770790100097656e-9), SC_(0.5421255889194592212629474063410851442735e-9), 
      SC_(8.35008525848388671875), SC_(0.9285605062636648199259070679545402526855e-9), SC_(0.1970657732000656088065688384526438968121e-8), 
      SC_(8.35008525848388671875), SC_(0.1698227447555211711005540564656257629395e-8), SC_(0.3604100142509616779027214782233014357877e-8), 
      SC_(8.35008525848388671875), SC_(0.339355921141759608872234821319580078125e-8), SC_(0.7202054870267985344182616265018696491029e-8), 
      SC_(8.35008525848388671875), SC_(0.6313728651008432279922999441623687744141e-8), SC_(0.1339944803923258042086625536682875546438e-7), 
      SC_(8.35008525848388671875), SC_(0.8383264749056706932606175541877746582031e-8), SC_(0.1779156610299678955518645324556312923063e-7), 
      SC_(8.35008525848388671875), SC_(0.1962631124285962869180366396903991699219e-7), SC_(0.4165236676325977052853867589868067553003e-7), 
      SC_(8.35008525848388671875), SC_(0.5256384838503436185419559478759765625e-7), SC_(0.1115547766899315795021036787777738246793e-6), 
      SC_(8.35008525848388671875), SC_(0.116242290459922514855861663818359375e-6), SC_(0.2466977595594028290046253241160990171727e-6), 
      SC_(8.35008525848388671875), SC_(0.1776920584006802528165280818939208984375e-6), SC_(0.3771109067441330494076108837938934979323e-6), 
      SC_(8.35008525848388671875), SC_(0.246631174150024889968335628509521484375e-6), SC_(0.5234185104052220487989578542374244461787e-6), 
      SC_(8.35008525848388671875), SC_(0.7932688959044753573834896087646484375e-6), SC_(0.1683533584602828763167290870416711088711e-5), 
      SC_(8.35008525848388671875), SC_(0.1372093493046122603118419647216796875e-5), SC_(0.2911959497583481521632815773831789242539e-5), 
      SC_(8.35008525848388671875), SC_(0.214747751670074649155139923095703125e-5), SC_(0.4557541251741361946740815280463978788759e-5), 
      SC_(8.35008525848388671875), SC_(0.527022712049074470996856689453125e-5), SC_(0.1118491668149296147391580982027848644964e-4), 
      SC_(8.35008525848388671875), SC_(0.9233162927557714283466339111328125e-5), SC_(0.1959547282760206637598957101789474343893e-4), 
      SC_(8.35008525848388671875), SC_(0.269396477960981428623199462890625e-4), SC_(0.5717488787825101038116116448568043886015e-4), 
      SC_(8.35008525848388671875), SC_(0.3208058114978484809398651123046875e-4), SC_(0.6808602882722058087940799635524413983679e-4), 
      SC_(8.35008525848388671875), SC_(0.00010957030463032424449920654296875), SC_(0.0002325650011386463513065814423233172031071), 
      SC_(8.35008525848388671875), SC_(0.000126518702018074691295623779296875), SC_(0.0002685431183137980700956096910227791349613), 
      SC_(8.35008525848388671875), SC_(0.00028976381872780621051788330078125), SC_(0.0006151466916567491000405537059346943296718), 
      SC_(8.35008525848388671875), SC_(0.000687857042066752910614013671875), SC_(0.001460885623281189582659547621062792419205), 
      SC_(8.35008525848388671875), SC_(0.00145484809763729572296142578125), SC_(0.003092354511270021672959060525719939657619), 
      SC_(8.35008525848388671875), SC_(0.002847635187208652496337890625), SC_(0.006061754232424053942762224395926479452724), 
      SC_(8.35008525848388671875), SC_(0.0056468211114406585693359375), SC_(0.01205618582978607475974534372018331649234), 
      SC_(8.35008525848388671875), SC_(0.011621631681919097900390625), SC_(0.02497093962240100225008553038129746109154), 
      SC_(8.35008525848388671875), SC_(0.0257236398756504058837890625), SC_(0.05611021932688490941026883371307113047986), 
      SC_(8.35008525848388671875), SC_(0.0560617186129093170166015625), SC_(0.1263453655338174105049911829505423628754), 
      SC_(8.35008525848388671875), SC_(0.106835305690765380859375), SC_(0.2544955644502210185824015471606320029378), 
      SC_(8.35008525848388671875), SC_(0.2401093542575836181640625), SC_(0.6645874239603678819609156017830715676702), 
      SC_(8.35008525848388671875), SC_(0.438671648502349853515625), SC_(1.536995347819618393387328773272907959484), 
      SC_(8.35008525848388671875), SC_(0.903765499591827392578125), SC_(5.807575170920757391333663733086588298286), 
      SC_(9.05791950225830078125), SC_(0.1431564604442703013402649929484277542953e-29), SC_(0.3154652249272743515358442481050496705959e-29), 
      SC_(9.05791950225830078125), SC_(0.1791466932348087634896446282571611213266e-29), SC_(0.3947747220133110322799882025912166405739e-29), 
      SC_(9.05791950225830078125), SC_(0.6013619202535540063110633226832922483532e-29), SC_(0.1325184856112988269489259326446958165414e-28), 
      SC_(9.05791950225830078125), SC_(0.115805324961653822428570241697281798758e-28), SC_(0.2551931835519645397384620899693010823261e-28), 
      SC_(9.05791950225830078125), SC_(0.1422457400834001098175711728787848259007e-28), SC_(0.3134583256046998923120958035052167830755e-28), 
      SC_(9.05791950225830078125), SC_(0.4970121018327539153628705477876439795096e-28), SC_(0.1095235478787799744975907281941028579988e-27), 
      SC_(9.05791950225830078125), SC_(0.9660079415057497591758174164417478444323e-28), SC_(0.2128733216809857444780474082092193404369e-27), 
      SC_(9.05791950225830078125), SC_(0.1232929313253182131376331095427391968754e-27), SC_(0.2716931683821973114809547479629023087336e-27), 
      SC_(9.05791950225830078125), SC_(0.3296523285617759312781860549364832953326e-27), SC_(0.7264348786971048930313153654157758299519e-27), 
      SC_(9.05791950225830078125), SC_(0.528364435768055252017009628713605422886e-27), SC_(0.1164324718953423812840797502181511074885e-26), 
      SC_(9.05791950225830078125), SC_(0.886586057273120049620324386849842094685e-27), SC_(0.1953716018872443147401499351362959093991e-26), 
      SC_(9.05791950225830078125), SC_(0.2499669674831043259218157022821422146034e-26), SC_(0.550837072785437889109549468827089227749e-26), 
      SC_(9.05791950225830078125), SC_(0.4131050397232622964314362671638736040881e-26), SC_(0.9103345659040111376422713882766452663518e-26), 
      SC_(9.05791950225830078125), SC_(0.7679738097881433551381658732998641759182e-26), SC_(0.1692337390092051803145956763830592214146e-25), 
      SC_(9.05791950225830078125), SC_(0.199929739820949207249437007767740538737e-25), SC_(0.440573063531562974560809957989790578186e-25), 
      SC_(9.05791950225830078125), SC_(0.5151477415246978459754129800826163591626e-25), SC_(0.11351998900121629038834906490214739902e-24), 
      SC_(9.05791950225830078125), SC_(0.101200734533556026342258477595279955025e-24), SC_(0.2230099318141629322715485110014889841194e-24), 
      SC_(9.05791950225830078125), SC_(0.2064292695896540981798546456623054911033e-24), SC_(0.4548956837894464751787453186486142389963e-24), 
      SC_(9.05791950225830078125), SC_(0.4063294332896333395257434433879773416284e-24), SC_(0.8954035722138203560380363554731750197979e-24), 
      SC_(9.05791950225830078125), SC_(0.8138195767936862452966745688936976428456e-24), SC_(0.1793364931255639284980704007276052736735e-23), 
      SC_(9.05791950225830078125), SC_(0.9575550627132253801929510132578249716542e-24), SC_(0.2110106119567503000298650286554781054522e-23), 
      SC_(9.05791950225830078125), SC_(0.2855160956298500804375620841706273850616e-23), SC_(0.6291745342732299276098361662975120213963e-23), 
      SC_(9.05791950225830078125), SC_(0.65201444297915461398563707001320281266e-23), SC_(0.1436804753847090284841789400547023868969e-22), 
      SC_(9.05791950225830078125), SC_(0.1310988374636350038320977491775043421995e-22), SC_(0.2888945711553814592199431071395309118877e-22), 
      SC_(9.05791950225830078125), SC_(0.2590288837798696209228010176465529547374e-22), SC_(0.5708062691036443881725807225889040536907e-22), 
      SC_(9.05791950225830078125), SC_(0.2937779542193655202274099291941187976629e-22), SC_(0.6473806918589257929168848362981256399467e-22), 
      SC_(9.05791950225830078125), SC_(0.7863513178004503049754083414326234074965e-22), SC_(0.1732834791890848767051681603427277899451e-21), 
      SC_(9.05791950225830078125), SC_(0.1903818607087388763706780167350761726053e-21), SC_(0.4195329803780372971638862970337421704626e-21), 
      SC_(9.05791950225830078125), SC_(0.3812242142377350870566942975497647799754e-21), SC_(0.84008072090499869521429088340237744825e-21), 
      SC_(9.05791950225830078125), SC_(0.5493133580141330277178034419485741501887e-21), SC_(0.1210488590621064027940195119088838608895e-20), 
      SC_(9.05791950225830078125), SC_(0.9672153634284186955666772243312215295852e-21), SC_(0.2131393939401312961699471075038214057137e-20), 
      SC_(9.05791950225830078125), SC_(0.1702169477623814384559878647986894129041e-20), SC_(0.3750967825388348419801806653021941193416e-20), 
      SC_(9.05791950225830078125), SC_(0.4817114569977399785676754474621208412799e-20), SC_(0.1061518374093883020098862460032696167784e-19), 
      SC_(9.05791950225830078125), SC_(0.7538352992756463183303278501219690799218e-20), SC_(0.1661181210446929638476935167073047556707e-19), 
      SC_(9.05791950225830078125), SC_(0.2596305715949999708394617609422128090557e-19), SC_(0.5721321721145671001227774122509306055176e-19), 
      SC_(9.05791950225830078125), SC_(0.4444587480324321591032923385589104015025e-19), SC_(0.9794268346941221703798574479056325989627e-19), 
      SC_(9.05791950225830078125), SC_(0.9715574921498573937069095571295029856174e-19), SC_(0.2140962425584338463294826233113563101289e-18), 
      SC_(9.05791950225830078125), SC_(0.2036598542733453787268262970278076551267e-18), SC_(0.4487928909223618938299710968847143982242e-18), 
      SC_(9.05791950225830078125), SC_(0.4248971931658660264162106698360155121463e-18), SC_(0.9363202205269548994010977882112360425486e-18), 
      SC_(9.05791950225830078125), SC_(0.6521097487613458963613731825259556273977e-18), SC_(0.1437014773429318969440529662554354621061e-17), 
      SC_(9.05791950225830078125), SC_(0.1436126164096190058281493628911107407475e-17), SC_(0.3164704282115974280634551112581105735532e-17), 
      SC_(9.05791950225830078125), SC_(0.3118908901459261162419055180006211003274e-17), SC_(0.6872950721700411519896314616257619649159e-17), 
      SC_(9.05791950225830078125), SC_(0.3593346613595175715618300349429858897565e-17), SC_(0.7918440384608098105284168669591190857757e-17), 
      SC_(9.05791950225830078125), SC_(0.9445874854124767215374919304693435151421e-17), SC_(0.2081530254550673990000527907345002592813e-16), 
      SC_(9.05791950225830078125), SC_(0.2566182432094081539023303073498993853718e-16), SC_(0.5654940864231045623097989372344393889809e-16), 
      SC_(9.05791950225830078125), SC_(0.3363765695149349330660137891158001366421e-16), SC_(0.7412526813877441309836883895951271491157e-16), 
      SC_(9.05791950225830078125), SC_(0.1073581901339262605326457800103412409953e-15), SC_(0.2365787439370827419417927163994335313562e-15), 
      SC_(9.05791950225830078125), SC_(0.186668406231853462907965823802669547149e-15), SC_(0.4113498655666461529741469728959584605667e-15), 
      SC_(9.05791950225830078125), SC_(0.3727540802657755688795382376099496468669e-15), SC_(0.8214155994683834723574831964831005243093e-15), 
      SC_(9.05791950225830078125), SC_(0.6211646767866855090717281839829411183018e-15), SC_(0.1368822991789976842621256917018358658063e-14), 
      SC_(9.05791950225830078125), SC_(0.1561186859754253464932505224282976996619e-14), SC_(0.3440292965734564418953577846754869711643e-14), 
      SC_(9.05791950225830078125), SC_(0.3092010764722992466335682593125966377556e-14), SC_(0.681367692623707945134611894380943253949e-14), 
      SC_(9.05791950225830078125), SC_(0.6192850577371690132255643845837767003104e-14), SC_(0.13646809890861383761353246234604301386e-13), 
      SC_(9.05791950225830078125), SC_(0.1047879028014987723427253740737796761096e-13), SC_(0.2309147573525077750995455599061229154568e-13), 
      SC_(9.05791950225830078125), SC_(0.1978473638988408750405412206418986897916e-13), SC_(0.4359842577828666553912992682044051988355e-13), 
      SC_(9.05791950225830078125), SC_(0.4041816252346730475863978426787070930004e-13), SC_(0.890670577635369247768708968495975078987e-13), 
      SC_(9.05791950225830078125), SC_(0.9410302262901834580155480125540634617209e-13), SC_(0.2073691337986060222984783598283581979693e-12), 
      SC_(9.05791950225830078125), SC_(0.1334530223958893535574077304772799834609e-12), SC_(0.2940823459639750406216533544479977491992e-12), 
      SC_(9.05791950225830078125), SC_(0.266297021326439287136622624529991298914e-12), SC_(0.5868226237888610086993479571271314918231e-12), 
      SC_(9.05791950225830078125), SC_(0.5920415525016708979677559909760020673275e-12), SC_(0.1304646125971113163641907177702515707777e-11), 
      SC_(9.05791950225830078125), SC_(0.155163989296047688526414276566356420517e-11), SC_(0.3419254893007777507867868413269779363349e-11), 
      SC_(9.05791950225830078125), SC_(0.326923297461201300961874949280172586441e-11), SC_(0.7204210780838943251938695589081939118372e-11), 
      SC_(9.05791950225830078125), SC_(0.3753785910581841633870681107509881258011e-11), SC_(0.8271990750123550233372114200470324412159e-11), 
      SC_(9.05791950225830078125), SC_(0.9579165585749116473834874341264367103577e-11), SC_(0.2110902726134560402003355275619485042681e-10), 
      SC_(9.05791950225830078125), SC_(0.1858167439361402273334533674642443656921e-10), SC_(0.4094731089351581985650299702999672083823e-10), 
      SC_(9.05791950225830078125), SC_(0.5449485307451595872407779097557067871094e-10), SC_(0.1200870085095152217108006375172095614186e-9), 
      SC_(9.05791950225830078125), SC_(0.6089519166696533147842274047434329986572e-10), SC_(0.134191047179292714125351544086056023465e-9), 
      SC_(9.05791950225830078125), SC_(0.1337744776064297980155970435589551925659e-9), SC_(0.2947907173933645856408143378003830948434e-9), 
      SC_(9.05791950225830078125), SC_(0.2554458866654840676346793770790100097656e-9), SC_(0.562910635442301128672721898229300339369e-9), 
      SC_(9.05791950225830078125), SC_(0.9285605062636648199259070679545402526855e-9), SC_(0.2046212572996540499923424107815318022543e-8), 
      SC_(9.05791950225830078125), SC_(0.1698227447555211711005540564656257629395e-8), SC_(0.374228101938620755446651962929752618141e-8), 
      SC_(9.05791950225830078125), SC_(0.339355921141759608872234821319580078125e-8), SC_(0.7478181009885443030654062342037974946626e-8), 
      SC_(9.05791950225830078125), SC_(0.6313728651008432279922999441623687744141e-8), SC_(0.1391318168034338234837029722532165802178e-7), 
      SC_(9.05791950225830078125), SC_(0.8383264749056706932606175541877746582031e-8), SC_(0.1847369315996482893977157656361806551841e-7), 
      SC_(9.05791950225830078125), SC_(0.1962631124285962869180366396903991699219e-7), SC_(0.4324931480838746041625206349890339471437e-7), 
      SC_(9.05791950225830078125), SC_(0.5256384838503436185419559478759765625e-7), SC_(0.1158317770831471770924205695789568537273e-6), 
      SC_(9.05791950225830078125), SC_(0.116242290459922514855861663818359375e-6), SC_(0.2561561307737854087185767462770208718312e-6), 
      SC_(9.05791950225830078125), SC_(0.1776920584006802528165280818939208984375e-6), SC_(0.39156930795883500037782346733448271256e-6), 
      SC_(9.05791950225830078125), SC_(0.246631174150024889968335628509521484375e-6), SC_(0.5434863346610004614961081293368009808849e-6), 
      SC_(9.05791950225830078125), SC_(0.7932688959044753573834896087646484375e-6), SC_(0.1748080167846223040853665963955169181426e-5), 
      SC_(9.05791950225830078125), SC_(0.1372093493046122603118419647216796875e-5), SC_(0.3023603932659035997865798381293341205636e-5), 
      SC_(9.05791950225830078125), SC_(0.214747751670074649155139923095703125e-5), SC_(0.4732277388528918566050741155653964028704e-5), 
      SC_(9.05791950225830078125), SC_(0.527022712049074470996856689453125e-5), SC_(0.1161374787529083016181945296971724928477e-4), 
      SC_(9.05791950225830078125), SC_(0.9233162927557714283466339111328125e-5), SC_(0.2034676914359927680695331543608768988425e-4), 
      SC_(9.05791950225830078125), SC_(0.269396477960981428623199462890625e-4), SC_(0.5936703302530892369764216926987806807892e-4), 
      SC_(9.05791950225830078125), SC_(0.3208058114978484809398651123046875e-4), SC_(0.7069653334490330852794441936131793027422e-4), 
      SC_(9.05791950225830078125), SC_(0.00010957030463032424449920654296875), SC_(0.0002414825989570633912995913519215386700583), 
      SC_(9.05791950225830078125), SC_(0.000126518702018074691295623779296875), SC_(0.0002788404727430997089747468089908789181372), 
      SC_(9.05791950225830078125), SC_(0.00028976381872780621051788330078125), SC_(0.000638738891360736008829881157043723842803), 
      SC_(9.05791950225830078125), SC_(0.000687857042066752910614013671875), SC_(0.001516938308174365641969878913071045025617), 
      SC_(9.05791950225830078125), SC_(0.00145484809763729572296142578125), SC_(0.00321110526183416990135343809918247336095), 
      SC_(9.05791950225830078125), SC_(0.002847635187208652496337890625), SC_(0.006294891333644249781144157926744671614596), 
      SC_(9.05791950225830078125), SC_(0.0056468211114406585693359375), SC_(0.01252130101261455228056921833268729571347), 
      SC_(9.05791950225830078125), SC_(0.011621631681919097900390625), SC_(0.02594063673570323792964668118914872441002), 
      SC_(9.05791950225830078125), SC_(0.0257236398756504058837890625), SC_(0.05832305078727693333226267606064661195371), 
      SC_(9.05791950225830078125), SC_(0.0560617186129093170166015625), SC_(0.1314950543792148978624394628956067400208), 
      SC_(9.05791950225830078125), SC_(0.106835305690765380859375), SC_(0.2654483612415900617941432175085551926815), 
      SC_(9.05791950225830078125), SC_(0.2401093542575836181640625), SC_(0.6974284805143726880282057511738019555138), 
      SC_(9.05791950225830078125), SC_(0.438671648502349853515625), SC_(1.629185635178873766677925251039821143485), 
      SC_(9.05791950225830078125), SC_(0.903765499591827392578125), SC_(6.32705246632545632147034161046161342959), 
      SC_(9.133758544921875), SC_(0.1431564604442703013402649929484277542953e-29), SC_(0.3166588379386359837662710450824745998242e-29), 
      SC_(9.133758544921875), SC_(0.1791466932348087634896446282571611213266e-29), SC_(0.3962684151608216310450856914760484352206e-29), 
      SC_(9.133758544921875), SC_(0.6013619202535540063110633226832922483532e-29), SC_(0.1330198904450900967019080660951867562253e-28), 
      SC_(9.133758544921875), SC_(0.115805324961653822428570241697281798758e-28), SC_(0.2561587476783072877437995061182958780849e-28), 
      SC_(9.133758544921875), SC_(0.1422457400834001098175711728787848259007e-28), SC_(0.3146443451922714064504469391515819608822e-28), 
      SC_(9.133758544921875), SC_(0.4970121018327539153628705477876439795096e-28), SC_(0.1099379476967907805024633596202812551352e-27), 
      SC_(9.133758544921875), SC_(0.9660079415057497591758174164417478444323e-28), SC_(0.213678761857755691115389300048368707761e-27), 
      SC_(9.133758544921875), SC_(0.1232929313253182131376331095427391968754e-27), SC_(0.2727211628337373159341633878514945047035e-27), 
      SC_(9.133758544921875), SC_(0.3296523285617759312781860549364832953326e-27), SC_(0.7291834609641984080820739719339622110891e-27), 
      SC_(9.133758544921875), SC_(0.528364435768055252017009628713605422886e-27), SC_(0.116873012729696839367246878018486989899e-26), 
      SC_(9.133758544921875), SC_(0.886586057273120049620324386849842094685e-27), SC_(0.1961108215147545731654019661289312160823e-26), 
      SC_(9.133758544921875), SC_(0.2499669674831043259218157022821422146034e-26), SC_(0.5529212527370272487719402384883321506077e-26), 
      SC_(9.133758544921875), SC_(0.4131050397232622964314362671638736040881e-26), SC_(0.9137789619790643132454043156307219409037e-26), 
      SC_(9.133758544921875), SC_(0.7679738097881433551381658732998641759182e-26), SC_(0.1698740618621894793643662785149599259976e-25), 
      SC_(9.133758544921875), SC_(0.199929739820949207249437007767740538737e-25), SC_(0.4422400420113873694244706119533817266574e-25), 
      SC_(9.133758544921875), SC_(0.5151477415246978459754129800826163591626e-25), SC_(0.1139495099918507361927196722899871079642e-24), 
      SC_(9.133758544921875), SC_(0.101200734533556026342258477595279955025e-24), SC_(0.2238537254726798757161646422414983289474e-24), 
      SC_(9.133758544921875), SC_(0.2064292695896540981798546456623054911033e-24), SC_(0.4566168541882075534287139119305248572501e-24), 
      SC_(9.133758544921875), SC_(0.4063294332896333395257434433879773416284e-24), SC_(0.8987914744919890138736138418319428222193e-24), 
      SC_(9.133758544921875), SC_(0.8138195767936862452966745688936976428456e-24), SC_(0.1800150413606538243120530969681451426396e-23), 
      SC_(9.133758544921875), SC_(0.9575550627132253801929510132578249716542e-24), SC_(0.21180900427408106498072807137975544998e-23), 
      SC_(9.133758544921875), SC_(0.2855160956298500804375620841706273850616e-23), SC_(0.6315551165091928672573973671753279421403e-23), 
      SC_(9.133758544921875), SC_(0.65201444297915461398563707001320281266e-23), SC_(0.144224113387716649510960987260862116892e-22), 
      SC_(9.133758544921875), SC_(0.1310988374636350038320977491775043421995e-22), SC_(0.2899876498588179380205715603124867921109e-22), 
      SC_(9.133758544921875), SC_(0.2590288837798696209228010176465529547374e-22), SC_(0.5729660056956125438944678043621476099764e-22), 
      SC_(9.133758544921875), SC_(0.2937779542193655202274099291941187976629e-22), SC_(0.6498301599969281622277012892067585290436e-22), 
      SC_(9.133758544921875), SC_(0.7863513178004503049754083414326234074965e-22), SC_(0.1739391248801805801607283296725349924552e-21), 
      SC_(9.133758544921875), SC_(0.1903818607087388763706780167350761726053e-21), SC_(0.4211203503462802062650808793819815783631e-21), 
      SC_(9.133758544921875), SC_(0.3812242142377350870566942975497647799754e-21), SC_(0.8432593003484141482284951403272575802672e-21), 
      SC_(9.133758544921875), SC_(0.5493133580141330277178034419485741501887e-21), SC_(0.1215068667338563382528779550284588948439e-20), 
      SC_(9.133758544921875), SC_(0.9672153634284186955666772243312215295852e-21), SC_(0.2139458408437458632544702298187423963293e-20), 
      SC_(9.133758544921875), SC_(0.1702169477623814384559878647986894129041e-20), SC_(0.3765160210627052717818735073383363610292e-20), 
      SC_(9.133758544921875), SC_(0.4817114569977399785676754474621208412799e-20), SC_(0.1065534798228777762983085692745067109424e-19), 
      SC_(9.133758544921875), SC_(0.7538352992756463183303278501219690799218e-20), SC_(0.1667466554600080042731918866079645111871e-19), 
      SC_(9.133758544921875), SC_(0.2596305715949999708394617609422128090557e-19), SC_(0.5742969254721264735800352737628286744035e-19), 
      SC_(9.133758544921875), SC_(0.4444587480324321591032923385589104015025e-19), SC_(0.9831326523918958338122603416318193669595e-19), 
      SC_(9.133758544921875), SC_(0.9715574921498573937069095571295029856174e-19), SC_(0.2149063098514620884683636668531297215924e-18), 
      SC_(9.133758544921875), SC_(0.2036598542733453787268262970278076551267e-18), SC_(0.450490970430602546519886934276976341266e-18), 
      SC_(9.133758544921875), SC_(0.4248971931658660264162106698360155121463e-18), SC_(0.9398629374723158741379479331377426086494e-18), 
      SC_(9.133758544921875), SC_(0.6521097487613458963613731825259556273977e-18), SC_(0.1442451948101993414376845665768396656522e-17), 
      SC_(9.133758544921875), SC_(0.1436126164096190058281493628911107407475e-17), SC_(0.3176678445699666680955154923506384383446e-17), 
      SC_(9.133758544921875), SC_(0.3118908901459261162419055180006211003274e-17), SC_(0.6898955627343371549445486787441361048131e-17), 
      SC_(9.133758544921875), SC_(0.3593346613595175715618300349429858897565e-17), SC_(0.7948401067199780099392046222922514366964e-17), 
      SC_(9.133758544921875), SC_(0.9445874854124767215374919304693435151421e-17), SC_(0.2089406056379377447514939065220814131497e-16), 
      SC_(9.133758544921875), SC_(0.2566182432094081539023303073498993853718e-16), SC_(0.5676337235243358738091088936640596066828e-16), 
      SC_(9.133758544921875), SC_(0.3363765695149349330660137891158001366421e-16), SC_(0.7440573291755156029522906020009751167445e-16), 
      SC_(9.133758544921875), SC_(0.1073581901339262605326457800103412409953e-15), SC_(0.2374738773611867553628158083140001517171e-15), 
      SC_(9.133758544921875), SC_(0.186668406231853462907965823802669547149e-15), SC_(0.4129062734143745293001314520947967973891e-15), 
      SC_(9.133758544921875), SC_(0.3727540802657755688795382376099496468669e-15), SC_(0.8245235564468013190217027819473057111831e-15), 
      SC_(9.133758544921875), SC_(0.6211646767866855090717281839829411183018e-15), SC_(0.1374002151976739526070303332333153231473e-14), 
      SC_(9.133758544921875), SC_(0.1561186859754253464932505224282976996619e-14), SC_(0.345330986307322779610281858961887866469e-14), 
      SC_(9.133758544921875), SC_(0.3092010764722992466335682593125966377556e-14), SC_(0.6839457560017699966998240266744866833624e-14), 
      SC_(9.133758544921875), SC_(0.6192850577371690132255643845837767003104e-14), SC_(0.1369844477344809694294274855536145555036e-13), 
      SC_(9.133758544921875), SC_(0.1047879028014987723427253740737796761096e-13), SC_(0.2317884601796733088332805910112079640838e-13), 
      SC_(9.133758544921875), SC_(0.1978473638988408750405412206418986897916e-13), SC_(0.4376338737839915159615291094424318821963e-13), 
      SC_(9.133758544921875), SC_(0.4041816252346730475863978426787070930004e-13), SC_(0.8940405718733956105008108936229491171121e-13), 
      SC_(9.133758544921875), SC_(0.9410302262901834580155480125540634617209e-13), SC_(0.2081537480023233882996487367026237796731e-12), 
      SC_(9.133758544921875), SC_(0.1334530223958893535574077304772799834609e-12), SC_(0.2951950534411155963930576171475953294381e-12), 
      SC_(9.133758544921875), SC_(0.266297021326439287136622624529991298914e-12), SC_(0.5890429608142096452582699368428277658628e-12), 
      SC_(9.133758544921875), SC_(0.5920415525016708979677559909760020673275e-12), SC_(0.1309582462746558652863594053840430217562e-11), 
      SC_(9.133758544921875), SC_(0.155163989296047688526414276566356420517e-11), SC_(0.3432192189441651910506223477029484359357e-11), 
      SC_(9.133758544921875), SC_(0.326923297461201300961874949280172586441e-11), SC_(0.7231469061768686221153900026064267317522e-11), 
      SC_(9.133758544921875), SC_(0.3753785910581841633870681107509881258011e-11), SC_(0.8303289146932668340064370413905394263433e-11), 
      SC_(9.133758544921875), SC_(0.9579165585749116473834874341264367103577e-11), SC_(0.2118889663395998649174305175749893352622e-10), 
      SC_(9.133758544921875), SC_(0.1858167439361402273334533674642443656921e-10), SC_(0.4110224157747644423583113222517509247955e-10), 
      SC_(9.133758544921875), SC_(0.5449485307451595872407779097557067871094e-10), SC_(0.1205413768662624216495148932262066295668e-9), 
      SC_(9.133758544921875), SC_(0.6089519166696533147842274047434329986572e-10), SC_(0.1346987804166698582103545800495053990511e-9), 
      SC_(9.133758544921875), SC_(0.1337744776064297980155970435589551925659e-9), SC_(0.2959061051069945958835846269141153207358e-9), 
      SC_(9.133758544921875), SC_(0.2554458866654840676346793770790100097656e-9), SC_(0.5650404976451556437547264787093649893696e-9), 
      SC_(9.133758544921875), SC_(0.9285605062636648199259070679545402526855e-9), SC_(0.2053954744750057116474127386247909619633e-8), 
      SC_(9.133758544921875), SC_(0.1698227447555211711005540564656257629395e-8), SC_(0.3756440536735007285724410041691924552336e-8), 
      SC_(9.133758544921875), SC_(0.339355921141759608872234821319580078125e-8), SC_(0.7506475900995156011288348709812761296664e-8), 
      SC_(9.133758544921875), SC_(0.6313728651008432279922999441623687744141e-8), SC_(0.1396582442346313363883316229808727647626e-7), 
      SC_(9.133758544921875), SC_(0.8383264749056706932606175541877746582031e-8), SC_(0.185435913261832036402039632866481232055e-7), 
      SC_(9.133758544921875), SC_(0.1962631124285962869180366396903991699219e-7), SC_(0.434129554949963243700876044278654143025e-7), 
      SC_(9.133758544921875), SC_(0.5256384838503436185419559478759765625e-7), SC_(0.1162700451183168473911017592713844196679e-6), 
      SC_(9.133758544921875), SC_(0.116242290459922514855861663818359375e-6), SC_(0.2571253384891851869983057722984941167289e-6), 
      SC_(9.133758544921875), SC_(0.1776920584006802528165280818939208984375e-6), SC_(0.3930508732020080511653669652574353308432e-6), 
      SC_(9.133758544921875), SC_(0.246631174150024889968335628509521484375e-6), SC_(0.5455427024829950357648610380554611106052e-6), 
      SC_(9.133758544921875), SC_(0.7932688959044753573834896087646484375e-6), SC_(0.1754694313872418709490561563784264799798e-5), 
      SC_(9.133758544921875), SC_(0.1372093493046122603118419647216796875e-5), SC_(0.3035044237918703040505553684515020241501e-5), 
      SC_(9.133758544921875), SC_(0.214747751670074649155139923095703125e-5), SC_(0.4750182757595584701061593856707247719575e-5), 
      SC_(9.133758544921875), SC_(0.527022712049074470996856689453125e-5), SC_(0.1165769059949704066228382252035286160394e-4), 
      SC_(9.133758544921875), SC_(0.9233162927557714283466339111328125e-5), SC_(0.2042375517716371738869107466887463011806e-4), 
      SC_(9.133758544921875), SC_(0.269396477960981428623199462890625e-4), SC_(0.5959166435958585870676376573465541576689e-4), 
      SC_(9.133758544921875), SC_(0.3208058114978484809398651123046875e-4), SC_(0.7096403444944712104973145464518363610699e-4), 
      SC_(9.133758544921875), SC_(0.00010957030463032424449920654296875), SC_(0.0002423963976274547113520639506749754456639), 
      SC_(9.133758544921875), SC_(0.000126518702018074691295623779296875), SC_(0.0002798956577974984275654773358319046607446), 
      SC_(9.133758544921875), SC_(0.00028976381872780621051788330078125), SC_(0.0006411564364665104093195535580417210607387), 
      SC_(9.133758544921875), SC_(0.000687857042066752910614013671875), SC_(0.001522682253857451785825717819605251309824), 
      SC_(9.133758544921875), SC_(0.00145484809763729572296142578125), SC_(0.003223274550919413078558922991480584957931), 
      SC_(9.133758544921875), SC_(0.002847635187208652496337890625), SC_(0.006318784151197779190311758585128772805171), 
      SC_(9.133758544921875), SC_(0.0056468211114406585693359375), SC_(0.01256897385147959593489613597170196869447), 
      SC_(9.133758544921875), SC_(0.011621631681919097900390625), SC_(0.02604005426676951113278576709191952468673), 
      SC_(9.133758544921875), SC_(0.0257236398756504058837890625), SC_(0.05855006332475848314609853654570300938235), 
      SC_(9.133758544921875), SC_(0.0560617186129093170166015625), SC_(0.1320240756877234443504850105930086569822), 
      SC_(9.133758544921875), SC_(0.106835305690765380859375), SC_(0.2665760915709943020795338378823706112721), 
      SC_(9.133758544921875), SC_(0.2401093542575836181640625), SC_(0.7008301174292642126150669803023638485546), 
      SC_(9.133758544921875), SC_(0.438671648502349853515625), SC_(1.638819662694764963830781936726649515985), 
      SC_(9.133758544921875), SC_(0.903765499591827392578125), SC_(6.382473545580206211190860756907471309227), 
      SC_(9.68867778778076171875), SC_(0.1431564604442703013402649929484277542953e-29), SC_(0.3251023041419442629663190449032320727026e-29), 
      SC_(9.68867778778076171875), SC_(0.1791466932348087634896446282571611213266e-29), SC_(0.4068346099736040733007820502305239840058e-29), 
      SC_(9.68867778778076171875), SC_(0.6013619202535540063110633226832922483532e-29), SC_(0.1365667642877792388513882350014289657906e-28), 
      SC_(9.68867778778076171875), SC_(0.115805324961653822428570241697281798758e-28), SC_(0.2629890251554282451955534879170468653276e-28), 
      SC_(9.68867778778076171875), SC_(0.1422457400834001098175711728787848259007e-28), SC_(0.3230340964841896705434811777933594702467e-28), 
      SC_(9.68867778778076171875), SC_(0.4970121018327539153628705477876439795096e-28), SC_(0.1128693591548812362301112581850657975072e-27), 
      SC_(9.68867778778076171875), SC_(0.9660079415057497591758174164417478444323e-28), SC_(0.2193763429385665316044594844377369163394e-27), 
      SC_(9.68867778778076171875), SC_(0.1232929313253182131376331095427391968754e-27), SC_(0.279993064468643928226977638193225502021e-27), 
      SC_(9.68867778778076171875), SC_(0.3296523285617759312781860549364832953326e-27), SC_(0.748626581354401062400285076958412022658e-27), 
      SC_(9.68867778778076171875), SC_(0.528364435768055252017009628713605422886e-27), SC_(0.1199893424032530976049611749648424671609e-26), 
      SC_(9.68867778778076171875), SC_(0.886586057273120049620324386849842094685e-27), SC_(0.2013399668761852677111403309217419036542e-26), 
      SC_(9.68867778778076171875), SC_(0.2499669674831043259218157022821422146034e-26), SC_(0.5676644758883756040892047084598038565024e-26), 
      SC_(9.68867778778076171875), SC_(0.4131050397232622964314362671638736040881e-26), SC_(0.9381441804993808004633913547851406415649e-26), 
      SC_(9.68867778778076171875), SC_(0.7679738097881433551381658732998641759182e-26), SC_(0.1744036240543871557809919087250212910793e-25), 
      SC_(9.68867778778076171875), SC_(0.199929739820949207249437007767740538737e-25), SC_(0.4540320351633245557172484303204920023135e-25), 
      SC_(9.68867778778076171875), SC_(0.5151477415246978459754129800826163591626e-25), SC_(0.1169878866964547547221995022196319253455e-24), 
      SC_(9.68867778778076171875), SC_(0.101200734533556026342258477595279955025e-24), SC_(0.229822614191584026946923327297559366982e-24), 
      SC_(9.68867778778076171875), SC_(0.2064292695896540981798546456623054911033e-24), SC_(0.4687921940628085039658292492167189278115e-24), 
      SC_(9.68867778778076171875), SC_(0.4063294332896333395257434433879773416284e-24), SC_(0.9227570630986311860335595959287304929927e-24), 
      SC_(9.68867778778076171875), SC_(0.8138195767936862452966745688936976428456e-24), SC_(0.1848150050304199833133111992475494790028e-23), 
      SC_(9.68867778778076171875), SC_(0.9575550627132253801929510132578249716542e-24), SC_(0.2174567296961365499627397209048994087595e-23), 
      SC_(9.68867778778076171875), SC_(0.2855160956298500804375620841706273850616e-23), SC_(0.6483950516156469579872053996346377811741e-23), 
      SC_(9.68867778778076171875), SC_(0.65201444297915461398563707001320281266e-23), SC_(0.14806973928282357752174713141185757796e-22), 
      SC_(9.68867778778076171875), SC_(0.1310988374636350038320977491775043421995e-22), SC_(0.2977199491904860757140525556186179527699e-22), 
      SC_(9.68867778778076171875), SC_(0.2590288837798696209228010176465529547374e-22), SC_(0.5882437068841482784486186135173126954487e-22), 
      SC_(9.68867778778076171875), SC_(0.2937779542193655202274099291941187976629e-22), SC_(0.6671573851883745565690238915558861233449e-22), 
      SC_(9.68867778778076171875), SC_(0.7863513178004503049754083414326234074965e-22), SC_(0.1785770788748308935419019642128233135009e-21), 
      SC_(9.68867778778076171875), SC_(0.1903818607087388763706780167350761726053e-21), SC_(0.4323492030409370546305056478788417643006e-21), 
      SC_(9.68867778778076171875), SC_(0.3812242142377350870566942975497647799754e-21), SC_(0.8657441659200391598244521390735246465316e-21), 
      SC_(9.68867778778076171875), SC_(0.5493133580141330277178034419485741501887e-21), SC_(0.124746754587344929524756930088081819053e-20), 
      SC_(9.68867778778076171875), SC_(0.9672153634284186955666772243312215295852e-21), SC_(0.2196505433818528328509987828061540879135e-20), 
      SC_(9.68867778778076171875), SC_(0.1702169477623814384559878647986894129041e-20), SC_(0.3865555333641529419702847527826614072778e-20), 
      SC_(9.68867778778076171875), SC_(0.4817114569977399785676754474621208412799e-20), SC_(0.1093946470285242032028108746437167153554e-19), 
      SC_(9.68867778778076171875), SC_(0.7538352992756463183303278501219690799218e-20), SC_(0.1711928277476865895025016138793163450484e-19), 
      SC_(9.68867778778076171875), SC_(0.2596305715949999708394617609422128090557e-19), SC_(0.5896101146205924080209010975883278777754e-19), 
      SC_(9.68867778778076171875), SC_(0.4444587480324321591032923385589104015025e-19), SC_(0.1009347134128383557596449566027375645171e-18), 
      SC_(9.68867778778076171875), SC_(0.9715574921498573937069095571295029856174e-19), SC_(0.2206366225625197607062393262722651761487e-18), 
      SC_(9.68867778778076171875), SC_(0.2036598542733453787268262970278076551267e-18), SC_(0.4625029682907836869149984686402911484836e-18), 
      SC_(9.68867778778076171875), SC_(0.4248971931658660264162106698360155121463e-18), SC_(0.9649236652888796690140481551193469292926e-18), 
      SC_(9.68867778778076171875), SC_(0.6521097487613458963613731825259556273977e-18), SC_(0.1480913828253450121139914131430527700071e-17), 
      SC_(9.68867778778076171875), SC_(0.1436126164096190058281493628911107407475e-17), SC_(0.326138215164909851026584251749321471815e-17), 
      SC_(9.68867778778076171875), SC_(0.3118908901459261162419055180006211003274e-17), SC_(0.7082911013072682747000780572255711975698e-17), 
      SC_(9.68867778778076171875), SC_(0.3593346613595175715618300349429858897565e-17), SC_(0.8160339114525799062082084735226523498546e-17), 
      SC_(9.68867778778076171875), SC_(0.9445874854124767215374919304693435151421e-17), SC_(0.2145118473998511274881313602044206744275e-16), 
      SC_(9.68867778778076171875), SC_(0.2566182432094081539023303073498993853718e-16), SC_(0.5827692434789835381847157763572645320376e-16), 
      SC_(9.68867778778076171875), SC_(0.3363765695149349330660137891158001366421e-16), SC_(0.7638970499081317392873833048234230607465e-16), 
      SC_(9.68867778778076171875), SC_(0.1073581901339262605326457800103412409953e-15), SC_(0.2438059370337366694971865890221449361133e-15), 
      SC_(9.68867778778076171875), SC_(0.186668406231853462907965823802669547149e-15), SC_(0.4239161040175672541184411515854470301012e-15), 
      SC_(9.68867778778076171875), SC_(0.3727540802657755688795382376099496468669e-15), SC_(0.8465088477085573603406429900262807830109e-15), 
      SC_(9.68867778778076171875), SC_(0.6211646767866855090717281839829411183018e-15), SC_(0.1410638870563247541019592683730600534645e-14), 
      SC_(9.68867778778076171875), SC_(0.1561186859754253464932505224282976996619e-14), SC_(0.3545389734610116283745403992517326218345e-14), 
      SC_(9.68867778778076171875), SC_(0.3092010764722992466335682593125966377556e-14), SC_(0.7021826475197518343268876740635260691793e-14), 
      SC_(9.68867778778076171875), SC_(0.6192850577371690132255643845837767003104e-14), SC_(0.1406370334710871427355144538953530568614e-13), 
      SC_(9.68867778778076171875), SC_(0.1047879028014987723427253740737796761096e-13), SC_(0.2379689225428403830805824843435024125941e-13), 
      SC_(9.68867778778076171875), SC_(0.1978473638988408750405412206418986897916e-13), SC_(0.4493030469761010753770634758104886456194e-13), 
      SC_(9.68867778778076171875), SC_(0.4041816252346730475863978426787070930004e-13), SC_(0.9178794812881475840845530325584300055822e-13), 
      SC_(9.68867778778076171875), SC_(0.9410302262901834580155480125540634617209e-13), SC_(0.2137040088060036565135543283518672527502e-12), 
      SC_(9.68867778778076171875), SC_(0.1334530223958893535574077304772799834609e-12), SC_(0.3030662042144195267231304473289146643627e-12), 
      SC_(9.68867778778076171875), SC_(0.266297021326439287136622624529991298914e-12), SC_(0.6047493417392123849215205373703035008782e-12), 
      SC_(9.68867778778076171875), SC_(0.5920415525016708979677559909760020673275e-12), SC_(0.1344501479492258945113336046310014816028e-11), 
      SC_(9.68867778778076171875), SC_(0.155163989296047688526414276566356420517e-11), SC_(0.3523708974330744853236964747567641430526e-11), 
      SC_(9.68867778778076171875), SC_(0.326923297461201300961874949280172586441e-11), SC_(0.7424290664415293228539717523609736893227e-11), 
      SC_(9.68867778778076171875), SC_(0.3753785910581841633870681107509881258011e-11), SC_(0.8524690014014443239947636563728062120671e-11), 
      SC_(9.68867778778076171875), SC_(0.9579165585749116473834874341264367103577e-11), SC_(0.217538823889157646975210405182959223965e-10), 
      SC_(9.68867778778076171875), SC_(0.1858167439361402273334533674642443656921e-10), SC_(0.4219820147522043070128530753246849254589e-10), 
      SC_(9.68867778778076171875), SC_(0.5449485307451595872407779097557067871094e-10), SC_(0.123755520669603051486977872131239776802e-9), 
      SC_(9.68867778778076171875), SC_(0.6089519166696533147842274047434329986572e-10), SC_(0.1382904205791782327650424561456364420732e-9), 
      SC_(9.68867778778076171875), SC_(0.1337744776064297980155970435589551925659e-9), SC_(0.3037962155314103466980398435578035811679e-9), 
      SC_(9.68867778778076171875), SC_(0.2554458866654840676346793770790100097656e-9), SC_(0.5801068712155522869254305125924800021551e-9), 
      SC_(9.68867778778076171875), SC_(0.9285605062636648199259070679545402526855e-9), SC_(0.2108721880262844458250615511467757397605e-8), 
      SC_(9.68867778778076171875), SC_(0.1698227447555211711005540564656257629395e-8), SC_(0.3856603156494151031626014622236295405284e-8), 
      SC_(9.68867778778076171875), SC_(0.339355921141759608872234821319580078125e-8), SC_(0.7706630351863533996193594586001707313967e-8), 
      SC_(9.68867778778076171875), SC_(0.6313728651008432279922999441623687744141e-8), SC_(0.1433821247406628873576148497830599721119e-7), 
      SC_(9.68867778778076171875), SC_(0.8383264749056706932606175541877746582031e-8), SC_(0.1903804203900791197963208573178075630015e-7), 
      SC_(9.68867778778076171875), SC_(0.1962631124285962869180366396903991699219e-7), SC_(0.4457052884132293020120854209156627105415e-7), 
      SC_(9.68867778778076171875), SC_(0.5256384838503436185419559478759765625e-7), SC_(0.1193702973058086549839229184033913497394e-6), 
      SC_(9.68867778778076171875), SC_(0.116242290459922514855861663818359375e-6), SC_(0.2639813902774001345761419758813907104525e-6), 
      SC_(9.68867778778076171875), SC_(0.1776920584006802528165280818939208984375e-6), SC_(0.4035312768290402179724322739059024257911e-6), 
      SC_(9.68867778778076171875), SC_(0.246631174150024889968335628509521484375e-6), SC_(0.5600891862976372697203768774599773207812e-6), 
      SC_(9.68867778778076171875), SC_(0.7932688959044753573834896087646484375e-6), SC_(0.1801481940511445476740974446351987487537e-5), 
      SC_(9.68867778778076171875), SC_(0.1372093493046122603118419647216796875e-5), SC_(0.3115971501912317630070527687604350940031e-5), 
      SC_(9.68867778778076171875), SC_(0.214747751670074649155139923095703125e-5), SC_(0.487684306379282015062143358653997082528e-5), 
      SC_(9.68867778778076171875), SC_(0.527022712049074470996856689453125e-5), SC_(0.1196853588100977502339779488819564685125e-4), 
      SC_(9.68867778778076171875), SC_(0.9233162927557714283466339111328125e-5), SC_(0.2096834472868302910410196802704006676419e-4), 
      SC_(9.68867778778076171875), SC_(0.269396477960981428623199462890625e-4), SC_(0.6118067920858328215848432186906668843509e-4), 
      SC_(9.68867778778076171875), SC_(0.3208058114978484809398651123046875e-4), SC_(0.728563051879196577689828782571903259951e-4), 
      SC_(9.68867778778076171875), SC_(0.00010957030463032424449920654296875), SC_(0.0002488605167115786508690708436538612267867), 
      SC_(9.68867778778076171875), SC_(0.000126518702018074691295623779296875), SC_(0.0002873599340757661525877158215704810459439), 
      SC_(9.68867778778076171875), SC_(0.00028976381872780621051788330078125), SC_(0.0006582580089946178862548800972604646079047), 
      SC_(9.68867778778076171875), SC_(0.000687857042066752910614013671875), SC_(0.001563315133764469862768389959281037577749), 
      SC_(9.68867778778076171875), SC_(0.00145484809763729572296142578125), SC_(0.003309362765364723055307278727456538237784), 
      SC_(9.68867778778076171875), SC_(0.002847635187208652496337890625), SC_(0.006487815096079959006202372484557638978379), 
      SC_(9.68867778778076171875), SC_(0.0056468211114406585693359375), SC_(0.01290626953766988799660445158177424847916), 
      SC_(9.68867778778076171875), SC_(0.011621631681919097900390625), SC_(0.02674359636941844621991721651837741451166), 
      SC_(9.68867778778076171875), SC_(0.0257236398756504058837890625), SC_(0.06015731222731434671073651753326632370858), 
      SC_(9.68867778778076171875), SC_(0.0560617186129093170166015625), SC_(0.1357733748610437010694097215927829841061), 
      SC_(9.68867778778076171875), SC_(0.106835305690765380859375), SC_(0.2745822634838372037418174975910269667509), 
      SC_(9.68867778778076171875), SC_(0.2401093542575836181640625), SC_(0.7250883224868308476529921117348408126984), 
      SC_(9.68867778778076171875), SC_(0.438671648502349853515625), SC_(1.707985158316040663526167707818201486792), 
      SC_(9.68867778778076171875), SC_(0.903765499591827392578125), SC_(6.786671267376742594913276377075578069747), 
   };
#undef SC_


   using namespace std;

   typedef T (*func_t)(const T&);
   func_t f = &boost::math::sqrt1pm1<T>;

   boost::math::tools::test_result<T> result = boost::math::tools::test(
      sqrtp1m1_data, 
      bind_func(f, 0), 
      extract_result(1));

   std::cout << "~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
      "Test results for type " << type_name << std::endl << std::endl;
   handle_test_result(result, sqrtp1m1_data[result.worst()], result.worst(), type_name, "boost::math::sqrt1pm1", "sqrt1pm1");

   typedef T (*func2_t)(T const, T const);
   func2_t f2 = &boost::math::powm1<T,T>;
   result = boost::math::tools::test(
      powm1_data, 
      bind_func(f2, 0, 1), 
      extract_result(2));
   handle_test_result(result, powm1_data[result.worst()], result.worst(), type_name, "boost::math::powm1", "powm1");
}

int test_main(int, char* [])
{
   expected_results();
   BOOST_MATH_CONTROL_FP;
   test_powm1_sqrtp1m1(1.0F, "float");
   test_powm1_sqrtp1m1(1.0, "double");
#ifndef BOOST_MATH_NO_LONG_DOUBLE_MATH_FUNCTIONS
   test_powm1_sqrtp1m1(1.0L, "long double");
#ifndef BOOST_MATH_NO_REAL_CONCEPT_TESTS
   test_powm1_sqrtp1m1(boost::math::concepts::real_concept(), "real_concept");
#endif
#else
   std::cout << "<note>The long double tests have been disabled on this platform "
      "either because the long double overloads of the usual math functions are "
      "not available at all, or because they are too inaccurate for these tests "
      "to pass.</note>" << std::cout;
#endif
   return 0;
}


