/*
 * Copyright (C) 2002-4 Edscott Wilson Garcia
 * EMail: edscott@imp.mx
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <dirent.h>
#include <limits.h>

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>

#if 0
#ifdef HAVE_LIBRSVG
#include <librsvg/rsvg.h>
#endif
#endif


#include <libxml/parser.h>
#include <libxml/tree.h>

#include "glade_support.h"

#include "constants.h"
#include "types.h"
#include "modules.h"

#include "basic_row.h"

#include "icons.h"

#include "entry.h"
#include "misc.h"
#include "pasteboard.h"
#include "settings.h"
#include "toolbar.h"

/* if themes.mime.xml does not exist...*/
#define FALLBACK 1


#define PIXBUF_HASH
#ifdef PIXBUF_HASH
static GHashTable *pixbuf_hash=NULL;
typedef struct pixbuf_hash_t {
    GdkPixbuf *pixbuf;
    int size;
    struct pixbuf_hash_t *next;
}pixbuf_hash_t;
#endif

static GtkStyle *style;
static gchar *icon_theme=NULL;
static const gchar *composite_type_id(const gchar *path,int type,int subtype,const gchar *id){
	static gchar *composite_type=NULL;
	gboolean greenball=FALSE,redball=FALSE;
	gchar *p;
	g_free(composite_type);
	composite_type=g_strdup(id);
	if (path){
	    if (IS_DIR(type) && FSTAB_is_mounted(path)==1) greenball=TRUE;
	    else if (FSTAB_is_in_fstab(path) && !IS_XF_FSTAB(type)) redball=TRUE;
	}
	if (IS_NOWRITE(type)||IS_XF_LNK(type)||HAS_HIDDEN(type)||greenball||redball||IS_NETREADONLY(subtype)||IS_XF_NETSHARE(subtype)){
	    p=composite_type;
	    composite_type = g_strconcat(p,"/composite",NULL);
	    g_free(p);	
	}
	if(HAS_HIDDEN(type)){
	    p=composite_type;
	    if(SHOWS_HIDDEN(type)) composite_type = g_strconcat(p,"/shows_hidden",NULL);
	    else composite_type = g_strconcat(p,"/has_hidden",NULL);
	    g_free(p);
	}
	if (IS_NOWRITE(type)||IS_NETREADONLY(subtype)){
	    p=composite_type;
	    composite_type = g_strconcat(p,"/no_write",NULL);
	    g_free(p);	
	}
	if (IS_XF_LNK(type)||IS_XF_NETSHARE(subtype)){
	    p=composite_type;
	    composite_type = g_strconcat(p,"/link",NULL);
	    g_free(p);	
	}
	/*if (path && IS_FSTAB_TYPE(type) && IS_XF_FSTAB(type)){*/
	if (path && greenball){
	    p=composite_type;
	    composite_type = g_strconcat(p,"/fstab_mounted",NULL);
	    g_free(p);	
	} else if (path && redball){
	    p=composite_type;
	    composite_type = g_strconcat(p,"/fstab_unmounted",NULL);
	    g_free(p);	
	}
#ifdef DEBUG
	printf("DBG 1 s:%s composite type=%s type=0x%x (link=%d)\n",path,composite_type,type,IS_XF_LNK(type));
#endif 

	return composite_type;
}

static gboolean set_size_icons(GtkTreeModel * treemodel, GtkTreePath * treepath, GtkTreeIter * iter, gpointer data)
{
    set_icon(treemodel,  iter);
    return FALSE;
}

G_MODULE_EXPORT
gboolean is_image (gchar *file){
    static GSList *pix_formats=NULL;
    GSList *l;
    gboolean result=FALSE;
    
    const gchar *mimetype = MIME_get_type(file,TRUE);
    /* check for image support types */

    if (!pix_formats) pix_formats = gdk_pixbuf_get_formats();
    for(l = pix_formats; l; l = l->next) {
	gchar **pix_mimetypes;
	int i;
	GdkPixbufFormat *fmt = l->data;
	pix_mimetypes = gdk_pixbuf_format_get_mime_types(fmt);
	for(i = 0; pix_mimetypes[i]; i++) {
		if(!g_ascii_strcasecmp(pix_mimetypes[i],mimetype)) {
		    result=TRUE;
		    break;
		}
	}
	g_strfreev(pix_mimetypes);
	if (result) break;
    }
    /* let's keep this one in memory to reduce overhead */
    /*g_slist_free(pix_formats);*/
    return result;
}

static
const gchar *get_thumbnail_path(char *file, int size){
    gchar *xdg_dir;
    gchar *cache_dir,*filename;
    static gchar *thumbnail_path=NULL;
    GString *gs;
    gchar key[11];

    xdg_dir=xfce_resource_save_location (XFCE_RESOURCE_CACHE,"/",TRUE);
    cache_dir = g_build_filename(xdg_dir,"xfce4","xffm","thumbnails",NULL);  
    g_free(xdg_dir);
    mkdir(cache_dir, 0770);
    if (!g_file_test(cache_dir,G_FILE_TEST_IS_DIR)) {
	g_free(cache_dir);
	return NULL;
    }

    /* thumbnails are not subject to thumbnailization: */
    filename=g_path_get_dirname(file);
    if (strcmp(cache_dir,filename)==0){
	DBG("thumbnails cannot be thumbnailed:%s",file);
	g_free(cache_dir);
	g_free(filename);
	return NULL;
    }
    g_free(filename);
    
    
    gs = g_string_new(file);
    sprintf(key, "%10u", g_string_hash(gs));
    g_string_free(gs, TRUE);

    filename=g_strdup_printf("%s-%d.png",key,size);
    thumbnail_path = g_build_filename(cache_dir,filename,NULL);
    g_free(filename);
    g_free(cache_dir);
    
    return (const gchar *)thumbnail_path;
 }

/* some of the code in this function contributed by Thomas Leonard: */
static
void save_thumbnail(const gchar *thumbnail_path, const gchar *original_path, GdkPixbuf *tgt){
	int original_width, original_height;
	char *swidth, *sheight, *ssize, *smtime;
	struct stat st;
	GError *error=NULL;
	
	if (!thumbnail_path || !original_path) return;
	DBG("creating thumbnail for %s (%s)",original_path,thumbnail_path);
	
	original_width = gdk_pixbuf_get_width(tgt);
	original_height = gdk_pixbuf_get_height(tgt);

	if (stat(original_path, &st) != 0) return;

	swidth = g_strdup_printf("%d", original_width);
	sheight = g_strdup_printf("%d", original_height);
	ssize = g_strdup_printf("%ld", (long int)st.st_size);
	smtime = g_strdup_printf("%ld", st.st_mtime);
	
	gdk_pixbuf_save(tgt,
			thumbnail_path,
			"png",
			&error,
			"tEXt::Thumb::Image::Width", swidth,
			"tEXt::Thumb::Image::Height", sheight,
			"tEXt::Thumb::Size", ssize,
			"tEXt::Thumb::MTime", smtime,
			"tEXt::Software", "Xfce4",
			NULL);
	if (error) g_error_free (error);
	g_free(swidth);
	g_free(sheight);
	g_free(ssize);
	g_free(smtime);
}

#define BIG_ENOUGH (96*96*3)
G_MODULE_EXPORT
GdkPixbuf *create_preview (char *file, int size)
{
    GdkPixbuf *src, *tgt;
    int max_w, w, h;
    int area,pw, ph;
    double scaleW, scaleH, scale;
    GError *error = NULL;
    const gchar *thumbnail_path=get_thumbnail_path(file,size);

    TRACE("preview %s",file);
    
    if (thumbnail_path && g_file_test(thumbnail_path,G_FILE_TEST_EXISTS)){
	TRACE("retrieving thumbnail for %s (%s)",file,thumbnail_path);
	tgt = gdk_pixbuf_new_from_file(thumbnail_path, NULL);
	if (tgt) return tgt;
    }

    switch (size)
    {
	default:
	    max_w = w = 96, h = 48;
	    break;
	case SMALL:
	    max_w = w = 48, h = 24;
	    break;
	case MEDIUM:
	    max_w = w = 100, h = 50;
	    break;
	case BIG:
	    max_w = w = 100, h = 50;
	    break;
	case REAL_BIG:
	    max_w = w = 200, h = 100;
	    break;
    }

#if GTK_CHECK_VERSION (2,4,0)
    src = gdk_pixbuf_new_from_file_at_size (file,w,h,&error);
    if(error){
	g_warning(error->message);
	g_error_free(error);
	error=NULL;
	return NULL;
    }
#else /* gtk < 2.4 */
    src = gdk_pixbuf_new_from_file(file, &error);
    if(error || !src){
	if (error) {
	    g_warning("file=%s (%s)",file,error->message);
	    g_error_free(error);
	}
	error=NULL;
	return NULL;
    }
#endif

    ph = gdk_pixbuf_get_height(src);
    pw = gdk_pixbuf_get_width(src);
    if (ph <= 0 || pw <= 0 || scale <=0) {
	g_object_unref(src);
	return NULL;
    }

    area=ph*pw;
    if ((ph > h || pw > w) && pw > 0 && ph > 0){
      scaleH = (double)h / ph;
      scaleW = (double)w / pw;
      if (scaleW*pw > max_w) scaleW=(double)max_w / pw;
      scale = (scaleH < scaleW) ? scaleH : scaleW;
      h = scale * ph;
      w = scale * pw;
    } else {
	if (area > BIG_ENOUGH) save_thumbnail(thumbnail_path,file,src);
	return (src);
    }
    if(w < 10 || h < 10)
    {
	if (src) g_object_unref(G_OBJECT(src));
	return NULL;
    }

    if(!src)
	return NULL;
    tgt = gdk_pixbuf_scale_simple(src, w, h, GDK_INTERP_BILINEAR);
    if (tgt) g_object_unref(G_OBJECT(src));

    if (area > BIG_ENOUGH) save_thumbnail(thumbnail_path,file,tgt);
    return tgt;
}
G_MODULE_EXPORT
GdkPixbuf *create_full_pixbuf (char *file)
{
    GdkPixbuf *src;
    GError *error = NULL;

    src = gdk_pixbuf_new_from_file(file, &error);
    if(!src)
    {
	printf("DBG: pixbuf error, file=%s\n",file); 
	return NULL;
    }
    if(error)
	g_error_free(error);

    return src;
}


/*void */
G_MODULE_EXPORT
gboolean
increase_size (GtkTreeModel * treemodel, GtkTreePath * path, GtkTreeIter * iter, gpointer data)
{
    tree_entry_t *en;
    gtk_tree_model_get(treemodel, iter, ENTRY_COLUMN, &en, -1);
    if(!en)
	return(FALSE);
    set_icon(treemodel,  iter);
    return FALSE;
}

/*void */
G_MODULE_EXPORT
gboolean
decrease_size (GtkTreeModel * treemodel, GtkTreePath * path, GtkTreeIter * iter, gpointer data)
{
    tree_entry_t *en;
    gtk_tree_model_get(treemodel, iter, ENTRY_COLUMN, &en, -1);
    if (!en) return(FALSE);   
    set_icon(treemodel, iter);
    return FALSE;
}


G_MODULE_EXPORT
const gchar *resolve_folder_icon (tree_entry_t * en)
{
    int type = en->type;
    const gchar *basic_type;
 
    if (IS_NOACCESS(en->type)){
	    return "xfce/no-access";
    }
    
    if (en->path && strcmp(g_get_home_dir(),en->path)==0){
	/*printf("%s==%s\n",g_get_home_dir(),en->path);*/
	basic_type="xfce/b-home";
    }
   
    else if(IS_EXPANDED(type)) basic_type="xfce/open_folder";
    else basic_type="xfce/closed_folder";
    
    return (basic_type);

}
       

G_MODULE_EXPORT
void 
insert_pixbuf_tag (		const gchar *id, 
				GdkPixbuf **tgt,
			        int size,	
				int scale, 
				const gchar *where){
    int x_tgt=0,y_tgt=0; /* default: NW */
    GdkPixbuf *tag,*otag;
    int width, height;
   
    if (!(*tgt)) return;
    width=gdk_pixbuf_get_width(*tgt);
    height=gdk_pixbuf_get_height(*tgt);

    /* only use stock gtk here: */
    if (strncmp(id,"gtk-",strlen("gtk-"))==0){
	otag = gtk_widget_render_icon(tree_details->window, id ,size, NULL);
    }
    else  otag = icon_tell(size,id);
    if (!otag) return;
    tag = gdk_pixbuf_scale_simple(otag, width/scale,height/scale, GDK_INTERP_BILINEAR);	
    g_object_unref(G_OBJECT(otag));
    if (!tag) return;
    
    if (strcmp(where,"NE")==0) 
	x_tgt=(scale-1)*width/scale, y_tgt=0;
    else if (strcmp(where,"SW")==0)
       	x_tgt=0, y_tgt=(scale-1)*width/scale;
    else if (strcmp(where,"SE")==0) 
	x_tgt=(scale-1)*width/scale, y_tgt=(scale-1)*width/scale;
    else if (strcmp(where,"E")==0) 
	x_tgt=(scale-1)*width/scale, y_tgt=(scale-1)*width/scale/2;

    
#if 0 
    {
	/* with this we get the tag to have a transparent part
	 * over the icon, but where the icon *is* transparent,
	 * and the tag as well, the resulting pixbuf will be *not*
	 * transparent here! (It will have the color of the filled
	 * in rectangle.*/
	GdkBitmap *mask_return;
	GdkPixmap *sandbox,*pixmap;
	static GdkGC *penGC=NULL;
	GdkColormap *cmap=gdk_colormap_get_system();

	
	GdkPixbuf *pb=gdk_pixbuf_new_subpixbuf (*tgt,
		x_tgt,y_tgt,
		width/scale,height/scale);
	
	gdk_pixbuf_render_pixmap_and_mask_for_colormap (pb, 
		cmap, 
		&pixmap, NULL, 128);
	
	sandbox = gdk_pixmap_new((GdkDrawable *)pixmap, width/scale,height/scale,-1);
	g_object_unref(pixmap);

	if (!penGC) {
	    GdkColor pen_color= {0,65535,65535,65535};
	    gdk_colormap_alloc_color(cmap,&pen_color,TRUE,TRUE);
	    penGC = gdk_gc_new ((GdkDrawable *)sandbox);
	    gdk_gc_set_colormap (penGC,cmap);
	    gdk_gc_set_foreground (penGC, &pen_color);
	}
	gdk_draw_rectangle (sandbox, penGC, TRUE, 0,0,width/scale,height/scale);
	gdk_draw_pixbuf ((GdkDrawable *)sandbox,
                         penGC,
                         *tgt,x_tgt,y_tgt,
			 0,0, /* dest */
			 width/scale,height/scale,
                         GDK_RGB_DITHER_NONE,0,0);
	
	
		
	gdk_pixbuf_render_pixmap_and_mask_for_colormap (tag, 
		cmap, 
		NULL, &mask_return, 128);
	
	gdk_gc_set_clip_mask   (penGC,mask_return);

	gdk_draw_pixbuf ((GdkDrawable *)sandbox,
                         penGC,
                         tag,
                         0,0, /* src */
			 0,0, /* dest */
			 width/scale,height/scale,
                         GDK_RGB_DITHER_NONE,0,0);
	gdk_gc_set_clip_mask   (penGC,NULL);
	
	gdk_pixbuf_get_from_drawable (tag,
		(GdkDrawable *)sandbox,
                cmap,
                0,0,0,0,
		width/scale,height/scale);
	g_object_unref(G_OBJECT(sandbox));
	g_object_unref(G_OBJECT(pb));
	g_object_unref(G_OBJECT(mask_return));

    }
#endif
    gdk_pixbuf_copy_area (tag,0,0,width/scale,height/scale,*tgt,x_tgt,y_tgt);
    g_object_unref(G_OBJECT(tag));
}

#ifdef FALLBACK
GHashTable *fallback_hash(void){
    static GHashTable *hardfallback_hash=NULL;
    if (hardfallback_hash) return hardfallback_hash;
    else {
 	gchar **p;
	const gchar *hard_fallback[]={
	      "application", "d-application.png",
	      "audio", "d-audio.png",
	      "image", "d-image.png",
	      "inode", "f-memory.png",
	      "message", "d-message.png",
	      "model", "d-model.png",
	      "multipart", "d-multipart.png",
	      "text", "d-file.png",
	      "video", "d-video.png",
	      "xfce/default" , "d-xfce.png",
	      "xfce/b-book" , "b-book-closed.png", 
	      "xfce/find_result" , "b-find.png", 
	      "xfce/b-find" , "b-find.png", 
	      "xfce/b-frequent" , "b-frequent.png", 
	      "xfce/b-fstab" , "b-fstab.png", 
	      "xfce/b-home" , "b-home.png", 
	      "xfce/b-network" , "b-network.png", 
	      "xfce/b-recent" , "b-recent.png", 
	      "xfce/b-trash" , "b-trash.png", 
	      "xfce/b-xffm" , "xffm_icon.xpm", 
	      "xfce/broken" , "d-symlink-broken.png", 
	      "xfce/c-gid" , "c-gid.png", 
	      "xfce/c-mode" , "c-mode.png", 
	      "xfce/c-name" , "c-name.png", 
	      "xfce/c-size" , "c-size.png", 
	      "xfce/c-time" , "c-time.png", 
	      "xfce/c-time32" , "c-time.png", 
	      "xfce/c-uid" , "c-uid.png", 
	      "xfce/cdrom" , "f-cdrom.png", 
	      "xfce/closed_folder" , "dir-closed.png", 
	      "xfce/disk" , "f-disk.png", 
	      "xfce/downarrow" , "downarrow.png", 
	      "xfce/dvd" , "f-dvd.png", 
	      "xfce/error" , "gtk-dialog-error", 
	      "xfce/executable" , "d-application.png", 
	      "xfce/floppy" , "f-floppy.png", 
	      "xfce/go-back" , "go-back.png", 
	      "xfce/go-down" , "downarrow.png", 
	      "xfce/go-forward" , "go-forward.png", 
	      "xfce/go-home" , "go-home.png", 
	      "xfce/go-jump" , "go-jump.png", 
	      "xfce/go-start" , "go-start.png", 
	      "xfce/go-up" , "go-up.png", 
	      "xfce/info" , "a-info.png", 
	      "xfce/ipc" , "d-symlink.png", 
	      "xfce/link" , "go-forward.png", 
	      "xfce/netws" , "p-terminal.xpm", 
	      "xfce/nfs" , "f-netdisk.png", 
	      "xfce/no-access" , "d-no-access.png", 
	      "xfce/open_folder" , "dir-open.png", 
#ifdef USE_TOOLBAR_PANEL
	      "xfce/p-edit" , "p-edit.png", 
	      "xfce/p-file1" , "p-file1.png", 
	      "xfce/p-file2" , "p-file2.png", 
	      "xfce/p-games" , "p-games.png", 
	      "xfce/p-man" , "d-help.png", 
	      "xfce/p-multimedia" , "p-multimedia.png", 
	      "xfce/p-network" , "p-network.png", 
	      "xfce/p-newmail" , "p-newmail.png", 
	      "xfce/p-nomail" , "p-nomail.png", 
	      "xfce/p-oldmail" , "p-oldmail.png", 
	      "xfce/p-paint" , "p-paint.png", 
	      "xfce/p-print" , "p-print.png", 
	      "xfce/p-schedule" , "p-schedule.png", 
	      "xfce/p-sound" , "p-sound.png", 
	      "xfce/p-terminal" , "p-terminal.png", 
	      "xfce/p-unknown" , "p-unknown.png", 
#endif
	      "xfce/process" , "f-memory.png", 
	      "xfce/question" , "gtk-dialog-question", 
	      "xfce/share" , "d-symlink.png", 
	      "xfce/share_open" , "d-symlink.png", 
	      "xfce/t-exec" , "t-exec.png", 
	      "xfce/t-find" , "t-find.png", 
	      "xfce/t-new-directory" , "t-new-directory.png", 
	      "xfce/t-new-file" , "t-new-file.png", 
	      "xfce/t-preferences" , "t-preferences.png", 
	      "xfce/t-quit" , "t-quit.png", 
	      "xfce/t-reload" , "t-reload.png", 
	      "xfce/t-terminal" , "t-terminal.png", 
	      "xfce/t-xfce-setting-show" , "t-xfce-setting-show.png", 
	      "xfce/t-xfdiff" , "t-xfdiff.png", 
	      "xfce/t-xfmime-edit" , "t-xfmime-edit.png", 
	      "xfce/t-zoom-in" , "t-zoom-in.png", 
	      "xfce/t-zoom-out" , "t-zoom-out.png", 
	      "xfce/t-zoom-view" , "t-zoom-view.png", 
	      "xfce/unknown" , "d-xfce.png", 
	      "xfce/warning" , "gtk-dialog-warning", 
	      "xfce/waste_basket_empty" , "d-waste_basket_empty.png", 
	      "xfce/waste_basket_full" , "d-waste_basket_full.png", 
	      "xfce/xfdiff" , "xfdiff.png", 
	      "xfce/xffm" , "xffm_icon.xpm", 
	      "t-pastelink.png" , "t-pastelink.png", /* FIXME: add type */
	      "downarrow.png" , "downarrow.png", /* FIXME: add type */
	      NULL,NULL
	};
	hardfallback_hash = g_hash_table_new(g_str_hash, g_str_equal);
	
	for (p=(gchar **)hard_fallback; *p; p+=2){
	    g_hash_table_insert(hardfallback_hash, *p,*(p+1));	
	}
	return hardfallback_hash;
    }   
}
    
#endif
static GdkPixbuf *icon_tell_cut(int size, const gchar *id, gboolean cut)
{
    GtkIconSet *iconset;
    int gtksize;
    GdkPixbuf *pixbuf;
    
    if (!id || !strlen(id)) return NULL;
#ifdef PIXBUF_HASH
    if (pixbuf_hash && !cut){
	pixbuf_hash_t *pixbuf_hash_p = g_hash_table_lookup(pixbuf_hash,id);
	if (pixbuf_hash_p){
	    TRACE("pixbuf found for %s",id);
	    for (;pixbuf_hash_p; pixbuf_hash_p= pixbuf_hash_p->next){
		if (pixbuf_hash_p->size==size) break;
	    }
	    if (pixbuf_hash_p){
		pixbuf=pixbuf_hash_p->pixbuf;
		g_object_ref(pixbuf);
		return pixbuf;
	    } else DBG("pixbuf size %d not found",size);
	}
	
    }
#endif

    if (strncmp(id,"gtk-",strlen("gtk-"))==0){
	pixbuf=load_stock_icon(id ,size);
	goto pixbuf_done;
	/*return gtk_widget_render_icon(tree_details->window, id ,size, NULL);*/
    } 	
    

    if (strstr(id,"/composite")) {
	gchar *basic_type=g_strdup(id);
	
	GdkPixbuf *folder,*ofolder;

	*(strstr(basic_type,"/composite"))=0;
	
	ofolder = icon_tell(size,basic_type);
	g_free(basic_type);
	if (ofolder) {
	  folder=gdk_pixbuf_copy(ofolder);
	  g_object_unref(G_OBJECT(ofolder));
	} else folder=NULL;
	
	
	if (strstr(id,"/shows_hidden")){
	    insert_pixbuf_tag("gtk-add",&folder,size,4,"E");
	}
	else if (strstr(id,"/has_hidden")) {
	    insert_pixbuf_tag("gtk-remove",&folder,size,4,"E");
	}
	
	if (strstr(id,"/no_write")){
	    insert_pixbuf_tag("gtk-cancel",&folder,size,4,"SE");
	}
	if (strstr(id,"/link")){
	    insert_pixbuf_tag("xfce/link",&folder,size,3,"SW");
	}
	if (strstr(id,"/fstab_mounted")){
	    insert_pixbuf_tag("gtk-yes",&folder,size,5,"NW");
	}
	else if (strstr(id,"/fstab_unmounted")){
	    insert_pixbuf_tag("gtk-no",&folder,size,5,"NW");
	}
	if (cut && folder){   
	    iconset=gtk_icon_set_new_from_pixbuf(folder);
	    g_object_unref(G_OBJECT(folder));
	} 
	else {
	    pixbuf=folder;
	    goto pixbuf_done;
	}
    }
    
    iconset = MIME_ICON_get_iconset(id,tree_details->window);
#ifdef FALLBACK
    if (!iconset){
	GdkPixbuf *pb;
	gchar *b_id=g_strdup(id);
	if (!strstr(b_id,"xfce/") && strchr(b_id,'/')) *strchr(b_id,'/')=0;
	pb = MIME_ICON_create_pixbuf(g_hash_table_lookup(fallback_hash(),b_id));
	g_free(b_id);
	if (pb) {
	    iconset=gtk_icon_set_new_from_pixbuf(pb);
	    g_object_unref(G_OBJECT(pb));
	}
    }
#endif
    
    if(!iconset) 
    {
        iconset = MIME_ICON_get_iconset("d-file.png",tree_details->window);
/*#ifdef DEBUG
        printf("icon for %s not found in $theme/mime.xml\n",id); 
        printf("DBG: looking for iconset %s...%s\n",id,(iconset)?"found":"not found"); 
#endif*/
    }
    

    if(!style) style = gtk_style_new();
    switch (size)
    {
	case REAL_BIG:
	    gtksize = GTK_ICON_SIZE_DIALOG;
	    break;
	case BIG:
	    gtksize = GTK_ICON_SIZE_DND;
	    break;
	case MEDIUM:
	default:
	    gtksize = GTK_ICON_SIZE_LARGE_TOOLBAR;
	    break;
	case SMALL:
	    gtksize = GTK_ICON_SIZE_BUTTON;
	    break;
	    /*gtksize=GTK_ICON_SIZE_SMALL_TOOLBAR; break;*/
    }
    if(!iconset) return NULL;
    pixbuf = gtk_icon_set_render_icon(iconset, style, GTK_DIR_RIGHT, (cut) ? GTK_STATE_INSENSITIVE : GTK_STATE_NORMAL, gtksize, NULL, NULL);
pixbuf_done:
#ifdef PIXBUF_HASH
    if (!cut) {
	pixbuf_hash_t *pixbuf_hash_p,*p;
	g_object_ref(pixbuf);
	if (!pixbuf_hash){
	    pixbuf_hash = g_hash_table_new(g_str_hash, g_str_equal);
	}
	pixbuf_hash_p = g_hash_table_lookup(pixbuf_hash,id);
	if (!pixbuf_hash_p) {
	    DBG("new pixbuf for %s",id);
	    pixbuf_hash_p=(pixbuf_hash_t *)malloc(sizeof(pixbuf_hash_t));
	    pixbuf_hash_p->size=size;
	    pixbuf_hash_p->pixbuf=pixbuf;
	    pixbuf_hash_p->next=NULL;
	    g_object_ref(pixbuf);	    
	    g_hash_table_insert(pixbuf_hash, g_strdup(id), (gpointer) pixbuf_hash_p);
	    return pixbuf;
	}
	for (p=pixbuf_hash_p; p && p->next; p=p->next);
	
	/*if (p->size==size) g_warning("this does not happen");*/
	
	pixbuf_hash_p = p->next = (pixbuf_hash_t *)malloc(sizeof(pixbuf_hash_t));
	pixbuf_hash_p->size=size;
	pixbuf_hash_p->pixbuf=pixbuf;
	pixbuf_hash_p->next=NULL;
	g_object_ref(pixbuf);	    
	return pixbuf;    
    }
#endif
    return pixbuf;
}

G_MODULE_EXPORT
GdkPixbuf *icon_tell(int size, const gchar *id)
{
    return icon_tell_cut(size, id, FALSE);
}




G_MODULE_EXPORT
const gchar *resolve_icon_id (tree_entry_t * en)
{
    if(!en) assert_not_reached();
    if(IS_ROOT_TYPE(en->type))
    {
	if(IS_NETWORK_TYPE(en->type)) return ("xfce/b-network");
	else if(IS_FIND_TYPE(en->type)) return ("xfce/b-find");
	else if(IS_FSTAB_TYPE(en->type)) return ("xfce/b-fstab");
	else if(IS_BOOKMARK_TYPE(en->type)) 
	{
	    if(IS_EXPANDED(en->type)) return ("xfce/b-book");
	    return ("xfce/b-book");
	}
	else if(IS_RECENT_TYPE(en->type)) return ("xfce/b-recent");
	else if(IS_FREQUENT_TYPE(en->type)) return ("xfce/b-frequent");
	else if(IS_TRASH_TYPE(en->type))
	{ /*FIXME: monitor trash icon
	    if(IS_EXPANDED(en->type)) return (""); 
	    return ("xfce/trash_closed");*/
		return ("xfce/b-trash");
	}
	else if(!IS_LOCAL_TYPE(en->type)) return (NULL);
    }
    
    if(IS_NETWORK_TYPE(en->type)||IS_BOOKMARK_TYPE(en->type)) {
      if(IS_XF_NETWG(en->subtype)) return ("xfce/b-network");
      if(IS_XF_NETWS(en->subtype)) return ("xfce/netws");
      if(IS_XF_NETSHARE(en->subtype)) {
	      if(IS_EXPANDED(en->type)) return ("xfce/share_open");
	      else return ("xfce/share");
      }
      if(IS_XF_NETIPC(en->subtype)) return ("xfce/ipc");
      if(IS_XF_NETPRINT(en->subtype)) return ("xfce/p-print");
      if(IS_NETDIR(en->subtype)) { 
	 if (IS_EXPANDED(en->type)){
	      return "xfce/open_folder";
   	 } else {
	      return "xfce/closed_folder";
	 }
      }
      if (IS_NETWORK_TYPE(en->type)) {
	    const gchar *g=MIME_get_type(en->path,FALSE);
	    if (strcmp(g,"undetermined type")==0) return "application/default";
	    else return g;
      }
    }

    if(IS_FSTAB_TYPE(en->type) && IS_XF_FSTAB(en->type)){
	const gchar *t="xfce/disk";
	if (IS_NFS_TYPE(en->subtype)) t = "xfce/nfs";
	else if (IS_PROC_TYPE(en->subtype)) t = "xfce/process";
	else if (IS_CDFS_TYPE(en->subtype)) t = "xfce/cdrom";
	else if (strstr(en->path,"floppy")) t = "xfce/floppy";
	else if (strstr(en->path,"cdrom")) t = "xfce/cdrom";
	else if (strstr(en->path,"cdrw")) t = "xfce/cdrom";
	else if (strstr(en->path,"dvd")) t = "xfce/dvd";
	return t;
    }
    

    /* broken links : */
    if(IS_BROKEN_LNK(en->type)) return "xfce/broken";

    /* find results */
    if(IS_XF_FND(en->type)) return "xfce/find_result";

    if(IS_DIR(en->type))
    {
	/*printf("folder %s link=%d\n",en->path,IS_XF_LNK(en->type)); */
	/* strstr makes all folders inside wastebasket use this icon */
	if(strstr(en->path, "/..Wastebasket"))
	{
	    if(en->count)
		return "xfce/waste_basket_full";
	    return "xfce/waste_basket_empty";
	}

	return resolve_folder_icon(en);
    }

    /* non-broken links: */
    /*printf("file %s link=%d\n",en->path,IS_XF_LNK(en->type)); */
    /*if(IS_XF_LNK(en->type)) return "inode/symlink";*/

    /* d_types: */
    else if(IS_XF_CHR(en->type)) return ("inode/chardevice");
    else if(IS_XF_BLK(en->type)) return ("inode/blockdevice");
    else if(IS_XF_FIFO(en->type)) return ("inode/fifo");
    else if(IS_XF_SOCK(en->type)) return ("inode/socket");

    if (IS_NOACCESS(en->type)) return ("xfce/no-access");


    /*if all else fails: */
    return NULL;
}

 /* 
  *options for icons... use GTK_STATE_INSENSITIVE for files that
  have been cut to the pasteboard, until the pasteboard has been
  found invalid... 
  *  GTK_STATE_NORMAL,
  GTK_STATE_ACTIVE,
  GTK_STATE_PRELIGHT,
  GTK_STATE_SELECTED,
  GTK_STATE_INSENSITIVE*/
/* use parameter for iconview size */
static GdkPixbuf *resolve_icon (tree_entry_t * en, tree_entry_t * p_en, int s)
{
    int gtksize, size;
    char *loc = NULL;
    const gchar *id=NULL;
    int max_preview_size=256;
    gboolean cut = FALSE;
    /*GdkPixbuf *Icon;
    GtkIconSet *iconset;
    char *l, *stock_id = NULL;*/
       const gchar *mimetype;


    if(!style) style = gtk_style_new();

    if (s<0) {
	gtksize = GTK_ICON_SIZE_DIALOG;
	size = REAL_BIG;
    }
    else switch (tree_details->icon_size){
      case 3:
	gtksize = GTK_ICON_SIZE_DIALOG;
	size = REAL_BIG;
	break;
      case 2:
	gtksize = GTK_ICON_SIZE_DND;
	size = BIG;
	break;
      case 1:
	gtksize = GTK_ICON_SIZE_LARGE_TOOLBAR;
	size = MEDIUM;
	break;
      case 0:
      default:
	gtksize = GTK_ICON_SIZE_BUTTON;
	size = SMALL;
	break;
    }
    if(valid_pasteboard() == 2)
    {
	cut = in_pasteboard(en);
	if(cut){  SET_CUT(en->type); }
    }
    else UNSET_CUT(en->type);

    if (!id) id = resolve_icon_id(en);
    if(id){
#ifdef DEBUG
	printf("ICON RESOLVED: %s --> %s\n",en->path,id);
#endif
     	return icon_tell_cut(size, composite_type_id(en->path,en->type,en->subtype,id), cut);
    }

    /* image previews before executable */
    if(en->path) loc = strrchr(en->path, '/');
    if (getenv("XFFM_MAX_PREVIEW_SIZE") && strlen( getenv("XFFM_MAX_PREVIEW_SIZE"))){
	if (is_number(getenv("XFFM_MAX_PREVIEW_SIZE"))) {   
	   max_preview_size=(atoi(getenv("XFFM_MAX_PREVIEW_SIZE")));
	}	   
    }
    
    if ( !cut && loc && IS_FILE(en->type) && !IS_NETWORK_TYPE(en->type) &&
	 !IS_TRASH_TYPE(en->type) && !strstr(en->path, "/..Wastebasket") &&
	 en->st->st_size <= max_preview_size*1024 && 
	  ((p_en && SHOWS_IMAGES(p_en->type)) 
	 ) && (IS_IMAGE(loc))
	)
    {
 	GdkPixbuf *tgt;
        process_pending_gtk(); 
	if (s>=0) tgt = create_preview(en->path, size);
	else tgt = create_preview(en->path, s);
	if(tgt) return tgt;
    }
    

    /* mime type (before executables) : */
#if 0
       gchar *p=strrchr(en->path,G_DIR_SEPARATOR);
       if (p && strchr(p,'-')){
	       p=g_strdup(en->path);
	       *strrchr(p,'-')=0;
	       mimetype = MIME_get_type(p,FALSE); /* don't try magic here, too expensive */
	       g_free(p);
       } 
       else 
#endif
       mimetype = MIME_get_type(en->path,FALSE);
       if ((strcmp("application/octet-stream",mimetype)==0 ||strcmp("text/plain",mimetype)==0) && IS_EXE(en->type)){
	   return icon_tell_cut(size, composite_type_id(en->path,en->type,en->subtype,"xfce/executable"), cut); 
       } else if (strcmp("undetermined type",mimetype)==0) {
    /* icon determination has failed, return the default icon */
	   return icon_tell_cut(size, composite_type_id(NULL,en->type,en->subtype,"xfce/default"), cut);
       }
       
 
    return  icon_tell_cut(size, composite_type_id(en->path,en->type,en->subtype,mimetype), cut);
    
}

G_MODULE_EXPORT
GdkPixbuf *resolve_icon_size (tree_entry_t * en,int size)
{
    int gtksize;
    static GtkStyle *style = NULL;
    char *loc;
    const gchar *id;


    if(!en || !en->path) return NULL;

    if(!style) style = gtk_style_new();
    gtksize = GTK_ICON_SIZE_LARGE_TOOLBAR;
    /*size = SMALL;*/

    id = resolve_icon_id(en);
    if(!id && IS_EXE(en->type)) id = "xfce/executable";
    if(id) {
	return icon_tell(size, id);
    }

    /* mime type: */
    loc = strrchr(en->path, '/');
    /*printf("path=%s, loc=%s\n",en->path,(loc)?loc:"null");*/
    if(loc)
    {
	GdkPixbuf *Icon;
	GtkIconSet *iconset;
	const gchar *stock_id = NULL;
	stock_id = MIME_get_type(en->path,FALSE);
	if(stock_id)
	{
	    iconset = MIME_ICON_get_iconset(stock_id,tree_details->window);
	    if(!iconset) return NULL;
	    Icon = gtk_icon_set_render_icon(iconset, style, GTK_DIR_RIGHT, GTK_STATE_NORMAL, gtksize, NULL, NULL);
	    return Icon;
	}
	/*else printf("key=%s not found loco\n",loc); */
    }
    return icon_tell(size, "xfce/default");

}


G_MODULE_EXPORT
void set_icon (GtkTreeModel *treemodel, GtkTreeIter * iterator)
{
    GdkPixbuf *Icon;
    tree_entry_t *en;
    tree_entry_t *p_en = NULL;
    GtkTreeIter parent;
    gtk_tree_model_get(treemodel, iterator, ENTRY_COLUMN, &en, -1);
    if (!en) return;   
    if(gtk_tree_model_iter_parent(treemodel, &parent, iterator))
	gtk_tree_model_get(treemodel, &parent, ENTRY_COLUMN, &p_en, -1);

    if(!en) return;
    if(IS_DUMMY_TYPE(en->type))
	return;
    /*if (en->path) printf("DBG: icon setting for %s\n",en->path); */
    set_font(treemodel,iterator); 
    Icon = resolve_icon(en, p_en,0);
    update_iconcolumn_for_row(treemodel, iterator,Icon);	
    if (Icon) g_object_unref (G_OBJECT (Icon));    
}

G_MODULE_EXPORT
GdkPixbuf *get_icon (tree_entry_t *en,tree_entry_t *p_en)
{
    GdkPixbuf *Icon;
    if (!en) return NULL;  
    TRACE("shows images=%d",SHOWS_IMAGES(p_en->type)); 
    Icon = resolve_icon(en, p_en,-1);
    return Icon;
}

G_MODULE_EXPORT
void update_icon (GtkTreeModel *treemodel, GtkTreeIter * iterator)
{
    GtkTreeIter parent;
    GdkPixbuf *Icon;
    tree_entry_t *en;
    tree_entry_t *p_en = NULL;
    gtk_tree_model_get(treemodel, iterator, ENTRY_COLUMN, &en, -1);
    if (!en) return;   
    if(gtk_tree_model_iter_parent(treemodel, &parent, iterator))
	gtk_tree_model_get(treemodel, &parent, ENTRY_COLUMN, &p_en, -1);    

    if(!tree_details || !tree_details->window)
	return;
    if(!en)
	return;

    if(IS_DUMMY_TYPE(en->type) || IS_XF_BLK(en->type) || IS_XF_CHR(en->type))
	return;
    if(IS_DIR(en->type))
    {
	struct stat st;
	stat(en->path, &st);
	if(st.st_mtime != en->st->st_mtime || st.st_ctime != en->st->st_ctime)
	{
	    memcpy(en->st, &st, sizeof(struct stat));
	    en->count = count_files(en->path);
	    Icon = resolve_icon(en, p_en,0);
	    D(printf(".");fflush(NULL);)
	    update_iconcolumn_for_row(treemodel, iterator,Icon);	
	    update_text_cell_for_row(SIZE_COLUMN,treemodel, iterator,sizetag((off_t)-1, en->count)); 

    	    if (Icon) g_object_unref (G_OBJECT (Icon));    
	}

    }
    /*in_pasteboard(en)==2 implies cut, ==1 implies copy */
    if(IS_CUT(en->type) || in_pasteboard(en)==2)
    {
	/*printf("dbg update icon %s\n",en->path); */
	Icon = resolve_icon(en, p_en,0);
        update_iconcolumn_for_row(treemodel, iterator,Icon);	
	if (Icon) g_object_unref (G_OBJECT (Icon));    
	if(in_pasteboard(en)==2){
	    SET_CUT(en->type);
	} else {
	    UNSET_CUT(en->type);
	}

    }
    return;
}


G_MODULE_EXPORT
GtkWidget *icon_image (char *id)
{

    if(id)
    {
	GdkPixbuf *Icon;
	GtkIconSet *iconset;
	if(!style) style = gtk_style_new();
	if (strncmp(id,"gtk-",strlen("gtk-"))==0) {
	  Icon = load_stock_icon(id, GTK_ICON_SIZE_SMALL_TOOLBAR);
	  /*Icon = gtk_widget_render_icon(tree_details->window, id, GTK_ICON_SIZE_SMALL_TOOLBAR, NULL);*/
	} else {
	  iconset = MIME_ICON_get_iconset(id,tree_details->window);
	  if(!iconset) return NULL;
	  Icon = gtk_icon_set_render_icon(iconset, style, GTK_DIR_RIGHT, GTK_STATE_NORMAL, GTK_ICON_SIZE_SMALL_TOOLBAR, NULL, NULL);
	}
	if(Icon){
	   GtkWidget *image=gtk_image_new_from_pixbuf(Icon);
	   g_object_unref(G_OBJECT(Icon));
	   return image;
	}
    }
    return NULL;
}


G_MODULE_EXPORT
void recreate_icons (GtkTreeView *treeview)
{
    int theme_status;
    GtkTreeModel *treemodel = gtk_tree_view_get_model(treeview);
    /* create new icons */
    if (icon_theme){
	    g_free(icon_theme);
    }
    icon_theme=g_strdup(tree_details->theme);
    theme_status=MIME_ICON_load_theme();
    switch (theme_status) {
	case 0:
	    g_warning("Your mime-icons installation is somehow broken");
	    break;
	case -1:
#ifdef DEBUG
	    printf("DBG: theme has not changed\n");
#endif
	    break;
	case 1:
	    /* reset icons */
	    gtk_widget_freeze_child_notify((GtkWidget *) (treeview));
	    gtk_tree_model_foreach(treemodel, set_size_icons, (gpointer) (tree_details));
	    gtk_widget_thaw_child_notify((GtkWidget *) (treeview));
	    break;
   }
}

G_MODULE_EXPORT
GdkPixbuf *load_stock_icon (const gchar *id, int size){
    /*gchar *s,*p=NULL;*/
    GdkPixbuf *pixbuf=NULL;
    if (strncmp(id,"gtk-",strlen("gtk-"))!=0){
	g_warning("load_stock_icon called on non stock item %s",id);
	return NULL;
    }
#if 0
    s = g_strconcat("stock_",id+strlen("gtk-"),".png",NULL);
    p=MIME_ICON_find_pixmap_file ((const gchar *)s);
    if (p) {
      pixbuf = icon_tell(size,s);
      g_free(p);
    }
    D(else g_warning("cannot locate %s",s);)
    g_free(s);
    if (pixbuf) return pixbuf;
#endif
    pixbuf=gtk_widget_render_icon(tree_details->window,id,size, NULL);
    return pixbuf;
}

G_MODULE_EXPORT
void on_full_preview1_activate (GtkMenuItem * menuitem, gpointer user_data)
{

}
