/* NVTV Brooktree encoder data -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id: data_bt.c,v 1.30 2003/06/11 16:06:21 dthierbach Exp $
 *
 * Contents:
 *
 * Data tables and routines for the Brooktree (and Conexant) chips.
 *
 */

#include <stddef.h>
#include <stdlib.h>
#include <math.h>

#include "data_bt.h"
#include "data_nv.h"

/* -------- BT -------- NTSC -------- */

/* -------- 640x480 -------- */

TVBtRegs bt_ntsc_small_a = { /* over comp 13.79 13.58 % (std 0) */
  hsynoffset   : 0, 
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 1792,
  h_active     : 640,
  hsync_width  : 132,
  hburst_begin : 150,
  hburst_end   : 96,
  h_blanko     : 381,
  v_blanko     : 34,
  v_activeo    : 212,
  h_fract      : 0,
  h_clki       : 784,
  h_blanki     : 126,
  v_linesi     : 600,
  v_blanki     : 75,
  v_activei    : 480,
  v_scale      : 5266,
  pll_fract    : 34830,
  pll_int      : 12,
  sync_amp     : 229,
  bst_amp      : 118,
  mcr          : 121,
  mcb          : 68,
  my           : 133,
  msc          : 545259520,
  flags1       : BT_FLAG1_NTSC,
};

TVBtRegs bt_ntsc_normal_a = { /* over comp 10.59 08.23 % (status voc=08.64) */
  hsynoffset   : 0,	/* NV: 12 */
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 1728,
  h_active     : 640,
  hsync_width  : 128,
  hburst_begin : 144,
  hburst_end   : 88,
  h_blanko     : 345,
  v_blanko     : 28,
  v_activeo    : 224,
  h_fract      : 0,
  h_clki       : 800,
  h_blanki     : 140,
  v_linesi     : 567,
  v_blanki     : 58,
  v_activei    : 480,
  v_scale      : 4751,
  pll_fract    : 5499,
  pll_int      : 12,
  sync_amp     : 229,
  bst_amp      : 118,
  mcr          : 121,
  mcb          : 68,
  my           : 133,
  msc          : 565454317,
  flags1       : BT_FLAG1_NTSC,
};

TVBtRegs bt_ntsc_huge_a = { /* over comp 02.46 01.23 % */
  hsynoffset   : 0,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 1584,
  h_active     : 640,
  hsync_width  : 118,
  hburst_begin : 132,
  hburst_end   : 70,
  h_blanko     : 263,
  v_blanko     : 19,
  v_activeo    : 242,
  h_fract      : 0,
  h_clki       : 792,
  h_blanki     : 133,
  v_linesi     : 525,
  v_blanki     : 36,
  v_activei    : 480,
  v_scale      : 4096,
  pll_fract    : 5041,
  pll_int      : 11,
  sync_amp     : 229,
  bst_amp      : 118,
  mcr          : 122,
  mcb          : 68,
  my           : 133,
  msc          : 616859255,
  flags1       : BT_FLAG1_NTSC,
};

/* -------- 800x600 -------- */

TVBtRegs bt_ntsc_small_b = { /* over comp 21.62 11.52 % (std 2) */
  hsynoffset   : 0,	/* NV: -8 */
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 2464,
  h_active     : 800,
  hsync_width  : 182,
  hburst_begin : 206,
  hburst_end   : 180,
  h_blanko     : 597,
  v_blanko     : 32,
  v_activeo    : 216,
  h_fract      : 0,
  h_clki       : 880,
  h_blanki     : 66,
  v_linesi     : 735,
  v_blanki     : 86,
  v_activei    : 600,
  v_scale      : 7373,
  pll_fract    : 15124,
  pll_int      : 17,
  sync_amp     : 229,
  bst_amp      : 116,
  mcr          : 119,
  mcb          : 67,
  my           : 133,
  msc          : 396552378,
  flags1       : BT_FLAG1_NTSC,
};

TVBtRegs bt_ntsc_normal_b = { /* over comp 11.90 05.35 % */
  hsynoffset   : 0,	/* NV: -62 */
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 2192,
  h_active     : 800,
  hsync_width  : 162,
  hburst_begin : 182,
  hburst_end   : 146,
  h_blanko     : 445, /* tool: 480 */
  v_blanko     : 24,
  v_activeo    : 232,
  h_fract      : 0,
  h_clki       : 840,
  h_blanki     : 26,
  v_linesi     : 685,
  v_blanki     : 60,
  v_activei    : 600,
  v_scale      : 6593,
  pll_fract    : 21540,
  pll_int      : 15,
  sync_amp     : 229,
  bst_amp      : 117,
  mcr          : 120,
  mcb          : 67,
  my           : 133,
  msc          : 445759608,
  flags1       : BT_FLAG1_NTSC,
};

TVBtRegs bt_ntsc_huge_b = { /* over comp 07.15 00.004 % */
  hsynoffset   : 0,	/* NV: -60 */
  vsynoffset   : 0,
  hsynwidth    : 1,
  vsynwidth    : 2,
  h_clko       : 2080,
  h_active     : 800,
  hsync_width  : 154,
  hburst_begin : 174,
  hburst_end   : 132,
  h_blanko     : 383, /* calc: 383, tool: 400 */
  v_blanko     : 18,
  v_activeo    : 244,
  h_fract      : 0,
  h_clki       : 840,
  h_blanki     : 26,
  v_linesi     : 650,
  v_blanki     : 42,
  v_activei    : 600,
  v_scale      : 6046,
  pll_fract    : 35747,
  pll_int      : 14,
  sync_amp     : 229,
  bst_amp      : 117,
  mcr          : 120,
  mcb          : 67,
  my           : 133,
  msc          : 469762048,
  flags1       : BT_FLAG1_NTSC,
};

/* -------- 720x480 -------- */

TVBtRegs bt_ntsc_normal_c = { /* Overscan 08.762 x 18.107 */
  hsynoffset   : 0,	/* NV: -4 */
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 2048,
  h_active     : 720,
  hsync_width  : 152,
  hburst_begin : 170,
  hburst_end   : 128,
  h_blanko     : 445,
  v_blanko     : 34,
  v_activeo    : 212,
  h_fract      : 0,
  h_clki       : 896,
  h_blanki     : 154,
  v_linesi     : 600,
  v_blanki     : 75,
  v_activei    : 480,
  v_scale      : 5266,
  pll_fract    : 21081,
  pll_int      : 14,
  sync_amp     : 229,
  bst_amp      : 117,
  mcr          : 120,
  mcb          : 68,
  my           : 133,
  msc          : 477102080,
  flags1       : BT_FLAG1_NTSC,
};

/* -------- 720x480 DVD -------- */

TVBtRegs bt_ntsc_dvd_a = { /* Overscan 01.245 x 01.235 */
  hsynoffset   : 0,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 1760,
  h_active     : 720,
  hsync_width  : 130,
  hburst_begin : 146,
  hburst_end   : 92,
  h_blanko     : 283,
  v_blanko     : 19,
  v_activeo    : 242,
  h_fract      : 0,
  h_clki       : 880,
  h_blanki     : 140,
  v_linesi     : 525,
  v_blanki     : 36,
  v_activei    : 480,
  v_scale      : 4096,
  pll_fract    : 20165,
  pll_int      : 12,
  sync_amp     : 229,
  bst_amp      : 118,
  mcr          : 121,
  mcb          : 68,
  my           : 133,
  msc          : 555173329,
  flags1       : BT_FLAG1_NTSC,
};

TVBtRegs bt_ntsc_dvd_b = { /* Beos NTSC720  hoc=02.135  voc=01.235 */
  hsynoffset   : 4,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 1776,
  h_active     : 720,
  hsync_width  : 131,
  hburst_begin : 152,
  hburst_end   : 94,
  h_blanko     : 296,
  v_blanko     : 24,
  v_activeo    : 242,
  h_fract      : 0,
  h_clki       : 888,
  h_blanki     : 144,
  v_linesi     : 525,
  v_blanki     : 26,
  v_activei    : 480,
  v_scale      : 4096,
  pll_fract    : 28312,
  pll_int      : 12,
  sync_amp     : 229,
  bst_amp      : 117,
  mcr          : 120,
  mcb          : 68,
  my           : 133,
  msc          : 549622076,
  flags1       : BT_FLAG1_NTSC,
};

/* -------- BT -------- PAL -------- */

/* -------- 640x480 -------- */

TVBtRegs bt_pal_def_a = {
  hsynoffset   : 0,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 1888,
  h_active     : 640,
  hsync_width  : 138,
  hburst_begin : 166,
  hburst_end   : 104,
  h_blanko     : 449,
  v_blanko     : 46,
  v_activeo    : 242,
  h_fract      : 0,
  h_clki       : 944,
  h_blanki     : 266,
  v_linesi     : 625,
  v_blanki     : 90,
  v_activei    : 480,
  v_scale      : 4096,
  pll_fract    : 7282,
  pll_int      : 13,
  sync_amp     : 240,
  bst_amp      : 88,
  mcr          : 129,
  mcb          : 73,
  my           : 140,
  msc          : 645499916,
  flags1       : BT_FLAG1_PAL_BDGHI,
};

TVBtRegs bt_pal_small_a = { /* over comp 16.56 16.67 % (std 1) */
  hsynoffset   : 0,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 1888,
  h_active     : 640,
  hsync_width  : 138,
  hburst_begin : 166,
  hburst_end   : 104,
  h_blanko     : 449,
  v_blanko     : 46,
  v_activeo    : 242,
  h_fract      : 0,
  h_clki       : 944,
  h_blanki     : 266,
  v_linesi     : 625,
  v_blanki     : 90,
  v_activei    : 480,
  v_scale      : 4096,
  pll_fract    : 7282,
  pll_int      : 13,
  sync_amp     : 240,
  bst_amp      : 88,
  mcr          : 129,
  mcb          : 73,
  my           : 140,
  msc          : 645499916,
  flags1       : BT_FLAG1_PAL_BDGHI,
};

TVBtRegs bt_pal_normal_a = { /* over comp 12.87 07.64 % */
  hsynoffset   : 0,  /* ? 208 */
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 1808,
  h_active     : 640,
  hsync_width  : 132,
  hburst_begin : 158,
  hburst_end   : 94,
  h_blanko     : 403, /* 416 */
  v_blanko     : 33,
  v_activeo    : 268,
  h_fract      : 0,
  h_clki       : 1000,
  h_blanki     : 315,
  v_linesi     : 565,
  v_blanki     : 57,
  v_activei    : 480,
  v_scale      : 3310,
  pll_fract    : 36409,
  pll_int      : 12,
  sync_amp     : 240,
  bst_amp      : 88,
  mcr          : 130,
  mcb          : 73,
  my           : 140,
  msc          : 674061860,
  flags1       : BT_FLAG1_PAL_BDGHI,
};

TVBtRegs bt_pal_huge_a = { /* over comp 06.22 00.69 % */
  hsynoffset   : 0, /* ? 208, */
  vsynoffset   : 0,
  hsynwidth    : 1,
  vsynwidth    : 1,
  h_clko       : 1680,
  h_active     : 640,
  hsync_width  : 124,
  hburst_begin : 146, /* 148, */
  hburst_end   : 78,
  h_blanko     : 331, /* 344, */
  v_blanko     : 23,
  v_activeo    : 288,
  h_fract      : 0,
  h_clki       : 1000,
  h_blanki     : 299, /* 312, */
  v_linesi     : 525,
  v_blanki     : 36,
  v_activei    : 480,
  v_scale      : 2785,
  pll_fract    : 43691,
  pll_int      : 11,
  sync_amp     : 240,
  bst_amp      : 89,
  mcr          : 131,
  mcb          : 73,
  my           : 140,
  msc          : 725418954,
  flags1       : BT_FLAG1_PAL_BDGHI,
};

/* -------- 800x600 -------- */

TVBtRegs bt_pal_def_b = {
  hsynoffset   : 0,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 2304,
  h_active     : 800,
  hsync_width  : 170,
  hburst_begin : 202,
  hburst_end   : 154,
  h_blanko     : 525,
  v_blanko     : 41,
  v_activeo    : 252,
  h_fract      : 0,
  h_clki       : 960,
  h_blanki     : 140,
  v_linesi     : 750,
  v_blanki     : 95,
  v_activei    : 600,
  v_scale      : 5734,
  pll_fract    : 0,
  pll_int      : 16,
  sync_amp     : 240,
  bst_amp      : 87,
  mcr          : 128,
  mcb          : 72,
  my           : 140,
  msc          : 528951320,
  flags1       : BT_FLAG1_PAL_BDGHI,
};

TVBtRegs bt_pal_small_b = {  /* over comp 14.53 13.19 % (std 3) */
  hsynoffset   : 0,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 2304,
  h_active     : 800,
  hsync_width  : 170,
  hburst_begin : 202,
  hburst_end   : 154,
  h_blanko     : 525, /* 530 */
  v_blanko     : 41,
  v_activeo    : 252,
  h_fract      : 0,
  h_clki       : 960,
  h_blanki     : 135, /* tuned */ /* 140, 141 */
  v_linesi     : 750,
  v_blanki     : 95,
  v_activei    : 600,
  v_scale      : 5734, 
  pll_fract    : 0,
  pll_int      : 16,
  sync_amp     : 240,
  bst_amp      : 87,
  mcr          : 128,
  mcb          : 72,
  my           : 140,
  msc          : 528951320,  
  flags1       : BT_FLAG1_PAL_BDGHI,
};

TVBtRegs bt_pal_normal_b = { /* over comp 10.81 05.56 % */
  hsynoffset   : 0,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 2208, 
  h_active     : 800,
  hsync_width  : 162,  
  hburst_begin : 194,  
  hburst_end   : 142, 
  h_blanko     : 471,  /* 478, */
  v_blanko     : 30,   
  v_activeo    : 274,  
  h_fract      : 0,
  h_clki       : 1000, 
  h_blanki     : 175,  /* 100, 140 */
  v_linesi     : 690,  
  v_blanki     : 64,   
  v_activei    : 600,
  v_scale      : 4948, 
  pll_fract    : 21845, 
  pll_int      : 15,    
  sync_amp     : 240,
  bst_amp      : 87,
  mcr          : 128,
  mcb          : 72,
  my           : 140,
  msc          : 551949204, 
  flags1       : BT_FLAG1_PAL_BDGHI,
};

TVBtRegs bt_pal_large_b = { /* over comp 07.461 x 02.083 */
  hsynoffset   : 0,	/* NV: -8 */
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 2128,
  h_active     : 800,
  hsync_width  : 156,
  hburst_begin : 186,
  hburst_end   : 134,
  h_blanko     : 425,
  v_blanko     : 25,
  v_activeo    : 284,
  h_fract      : 0,
  h_clki       : 1000,
  h_blanki     : 190,
  v_linesi     : 665,
  v_blanki     : 51,
  v_activei    : 600,
  v_scale      : 4620,
  pll_fract    : 50972,
  pll_int      : 14,
  sync_amp     : 240,
  bst_amp      : 87,
  mcr          : 128,
  mcb          : 72,
  my           : 140,
  msc          : 572699174,
  flags1       : BT_FLAG1_PAL_BDGHI,
};

TVBtRegs bt_pal_huge_b = { /* over comp 00.039 00.00 % (no calc'd mode) */
  hsynoffset   : 0,	/* NV: 14 */
  vsynoffset   : 0,
  hsynwidth    : 1,
  vsynwidth    : 1,
  h_clko       : 1970,  
  h_active     : 800,	
  hsync_width  : 144,   
  hburst_begin : 172,   
  hburst_end   : 114,   
  h_blanko     : 303,   
  v_blanko     : 22,    
  v_activeo    : 309,   
  h_fract      : 0,  	
  h_clki       : 945,   
  h_blanki     : 101,   
  v_linesi     : 652,   
  v_blanki     : 133,   
  v_activei    : 600,	
  v_scale      : 4438,  
  pll_fract    : 44478, 
  pll_int      : 13,    
  sync_amp     : 240,	
  bst_amp      : 88,    
  mcr          : 128,
  mcb          : 72,
  my           : 140,
  msc          : 618715949, 
  flags1       : BT_FLAG1_PAL_BDGHI,
};

/* -------- 768x576 -------- */

/* 768 x 576 (720x576 scaled by xine) */

TVBtRegs bt_pal_small_c = { /* hoc = 13.122 voc = 07.986 */
  hsynoffset   : 0,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 2176,
  h_active     : 768,
  hsync_width  : 160,
  hburst_begin : 190,
  hburst_end   : 138,
  h_blanko     : 485,
  v_blanko     : 34,
  v_activeo    : 267,
  h_fract      : 0,
  h_clki       : 1000,
  h_blanki     : 203,
  v_linesi     : 680,
  v_blanki     : 71,
  v_activei    : 576,
  v_scale      : 4817,
  pll_fract    : 7282,
  pll_int      : 15,
  sync_amp     : 240,
  bst_amp      : 87,
  mcr          : 128,
  mcb          : 72,
  my           : 140,
  msc          : 560066112,
  flags1       : BT_FLAG1_PAL_BDGHI,
};

TVBtRegs bt_pal_normal_c = { /* hoc =  09.806 voc = 04.514 */
  hsynoffset   : 0,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 2096,
  h_active     : 768,
  hsync_width  : 154,
  hburst_begin : 184,
  hburst_end   : 130,
  h_blanko     : 439,
  v_blanko     : 29,
  v_activeo    : 277,
  h_fract      : 0,
  h_clki       : 1000,
  h_blanki     : 203,
  v_linesi     : 655,
  v_blanki     : 58,
  v_activei    : 576,
  v_scale      : 4489,
  pll_fract    : 36409,
  pll_int      : 14,
  sync_amp     : 240,
  bst_amp      : 87,
  mcr          : 128,
  mcb          : 72,
  my           : 140,
  msc          : 581442673,
  flags1       : BT_FLAG1_PAL_BDGHI,
};

TVBtRegs bt_pal_huge_c = { /* hoc =  00.000 voc = 00.000 */
  hsynoffset   : 2,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 1890,
  h_active     : 768,
  hsync_width  : 138,
  hburst_begin : 166,
  hburst_end   : 104,
  h_blanko     : 323,
  v_blanko     : 22,
  v_activeo    : 290,
  h_fract      : 0,
  h_clki       : 945,
  h_blanki     : 154,
  v_linesi     : 625,
  v_blanki     : 42,
  v_activei    : 576,
  v_scale      : 4096,
  pll_fract    : 8192,
  pll_int      : 13,
  sync_amp     : 240,
  bst_amp      : 88,
  mcr          : 129,
  mcb          : 73,
  my           : 140,
  msc          : 644816848,
  flags1       : BT_FLAG1_PAL_BDGHI,
};

TVBtRegs bt_pal_normal1_c = { /* hoc = 11.163 voc = 05.903 */
  hsynoffset   : 0,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 2128,
  h_active     : 768,
  hsync_width  : 156,
  hburst_begin : 186,
  hburst_end   : 134,
  h_blanko     : 457,
  v_blanko     : 31,
  v_activeo    : 273,
  h_fract      : 0,
  h_clki       : 1000,
  h_blanki     : 203,
  v_linesi     : 665,
  v_blanki     : 63,
  v_activei    : 576,
  v_scale      : 4620,
  pll_fract    : 50972,
  pll_int      : 14,
  sync_amp     : 240,
  bst_amp      : 87,
  mcr          : 128,
  mcb          : 72,
  my           : 140,
  msc          : 572699174,
  flags1       : BT_FLAG1_PAL_BDGHI,
};

/* -------- 800x576 -------- */

/* 800 x 576 (for 1024x576 widescreen scaled by xine) */

TVBtRegs bt_pal_normal_d = { /* hoc = 08.83 voc = 07.29 */
  hsynoffset   : 0,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 2160,
  h_active     : 800,
  hsync_width  : 158,
  hburst_begin : 188,
  hburst_end   : 136,
  h_blanko     : 443,
  v_blanko     : 33,
  v_activeo    : 269,
  h_fract      : 0,
  h_clki       : 1000,
  h_blanki     : 175,
  v_linesi     : 675,
  v_blanki     : 69,
  v_activei    : 576,
  v_scale      : 4751,
  pll_fract    : 0,
  pll_int      : 15,
  sync_amp     : 240,
  bst_amp      : 87,
  mcr          : 128,
  mcb          : 72,
  my           : 140,
  msc          : 564214742,
  flags1       : BT_FLAG1_PAL_BDGHI,
};

/* -------- 720x576 DVD -------- */

TVBtRegs bt_pal_dvd_a = { /* NV/Beos DVD, hoc = 00.21, voc = 00.00 */
  hsynoffset   : 0,	/* NV: 4 */
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 1776,
  h_active     : 720,
  hsync_width  : 130,
  hburst_begin : 156,
  hburst_end   : 90,
  h_blanko     : 305,
  v_blanko     : 22,
  v_activeo    : 290,
  h_fract      : 0,
  h_clki       : 888,
  h_blanki     : 147,
  v_linesi     : 625,
  v_blanki     : 42,
  v_activei    : 576,
  v_scale      : 4096,
  pll_fract    : 21845,
  pll_int      : 12,
  sync_amp     : 240,
  bst_amp      : 89,
  mcr          : 130,
  mcb          : 73,
  my           : 140,
  msc          : 686207118,
  flags1       : BT_FLAG1_PAL_BDGHI,
};

/* -------- BT -------- PAL-60 -------- */

/* -------- 640x480 -------- */

TVBtRegs bt_pal60_small_a = { /* 16.758 x 14.815 */
  hsynoffset   : 0,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 1856,
  h_active     : 640,
  hsync_width  : 138,
  hburst_begin : 164,
  hburst_end   : 102,
  h_blanko     : 417,
  v_blanko     : 36,
  v_activeo    : 209,
  h_fract      : 0,
  h_clki       : 800,
  h_blanki     : 140,
  v_linesi     : 609,
  v_blanki     : 81,
  v_activei    : 480,
  v_scale      : 5407,
  pll_fract    : 64161,
  pll_int      : 12,
  sync_amp     : 240,
  bst_amp      : 88,
  mcr          : 129,
  mcb          : 73,
  my           : 140,
  msc          : 652069372,
  flags1       : BT_FLAG1_PAL_60,
};

TVBtRegs bt_pal60_normal_a = { /* 10.592 x 08.230 */
  hsynoffset   : 0,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 1728,
  h_active     : 640,
  hsync_width  : 128,
  hburst_begin : 152,
  hburst_end   : 86,
  h_blanko     : 345,
  v_blanko     : 28,
  v_activeo    : 224,
  h_fract      : 0,
  h_clki       : 800,
  h_blanki     : 145, /* tuned */ /* 140, */
  v_linesi     : 567,
  v_blanki     : 58,
  v_activei    : 480,
  v_scale      : 4751,
  pll_fract    : 5500,
  pll_int      : 12,
  sync_amp     : 240,
  bst_amp      : 89,
  mcr          : 130,
  mcb          : 73,
  my           : 140,
  msc          : 700370807,
  flags1       : BT_FLAG1_PAL_60,
};

TVBtRegs bt_pal60_huge_a = { /* 01.469 x 01.235 */
  hsynoffset   : 0,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 1568,
  h_active     : 640,
  hsync_width  : 116,
  hburst_begin : 138,
  hburst_end   : 66,
  h_blanko     : 255,
  v_blanko     : 19,
  v_activeo    : 242,
  h_fract      : 0,
  h_clki       : 784,
  h_blanki     : 126,
  v_linesi     : 525,
  v_blanki     : 36,
  v_activei    : 480,
  v_scale      : 4096,
  pll_fract    : 63245,
  pll_int      : 10,
  sync_amp     : 240,
  bst_amp      : 90,
  mcr          : 131,
  mcb          : 74,
  my           : 140,
  msc          : 771837216,
  flags1       : BT_FLAG1_PAL_60,
};

/* -------- 800x600 -------- */

TVBtRegs bt_pal60_small_b = { /* 21.623 x 11.523 */
  hsynoffset   : 0,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 2464,
  h_active     : 800,
  hsync_width  : 182,
  hburst_begin : 218,
  hburst_end   : 176,
  h_blanko     : 597,
  v_blanko     : 32,
  v_activeo    : 216,
  h_fract      : 0,
  h_clki       : 880,
  h_blanki     : 66,
  v_linesi     : 735,
  v_blanki     : 86,
  v_activei    : 600,
  v_scale      : 7373,
  pll_fract    : 15124,
  pll_int      : 17,
  sync_amp     : 240,
  bst_amp      : 87,
  mcr          : 127,
  mcb          : 72,
  my           : 140,
  msc          : 491169137,
  flags1       : BT_FLAG1_PAL_60,
};

TVBtRegs bt_pal60_normal_b = { /* 11.897 x 05.350 */
  hsynoffset   : 0,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 2192,
  h_active     : 800,
  hsync_width  : 162,
  hburst_begin : 194,
  hburst_end   : 142,
  h_blanko     : 445,
  v_blanko     : 24,
  v_activeo    : 232,
  h_fract      : 0,
  h_clki       : 840,
  h_blanki     : 26,
  v_linesi     : 685,
  v_blanki     : 60,
  v_activei    : 600,
  v_scale      : 6593,
  pll_fract    : 21540,
  pll_int      : 15,
  sync_amp     : 240,
  bst_amp      : 87,
  mcr          : 128,
  mcb          : 72,
  my           : 140,
  msc          : 552117133,
  flags1       : BT_FLAG1_PAL_60,
};

TVBtRegs bt_pal60_huge_b = { /* 07.862 x 00.823 */
  hsynoffset   : 0,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 2096,
  h_active     : 800,
  hsync_width  : 156,
  hburst_begin : 184,
  hburst_end   : 130,
  h_blanko     : 391,
  v_blanko     : 19,
  v_activeo    : 242,
  h_fract      : 0,
  h_clki       : 840,
  h_blanki     : 26,
  v_linesi     : 655,
  v_blanki     : 44,
  v_activei    : 600,
  v_scale      : 6124,
  pll_fract    : 43080,
  pll_int      : 14,
  sync_amp     : 240,
  bst_amp      : 87,
  mcr          : 128,
  mcb          : 72,
  my           : 140,
  msc          : 577404940,
  flags1       : BT_FLAG1_PAL_60,
};

/* -------- 720x480 DVD -------- */

TVBtRegs bt_pal60_dvd_a = { /* 01.245 x 01.235 */
  hsynoffset   : 0,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 1760,
  h_active     : 720,
  hsync_width  : 130,
  hburst_begin : 156,
  hburst_end   : 90,
  h_blanko     : 283,
  v_blanko     : 19,
  v_activeo    : 242,
  h_fract      : 0,
  h_clki       : 880,
  h_blanki     : 140,
  v_linesi     : 525,
  v_blanki     : 36,
  v_activei    : 480,
  v_scale      : 4096,
  pll_fract    : 20165,
  pll_int      : 12,
  sync_amp     : 240,
  bst_amp      : 89,
  mcr          : 130,
  mcb          : 73,
  my           : 140,
  msc          : 687636792,
  flags1       : BT_FLAG1_PAL_60,
};

/* -------- BT async -------- PAL -------- */

/* -------- 640x480 -------- */

TVBtRegs bt_async_pal_small_a = { /* over comp 16.56 16.67 % (std 1) */
  hsynoffset   : 0,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 1888,
  h_active     : 640,
  hsync_width  : 138,
  hburst_begin : 166,
  hburst_end   : 104,
  h_blanko     : 449,
  v_blanko     : 46,
  v_activeo    : 242,
  h_fract      : 0,
  h_clki       : 944,
  h_blanki     : 266,
  v_linesi     : 625,
  v_blanki     : 90,
  v_activei    : 480,
  v_scale      : 4096,
  pll_fract    : 7282,
  pll_int      : 13,
  sync_amp     : 240,
  bst_amp      : 88,
  mcr          : 129,
  mcb          : 73,
  my           : 140,
  msc          : 645499916,
  flags1       : BT_FLAG1_PAL_BDGHI | BT_FLAG1_EN_ASYNC,
};

/* -------- -------- */

#define PI 3.14159265358979323846

static int clamp_attenuate[8] = {-100, -90, -75, -50, -30, -20, -10, 0};
static int clamp_flicker[5]   = {0, 25, 50, 75, 100};
static int clamp_luma_bandwidth[4] = {0, 30, 60, 100};
static int clamp_chroma_bandwidth[3] = {0, 50, 100};

/* Convert NTSC to PAL-M. Values copied from calc_bt.c. FIXME refactor? */

/* TODO: Doesn't respect DoublePll / Mode2x */

void data_ntsc_palm_bt (TVBtRegs *r)
{
  double Fxtal = 13500000; 
  double Fclk;
  double dHCLKO;
  double dSINX;
  double dV100;
  register double f;

  Fclk = Fxtal * (r->pll_int + r->pll_fract / 65536.0) / 6.0;
  dHCLKO = r->h_clko;
  dV100 = 716.1;
  f = PI * 3575611.88 / Fclk;
  dSINX = sin (f) / f;

  r->hburst_begin = (int) ((0.09125874125874 * dHCLKO) + 0.5);
  r->hburst_end = 2 * (int) ((0.06545454545455 * dHCLKO) + 0.5) - 128;
  r->sync_amp = 240;
  r->bst_amp = (int) ((84.84 / dSINX) + 0.5);
  r->msc = (long) (0x80000000 * (454.5 / dHCLKO) + 0.5);
  r->mcr = (int) ((dV100 / (5.746659386525 * dSINX)) + 0.5);
  r->mcb = (int) ((dV100 / (10.22275919266 * dSINX)) + 0.5);
  r->my  = (int) ((dV100 / 5.099501953125) + 0.5);
  r->flags1 &= ~BT_FLAG1_SYSTEM;
  r->flags1 |= BT_FLAG1_PAL_M;
  r->macro = 4; /* FIXME test this */
}

void data_default_bt (TVSettings *s)
{
  s->tv_hoffset = s->mon_hoffset = 0;
  s->tv_voffset = s->mon_voffset = 0;
  s->brightness_sig = 0;
  s->contrast_sig = 0;
  s->saturation_sig = 0;
  s->contrast = 0;
  s->saturation = 0;
  s->phase = 0;
  s->hue = 0;
  s->flicker = 75;
  s->flicker_adapt = 0;
  s->sharpness = 50;
  s->cross_color = 100;
  s->luma_bandwidth = 30;
  s->chroma_bandwidth = 100;
  /* FIXME s->flags; */
}

void data_clamp_bt (TVSettings *s, TVRegs *r)
{
  if (s->tv_hoffset <= -100) s->tv_hoffset = -100;
  if (s->tv_hoffset >=  100) s->tv_hoffset =  100;
  if (s->tv_voffset <= -100) s->tv_voffset = -100;
  if (s->tv_voffset >=  100) s->tv_voffset =  100;
  if (s->mon_hoffset <= -100) s->mon_hoffset = -100;
  if (s->mon_hoffset >=  100) s->mon_hoffset =  100;
  if (s->mon_voffset <= -100) s->mon_voffset = -100;
  if (s->mon_voffset >=  100) s->mon_voffset =  100;
  s->tv_voffset = 0; 
  /* FIXME. Clamp to 0 for now, until some 'VTotal magic' */
  s->brightness_sig = 0;
  if (s->contrast_sig < -50) s->contrast_sig = -50;
  if (s->contrast_sig >  50) s->contrast_sig = 50;
  if (s->saturation_sig < -50) s->saturation_sig = -50;
  if (s->saturation_sig >  50) s->saturation_sig = 50;
  if (s->phase < -180) s->phase = -180;
  if (s->phase >  180) s->phase = 180;
  s->hue = 0;
  s->contrast   = data_clamp (s->contrast, 8, clamp_attenuate); 
  s->saturation = data_clamp (s->saturation, 8, clamp_attenuate); 
  s->sharpness = 50;
  s->flicker = data_clamp (s->flicker, 5, clamp_flicker); 
  s->flicker_adapt = 0;
  s->cross_color = 100;
  s->chroma_bandwidth = data_clamp (s->chroma_bandwidth, 3, 
				    clamp_chroma_bandwidth); 
  s->luma_bandwidth = data_clamp (s->luma_bandwidth, 4, 
				  clamp_luma_bandwidth); 
  /* FIXME: Operate on mode flags */
}

/* The mux setup is different for NV and Voodoo cards, so 
   we have extra routines for that. */

void data_mux_nv_bt (TVBtRegs *r)
{
  r->out_muxa = 0;
  r->out_muxb = 2;
  r->out_muxc = 1;
  r->out_muxd = 0; /* only available for the conexant chip */
}

void data_mux_tdfx_bt (TVBtRegs *r)
{
  r->out_muxa = 0;
  r->out_muxb = 1;
  r->out_muxc = 2;
  r->out_muxd = 0; /* only available for the conexant chip */
}

/* The init/setup routines without mux setup */

void data_init_nomux_bt (TVSystem system, TVBtRegs *r)
{
  if (!r) return;
  r->ylpf = 3;
  r->clpf = 0;
  r->flags2 = BT_FLAG2_DIS_GM;
  r->flags3 = 0;
  r->f_selc = 2;
  r->f_sely = 2;
  r->phase_off = 0; /* 1.406 degrees per bit incr */
  switch (system)
  {
    case TV_SYSTEM_NTSC_J: 
      r->flags1 &= ~BT_FLAG1_SYSTEM;
      r->flags1 |= BT_FLAG1_NTSC_J;
      break;
    case TV_SYSTEM_PAL_M: 
      r->flags1 &= ~BT_FLAG1_SYSTEM;
      r->flags1 |= BT_FLAG1_PAL_M;
      break;
    default:
      break;
  }
  switch (system)
  {
    case TV_SYSTEM_NONE:
    case TV_SYSTEM_NTSC:
    case TV_SYSTEM_NTSC_J: 
      r->macro = 0; break;
    case TV_SYSTEM_SECAM: /* makes most sense */
    case TV_SYSTEM_PAL:
    case TV_SYSTEM_PAL_X:
    case TV_SYSTEM_PAL_60: 
    case TV_SYSTEM_PAL_N:  
    case TV_SYSTEM_PAL_NC: 
    case TV_SYSTEM_PAL_M: /* FIXME test this */ 
    case TV_SYSTEM_PAL_M60:  
      r->macro = 4; break;
  }
}

void data_setup_nomux_bt (TVSettings *s, TVRegs *r)
{
  float f;

  r->enc.bt.hsynoffset += s->tv_hoffset * 2;
  f = (1.0 + 0.01 * s->contrast_sig) * r->enc.bt.my;
  if (f < 0.0) f = 0.0;
  if (f > 255.0) f = 255.0;
  r->enc.bt.my = (int) f;
  f = (1.0 + 0.01 * s->saturation_sig) * r->enc.bt.mcr;
  if (f < 0.0) f = 0.0;
  if (f > 255.0) f = 255.0;
  r->enc.bt.mcr = (int) f;
  f = (1.0 + 0.01 * s->saturation_sig) * r->enc.bt.mcb;
  if (f < 0.0) f = 0.0;
  if (f > 255.0) f = 255.0;
  r->enc.bt.mcb = (int) f;
  r->enc.bt.yattenuate = 7 - data_pick (s->contrast, 8, clamp_attenuate); 
  r->enc.bt.cattenuate = 7 - data_pick (s->saturation, 8, clamp_attenuate); 
  f = s->phase * 256.0 / 360.0;
  if (f < 0.0) f += 256.0;
  if (f < 0.0) f = 0.0;
  if (f > 255.0) f = 255.0;
  r->enc.bt.phase_off = (int) f;
  r->enc.bt.flags2 &= ~BT_FLAG2_DIS_FFILT;
  switch (data_pick (s->flicker, 5, clamp_flicker)) {
    case 0: r->enc.bt.flags2 |= BT_FLAG2_DIS_FFILT; break;
    case 1: r->enc.bt.f_sely = 1; r->enc.bt.f_selc = 1; break;
    case 2: r->enc.bt.f_sely = 2; r->enc.bt.f_selc = 1; break;
    case 3: r->enc.bt.f_sely = 3; r->enc.bt.f_selc = 2; break;
    case 4: r->enc.bt.f_sely = 0; r->enc.bt.f_selc = 4; break;
  }
  /* There is a special case here: For v_scale values less then 4096,
     f_sely = 3 gives bad results, so set it to 2 in this case */
  if (r->enc.bt.v_scale < 4096 && r->enc.bt.f_sely == 3) {
    r->enc.bt.f_sely = 2;
  }
  switch (data_pick (s->luma_bandwidth, 4, clamp_luma_bandwidth)) {
    case 0: r->enc.bt.ylpf = 2; break;
    case 1: r->enc.bt.ylpf = 3; break;
    case 2: r->enc.bt.ylpf = 1; break;
    case 3: r->enc.bt.ylpf = 0; break;
  }
  switch (data_pick (s->chroma_bandwidth, 3, clamp_chroma_bandwidth)) {
    case 0: r->enc.bt.clpf = 2; break;
    case 1: r->enc.bt.clpf = 3; break;
    case 2: r->enc.bt.clpf = 0; break;
  }
  r->enc.bt.macro = (r->enc.bt.macro & ~0x3) | 
                ((s->flags & TV_DESC_MACROVISION) ? 1 : 0);
  if (s->flags & TV_DESC_CARRIER_LOCK) {
    r->enc.bt.flags1 |= BT_FLAG1_DIS_SCRESET;
  } else {
    r->enc.bt.flags1 &= ~BT_FLAG1_DIS_SCRESET;
  }
  if (s->flags & TV_DESC_NONINTERLACED) {
    r->enc.bt.flags1 |= BT_FLAG1_NI_OUT;
  } else {
    r->enc.bt.flags1 &= ~BT_FLAG1_NI_OUT;
  }
  if (s->flags & TV_DESC_MONOCHROME) {
    r->enc.bt.flags2 |= BT_FLAG2_DIS_CHROMA;
  } else {
    r->enc.bt.flags2 &= ~BT_FLAG2_DIS_CHROMA;
  }
  r->enc.bt.flags3 &= ~BT_FLAG3_DAC;
  switch (s->connector) 
  {
    case CONNECT_COMPOSITE: 
      r->enc.bt.flags3 |= BT_FLAG3_COMPOSITE; 
      break;
    case CONNECT_SVIDEO: 
      r->enc.bt.flags3 |= BT_FLAG3_SVIDEO;
      break;
    case CONNECT_CONVERT:
      r->enc.bt.flags3 |= BT_FLAG3_CONVERT;
      r->enc.bt.out_muxa = 0;
      r->enc.bt.out_muxb = 0;
      r->enc.bt.out_muxc = 0;
      break;
    case CONNECT_BOTH:
    default:
      r->enc.bt.flags3 |= BT_FLAG3_BOTH;
      break;
  }
  /* FIXME dualview;     */
}

/* To be on the safe side, we have init/setup routines defaulting to
   the NV mux setup */

void data_init_bt (TVSystem system, TVBtRegs *r)
{
  data_mux_nv_bt (r);
  data_init_nomux_bt (system, r);
}

void data_setup_bt (TVSettings *s, TVRegs *r)
{
  data_mux_nv_bt (&r->enc.bt);
  data_setup_nomux_bt (s, r);
}

/*
Setup:

Coring = clamp value below threshold

X/Y Pos horiz instead of vertical (smaller field)

Macrovision mode CH regs/BT Flags
Interlace, doublescan -> CRT Regs

Async modes:
  en_async 
  set dis_scrst ???

*/

/* 

PAL-60:

640x480  16x16/13x13  12x7/10x8  6x0/2x1
  16.758 x 14.815
  10.592 x 08.230
  01.469 x 01.235

800x600  14x13/21x11  10x5/11x5  0x0/7x0
  16.180 x 13.992

720x480  0x0

*/
